/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-param-specs
 * @title:	Parameter Specification
 * @short_description:	Custom plugin parameter type
 * @include:	gnome-scan.h
 *
 * Instead of using a all-in-one option object (like in Gnome Scan
 * 0.4), Gnome Scan now use regular #GParamSpecs. Gnome Scan
 * #GParamSpecs (referred as #GSParamSpecs) have been designed in order
 * to build generic widget for plugins parameters (like Gegl and other
 * projects does). A common attribute added to each #GSParamSpec is
 * the @group quark. The @group quark is computed from the translated
 * group title.
 *
 * Also, #GSParamSpecs are much more high-level than it's parent
 * #GParamSpec in order to build complex param widgets (range, area,
 * paper-size, …).
 *
 * Current implementation makes a heavy use of
 * g_param_spec_set_qdata(), because #GParamSpec is a shallow
 * type. Future implementation might break thing for either creating
 * its own #GSParamSpec fundamental type or simply using a common
 * struct right after the @parent_instance field.
 **/

#include <glib/gi18n.h>
#include "gnome-scan-param-specs.h"
#include "gnome-scan-range-widget.h"
#include "gnome-scan-enum-widget.h"
#include "gnome-scan-number-widget.h"
#include "gnome-scan-string-widget.h"
#include "gnome-scan-boolean-widget.h"
#include "gnome-scan-paper-size-widget.h"
#include "gnome-scan-page-orientation-widget.h"
#include "gnome-scan-preview-plugin.h"

/* Quarks */
/* extended properties */
GS_DEFINE_QUARK (gs_group,			"group");
GS_DEFINE_QUARK	(gs_domain,		    "domain");
GS_DEFINE_QUARK (gs_unit,			"unit");
GS_DEFINE_QUARK (gs_index,			"index");
GS_DEFINE_QUARK (gs_widget_type,	"widget-type");
GS_DEFINE_QUARK (gs_formats,		"formats");

/* groups */
GS_DEFINE_QUARK (gs_scanner_front,	N_("Scan options"));
GS_DEFINE_QUARK (gs_format,	N_("Format"));
GS_DEFINE_QUARK (gs_sink_front,	N_("Output options"));
GS_DEFINE_QUARK (gs_preview,		N_("Preview"));
GS_DEFINE_QUARK (gs_hidden,		N_("Hidden"));

/* Common */

/**
 * gnome_scan_format_new:
 * @name:	format name
 * @domain:	translation domain
 * @description:description
 * @mime_types:	list of mime types for this format
 * @extensions:	list of suffixes for file of this format.
 *
 * Returns: a new #GnomeScanFormat
 **/
GnomeScanFormat* gnome_scan_format_new (gchar *name,
					gchar *domain,
					gchar *description,
					gchar **mime_types,
					gchar **extensions)
{
  GnomeScanFormat *format = g_new0 (GnomeScanFormat, 1);
  format->name		= name;
  format->domain		= domain;
  format->description	= description;
  format->mime_types	= mime_types;
  format->extensions	= extensions;
  return format;
}

/**
 * gs_param_spec_get_group:
 * @spec: The #GSParamSpec
 * 
 * Retrieve the @spec group. Since the @group is not store in a
 * struct, but using qdata, just cast @spec to #GParamSpec.
 * 
 * Returns: @spec group's quark
 **/
GQuark
gs_param_spec_get_group (GParamSpec *spec)
{
  return (GQuark) g_param_spec_get_qdata (spec, GS_PARAM_GROUP_QUARK);
}

/**
 * gs_param_spec_get_group_string:
 * @spec: The #GSParamSpec
 * 
 * Retrieve the @spec group string. Since the @group is not store in a
 * struct, but using qdata, just cast @spec to #GParamSpec. The string
 * is not translated.
 *
 * See: gs_param_spec_get_group()
 * Returns: @spec group's quark
 **/
const gchar*
gs_param_spec_get_group_string (GParamSpec *spec)
{
  return g_quark_to_string (gs_param_spec_get_group (spec));
}

/**
 * gs_param_spec_set_group:
 * @spec:	a #GSParamSpec
 * @group:	new @spec group's #GQuark.
 **/
void
gs_param_spec_set_group (GParamSpec *spec, GQuark group)
{

  g_param_spec_set_qdata (spec, GS_PARAM_GROUP_QUARK, (gpointer) group);
}

/**
 * gs_param_spec_set_group_from_string:
 * @spec:	a #GSParamSpec
 * @group:	new @spec group name
 **/
void
gs_param_spec_set_group_from_string (GParamSpec *spec, const gchar *group)
{
  gs_param_spec_set_group (spec, g_quark_from_string (group));
}


/**
 * gs_param_spec_set_domain:
 * @spec:	a #GSParamSpec
 * @domain:	@domain for translating @spec @nick and @blurb
 *
 **/
void
gs_param_spec_set_domain (GParamSpec *spec, const gchar *domain)
{
  g_param_spec_set_qdata (spec, GS_PARAM_DOMAIN_QUARK, (gpointer) domain);
}

/**
 * gs_param_spec_get_domain:
 * @spec:	a #GSParamSpec
 *
 **/
const gchar*
gs_param_spec_get_domain (GParamSpec *spec)
{
  return g_param_spec_get_qdata (spec, GS_PARAM_DOMAIN_QUARK);
}

/**
 * gs_param_spec_set_unit:
 * @spec:	a #GSParamSpec
 * @unit:	#GnomeScanUnit fo @spec
 *
 * Parameter unit is specified for UI, but can also be need for
 * e.g. length of scan ROI where unit translation might occur.
 **/
void
gs_param_spec_set_unit 				(GParamSpec *spec, GnomeScanUnit unit)
{
  g_param_spec_set_qdata (spec, GS_PARAM_UNIT_QUARK, (gpointer) unit);
}

/**
 * gs_param_spec_get_unit:
 * @spec:	a #GSParamSpec
 *
 * Returns: a #GnomeScanUnit
 **/
GnomeScanUnit
gs_param_spec_get_unit				(GParamSpec *spec)
{
  return (GnomeScanUnit) g_param_spec_get_qdata (spec, GS_PARAM_UNIT_QUARK);
}

/**
 * gs_param_spec_set_index:
 * @spec:	a #GSParamSpec
 * @index:	new @spec'index 
 **/
void
gs_param_spec_set_index (GParamSpec *spec, guint index)
{
  g_param_spec_set_qdata (spec, GS_PARAM_INDEX_QUARK, (gpointer) index);
}

/**
 * gs_param_spec_get_index:
 * @spec:	a #GSParamSpec
 *
 * Returns:	an integer.
 **/
guint
gs_param_spec_get_index (GParamSpec *spec)
{
  return (guint) g_param_spec_get_qdata (spec, GS_PARAM_INDEX_QUARK);
}



/**
 * gs_param_spec_set_widget_type:
 * @spec:	a #GSParamSpec
 * @type:	a widget #GType
 *
 * This part is the only bound from backend to frontend. It might be
 * removed later. The idea is to have widget per param, not per type.
 **/
void
gs_param_spec_set_widget_type (GParamSpec *spec,  GType type)
{
  g_param_spec_set_qdata (spec, GS_PARAM_WIDGET_TYPE_QUARK, (gpointer) type);
}

/**
 * gs_param_spec_get_widget_type:
 * @spec:	a #GSParamSpec
 *
 * Returns:	a #GType
 **/
GType
gs_param_spec_get_widget_type (GParamSpec *spec)
{
  return (GType) g_param_spec_get_qdata (spec, GS_PARAM_WIDGET_TYPE_QUARK);
}

/**
 * gs_param_spec_cmp_index:
 * @a:	a #GSParamSpec
 * @b:	another #GSParamSpec
 *
 * Compare two #GSParamSpec by index.
 *
 * Returns:	an integer nagtive if a is previous to b else positive.
 **/
gint
gs_param_spec_cmp_index (GParamSpec*a, GParamSpec *b)
{
  return gs_param_spec_get_index (a) - gs_param_spec_get_index (b);
}

/**
 * gs_param_spec_cmp_name:
 * @a:	a #GSParamSpec
 * @b:	another #GSParamSpec
 *
 * Compare two #GSParamSpec by name.
 *
 * Returns:	a nagtive integer if a is previous to b else a positive integer.
 **/
gint
gs_param_spec_cmp_name (GParamSpec*a, GParamSpec *b)
{
  return g_ascii_strcasecmp (g_param_spec_get_name (a), g_param_spec_get_name (b));
}

/**
 * gs_param_values_cmp:
 * @pspec:	a #GSParamSpec
 * @a:		a #GSParamSpec
 * @b:		another #GSParamSpec
 *
 * Compare two value of type determined by @pspec.
 *
 * Returns:	a negative integer if a < b, else a positive integer.
 **/
gint
gs_param_values_cmp (GParamSpec *pspec, GValue *a, GValue *b)
{
  GParamSpecClass *klass = G_PARAM_SPEC_GET_CLASS (pspec);
	
  return klass->values_cmp (pspec, a, b);
}


/**
 * gs_param_spec_boolean:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @default_value:	default value
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for boolean value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_boolean (const gchar *name,
		       const gchar *nick,
		       const gchar *blurb,
		       GQuark group,
		       gboolean default_value,
		       GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_boolean (name, nick, blurb,
					   default_value, flags);
  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_BOOLEAN_WIDGET);
  return spec;
}

/**
 * gs_param_spec_int:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @minimum:		minimum value
 * @maximum:		maximum value
 * @default_value:	default value
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for integer value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_int (const gchar *name,
		   const gchar *nick,
		   const gchar *blurb,
		   GQuark group,
		   gint minimum,
		   gint maximum,
		   gint default_value,
		   GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_int (name, nick, blurb,
				       minimum, maximum, default_value,
				       flags);
  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_NUMBER_WIDGET);
  return spec;
}

/**
 * gs_param_spec_double:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @minimum:		minimum value
 * @maximum:		maximum value
 * @default_value:	default value
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for double value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_double (const gchar *name,
		      const gchar *nick,
		      const gchar *blurb,
		      GQuark group,
		      gdouble minimum,
		      gdouble maximum,
		      gdouble default_value,
		      GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_double (name, nick, blurb,
					  minimum, maximum, default_value,
					  flags);
  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_NUMBER_WIDGET);
  return spec;
}

/**
 * gs_param_spec_string:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @default_value:	default value
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for string value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_string (const gchar *name,
		      const gchar *nick,
		      const gchar *blurb,
		      GQuark group,
		      const gchar* default_value,
		      GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_string (name, nick, blurb,
					  default_value, flags);
  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_STRING_WIDGET);
  return spec;
}

/**
 * gs_param_spec_pointer:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @widget:		widget type
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for boolean value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Other #GSParamSpec have predefined widget type. Pointer spec allow
 * to store alsmost everything and thus needs specific widget type.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_pointer (const gchar *name,
		       const gchar *nick,
		       const gchar *blurb,
		       GQuark group,
		       GType widget,
		       GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_pointer (name, nick, blurb, flags);
  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, widget);

  return spec;
}

/* RANGE */
GS_DEFINE_PARAM (gs, GS, range, Range, G_TYPE_INT);

/**
 * gs_param_spec_range:
 * @name:	spec name
 * @nick:	spec nick
 * @blurb:	spec blurb
 * @group:	The quark of the group the param belong to
 * @minimum:	value of the lowest allowed value
 * @maximum:	value of the highed allowed value
 * @step:	lenght between two values
 * @flags:	Param flags.
 * 
 * Create a new #GSParamSpecRange. This spec has been designed to
 * store both integer and float value. Ensure all #GValue have the
 * same type.
 * 
 * Returns: the new #GSParamSpecRange
 **/
GParamSpec*
gs_param_spec_range (const gchar *name,
		     const gchar *nick,
		     const gchar *blurb,
		     GQuark group,
		     GValue *minimum,
		     GValue *maximum,
		     GValue *step,
		     GValue *default_value,
		     GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_internal (GS_TYPE_PARAM_RANGE,
					    name, nick, blurb, flags);
  GSParamSpecRange *range = GS_PARAM_SPEC_RANGE(spec);
  range->minimum = g_boxed_copy (G_TYPE_VALUE, minimum);
  range->maximum = g_boxed_copy (G_TYPE_VALUE, maximum);
  range->step = g_boxed_copy (G_TYPE_VALUE, step);
  range->default_value = g_boxed_copy (G_TYPE_VALUE, default_value);

  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_RANGE_WIDGET);

  return G_PARAM_SPEC (spec);
}


static void
gs_param_range_value_set_default (GParamSpec *pspec, GValue *value)
{
  GValue *default_value = GS_PARAM_SPEC_RANGE (pspec)->default_value;
  g_value_unset (value);
  g_value_init (value, G_VALUE_TYPE (default_value));
	
  if (G_VALUE_HOLDS_INT(value)) {
    g_value_set_int (value,
		     g_value_get_int (default_value));
  }
  else if (G_VALUE_HOLDS_DOUBLE(value)) {
    g_value_set_double (value,
			g_value_get_double (default_value));
  }
  else {
    g_warning ("%s: does not support %s value type.",
	       G_PARAM_SPEC_TYPE_NAME (pspec),
	       G_VALUE_TYPE_NAME (value));
  }
}

static gint
gs_param_range_values_cmp (GParamSpec *pspec, const GValue *a, const GValue *b)
{
  switch (G_VALUE_TYPE (a)) {
  case G_TYPE_INT:
    return g_value_get_int (a) - g_value_get_int (b);
    break;
  case G_TYPE_DOUBLE:
    return (gint) g_value_get_double (a) - g_value_get_double (b);
    break;
  case G_TYPE_STRING:
    return g_ascii_strcasecmp (g_value_get_string (a),
			       g_value_get_string (b));
    break;
  default:
    g_warning ("%s does not support %s value type.",
	       G_PARAM_SPEC_TYPE_NAME (pspec),
	       g_type_name (G_PARAM_SPEC_VALUE_TYPE (pspec)));
    return 0;
    break;
  }
			
  return 0;
}


/* ENUM */
GS_DEFINE_PARAM (gs, GS, enum, Enum, G_TYPE_INT);

/**
 * gs_param_spec_enum:
 * @name:		spec name
 * @nick:		spec nick
 * @blurb:		spec blurb
 * @group: 		The quark of the group the param belong to
 * @values:		Allowed values
 * @default_value:	default value;
 * @flags:		Param flags.
 * 
 * Create a new #GSParamSpecEnum . This parameter spec has been
 * designed to handle integer, float and string value.
 * 
 * Returns: the new #GSParamSpecEnum
 **/
GParamSpec*
gs_param_spec_enum (const gchar *name,
		    const gchar *nick,
		    const gchar *blurb,
		    GQuark group,
		    GValueArray *values,
		    GValue *default_value,
		    GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_internal (GS_TYPE_PARAM_ENUM,
					    name, nick, blurb, flags);
  GSParamSpecEnum *enumeration = GS_PARAM_SPEC_ENUM(spec);
  enumeration->values = values;
  enumeration->default_value = default_value;
  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_ENUM_WIDGET);

  return G_PARAM_SPEC (spec);
}

static void
gs_param_enum_value_set_default (GParamSpec *pspec, GValue *value)
{
  GValue *default_value = GS_PARAM_SPEC_ENUM(pspec)->default_value;
  g_value_unset (value);
  g_value_init (value, G_VALUE_TYPE (default_value));
	
  if (G_VALUE_HOLDS_INT(value)) {
    g_value_set_int (value, g_value_get_int (default_value));
  }
  else if (G_VALUE_HOLDS_DOUBLE(value)) {
    g_value_set_double (value,
			g_value_get_double (default_value));
  }
  else if (G_VALUE_HOLDS_STRING (value)) {
    g_value_set_string (value,
			g_value_dup_string (default_value));
  }
  else {
    g_warning ("%s: does not support %s value type.",
	       G_PARAM_SPEC_TYPE_NAME (pspec),
	       G_VALUE_TYPE_NAME (value));
  }
}

static gint
gs_param_enum_values_cmp (GParamSpec *pspec, const GValue *a, const GValue *b)
{
  return gs_param_range_values_cmp (pspec, a, b);
}


/* PAPER_SIZE */
GS_DEFINE_PARAM (gs, GS, paper_size, PaperSize, GTK_TYPE_PAPER_SIZE);

/**
 * gs_param_spec_paper_size:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @default_ps:		default paper size
 * @enumeration:	available paper size
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for #GtkPaperSize value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_paper_size (const gchar *name,
			  const gchar *nick,
			  const gchar *blurb,
			  GQuark group,
			  GtkPaperSize* default_ps,
			  GSList* enumeration,
			  GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_internal (GS_TYPE_PARAM_PAPER_SIZE,
					    name, nick, blurb, flags);
  GSParamSpecPaperSize *psspec = GS_PARAM_SPEC_PAPER_SIZE (spec);
  psspec->default_ps = default_ps;
  psspec->enumeration = enumeration;

  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_PAPER_SIZE_WIDGET);
  return spec;
}

static void
gs_param_paper_size_value_set_default (GParamSpec *spec,
				       GValue *value)
{
  g_value_set_boxed (value,
		     GS_PARAM_SPEC_PAPER_SIZE (spec)->default_ps);
}

static gint
gs_param_paper_size_values_cmp (GParamSpec *spec,
				const GValue *a,
				const GValue *b)
{
  GtkPaperSize *pa = g_value_get_boxed (a);
  GtkPaperSize *pb = g_value_get_boxed (b);
  /* compare surfaces */
  return (gtk_paper_size_get_width (pa, GTK_UNIT_MM) * gtk_paper_size_get_height (pa, GTK_UNIT_MM))
    - (gtk_paper_size_get_width (pb, GTK_UNIT_MM) * gtk_paper_size_get_height (pb, GTK_UNIT_MM));
}

void
gs_param_paper_size_get_extent	(GSParamSpecPaperSize *ps,
				 GnomeScanUnit unit,
				 gdouble *width,
				 gdouble *height)
{
  GtkPaperSize *gps;
  GSList *node = ps->enumeration;

  for (; node; node = node->next) {
    if (g_str_equal (gtk_paper_size_get_name (node->data),
		     "maximal")) {
      gps = node->data;
      *width = gtk_paper_size_get_width (gps, unit);
      *height = gtk_paper_size_get_height (gps, unit);
      break;
    }
  }

  return;
}

/* PAGE_ORIENTATION */
GS_DEFINE_PARAM (gs, GS, page_orientation, PageOrientation, GTK_TYPE_PAGE_ORIENTATION);

/**
 * gs_param_spec_page_orientation:
 * @name:		param name
 * @nick:		param nickname
 * @blurb:		description of the parameter
 * @group:		param group quark
 * @default_value:	default value
 * @flags:		Flags
 *
 * Create a new #GSParamSpec of for #GtkPageOrientation value. @name is for use
 * internally (e.g. gconf key) while nick and @blurb should be marked
 * for translation.
 *
 * Returns:	a new #GSParamSpec
 **/
GParamSpec*
gs_param_spec_page_orientation (const gchar *name,
				const gchar *nick,
				const gchar *blurb,
				GQuark group,
				guint default_value,
				GParamFlags flags)
{
  GParamSpec *spec = g_param_spec_internal (GS_TYPE_PARAM_PAGE_ORIENTATION,
					    name, nick, blurb, flags);
  GSParamSpecPageOrientation *psspec = GS_PARAM_SPEC_PAGE_ORIENTATION (spec);
  psspec->default_value = default_value;

  gs_param_spec_set_group (spec, group);
  gs_param_spec_set_widget_type (spec, GNOME_TYPE_SCAN_PAGE_ORIENTATION_WIDGET);
  return spec;
}

static void
gs_param_page_orientation_value_set_default (GParamSpec *spec,
					     GValue *value)
{
  g_value_set_enum (value,
		    GS_PARAM_SPEC_PAGE_ORIENTATION (spec)->default_value);
}

static gint
gs_param_page_orientation_values_cmp (GParamSpec *spec,
				      const GValue *a,
				      const GValue *b)
{
  return 0;
}
