/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gnome-vfs-private-ops.c - Private synchronous operations for the GNOME
   Virtual File System.

   Copyright (C) 1999 Free Software Foundation

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Author: Ettore Perazzoli <ettore@gnu.org> */

/* This file provides private versions of the ops for internal use.  These are
   meant to be used within the GNOME VFS and its modules: they are not for
   public consumption through the external API.  */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gnome-vfs.h"
#include "gnome-vfs-private.h"

GnomeVFSResult
gnome_vfs_open_uri_cancellable (GnomeVFSHandle **handle,
				GnomeVFSURI *uri,
				GnomeVFSOpenMode open_mode,
				GnomeVFSContext *context)
{
	GnomeVFSMethodHandle *method_handle;
	GnomeVFSResult result;

	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (uri->method != NULL, GNOME_VFS_ERROR_BADPARAMS);

	result = uri->method->open (uri->method, &method_handle, uri, open_mode,
				    context);

	/* FIXME */
	if ((open_mode & GNOME_VFS_OPEN_RANDOM)
	    && result == GNOME_VFS_ERROR_NOTSUPPORTED)
		result = uri->method->open (uri->method, &method_handle, uri,
					    open_mode & ~GNOME_VFS_OPEN_RANDOM,
					    context);

	if (result != GNOME_VFS_OK)
		return result;

	*handle = gnome_vfs_handle_new (uri, method_handle, open_mode);
	
	return GNOME_VFS_OK;
}

GnomeVFSResult
gnome_vfs_create_uri_cancellable (GnomeVFSHandle **handle,
				  GnomeVFSURI *uri,
				  GnomeVFSOpenMode open_mode,
				  gboolean exclusive,
				  guint perm,
				  GnomeVFSContext *context)
{
	GnomeVFSMethodHandle *method_handle;
	GnomeVFSResult result;

	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BADPARAMS);

	result = uri->method->create (uri->method, &method_handle, uri, open_mode,
				      exclusive, perm, context);
	if (result != GNOME_VFS_OK)
		return result;

	*handle = gnome_vfs_handle_new (uri, method_handle, open_mode);

	return GNOME_VFS_OK;
}

GnomeVFSResult
gnome_vfs_close_cancellable (GnomeVFSHandle *handle,
			     GnomeVFSContext *context)
{
	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);

	return gnome_vfs_handle_do_close (handle, context);
}

GnomeVFSResult
gnome_vfs_read_cancellable (GnomeVFSHandle *handle,
			    gpointer buffer,
			    GnomeVFSFileSize bytes,
			    GnomeVFSFileSize *bytes_written,
			    GnomeVFSContext *context)
{
	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);

	return gnome_vfs_handle_do_read (handle, buffer, bytes, bytes_written,
					 context);
}

GnomeVFSResult
gnome_vfs_write_cancellable (GnomeVFSHandle *handle,
			     gconstpointer buffer,
			     GnomeVFSFileSize bytes,
			     GnomeVFSFileSize *bytes_written,
			     GnomeVFSContext *context)
{
	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);

	return gnome_vfs_handle_do_write (handle, buffer, bytes,
					  bytes_written, context);
}

GnomeVFSResult
gnome_vfs_seek_cancellable (GnomeVFSHandle *handle,
			    GnomeVFSSeekPosition whence,
			    GnomeVFSFileOffset offset,
			    GnomeVFSContext *context)
{
	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);

	return gnome_vfs_handle_do_seek (handle, whence, offset, context);
}

GnomeVFSResult
gnome_vfs_get_file_info_uri_cancellable (GnomeVFSURI *uri,
					 GnomeVFSFileInfo *info,
					 GnomeVFSFileInfoOptions options,
					 const gchar *meta_keys[],
					 GnomeVFSContext *context)
{
	GnomeVFSResult result;
	GList *meta_list;

	meta_list = gnome_vfs_string_list_from_string_array (meta_keys);

	result = uri->method->get_file_info (uri->method, uri, info, options, meta_list,
					     context);

	gnome_vfs_free_string_list (meta_list);
	return result;
}

GnomeVFSResult
gnome_vfs_get_file_info_from_handle_cancellable (GnomeVFSHandle *handle,
						 GnomeVFSFileInfo *info,
						 GnomeVFSFileInfoOptions options,
						 const gchar *meta_keys[],
						 GnomeVFSContext *context)

{
	GnomeVFSResult result;
	GList *meta_list;

	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);

	meta_list = gnome_vfs_string_list_from_string_array (meta_keys);

	result =  gnome_vfs_handle_do_get_file_info (handle, info,
						     options, meta_list,
						     context);

	gnome_vfs_free_string_list (meta_list);

	return result;
}

GnomeVFSResult
gnome_vfs_truncate_uri_cancellable (GnomeVFSURI *uri,
				    GnomeVFSFileSize length,
				    GnomeVFSContext *context)
{
	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BADPARAMS);
	if(uri->method->truncate == NULL)
		return GNOME_VFS_ERROR_NOTSUPPORTED;

	return uri->method->truncate(uri->method, uri, length, context);
}

GnomeVFSResult
gnome_vfs_truncate_handle_cancellable (GnomeVFSHandle *handle,
				       GnomeVFSFileSize length,
				       GnomeVFSContext *context)
{
	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BADPARAMS);

	return gnome_vfs_handle_do_truncate (handle, length, context);
}

GnomeVFSResult
gnome_vfs_make_directory_for_uri_cancellable (GnomeVFSURI *uri,
					      guint perm,
					      GnomeVFSContext *context)
{
	GnomeVFSResult result;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BADPARAMS);

	if (uri->method->make_directory == NULL)
		return GNOME_VFS_ERROR_NOTSUPPORTED;

	result = uri->method->make_directory (uri->method, uri, perm, context);
	return result;
}

GnomeVFSResult
gnome_vfs_remove_directory_from_uri_cancellable (GnomeVFSURI *uri,
						 GnomeVFSContext *context)
{
	GnomeVFSResult result;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BADPARAMS);

	if (uri->method->remove_directory == NULL)
		return GNOME_VFS_ERROR_NOTSUPPORTED;

	result = uri->method->remove_directory (uri->method, uri, context);
	return result;
}

GnomeVFSResult
gnome_vfs_unlink_from_uri_cancellable (GnomeVFSURI *uri,
				       GnomeVFSContext *context)
{
	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BADPARAMS);

	if (uri->method->unlink == NULL)
		return GNOME_VFS_ERROR_NOTSUPPORTED;

	return uri->method->unlink (uri->method, uri, context);
}

static gboolean
check_same_fs_in_uri (GnomeVFSURI *a,
		      GnomeVFSURI *b)
{
	if (a->method != b->method)
		return FALSE;
	if (strcmp (a->method_string, b->method_string) != 0)
		return FALSE;

	return TRUE;
}

GnomeVFSResult
gnome_vfs_move_uri_cancellable (GnomeVFSURI *old,
				GnomeVFSURI *new,
				gboolean force_replace,
				GnomeVFSContext *context)
{
	g_return_val_if_fail (old != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (new != NULL, GNOME_VFS_ERROR_BADPARAMS);

	if (! check_same_fs_in_uri (old, new))
		return GNOME_VFS_ERROR_NOTSAMEFS;

	if (old->method->move == NULL)
		return GNOME_VFS_ERROR_NOTSUPPORTED;

	return old->method->move (old->method, old, new, force_replace, context);
}

GnomeVFSResult
gnome_vfs_check_same_fs_uris_cancellable (GnomeVFSURI *a,
					  GnomeVFSURI *b,
					  gboolean *same_fs_return,
					  GnomeVFSContext *context)
{
	g_return_val_if_fail (a != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (b != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (same_fs_return != NULL, GNOME_VFS_ERROR_BADPARAMS);

	if (! check_same_fs_in_uri (a, b)) {
		*same_fs_return = FALSE;
		return GNOME_VFS_OK;
	}

	if (a->method->check_same_fs == NULL) {
		*same_fs_return = FALSE;
		return GNOME_VFS_OK;
	}

	return a->method->check_same_fs (a->method, a, b, same_fs_return, context);
}

GnomeVFSResult
gnome_vfs_set_file_info_cancellable (GnomeVFSURI *a,
				     const GnomeVFSFileInfo *info,
				     GnomeVFSSetFileInfoMask mask,
				     GnomeVFSContext *context)
{
	g_return_val_if_fail (a != NULL, GNOME_VFS_ERROR_BADPARAMS);
	g_return_val_if_fail (info != NULL, GNOME_VFS_ERROR_BADPARAMS);

	return a->method->set_file_info (a->method, a, info, mask, context);
}
