/* $Id: directory-handle.cc,v 1.1 2003/03/29 10:11:12 murrayc Exp $ */               

/* Copyright 2003 gnome-vfsmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libgnomevfsmm/directory-handle.h>
#include <libgnomevfs/gnome-vfs-directory.h>

namespace
{

//SignalProxy_Visit:

//This Signal Proxy allows the C++ coder to specify a SigC::Slot instead of a static function.
class SignalProxy_Visit
{
public:
  typedef Gnome::Vfs::DirectoryHandle::SlotVisit SlotType;

  SignalProxy_Visit(const SlotType& slot);
  ~SignalProxy_Visit();

  static gboolean c_callback(const gchar* rel_path, GnomeVFSFileInfo* info, gboolean recursing_will_loop, gpointer data, gboolean* recurse);
              
protected:
  SlotType slot_;
};

SignalProxy_Visit::SignalProxy_Visit(const SlotType& slot)
:
  slot_ (slot)
{}

SignalProxy_Visit::~SignalProxy_Visit()
{}

gboolean SignalProxy_Visit::c_callback(const gchar* rel_path, GnomeVFSFileInfo* info, gboolean recursing_will_loop, gpointer data, gboolean* recurse)
{
  SignalProxy_Visit *const self = static_cast<SignalProxy_Visit*>(data);
  gboolean bResult = false;
  try
  {
    bool bRecurse = false;
    Glib::ustring strTemp = Glib::convert_const_gchar_ptr_to_ustring(rel_path);
    bResult = (self->slot_)(strTemp, Glib::wrap(info), recursing_will_loop, bRecurse);
    *recurse = bRecurse;
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }

  return bResult;
}

} //anonymous namespace

namespace Gnome
{

namespace Vfs
{

GnomeVFSDirectoryHandle* DirectoryHandle::gobj()
{
  return gobj_;
}

const GnomeVFSDirectoryHandle* DirectoryHandle::gobj() const
{
  return gobj_;
}

DirectoryHandle::DirectoryHandle()
: gobj_(0)
{
}

DirectoryHandle::~DirectoryHandle()
{
  //TODO: close() it if it's open? murrayc.
}

void DirectoryHandle::open(const Glib::ustring& text_uri, GnomeVFSFileInfoOptions options) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_directory_open(&gobj_, text_uri.c_str(), options);
  handle_result(result);
}

void DirectoryHandle::open(const Glib::RefPtr<const Uri>& uri, GnomeVFSFileInfoOptions options) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_directory_open_from_uri(&gobj_, const_cast<GnomeVFSURI*>(uri->gobj()), options);
  handle_result(result);
}



void DirectoryHandle::close() throw(exception)
{
  GnomeVFSResult result = gnome_vfs_directory_close(gobj());
  handle_result(result);
}
  
Glib::RefPtr<FileInfo> DirectoryHandle::read_next(bool& file_exists) throw(exception)
{
  GnomeVFSFileInfo* fileinfo = gnome_vfs_file_info_new();
  GnomeVFSResult result = gnome_vfs_directory_read_next(gobj(), fileinfo);
  Glib::RefPtr<FileInfo> cppFileInfo = Glib::wrap(fileinfo); //Takes ownership, so we don't need to unref fileinfo.
  
  if(result == GNOME_VFS_ERROR_EOF) //there are no more files.
    file_exists = false;  //This isn't really an error. It's just the end of the file list.
  else
    handle_result(result);
    
  return cppFileInfo;
}

void DirectoryHandle::visit(const Glib::ustring& uri, GnomeVFSFileInfoOptions info_options, GnomeVFSDirectoryVisitOptions visit_options, const SlotVisit& slot) throw(exception)
{
  SignalProxy_Visit proxy(slot);
  GnomeVFSResult result = gnome_vfs_directory_visit(uri.c_str(), info_options, visit_options, &SignalProxy_Visit::c_callback, &proxy);
  handle_result(result);
}

void DirectoryHandle::visit(const Glib::RefPtr<const Uri>& uri, GnomeVFSFileInfoOptions info_options, GnomeVFSDirectoryVisitOptions visit_options, const SlotVisit& slot) throw(exception)
{
  SignalProxy_Visit proxy(slot);
  GnomeVFSResult result = gnome_vfs_directory_visit_uri(const_cast<GnomeVFSURI*>(uri->gobj()), info_options, visit_options, &SignalProxy_Visit::c_callback, &proxy);
  handle_result(result);
}
 



void DirectoryHandle::handle_result(GnomeVFSResult result) throw(exception)
{
  if(result != GNOME_VFS_OK)
    throw exception(result);
}
  
} //namespace Vfs

} //namespace Gnome



