/* GnoRPM - A GNOME front end for the Redhat Package Manager (RPM)
 * Copyright (C) 1998-1999  James Henstridge
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include <stdio.h>
#include <rpmlib.h>

#include "dbhandle.h"
#include "install.h"
#include "misc.h"

#include "rpmquerydlg.h"

#ifdef WITH_RPMFIND
#include "find/libfind.h"
#endif


/* from mainwin.c */
GtkWidget *create_main(char *app_id, DBHandle *dbhdl);

static void parseAnArg(poptContext con, enum poptCallbackReason reason,
		       const struct poptOption *opt,
		       const char *arg, void *data);

static const struct poptOption gnorpm_options[] = {
  { NULL, '\0', POPT_ARG_CALLBACK|POPT_CBFLAG_POST,
    &parseAnArg, 0, NULL },
  { "geometry", '\0', POPT_ARG_STRING, NULL, -1,
    N_("geometry of main window"), N_("wxh+x+y") },
  { "root", '\0', POPT_ARG_STRING, NULL, -2,
    N_("the file system root"), N_("ROOT") },
  { "query", 'q', POPT_ARG_NONE, NULL, -3,
    N_("query packages"), NULL },
  { "packages", 'p', POPT_ARG_NONE, NULL, -4,
    N_("packages are in files (rather than db)"), NULL },
  { "install", 'i', POPT_ARG_NONE, NULL, -5,
    N_("install packages"), NULL },
  { "upgrade", 'U', POPT_ARG_NONE, NULL, -6,
    N_("upgrade packages"), NULL },
  POPT_AUTOHELP
  { NULL, '\0', 0, NULL, 0 }
};

static gint main_x = -1, main_y = -1, main_w = -1, main_h = -1;
static gchar *root = "";
static GList *args = NULL;
static enum { MODE_UNSET, MODE_QUERY, MODE_INSTALL, MODE_UPGRADE}
  majorMode = MODE_UNSET;
static isFiles = FALSE;
static GdkWindow *main_window = NULL;

static void parseAnArg(poptContext con, enum poptCallbackReason reason,
		       const struct poptOption *opt,
		       const char *arg, void *data) {
  gchar *str;

  switch (reason) {
  case POPT_CALLBACK_REASON_OPTION:
    switch (opt->val) {
    case -1: /* --geometry */
      gnome_parse_geometry(arg, &main_x, &main_y, &main_w, &main_h); break;
    case -2: /* --root */
      root = g_strdup(arg); break;
    case -3: /* --query */
      majorMode = MODE_QUERY; break;
    case -4: /* -p */
      isFiles = TRUE; break;
    case -5: /* --install */
      majorMode = MODE_INSTALL; break;
    case -6: /* --upgrade */
      majorMode = MODE_UPGRADE; break;
    }
    break;
  case POPT_CALLBACK_REASON_PRE:
    break;
  case POPT_CALLBACK_REASON_POST:
    while ((str = poptGetArg(con)) != NULL)
      args = g_list_append(args, str);
    break;
  }
}

static gint save_state(GnomeClient *client, gint phase,
		       GnomeRestartStyle save_style, gint shutdown,
		       GnomeInteractStyle interact_style, gint fast,
		       gchar *prog_name);

static void update_func(void) {
  while (gtk_events_pending())
    gtk_main_iteration();
}

int main(int argc, char *argv[]) {
  GtkWidget *app;
  GnomeClient *client;
  DBHandle *dbhdl = NULL;
  GList *pkgs = NULL, *tmp;
  dbiIndexSet matches;
  gint i, installFlags = 0, res;
  gchar buf[512];

  /* i18n stuff */
  bindtextdomain(PACKAGE, GNOMELOCALEDIR);
  textdomain(PACKAGE);
  gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv,
			     gnorpm_options, 0, NULL);

  client = gnome_master_client();
  gtk_signal_connect(GTK_OBJECT(client), "save_yourself",
		     GTK_SIGNAL_FUNC(save_state), argv[0]);

  if (rpmReadConfigFiles(NULL, NULL, NULL, 0))
    return 1;
  setupErrorHandler();

  if (majorMode == MODE_UNSET || majorMode == MODE_QUERY) {
    dbhdl = db_handle_new(root);
    if (dbhdl == NULL)
      return 1;

#ifdef WITH_RPMFIND
    libfind_init(dbhdl->db);
    libfind_set_idle_func(update_func);
#endif
  }
  
  switch (majorMode) {
  case MODE_UNSET:
    app = create_main("gnorpm", dbhdl);
    if (main_w != -1 || main_h != -1)
      gtk_widget_set_usize(app, main_w, main_h);
    if (main_x != -1 || main_y != -1)
      gtk_widget_set_uposition(app, main_x, main_y);
    gtk_widget_show(app);
    main_window = app->window;
    gtk_main();
    break;
  case MODE_QUERY:
    if (isFiles)
      app = rpm_query_dialog_new_from_files(dbhdl, args);
    else {
      pkgs = NULL;
      for (tmp = args; tmp != NULL; tmp = tmp->next) {
	if (rpmdbFindByLabel(dbhdl->db, tmp->data, &matches))
	  continue;
	for (i = 0; i < matches.count; i++)
	  if (matches.recs[i].recOffset != 0)
	    pkgs = g_list_append(pkgs,
				 GINT_TO_POINTER(matches.recs[i].recOffset));
	dbiFreeIndexRecord(matches);
      }
      app = rpm_query_dialog_new(dbhdl, pkgs);
    }
    if (main_w != -1 || main_h != -1)
      gtk_widget_set_usize(app, main_w, main_h);
    if (main_x != -1 || main_y != -1)
      gtk_widget_set_uposition(app, main_x, main_y);
    gtk_signal_connect(GTK_OBJECT(app), "destroy",
		       GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
    gtk_widget_show(app);
    main_window = app->window;
    gtk_main();
    break;
  case MODE_UPGRADE:
    installFlags = installFlags | RPMINSTALL_UPGRADE;
  case MODE_INSTALL:
    res = do_install(root, pkgs, NULL, installFlags, 0);
    if (res) {
      g_snprintf(buf, 511, _("%s of %d packages failed."),
		 (majorMode==MODE_UPGRADE)?_("Upgrade"):_("Install"), res);
      message_box(buf);
    }
    break;
  }

  if (dbhdl != NULL)
    db_handle_free(dbhdl);
  return 0;
}

static gint save_state(GnomeClient *client, gint phase,
		       GnomeRestartStyle save_style, gint shutdown,
		       GnomeInteractStyle interact_style, gint fast,
		       gchar *prog_name) {
  gint x, y, w, h;
  gchar *geom = NULL;
  if (main_window)
    geom = gnome_geometry_string(main_window);
  if (majorMode == MODE_UNSET) {
    gchar *argv[6];
    gint i = 0;

    argv[i++] = prog_name;
    if (root && *root) {
      argv[i++] = "--root";
      argv[i++] = root;
    }
    argv[i++] = "--geometry";
    argv[i++] = geom;
    gnome_client_set_restart_command(client, i, argv);
    gnome_client_set_clone_command(client, 0, NULL);
    if (geom) g_free(geom);
    return TRUE;
  } else if (majorMode == MODE_QUERY) {
    gchar **argv = g_new(gchar *, 7 + g_list_length(args));
    GList *tmp;
    gint i = 0;
    argv[i++] = prog_name;
    if (root && *root) {
      argv[i++] = "--root";
      argv[i++] = root;
    }
    argv[i++] = "--geometry";
    argv[i++] = geom;
    if (isFiles) {
      argv[i++] = "--query";
      argv[i++] = "-p";
    } else
      argv[i++] = "--query";
    for (tmp = args; tmp != NULL; tmp = tmp->next)
      argv[i++] = tmp->data;
    gnome_client_set_restart_command(client, i, argv);
    gnome_client_set_clone_command(client, 0, NULL);
    g_free(argv);
    if (geom) g_free(geom);
    return TRUE;
  }
  if (geom) g_free(geom);
  return FALSE;
}

