/* GnoRPM - A GNOME front end for the Redhat Package Manager (RPM)
 * Copyright (C) 1998-1999  James Henstridge
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rpmprops.h"
#include <rpmlib.h>
#include <string.h>
#include "misc.h"
#include "install.h"

#ifdef WITH_RPMFIND
#include "find/libfind.h"
#endif

static GnomePropertyBoxClass *parent_class;

static void rpm_props_box_destroy(GtkObject *object);
static void rpm_props_box_apply(GnomePropertyBox *self, gint page_num);

static void rpm_props_box_class_init(RpmPropsBoxClass *klass);
static void rpm_props_box_init(RpmPropsBox *box);

guint rpm_props_box_get_type(void) {
  static guint propsbox_type = 0;
  if (!propsbox_type) {
    GtkTypeInfo propsbox_info = {
      "RpmPropsBox",
      sizeof(RpmPropsBox),
      sizeof(RpmPropsBoxClass),
      (GtkClassInitFunc) rpm_props_box_class_init,
      (GtkObjectInitFunc) rpm_props_box_init,
      (GtkArgSetFunc) NULL,
      (GtkArgGetFunc) NULL,
    };
    propsbox_type = gtk_type_unique(gnome_property_box_get_type(),
				    &propsbox_info);
  }
  return propsbox_type;
}

static void rpm_props_box_class_init(RpmPropsBoxClass *klass) {
  parent_class = GNOME_PROPERTY_BOX_CLASS(gtk_type_class(
				gnome_property_box_get_type()));
  GTK_OBJECT_CLASS(klass)->destroy = rpm_props_box_destroy;
  GNOME_PROPERTY_BOX_CLASS(klass)->apply = rpm_props_box_apply;
}

typedef struct {
  gint32 interfaceFlags, installFlags, uninstallFlags;
  const gchar *tag, *string;
} RpmFlagInfo;
const static RpmFlagInfo prop_flags[] = {
  /* install options */
  {INTER_NODEPS,  0, 0, "NoDependencies", N_("No dependency checks")},
  {INTER_NOORDER, 0, 0, "NoReorder", N_("No reordering")},
  {0, RPMINSTALL_NOSCRIPTS, RPMUNINSTALL_NOSCRIPTS, "NoScripts",
   N_("Don't run scripts")}, 
  /* upgrade options */
  {0, RPMINSTALL_REPLACEPKG, 0, "ReplacePkgs",
   N_("Allow replacement of packages")},
  {0, RPMINSTALL_REPLACEFILES, 0, "ReplaceFiles",
   N_("Allow replacement of files")},
  {0, RPMINSTALL_UPGRADETOOLD, 0, "UpgradeToOld",
   N_("Allow upgrade to old version")},
  {0, RPMINSTALL_KEEPOBSOLETE, 0, "KeepObsolete",
   N_("Keep packages made obsolete")},
  /* file options */
  {0, RPMINSTALL_NODOCS, 0, "NoDocumentation",
   N_("Don't install documentation")},
  {0, RPMINSTALL_ALLFILES, 0, "AllFiles",
   N_("Install all files")},
  /* database options */
  {0, RPMINSTALL_JUSTDB, RPMUNINSTALL_JUSTDB, "JustDB",
   N_("Just update database")},
  {0, RPMINSTALL_TEST, RPMUNINSTALL_TEST, "JustTest",
   N_("Just test")},
  /* architecture checks */
  {0, RPMINSTALL_NOARCH, 0, "NoArchCheck",
   N_("Don't check package architecture")},
  {0, RPMINSTALL_NOOS, 0, "NoOSCheck",
   N_("Don't check package OS")},
};
const int num_prop_flags = sizeof(prop_flags) / sizeof(RpmFlagInfo);

static GtkWidget *rpm_props_box_fill_frame(RpmPropsBox *self, gchar *name,
					   gint start, gint end);
#ifdef WITH_RPMFIND
static void rpm_props_dist_select_row(GtkCList *clist, gint row, gint col,
				      GdkEvent *event, RpmPropsBox *self);
static void rpm_props_dist_changed(GtkButton *button, RpmPropsBox *self);
static void rpm_props_upgrade_add(RpmPropsBox *self);
static void rpm_props_upgrade_del(RpmPropsBox *self);
#endif

static void rpm_props_box_init(RpmPropsBox *self) {
  GtkWidget *frame, *hbox, *vbox, *box, *table, *label, *wid;
  gchar *text;
  gchar *distTitles[4] = { N_("Name"), N_("ID"), N_("Rating"), N_("mirror") };
  GList *tmp;

  set_icon(GTK_WIDGET(self));
  gtk_window_set_title(GTK_WINDOW(self), _("Preferences"));

  self->interfaceFlags = self->installFlags = self->uninstallFlags = 0;
  self->flags = g_new(GtkWidget *, num_prop_flags);
  /* create the behviour page */
  hbox = gtk_hbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), GNOME_PAD);
  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show(vbox);

  frame = rpm_props_box_fill_frame(self, _("Install Options"), 0, 2);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  frame = rpm_props_box_fill_frame(self, _("Upgrade Options"), 3, 6);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show(vbox);
  
  frame = rpm_props_box_fill_frame(self, _("Other Options"), 7, 8);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  frame = rpm_props_box_fill_frame(self, _("Database Options"), 9, 10);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  frame = rpm_props_box_fill_frame(self, _("Architecture Options"), 11, 12);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  label = gtk_label_new(_("Behaviour"));
  gtk_widget_show(hbox);
  gtk_widget_show(label);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(self), hbox, label);

  /* the interface page */
  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
  frame = gtk_frame_new(_("Package Listing"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, TRUE, 0);
  gtk_widget_show(frame);

  box = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_set_border_width(GTK_CONTAINER(box), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), box);
  gtk_widget_show(box);

  self->asList = gtk_radio_button_new_with_label(NULL, _("View as list"));
  gtk_box_pack_start(GTK_BOX(box), self->asList, TRUE, TRUE, 0);
  gtk_widget_show(self->asList);
  gtk_signal_connect_object(GTK_OBJECT(self->asList), "toggled",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  self->asIcons = gtk_radio_button_new_with_label(
		GTK_RADIO_BUTTON(self->asList)->group, _("View as icons"));
  gtk_box_pack_start(GTK_BOX(box), self->asIcons, TRUE, TRUE, 0);
  gtk_widget_show(self->asIcons);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(self->asIcons),
		gnome_config_get_bool("/gnorpm/Layout/asIcons=true"));

  gtk_signal_connect_object(GTK_OBJECT(self->asList), "toggled",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_signal_connect_object(GTK_OBJECT(self->asIcons), "toggled",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));

  frame = gtk_frame_new(_("RPM Path"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, TRUE, 0);
  gtk_widget_show(frame);
  box = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_set_border_width(GTK_CONTAINER(box), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), box);
  gtk_widget_show(box);

  self->rpmPath = gtk_entry_new();
  text =gnome_config_get_string("/gnorpm/Paths/rpmDir=/mnt/cdrom/RedHat/RPMS");
  gtk_entry_set_text(GTK_ENTRY(self->rpmPath), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->rpmPath), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_box_pack_start(GTK_BOX(box), self->rpmPath, TRUE, TRUE, 0);
  gtk_widget_show(self->rpmPath);

  label = gtk_label_new(_("Interface"));
  gtk_widget_show(vbox);
  gtk_widget_show(label);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(self), vbox, label);

#ifdef WITH_RPMFIND
  /* the rpmfind page */
  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
  frame = gtk_frame_new(_("Rpmfind Options"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, TRUE, 0);
  gtk_widget_show(frame);
  table = gtk_table_new(6, 4, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(table), GNOME_PAD);
  gtk_table_set_col_spacings(GTK_TABLE(table), GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(table), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  label = gtk_label_new(_("Metadata Server:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 0,1,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->metadataServer = gtk_combo_new();
  gtk_combo_set_value_in_list(GTK_COMBO(self->metadataServer), FALSE, FALSE);
  gtk_combo_set_popdown_strings(GTK_COMBO(self->metadataServer),
				metadataGetList());
  text = gnome_config_get_string("/gnorpm/rpmfind/server=http://rufus.w3.org/linux/RDF");
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(self->metadataServer)->entry), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(self->metadataServer)->entry),
			    "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->metadataServer, 1,4, 0,1,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->metadataServer);
  self->metadataServer = GTK_COMBO(self->metadataServer)->entry;

  label = gtk_label_new(_("HTTP Proxy:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 1,2,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->httpProxy = gtk_entry_new();
  text = gnome_config_get_string("/gnorpm/rpmfind/http-proxy=");
  gtk_entry_set_text(GTK_ENTRY(self->httpProxy), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->httpProxy), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->httpProxy, 1,4, 1,2,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->httpProxy);

  label = gtk_label_new(_("FTP Proxy:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 2,3,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->ftpProxy = gtk_entry_new();
  text = gnome_config_get_string("/gnorpm/rpmfind/ftp-proxy=");
  gtk_entry_set_text(GTK_ENTRY(self->ftpProxy), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->ftpProxy), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->ftpProxy, 1,4, 2,3,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->ftpProxy);

  label = gtk_label_new(_("Download dir:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 3,4,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->downloadDir = gtk_entry_new();
  text = gnome_config_get_string("/gnorpm/rpmfind/rpm-dir=");
  gtk_entry_set_text(GTK_ENTRY(self->downloadDir), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->downloadDir), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->downloadDir, 1,4, 3,4,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->downloadDir);

  label = gtk_label_new(_("Local Hostname:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 4,5,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->hostname = gtk_entry_new();
  text = gnome_config_get_string("/gnorpm/rpmfind/hostname");
  gtk_entry_set_text(GTK_ENTRY(self->hostname), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->hostname), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->hostname, 1,2, 4,5,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->hostname);
  label = gtk_label_new(_("Cache expire:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 2,3, 4,5,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  hbox = gtk_hbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_table_attach(GTK_TABLE(table), hbox, 3,4, 4,5,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(hbox);
  self->expires = gtk_spin_button_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0,31.0, 1.0, 2.0, 1.0)),
	1.0, 0);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(self->expires),
			gnome_config_get_int("/gnorpm/rpmfind/expires=14"));
  gtk_signal_connect_object(GTK_OBJECT(self->expires), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_box_pack_start(GTK_BOX(hbox), self->expires, TRUE, TRUE, 0);
  gtk_widget_show(self->expires);
  label = gtk_label_new(_("days"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
  gtk_widget_show(label);

  label = gtk_label_new(_("Vendor:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 5,6,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->vendor = gtk_entry_new();
  text = gnome_config_get_string("/gnorpm/rpmfind/vendor");
  gtk_entry_set_text(GTK_ENTRY(self->vendor), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->vendor), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->vendor, 1,2, 5,6,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->vendor);
  label = gtk_label_new(_("Distrib:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 2,3, 5,6,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->distrib = gtk_entry_new();
  text = gnome_config_get_string("/gnorpm/rpmfind/distribution");
  gtk_entry_set_text(GTK_ENTRY(self->distrib), text);
  g_free(text);
  gtk_signal_connect_object(GTK_OBJECT(self->distrib), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->distrib, 3,4, 5,6,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distrib);

  self->wantSource = gtk_check_button_new_with_label(_("Want sources"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(self->wantSource),
		gnome_config_get_bool("/gnorpm/rpmfind/wantSources=false"));
  gtk_signal_connect_object(GTK_OBJECT(self->wantSource), "toggled",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->wantSource, 0,2, 6,7,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->wantSource);
  self->wantLatest = gtk_check_button_new_with_label(_("Want latest version"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(self->wantLatest),
	gnome_config_get_bool("/gnorpm/rpmfind/wantLatestVersion=false"));
  gtk_signal_connect_object(GTK_OBJECT(self->wantLatest), "toggled",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->wantLatest, 2,4, 6,7,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->wantLatest);

  label = gtk_label_new(_("Rpmfind"));
  gtk_widget_show(vbox);
  gtk_widget_show(label);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(self), vbox, label);
  
  /* distributions page */
  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
  frame = gtk_frame_new(_("Distribution Settings"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);
  table = gtk_table_new(6, 3, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(table), GNOME_PAD);
  gtk_table_set_col_spacings(GTK_TABLE(table), GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(table), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  wid = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(wid),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_table_attach(GTK_TABLE(table), wid, 0,3, 0,1,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(wid);
  distTitles[0] = _(distTitles[0]);
  distTitles[1] = _(distTitles[1]);
  distTitles[2] = _(distTitles[2]);
  distTitles[3] = _(distTitles[3]);
  self->distClist = gtk_clist_new_with_titles(4, distTitles);
  gtk_clist_column_titles_passive(GTK_CLIST(self->distClist));
  gtk_widget_set_usize(self->distClist, -1, 75);
  gtk_clist_set_selection_mode(GTK_CLIST(self->distClist),
			       GTK_SELECTION_BROWSE);
  gtk_clist_set_column_width(GTK_CLIST(self->distClist), 0, 150);
  gtk_clist_set_column_width(GTK_CLIST(self->distClist), 1, 70);
  gtk_clist_set_column_width(GTK_CLIST(self->distClist), 2, 20);
  gtk_clist_set_sort_column(GTK_CLIST(self->distClist), 0);
  gtk_clist_set_sort_type(GTK_CLIST(self->distClist), GTK_SORT_ASCENDING);
  gtk_clist_freeze(GTK_CLIST(self->distClist));
  for (tmp = distribGetList(); tmp; tmp = tmp->next) {
    rpmDistrib *distrib = distribInfoGet(tmp->data);
    gchar *row[4], buf[21];

    g_snprintf(buf, 20, "%d", distrib->rating);
    row[0] = distrib->name;
    row[1] = distrib->ID;
    row[2] = buf;
    row[3] = distrib->myMirror;
    gtk_clist_append(GTK_CLIST(self->distClist), row);
  }
  gtk_clist_sort(GTK_CLIST(self->distClist));
  gtk_clist_thaw(GTK_CLIST(self->distClist));
  gtk_signal_connect(GTK_OBJECT(self->distClist), "select_row",
		     GTK_SIGNAL_FUNC(rpm_props_dist_select_row), self);
  gtk_container_add(GTK_CONTAINER(wid), self->distClist);
  gtk_widget_show(self->distClist);

  label = gtk_label_new(_("Name:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 1,2,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->distName = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(self->distName), 0.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), self->distName, 1,3, 1,2,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distName);
  
  label = gtk_label_new(_("Origin:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 2,3,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->distOrigin = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(self->distOrigin), 0.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), self->distOrigin, 1,3, 2,3,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distOrigin);

  label = gtk_label_new(_("Sources:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 3,4,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->distSources = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(self->distSources), 0.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), self->distSources, 1,3, 3,4,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distSources);

  label = gtk_label_new(_("Rating:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 4,5,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->distRating = gtk_spin_button_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(0.0, -1.0,10000.0, 1.0, 10.0, 10.0)),
	1.0, 0);
  gtk_table_attach(GTK_TABLE(table), self->distRating, 1,2, 4,5,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distRating);

  label = gtk_label_new(_("Preferred Mirror:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 5,6,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(label);
  self->distMirror = gtk_combo_new();
  gtk_combo_set_value_in_list(GTK_COMBO(self->distMirror), FALSE, FALSE);
  gtk_table_attach(GTK_TABLE(table), self->distMirror, 1,2, 5,6,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distMirror);

  self->distChange = gtk_button_new_with_label(_("Change"));
  gtk_signal_connect(GTK_OBJECT(self->distChange), "clicked",
		     GTK_SIGNAL_FUNC(rpm_props_dist_changed), self);
  gtk_table_attach(GTK_TABLE(table), self->distChange, 2,3, 5,6,
		   0, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->distChange);

  gtk_clist_select_row(GTK_CLIST(self->distClist), 0, 0);

  label = gtk_label_new(_("Distributions"));
  gtk_widget_show(vbox);
  gtk_widget_show(label);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(self), vbox, label);

  /* the noupgrade pane */
  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
  frame = gtk_frame_new(_("Distribution Settings"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);
  table = gtk_table_new(3, 3, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(table), GNOME_PAD);
  gtk_table_set_col_spacings(GTK_TABLE(table), GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(table), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  label = gtk_label_new(_("Add packages you don't want upgraded with web find here"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0,3, 0,1,
		   GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_widget_show(label);
  self->addUpgradeEntry = gtk_entry_new();
  gtk_signal_connect_object(GTK_OBJECT(self->addUpgradeEntry), "activate",
			    GTK_SIGNAL_FUNC(rpm_props_upgrade_add),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), self->addUpgradeEntry, 0,1, 1,2,
		   GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_widget_show(self->addUpgradeEntry);

  wid = gtk_button_new_with_label(_("Add"));
  gtk_signal_connect_object(GTK_OBJECT(wid), "clicked",
			    GTK_SIGNAL_FUNC(rpm_props_upgrade_add),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), wid, 1,2, 1,2,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show(wid);

  wid = gtk_button_new_with_label(_("Delete"));
  gtk_signal_connect_object(GTK_OBJECT(wid), "clicked",
			    GTK_SIGNAL_FUNC(rpm_props_upgrade_del),
			    GTK_OBJECT(self));
  gtk_table_attach(GTK_TABLE(table), wid, 2,3, 1,2,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show(wid);

  wid = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(wid),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_table_attach(GTK_TABLE(table), wid, 0,3, 2,3,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(wid);

  self->upgradeList = gtk_list_new();
  gtk_list_set_selection_mode(GTK_LIST(self->upgradeList),
			      GTK_SELECTION_BROWSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(wid),
					self->upgradeList);
  gtk_widget_show(self->upgradeList);
  for (tmp = getNoUpgradeList(); tmp; tmp = tmp->next) {
    GtkWidget *item = gtk_list_item_new_with_label(tmp->data);

    gtk_container_add(GTK_CONTAINER(self->upgradeList), item);
    gtk_widget_show(item);
  }

  label = gtk_label_new(_("No Upgrade"));
  gtk_widget_show(vbox);
  gtk_widget_show(label);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(self), vbox, label);
#endif
}

GtkWidget *rpm_props_box_new(void) {
  RpmPropsBox *self = gtk_type_new(rpm_props_box_get_type());
  return GTK_WIDGET(self);
}

static void rpm_props_box_flag_toggled(GtkToggleButton *cb, RpmPropsBox *self){
  RpmFlagInfo *info = gtk_object_get_user_data(GTK_OBJECT(cb));
  if (cb->active) {
    self->interfaceFlags |= info->interfaceFlags;
    self->installFlags   |= info->installFlags;
    self->uninstallFlags |= info->uninstallFlags;
  } else {
    self->interfaceFlags &= ~info->interfaceFlags;
    self->installFlags   &= ~info->installFlags;
    self->uninstallFlags &= ~info->uninstallFlags;
  }
  gnome_property_box_changed(GNOME_PROPERTY_BOX(self));
}

static GtkWidget *rpm_props_box_fill_frame(RpmPropsBox *self, gchar *name,
					   gint start, gint end) {
  GtkWidget *frame, *box;
  gint i, state;
  gchar buf[1024];

  frame = gtk_frame_new(name);
  box = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_set_border_width(GTK_CONTAINER(box), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), box);
  gtk_widget_show(box);
  for (i = start; i < end+1; i++) {
    self->flags[i] = gtk_check_button_new_with_label(_(prop_flags[i].string));
    /* get old value */
    g_snprintf(buf, 1023, "/gnorpm/Flags/%s=false", prop_flags[i].tag);
    state = gnome_config_get_bool(buf);
    if (state) {
      self->interfaceFlags |= prop_flags[i].interfaceFlags;
      self->installFlags |= prop_flags[i].installFlags;
      self->uninstallFlags |= prop_flags[i].uninstallFlags;
    }
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(self->flags[i]), state);
    gtk_object_set_user_data(GTK_OBJECT(self->flags[i]),
			     (gpointer)&(prop_flags[i]));
    gtk_signal_connect(GTK_OBJECT(self->flags[i]), "toggled",
		       GTK_SIGNAL_FUNC(rpm_props_box_flag_toggled), self);
    gtk_box_pack_start(GTK_BOX(box), self->flags[i], TRUE, TRUE, 0);
    gtk_widget_show(self->flags[i]);
  }
  return frame;
}

#ifdef WITH_RPMFIND
static void rpm_props_dist_select_row(GtkCList *clist, gint row, gint col,
				      GdkEvent *event, RpmPropsBox *self) {
  gchar *id, *rating, *mirror;
  rpmDistrib *distrib;
  GtkWidget *wid;

  gtk_clist_get_text(clist, row, 1, &id);
  gtk_clist_get_text(clist, row, 2, &rating);
  gtk_clist_get_text(clist, row, 3, &mirror);
  distrib = distribInfoGet(id);

  gtk_label_set(GTK_LABEL(self->distName),
		distrib->name ? distrib->name : "");
  gtk_label_set(GTK_LABEL(self->distOrigin),
		distrib->origin ? distrib->origin : "");
  gtk_label_set(GTK_LABEL(self->distSources),
		distrib->sources ? distrib->sources : "");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(self->distRating),
			    strtol(rating, NULL, 0));
  if (distrib->mirrors)
    gtk_combo_set_popdown_strings(GTK_COMBO(self->distMirror),
				  distrib->mirrors);
  else
    gtk_list_clear_items(GTK_LIST(GTK_COMBO(self->distMirror)->list), 0, -1);
  if (distrib->origin) {
    wid = gtk_list_item_new_with_label(distrib->origin);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(self->distMirror)->list),
		      wid);
    gtk_widget_show(wid);
  }
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(self->distMirror)->entry),
		     mirror);
}

static void rpm_props_dist_changed(GtkButton *button, RpmPropsBox *self) {
  GtkCList *clist = GTK_CLIST(self->distClist);
  gint row = GPOINTER_TO_INT(clist->selection->data);

  gtk_clist_set_text(clist, row, 2,
	gtk_entry_get_text(GTK_ENTRY(self->distRating)));
  gtk_clist_set_text(clist, row, 3,
	gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(self->distMirror)->entry)));
  gnome_property_box_changed(GNOME_PROPERTY_BOX(self));
}

static void rpm_props_upgrade_add(RpmPropsBox *self) {
  gchar *name = gtk_entry_get_text(GTK_ENTRY(self->addUpgradeEntry));
  GtkWidget *item = gtk_list_item_new_with_label(name);

  gtk_container_add(GTK_CONTAINER(self->upgradeList), item);
  gtk_widget_show(item);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(self));
}

static void rpm_props_upgrade_del(RpmPropsBox *self) {
  GList *selection = GTK_LIST(self->upgradeList)->selection;

  if (!selection) return;

  gtk_container_remove(GTK_CONTAINER(self->upgradeList),
		       GTK_WIDGET(selection->data));
  gnome_property_box_changed(GNOME_PROPERTY_BOX(self));
}
#endif

static void rpm_props_box_destroy(GtkObject *object) {
  RpmPropsBox *self = RPM_PROPS_BOX(object);

  if (self->flags) g_free(self->flags);
  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (* GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}

static void rpm_props_box_apply(GnomePropertyBox *object, gint page_num) {
  RpmPropsBox *self = RPM_PROPS_BOX(object);
  gint i, rows;
  gchar buf[1024];
  GList *tmp;

  switch (page_num) {
  case RPM_PROPS_FLAGS:
    for (i = 0; i < num_prop_flags; i++) {
      g_snprintf(buf, 1023, "/gnorpm/Flags/%s", prop_flags[i].tag);
      gnome_config_set_bool(buf, GTK_TOGGLE_BUTTON(self->flags[i])->active);
    }
    break;
  case RPM_PROPS_INTERFACE:
    gnome_config_set_bool("/gnorpm/Layout/asIcons",
			  GTK_TOGGLE_BUTTON(self->asIcons)->active);
    gnome_config_set_string("/gnorpm/Paths/rpmDir",
			    gtk_entry_get_text(GTK_ENTRY(self->rpmPath)));
    break;
#ifdef WITH_RPMFIND
  case RPM_PROPS_RPMFIND:
    gnome_config_set_string("/gnorpm/rpmfind/server",
			gtk_entry_get_text(GTK_ENTRY(self->metadataServer)));
    gnome_config_set_string("/gnorpm/rpmfind/http-proxy",
			    gtk_entry_get_text(GTK_ENTRY(self->httpProxy)));
    gnome_config_set_string("/gnorpm/rpmfind/ftp-proxy",
			    gtk_entry_get_text(GTK_ENTRY(self->ftpProxy)));
    gnome_config_set_string("/gnorpm/rpmfind/rpm-dir",
			    gtk_entry_get_text(GTK_ENTRY(self->downloadDir)));
    gnome_config_set_string("/gnorpm/rpmfind/hostname",
			    gtk_entry_get_text(GTK_ENTRY(self->hostname)));
    gnome_config_set_int("/gnorpm/rpmfind/expires",
	 gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(self->expires)));
    gnome_config_set_string("/gnorpm/rpmfind/vendor",
			    gtk_entry_get_text(GTK_ENTRY(self->vendor)));
    gnome_config_set_string("/gnorpm/rpmfind/distribution",
			    gtk_entry_get_text(GTK_ENTRY(self->distrib)));
    gnome_config_set_bool("/gnorpm/rpmfind/wantSources",
			  GTK_TOGGLE_BUTTON(self->wantSource)->active);
    gnome_config_set_bool("/gnorpm/rpmfind/wantLatestVersion",
			  GTK_TOGGLE_BUTTON(self->wantLatest)->active);
  case RPM_PROPS_DISTRIBS:
    rows = GTK_CLIST(self->distClist)->rows;
    for (i = 0; i < rows; i++) {
      gchar *id, *rating, *mirror;
      rpmDistrib *distrib;

      gtk_clist_get_text(GTK_CLIST(self->distClist), i, 1, &id);
      gtk_clist_get_text(GTK_CLIST(self->distClist), i, 2, &rating);
      gtk_clist_get_text(GTK_CLIST(self->distClist), i, 3, &mirror);

      distribSetRating(id, strtol(rating, NULL, 0));
      distribSetMirror(id, mirror);
    }
  case RPM_PROPS_NOUPGRADE:
    if (gnome_config_has_section("/gnorpm/No-Upgrade"))
      gnome_config_clean_section("/gnorpm/No-Upgrade");
    gnome_config_push_prefix("/gnorpm/No-Upgrade/");
    for (i = 0, tmp = GTK_LIST(self->upgradeList)->children; tmp;
	 i++, tmp = tmp->next) {
      gchar key[13], *value;

      g_snprintf(key, sizeof(key), "%d", i);
      gtk_label_get(GTK_LABEL(GTK_BIN(tmp->data)->child), &value);
      gnome_config_set_string(key, value);
    }
    updateNoUpgradeList();
    break;
#endif
  default:
    break;
  }

  gnome_config_sync();
  if (parent_class->apply)
    (* parent_class->apply)(object, page_num);
}

void rpm_props_box_get_flags(gint32 *interfaceFlags, gint32 *installFlags,
			     gint32 *uninstallFlags) {
  gint i;
  gchar buf[1024];
  gint32 inter = 0, inst = 0, uninst = 0;

  for (i = 0; i < num_prop_flags; i++) {
    g_snprintf(buf, 1023, "/gnorpm/Flags/%s=false", prop_flags[i].tag);
    if (gnome_config_get_bool(buf)) {
      inter  |= prop_flags[i].interfaceFlags;
      inst   |= prop_flags[i].installFlags;
      uninst |= prop_flags[i].uninstallFlags;
    }
  }
  if (interfaceFlags) *interfaceFlags = inter;
  if (installFlags)   *installFlags = inst;
  if (uninstallFlags) *uninstallFlags = uninst;
}
