/*
* gok-spy.c
*
* Copyright 2002 Sun Microsystems, Inc.,
* Copyright 2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/*
 * utility for getting accessible application UI info
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <libspi/keymasks.h>
#ifndef  SPI_KEYMASK_MOD4
  #define  SPI_KEYMASK_MOD4 Mod4Mask
#endif
#include "gok-spy.h"
#include "gok-log.h"
#include "gok-modifier.h"
#include "gok-data.h"
#include "gok-modifier-keymasks.h"

/*
 * these definitions allow us to tweak the efficiency/accuracy of the
 * gui search
 */

#define GOKSPY_GUI_BREADTH 999
#define GOKSPY_GUI_DEPTH 999

/*
 * NOTES
 *
 * the at-spi currently requires Accessible_ref and Accessible_unref
 * calls to keep accessible pointers valid.  
 *
 * gok-spy assumes only onelistener is registered at a time.
 *
 */

/*
 * private prototypes
 */

static void gok_spy_focus_listener (const AccessibleEvent *event,
                                    void *user_data);

static void gok_spy_defunctness_listener (const AccessibleEvent *event,
                                    void *user_data);

static void gok_spy_process_accessible(Accessible* accessible);

static SPIBoolean gok_spy_mouse_listener (const AccessibleDeviceEvent *event,
					  void *user_data);

SPIBoolean gok_spy_key_listener (const AccessibleKeystroke *key,
                                 void *user_data);

AccessibleNode* gok_spy_create_list (Accessible *anAccessible,
                                     AccessibleRole anAccessiblerole);

static void gok_spy_find_children (AccessibleNode* pNode, AccessibleRole Role);

static gboolean gok_spy_has_child_aux (Accessible* accessible,
                                       AccessibleRole role, gint children,
                                       gint depth, gint cdepth,
                                       gboolean* found);

static AccessibleNode*  gok_spy_append_node (AccessibleNode* pnode,
                                             Accessible* pAccessible);

static void gok_spy_modifier_listener(const AccessibleEvent *event,
                                      void *user_data);

static gboolean gok_spy_worth_searching (Accessible* accessible);

static void gok_spy_window_activate_listener (const AccessibleEvent *event,
                                              void *user_data);

static void gok_spy_window_deactivate_listener (const AccessibleEvent *event,
                                                void *user_data);

/*
 * gok-spy variables
 */

static GQueue* eventQueue;
static void* m_ptheAppChangeListener;
static void* m_ptheWindowChangeListener;
static void* m_ptheMouseButtonListener;

/* pointer to the current window */
static Accessible* m_ptheWindowAccessible;

/* pointer to the current accessible with editable text */
static Accessible* m_ptheAccessibleWithEditableText;

static AccessibleEventListener* focusListener;
static AccessibleEventListener* defunctnessListener;
static AccessibleDeviceListener* mouseListener;
static AccessibleEventListener* modifierListener;
static AccessibleEventListener* windowActivateListener;
static AccessibleEventListener* windowDeactivateListener;
static gboolean m_gokSpyOpen = FALSE;
static gint explicitrefs;
static gint explicitunrefs;
static gint implicitrefs;

static guint spy_handler_id = 0;

/* 
 * for debugging
 */
void gok_spy_accessible_ref(Accessible* accessible)
{
	char *s;

	gok_log_enter ();

	if (accessible != NULL)
	{
		explicitrefs++;
		Accessible_ref (accessible);
		s = Accessible_getName (accessible);
		if (s == NULL)
		{
			gok_log ("%#x", accessible);
		}
		else
		{
			gok_log ("%#x %s", accessible, s);
			SPI_freeString (s);
		}
	}
	else
	{
		gok_log ("NULL");
	}

	gok_log_leave ();
}
void gok_spy_accessible_implicit_ref(Accessible* accessible)
{
	char *s;

	gok_log_enter ();

	if (accessible != NULL)
	{
		implicitrefs++;
		s = Accessible_getName (accessible);
		if (s == NULL)
		{
			gok_log ("%#x", accessible);
		}
		else
		{
			gok_log ("%#x %s", accessible, s);
			SPI_freeString (s);
		}
	}
	else
	{
		gok_log ("NULL");
	}

	gok_log_leave ();
}
void gok_spy_accessible_unref(Accessible* accessible)
{
	char *s;

	gok_log_enter ();

	if (accessible != NULL)
	{
		s = Accessible_getName (accessible);
		if (s == NULL)
		{
			gok_log ("%#x", accessible);
		}
		else
		{
			gok_log ("%#x %s", accessible, s);
			SPI_freeString (s);
		}
		explicitunrefs++;
		Accessible_unref (accessible);
	}
	else
	{
		gok_log ("NULL");
	}

	gok_log_leave ();
}

gboolean gok_spy_accessible_is_okay(Accessible* accessible)
{
	gboolean returncode = FALSE;
	gok_log_enter();
	if (accessible != NULL)
	{
		/*
		returncode = cspi_accessible_is_a (accessible, "IDL:Accessibility/Accessible:1.0");
		*/
		/*
		if (Accessible_isApplication(accessible) ||
		    Accessible_isComponent(accessible))
		{
			returncode = TRUE;
		}
		*/
			/*
	    	if (gok_spy_accessible_is_desktopChild(accessible) == FALSE)
	    	{
				accessible = NULL;
			}
			else
			{
				returncode = TRUE;
			}
			*/
		/* kludge - check to see if there is even a window */
		if (m_ptheWindowAccessible != NULL)
		{
			returncode = TRUE;
		}
	}
	gok_log("returning: %d",returncode);
	gok_log_leave();
	return returncode;
}

/**
 * gok_spy_open:
 *
 * Intializes gok spy.
 * Note: user must initialize the SPI prior to this call; call this only once.
 */
void gok_spy_open()
{
	gboolean success = FALSE;

	gok_log_enter ();
	if (m_gokSpyOpen != TRUE)
	{
		eventQueue = g_queue_new();
		explicitunrefs = 0;
		explicitrefs = 0;
		implicitrefs = 0;
		m_gokSpyOpen = TRUE;
		m_ptheMouseButtonListener =NULL;
		m_ptheAppChangeListener = NULL;
		m_ptheWindowChangeListener = NULL;
		m_ptheWindowAccessible = NULL;
		m_ptheAccessibleWithEditableText = NULL;

		focusListener = SPI_createAccessibleEventListener (
			gok_spy_focus_listener, NULL);
		defunctnessListener = SPI_createAccessibleEventListener (
			gok_spy_defunctness_listener, NULL);
		mouseListener = SPI_createAccessibleDeviceListener (
			gok_spy_mouse_listener, NULL);
		modifierListener = SPI_createAccessibleEventListener (
			gok_spy_modifier_listener, NULL);
		windowActivateListener = SPI_createAccessibleEventListener (
			gok_spy_window_activate_listener, NULL);
		windowDeactivateListener = SPI_createAccessibleEventListener (
			gok_spy_window_deactivate_listener, NULL);

		success = SPI_registerGlobalEventListener (focusListener, "focus:"); 
		gok_log ("SPI_registerGlobalEventListener for focus events returned: %d" , success);
		/* success = SPI_registerGlobalEventListener (defunctnessListener, "object:state-changed:defunct");
		gok_log ("SPI_registerGlobalEventListener for defunct state events returned: %d" , success); */
		success = SPI_registerDeviceEventListener (mouseListener,
			SPI_BUTTON_PRESSED | SPI_BUTTON_RELEASED, NULL);
		gok_log ("SPI_registerDeviceEventListener for mouse events returned: %d" , success);

		success = SPI_registerGlobalEventListener (modifierListener,
			"keyboard:modifiers");
		gok_log ("SPI_registerGlobalEventListener for keyboard modifiers returned: %d", success);

		success = SPI_registerGlobalEventListener (
			windowActivateListener, "window:activate");
		gok_log ("SPI_registerGlobalEventListener for window:activate returned: %d", success);

		success = SPI_registerGlobalEventListener (
			windowDeactivateListener, "window:deactivate");
		gok_log ("SPI_registerGlobalEventListener for window:deactivate returned: %d", success);

	}
	gok_log_leave ();
}

/** 
 * gok_spy_close:
 *
 * Frees any allocated memory.
 */
void gok_spy_close(void)
{
	EventNode* en;
	gboolean result = FALSE;	
	
	gok_log_enter ();

	if (m_gokSpyOpen == TRUE)
	{
		m_gokSpyOpen = FALSE;
		
		if (focusListener != NULL) 
		{
			result = SPI_deregisterGlobalEventListenerAll(focusListener);
			gok_log("deregistering focus listener returned: %d",result);
		}
		if (defunctnessListener != NULL) 
		{
			result = SPI_deregisterGlobalEventListenerAll(defunctnessListener);
			gok_log("deregistering defunctness listener returned: %d",result);
		}
		if (mouseListener != NULL)
		{
			result = SPI_deregisterDeviceEventListener (mouseListener, NULL);
			gok_log("deregistering mouse listener returned: %d",result);
		}
		if (modifierListener != NULL)
		{
			result = SPI_deregisterGlobalEventListenerAll(
				modifierListener);
			gok_log("deregistering modifier listener returned: %d",
				result);
		}
	
		AccessibleEventListener_unref(focusListener);
		AccessibleEventListener_unref(defunctnessListener);
		AccessibleDeviceListener_unref(mouseListener);
		AccessibleEventListener_unref(modifierListener);

		if (m_ptheWindowAccessible != NULL) gok_spy_accessible_unref(m_ptheWindowAccessible);
		if (m_ptheAccessibleWithEditableText != NULL) gok_spy_accessible_unref(m_ptheAccessibleWithEditableText);
		
		gok_log("accessible reference balance is  [%d] (should be zero)",explicitrefs + implicitrefs - explicitunrefs);
		gok_log("explicit accessible references   [%d]",explicitrefs);
		gok_log("explicit accessible dereferences [%d]",explicitunrefs);
		gok_log("implicit accessible references   [%d]",implicitrefs);

		gok_log("eventQueue->length = %d", eventQueue->length);

		/* iterate over event queue and dereference accessibles */
		while (g_queue_is_empty(eventQueue) == FALSE)
		{
			en = (EventNode*)g_queue_pop_head(eventQueue);
			gok_spy_accessible_unref(en->paccessible);
			g_free(en);
		}
		g_queue_free(eventQueue);

	}
	
	gok_log_leave ();
}


/** 
* gok_spy_register_appchangelistener
* 
* @callback: the listener to register
**/ 
void gok_spy_register_appchangelistener(AppChangeListener* callback)
{
	gok_log_x("this function deprecated, use gok_spy_register_windowchangelistener instead");
	/* m_ptheAppChangeListener = callback; */
}

/** 
* gok_spy_deregister_appchangelistener
* 
* @callback: the listener to deregister
**/ 
void gok_spy_deregister_appchangelistener(AppChangeListener* callback)
{
	gok_log_x("this function deprecated, use gok_spy_deregister_windowchangelistener instead");
	/* m_ptheAppChangeListener = NULL; */
}

/** 
* gok_spy_register_windowchangelistener
* 
* @callback: the listener to register
**/ 
void gok_spy_register_windowchangelistener(AppChangeListener* callback)
{
	m_ptheWindowChangeListener = callback;
}

/** 
* gok_spy_deregister_windowchangelistener
* 
* @callback: the listener to deregister
**/ 
void gok_spy_deregister_windowchangelistener(AppChangeListener* callback)
{
	m_ptheWindowChangeListener = NULL;
}

/** 
* gok_spy_register_mousebuttonlistener
* 
* @callback: the listener to register
**/ 
void gok_spy_register_mousebuttonlistener(MouseButtonListener* callback)
{
	m_ptheMouseButtonListener = callback;
}

/** 
* gok_spy_deregister_mousebuttonlistener
* 
* @callback: the listener to deregister
**/ 
void gok_spy_deregister_mousebuttonlistener(MouseButtonListener* callback)
{
	m_ptheMouseButtonListener = NULL;
}

/** 
* gok_spy_get_list 
* 
* @paccessible: The parent accessible to the list
* @role: The role of the list members
*
* User must call gok_spy_free when finished with this list.
* 
* Returns: pointer to the list or NULL
**/ 
AccessibleNode* gok_spy_get_list( Accessible* paccessible, AccessibleRole role )
{
	AccessibleNode* plist = NULL;
	gok_log_enter();
	plist = gok_spy_create_list( paccessible, role );
	gok_log_leave();
	return plist;
}



/** 
* gok_spy_refresh 
* 
* @plist: Pointer to the list to refresh
*
* not implemented.
*
* Returns: pointer to the refreshed list
**/ 
AccessibleNode* gok_spy_refresh( AccessibleNode* plist)
{
	gok_log_enter();
	gok_log_x(" this function not implemented ");
	gok_log_leave();
	return NULL;
}

/**
* gok_spy_free
*
* Frees the memory used by the given list. This must be called for every list
* that is created.
*
* @pNode: Pointer to the list that you want freed.
*
* Returns: void
**/
void gok_spy_free (AccessibleNode* pNode)
{
	AccessibleNode* pNodeToFree;
	pNodeToFree = NULL;
	
	gok_log_enter();
	while (pNode != NULL)
	{
		pNodeToFree = pNode;
		pNode = pNode->pnext;
		
		SPI_freeString (pNodeToFree->pname);
		gok_spy_accessible_unref (pNodeToFree->paccessible);
		g_free (pNodeToFree);
	}
	gok_log_leave();
}

/** 
* gok_spy_get_accessibleWithEditableText
*
* accessor
**/
Accessible* gok_spy_get_accessibleWithEditableText()
{
	return m_ptheAccessibleWithEditableText;
}

/** 
* gok_spy_has_child
* 
* @accessible: Pointer to the accessible.
* @role: The role you are searching for. 
*
* If anything under this accessible* has the specified role, then returns true.
* 
* Returns: true if a child exists, false otherwise. 
**/ 
gboolean gok_spy_has_child (Accessible* accessible, AccessibleRole role) 
{
	gboolean bfound = FALSE;
	gok_log_enter();
	if (gok_spy_accessible_is_okay(accessible) == TRUE)
	{
		gok_spy_has_child_aux (accessible, role, GOKSPY_GUI_BREADTH, GOKSPY_GUI_DEPTH, 0, &bfound);
	}
	else
	{
		gok_log("accessible is NULL");
	}
	gok_log_leave();
	return bfound;
}


/** 
* gok_spy_check_queues
*
* this should be called during idle time. 
**/ 
void gok_spy_check_queues(void)
{
	EventNode* en = NULL;
	char *s;

	gok_log_enter();
	while (g_queue_is_empty(eventQueue) == FALSE)
	{
		en = (EventNode*)g_queue_pop_head(eventQueue);

		gok_log ("eventQueue->length = %d", eventQueue->length);

		if (en->paccessible != NULL)
		{
			s = Accessible_getName (en->paccessible);
			if (s == NULL)
			{
				gok_log ("%#x", en->paccessible);
			}
			else
			{
				gok_log ("%#x %s", en->paccessible, s);
				SPI_freeString (s);
			}
		}

		switch (en->type)
		{
		case GOKSPY_FOCUS_EVENT:
			gok_log("GOKSPY_FOCUS_EVENT");
			gok_spy_process_accessible(en->paccessible);
			break;
		/*
		 *  TODO: revisit this after CSUN
		 *
		case GOKSPY_DEFUNCT_EVENT:
			gok_log("GOKSPY_DEFUNCT_EVENT");
			gok_spy_accessible_unref (m_ptheWindowAccessible);
			m_ptheWindowAccessible = NULL;
			((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible);
			break;
		case GOKSPY_WINDOW_ACTIVATE_EVENT:
			gok_log("GOKSPY_WINDOW_ACTIVATE_EVENT");
			gok_spy_accessible_unref (m_ptheWindowAccessible);
			m_ptheWindowAccessible = en->paccessible;
			gok_spy_accessible_ref (m_ptheWindowAccessible);
			((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible);
			break;
		*/
		case GOKSPY_WINDOW_DEACTIVATE_EVENT:
			gok_log("GOKSPY_WINDOW_DEACTIVATE_EVENT");

			/* TODO: Determine if this workaround is necessary
			 *       or if at-spi should be modified to
			 *       not send out of order activate/deactivate
			 *       events
			 *
			 * The test below is needed to workaround the case
			 * that a window:deactivate event may arrive
			 * after a window:activate has been emitted by
			 * another window. The test ignores window:deactivate
			 * events unless they come from what gok thinks
			 * is the current window.
			 */
			if (en->paccessible == m_ptheWindowAccessible) {
				gok_spy_accessible_unref (m_ptheWindowAccessible);
				m_ptheWindowAccessible = NULL;
				((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible);
			}
			break;
		default:
			gok_log_x("unknown event type in internal gok event queue!");
			break;
		}
		gok_spy_accessible_unref (en->paccessible);
		g_free(en);
	}
	gok_log_leave();
}

/** 
* gok_spy_check_window:
*
* @role: the role to check.
*
* This function decides if the role corresponds to something that looks like a window to the user.
*
* Returns: boolean success. 
*/ 
gboolean gok_spy_check_window(AccessibleRole role)
{
	if ((role == SPI_ROLE_WINDOW) ||
	    (role == SPI_ROLE_DIALOG) ||
	    (role == SPI_ROLE_FILE_CHOOSER) ||
	    (role == SPI_ROLE_FRAME) ||	
	    (role == SPI_ROLE_DESKTOP_FRAME))
	{
		return TRUE;
	}
	return FALSE;
}

/** 
* gok_spy_worth_searching:
*
* @accessible: the accessible (possibly a parent) to examine for worthiness.
*
* This function decides if the accessible might have a subtree worth searching.
*
* Returns: boolean success. 
*/ 
gboolean gok_spy_worth_searching (Accessible* accessible)
{
	gboolean worthiness;
	AccessibleStateSet* ass = NULL;
	worthiness = TRUE;
	gok_log_enter();
	ass = Accessible_getStateSet(accessible);
	if (ass != NULL)
	{
		if ( AccessibleStateSet_contains( ass, SPI_STATE_MANAGES_DESCENDANTS ) == TRUE)
		{
			worthiness = FALSE;
		}
		AccessibleStateSet_unref(ass);
	}
	gok_log_leave();
	return worthiness;
}

	
/** 
 * gok_spy_has_child_aux:
 * @accessible: Pointer to the accessible.
 * @role: The role you are searching for.
 * @children: the breadth of the gui tree to search (for optimization).
 * @depth: how deep to look in the gui tree (for optimization).
 * @cdepth: recursive variable that keeps track of current depth
 * (pass in zero).
 * @found: out variable to hold the success boolean.
 * 
 * If anything under this accessible* has the specified role, then
 * returns true.
 *
 * Returns: true if a child exists, false otherwise.
 */
gboolean gok_spy_has_child_aux (Accessible* accessible, AccessibleRole role, gint children, gint depth, gint cdepth, gboolean* found) 
{ 
    Accessible* pAccessibleChild; 
	int child_count;
    int i; 

    if (accessible == NULL) return *found; 
	if (cdepth++ == depth) return *found;
	
	if (gok_spy_worth_searching(accessible) == FALSE) return *found;
	
    child_count = Accessible_getChildCount (accessible);
    
    if (child_count < 0)
    {
    	gok_log_x("exception occurred when trying to get the child count");
    	return *found;
    } 
    
    if (child_count > children) child_count = children;
    
    /* look through all the children of the given node */ 
	for (i=0; i<child_count; i++) 
	{ 
		pAccessibleChild = Accessible_getChildAtIndex (accessible, i);
		gok_spy_accessible_implicit_ref(pAccessibleChild); 
		if (pAccessibleChild != NULL) 
		{ 
			/* does this child have the role we're looking for? */ 
		    if (Accessible_getRole (pAccessibleChild) == role) 
		    {
		    	gok_log ("gok_spy_has_child_aux: found child at depth %d", cdepth);
		    	gok_spy_accessible_unref(pAccessibleChild);
		    	*found = TRUE;
	    		return *found;
		    } 
		    else /* search through the children of this child */ 
		    { 
		    	if (*found == TRUE)
		    	{
			    	gok_spy_accessible_unref(pAccessibleChild);
		    		return *found;
		    	}
		    	gok_spy_has_child_aux (pAccessibleChild, role, children, depth, cdepth, found);
		    	gok_spy_accessible_unref(pAccessibleChild);
		    } 
		} 
	}
	
	return *found;
} 

/**
 * gok_spy_find_children:
 * @pNode: Before execution of this function it is the node that you
 * would like to search for children under.  After execution of this
 * function pNode->pnext is the start of the list of children that
 * satisfy the role requested.  Please note that this function may
 * also modify pNode->paccessible.
 * @role: The role you are searching for.
 *
 * Finds the children of the given node that have the given role.
 * These children are added to the list of children on the node,
 * beginning at pNode->pnext.  Please note that this function may also
 * modify pNode->paccessible.  This is a recursive function.
 */
void gok_spy_find_children (AccessibleNode* pNode, AccessibleRole role) 
{ 
	Accessible* pAccessibleChild;
	Accessible* pAccessibleParent;
	long child_count;
	long i;

	g_assert(pNode != NULL);
	g_assert(pNode->paccessible != NULL); 

	pAccessibleParent = pNode->paccessible; 

	if (gok_spy_worth_searching(pAccessibleParent) == FALSE) return;

	child_count = Accessible_getChildCount (pAccessibleParent);

	if (child_count < 0)
	{
		gok_log_x("exception occurred trying to get the child count");
	}

	/* look through all the children of the given node */
	for (i=0; i<child_count; i++)
	{
		pAccessibleChild
			= Accessible_getChildAtIndex (pAccessibleParent, i);
		gok_spy_accessible_implicit_ref (pAccessibleChild);

		if (pAccessibleChild != NULL)
		{
			/* does this child have the role we're looking for? */

			/*
			if (((role == SPI_ROLE_MENU_ITEM) && 
			    ((Accessible_getRole (pAccessibleChild) ==  SPI_ROLE_MENU_ITEM) ||
			     (Accessible_getRole (pAccessibleChild) ==  SPI_ROLE_CHECK_MENU_ITEM) ||
			     (Accessible_getRole (pAccessibleChild) ==  SPI_ROLE_MENU))) ||
			*/
			if ((role == SPI_ROLE_MENU_ITEM)  /* always show a menu item */
			    || (Accessible_getRole (pAccessibleChild) == role))
			{
				gok_spy_append_node (pNode, pAccessibleChild);
			}
			else /* search through the children of this child */
			{
				pNode->paccessible = pAccessibleChild;
				gok_spy_find_children (pNode, role);
			}

		gok_spy_accessible_unref (pAccessibleChild);

		}
	}
}
 
/**
 * gok_spy_create_list:
 * @anAccessible: Pointer to parent accessible object.
 * @anAccessibleRole: The role you are searching for.
 *
 * Creates a list of accessible pointers of the given role.
 *
 * Returns: First node in the list of children. Will be NULL if no
 * children found.
 */
AccessibleNode* gok_spy_create_list (Accessible *anAccessible,
                                     AccessibleRole anAccessibleRole)
{
	AccessibleNode nodeRoot;

	gok_log_enter();

	g_assert(anAccessible != NULL);

	nodeRoot.paccessible = anAccessible;
	nodeRoot.pnext = NULL;
	nodeRoot.pname = NULL;

	gok_spy_find_children (&nodeRoot, anAccessibleRole);

	gok_log_leave();
	return nodeRoot.pnext;
}

/**
 * gok_spy_append_node:
 * @pNode: An existing list of AccessibleNode to append to, must be
 * non-NULL.
 * @pAccessible: The Accessible to append to the list.
 *
 * Creates a new AccessibleNode for pAccessible and appends it to the
 * existing list of AccessibleNode at pNode.  Please note that this
 * function calls gok_spy_accessible_ref (pAccessible) when it
 * attaches it to the AccessibleNode created for it.  Please also note
 * that the list must exist already and pNode must be non-NULL.
 *
 * Returns: the address of the AccessibleNode created for pAccessible
 * or NULL if problems occur.
 */
AccessibleNode* gok_spy_append_node (AccessibleNode* pNode,
                                     Accessible* pAccessible)
{
	AccessibleNode* pLastNode = NULL;
	AccessibleNode* pNewNode = NULL;
	char* pName = NULL;

	g_assert (pNode != NULL);

	gok_log_enter();

	pName = Accessible_getName (pAccessible);

	/* if the name is empty then ignore this child */ 
	if ( (pName !=NULL) && (strlen (pName) != 0) )
	{
		/* create a new node */
		pNewNode = (AccessibleNode*)g_malloc(sizeof(AccessibleNode));

		if (pNewNode != NULL)
		{
			pNewNode->paccessible = pAccessible;
			gok_spy_accessible_ref (pAccessible);

			pNewNode->pname = pName;
			gok_log("pNewNode->pname = %s",pNewNode->pname);
			pNewNode->pnext = NULL;

			pLastNode = pNode;
			while (pLastNode->pnext != NULL)
			{
				pLastNode = pLastNode->pnext;
			}
			pLastNode->pnext = pNewNode;
		}
		else
		{
			SPI_freeString (pName);
		}
	}
	else if (pName != NULL)
	{
		SPI_freeString (pName);
	}

	gok_log_leave();

	return pNewNode;
}

/** 
* gok_spy_focus_listener 
*
* callback for focus events in the at-spi 
**/ 
void gok_spy_focus_listener (const AccessibleEvent *event, void *user_data) 
{ 
	EventNode* en; 
	gok_log_enter();
	gok_spy_accessible_ref(event->source);
	
	en = (EventNode*)g_malloc(sizeof(EventNode));
	en->paccessible = event->source;
	en->type = GOKSPY_FOCUS_EVENT;
    g_queue_push_tail( eventQueue, en );

	gok_spy_add_idle_handler ();
    gok_log_leave();
}

/** 
* gok_spy_defunctness_listener 
*
* callback for when objects go defunct in the at-spi 
**/ 
void gok_spy_defunctness_listener (const AccessibleEvent *event, void *user_data) 
{
	EventNode* en; 
	gok_log_enter();

    if (event->source == m_ptheWindowAccessible)
    {
		en = (EventNode*)g_malloc(sizeof(EventNode));
		en->paccessible = NULL;/*event->source;*/
		en->type = GOKSPY_DEFUNCT_EVENT;
		g_queue_push_tail ( eventQueue, en );
		gok_spy_add_idle_handler ();
    	gok_log_x("our window is gone!");
    }

    gok_log_leave();
}

static void
gok_spy_window_activate_listener(const AccessibleEvent *event,
                                 void *user_data)
{
	EventNode *en;
	gok_log_enter ();

	gok_spy_accessible_ref (event->source);
	en = (EventNode*)g_malloc (sizeof (EventNode));
	en->paccessible = event->source;
	en->type = GOKSPY_WINDOW_ACTIVATE_EVENT;
	g_queue_push_tail (eventQueue, en);
	gok_spy_add_idle_handler ();

	gok_log_leave ();
}

static void
gok_spy_window_deactivate_listener(const AccessibleEvent *event,
                                   void *user_data)
{
	EventNode *en;
	gok_log_enter ();

	gok_spy_accessible_ref (event->source);
	en = (EventNode*)g_malloc (sizeof (EventNode));
	en->paccessible = event->source;
	en->type = GOKSPY_WINDOW_DEACTIVATE_EVENT;
	g_queue_push_tail (eventQueue, en);
	gok_spy_add_idle_handler ();

	gok_log_leave ();
}

/** 
* gok_spy_process_accessible 
* 
* @accessible: pointer to the accessible to process, assumed to have no references (see Accessible_ref in cspi)
* 
* side effects: sets global for foreground accessible container.  calls registered listener when
* the foreground window has changed.
**/ 
void gok_spy_process_accessible (Accessible* accessible) 
{ 
    Accessible* pnewWindowAccessible;
    Accessible* ptempAccessible;

    gok_log_enter();
    g_assert(accessible != NULL);

    pnewWindowAccessible = NULL;
	ptempAccessible = NULL;
	
	/* start by assuming we want to keep this accessible around */
	gok_spy_accessible_ref(accessible); 

	/*
	 *  check to see if focus is on an editable text object 
	 */
	gok_log("Checking for editable text...");
	gok_log("m_ptheAccessibleWithEditableText [%d]",m_ptheAccessibleWithEditableText);
	gok_log("                      accessible [%d]",accessible);
	if (Accessible_isEditableText(accessible) == TRUE)
	{
		gok_log("this thing has an editable text interface");
		if (accessible != m_ptheAccessibleWithEditableText)
		{
			gok_spy_accessible_ref(accessible); 
			gok_spy_accessible_unref(m_ptheAccessibleWithEditableText);
			m_ptheAccessibleWithEditableText = accessible;
			gok_log("found a (new) editable text interface"); 
		}
	}
	else
	{
		gok_log("no editable text interface on this thing");
		/* we might not want to do this - needs user testing */
		if (m_ptheAccessibleWithEditableText != NULL)
		{
			gok_spy_accessible_unref(m_ptheAccessibleWithEditableText);
		}
		m_ptheAccessibleWithEditableText = NULL;
	}

	/* bug with gedit toolbar has to do with needing to carry on up the parent chain 
	   if we do this we might lose context - need to balance pros and cons...
	*/	

	/*
	 *  find the container for the focussed accessible thing 
	 */
	
	gok_log("Looking for container...");
	gok_log("accessible [%d]",accessible);
	pnewWindowAccessible = accessible;
	while (gok_spy_check_window(Accessible_getRole(pnewWindowAccessible)) != TRUE)
	 /*|| (ptempAccessible != NULL))*/
	
	{
		gok_log("*");
    	ptempAccessible = Accessible_getParent(pnewWindowAccessible);
	    gok_spy_accessible_implicit_ref(ptempAccessible);
	    if (ptempAccessible == NULL)
	    {
	    	break;
	    } 
    	gok_spy_accessible_unref(pnewWindowAccessible);
   	 pnewWindowAccessible = ptempAccessible;
    }
	 
	/*
     * check to see if we have a new window 
	 */
	gok_log("Comparing this container with existing...");
    if (pnewWindowAccessible != m_ptheWindowAccessible)
    {
    	gok_log("different.");
		gok_spy_accessible_unref (m_ptheWindowAccessible);
    	m_ptheWindowAccessible = pnewWindowAccessible;
		((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible); 
	}
	else
	{
    	gok_log("same.");
		/* we have one ref too many (this isn't a different window) */
		gok_spy_accessible_unref(pnewWindowAccessible);
	}
	gok_log_leave();
} 

/**
 * gok_spy_button_is_switch_trigger:
 *
 * @button: an int representing logical mouse button number to query.
 *  
 * Returns: TRUE if the button is used by GOK as a switch, FALSE otherwise.
 **/
static gboolean
gok_spy_button_is_switch_trigger (int button)
{
        if (!strcmp (gok_data_get_name_accessmethod (), "directselection"))
	{
		if (gok_main_window_contains_pointer () == TRUE)
			return TRUE;
		else
			return FALSE;	
	}
	else if (gok_data_get_drive_corepointer() == TRUE) {
		gok_log("gok_data_get_drive_corepointer() is TRUE");
		return FALSE;
	}
	else {
		gok_log("gok_data_get_drive_corepointer() is FALSE");
		return gok_scanner_current_state_uses_core_mouse_button(button);
	}
}

/** 
* gok_spy_mouse_listener 
* 
* callback for mouse events in the at-spi 
* 
* @event: event structure
* @user_data: not used here
*
*/ 
SPIBoolean 
gok_spy_mouse_listener (const AccessibleDeviceEvent *event, void *user_data) 
{ 
	gint button = 0;
	gint state = 0;
	gboolean is_switch_trigger;
	gok_log_enter();

	/* 
	 * must check now and save for later, since a branch action 
	 * may change this by altering the window size, etc.
	 */
	button = event->keycode;
	is_switch_trigger = gok_spy_button_is_switch_trigger (button);

	if (m_ptheMouseButtonListener != NULL)
	{
		gok_log("mouse event %ld %x %x", 
			event->keyID, (unsigned) event->type, 
			(unsigned) event->modifiers);

			if (event->type == SPI_BUTTON_PRESSED)
			{
				state = 1;
			}
			((MouseButtonListenerCB)(m_ptheMouseButtonListener))(button, 
									     state, 
									     (long) event->modifiers,
									     event->timestamp);
	}
	gok_log_leave();
        return is_switch_trigger;
}

/** 
 * gok_spy_modifier_listener:
 * @event: the keyboard modifier event.
 * @user_data: not used.
 *
 * Callback for keyboard modifier events from the at-spi.
 */
void gok_spy_modifier_listener(const AccessibleEvent *event, void *user_data)
{
	const struct gok_modifier_keymask *m;

	gok_log_enter();
	gok_log ("event->detail2 = %d", event->detail2);

	for (m = gok_modifier_keymasks; m->name != NULL; m++)
	{
		if (event->detail2 & m->keymask)
		{
			gok_log ("%s on", m->name);
			gok_modifier_set_state (m->name, MODIFIER_STATE_ON);
		}
		else
		{
			gok_log ("%s off", m->name);
			gok_modifier_set_state (m->name, MODIFIER_STATE_OFF);
		}
	}

	gok_log_leave();
}

static gboolean
gok_spy_idle_handler (gpointer data)
{
	spy_handler_id = 0;
	gok_spy_check_queues ();
	return FALSE;
}

/** 
 * gok_spy_add_idle_handler:
 *
 * Adds the idle handler to deal with spy events
 */
void gok_spy_add_idle_handler (void)
{
	if (spy_handler_id == 0) {
		spy_handler_id = g_idle_add (gok_spy_idle_handler, NULL);
	}
}

/** 
 * gok_spy_remove_idle_handler:
 *
 * Removes the idle handler to deal with spy events
 */
void gok_spy_remove_idle_handler (void)
{
	if (spy_handler_id != 0) {
		gtk_idle_remove (spy_handler_id);
	}    
}


/*
gboolean gok_spy_is_desktop(Accessible* pAccessible)
{
	gint i;
	gint ndesktops;
	Accessible* desktop;

	gok_log_enter();

	ndesktops = SPI_getDesktopCount();
	for (i = 0; i < ndesktops; i++)
	{
		desktop = SPI_getDesktop(i);
		gok_spy_accessible_implicit_ref(desktop);
		if (pAccessible == desktop)
		{
			gok_spy_accessible_unref(desktop);
			gok_log_leave();
			return TRUE;
		}
	}
	gok_spy_accessible_unref(desktop);
	return FALSE;
}

gboolean gok_spy_accessible_is_desktopChild(Accessible* accessible)
{
	gint i;
	gint j;
	gint ndesktops;
	gint nchildren;
	Accessible* desktop;
	Accessible* child;
	gboolean returnCode = FALSE;
	
	gok_log_enter();

	ndesktops = SPI_getDesktopCount();
	for (i = 0; i < ndesktops; i++)
	{
		desktop = SPI_getDesktop(i);
		gok_spy_accessible_implicit_ref(desktop);
		if (desktop == NULL)
		{
			gok_log("desktop disappeared!?");
			break;
		}
		nchildren = Accessible_getChildCount(desktop);
		for (j = 0; j < nchildren; j++)
		{
			child = Accessible_getChildAtIndex(desktop, j);
			gok_spy_accessible_implicit_ref(child);
			if (child != NULL)
			{
				if (child == accessible)
				{
					gok_spy_accessible_unref(child);
					returnCode = TRUE;
					break;
				}
			}
		}
		if (returnCode == TRUE)
		{
			gok_spy_accessible_unref(desktop);
			break;
		}
		gok_spy_accessible_unref(desktop);
	}
	return returnCode;
	gok_log_leave();
}
*/
/* 
* gok_spy_get_applications 
* 
* User must call gok_spy_free when finished with this list.
* 
* Returns: pointer to the list or NULL
*/
/*
AccessibleNode* gok_spy_get_applications( )
{
	gint i;
	gint j;
	gint ndesktops;
	gint nchildren;
	Accessible* desktop;
	Accessible* child;
	AccessibleNode* plist;

	gok_log_enter();

	plist = NULL;
	ndesktops = SPI_getDesktopCount();
	for (i = 0; i < ndesktops; i++)
	{
		desktop = SPI_getDesktop(i);
		gok_spy_accessible_implicit_ref(desktop);
		if (desktop == NULL)
		{
			gok_log("desktop disappeared!?");
			break;
		}
		nchildren = Accessible_getChildCount(desktop);
		for (j = 0; j < nchildren; j++)
		{
			child = Accessible_getChildAtIndex(desktop, j);
			gok_spy_accessible_implicit_ref(child);
			if (child != NULL)
			{
				plist = gok_spy_append_node(plist, child);
				if (plist == NULL)
				{
					gok_spy_accessible_unref(child);
				}
			}
		}
		gok_spy_accessible_unref(desktop);
	}
	gok_log_leave();
	return plist;
}
*/

