/*
* gok-spy.c
*
* Copyright 2002 Sun Microsystems, Inc.,
* Copyright 2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/*
 * utility for getting accessible application UI info
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <libspi/keymasks.h>
#ifndef  SPI_KEYMASK_MOD4
  #define  SPI_KEYMASK_MOD4 Mod4Mask
#endif
#include "main.h"
#include "gok-spy.h"
#include "gok-log.h"
#include "gok-modifier.h"
#include "gok-data.h"
#include "gok-modifier-keymasks.h"
#include "gok-gconf-keys.h"

/*
 * NOTE: we can tweak the efficiency/accuracy of the
 * gui search by adjusting using the spy/gui_search_depth and 
 * spy/gui_search_breadth gconf keys.
 */

/*
 * NOTES
 *
 * the at-spi currently requires Accessible_ref and Accessible_unref
 * calls to keep accessible pointers valid.  
 *
 * gok-spy assumes only onelistener is registered at a time.
 *
 */

/*
 * private prototypes
 */
static void gok_spy_focus_listener (const AccessibleEvent *event,
                                    void *user_data);

static void gok_spy_defunctness_listener (const AccessibleEvent *event,
                                    void *user_data);

static void gok_spy_process_accessible(Accessible* accessible);

static SPIBoolean gok_spy_mouse_listener (const AccessibleDeviceEvent *event,
					  void *user_data);

SPIBoolean gok_spy_key_listener (const AccessibleKeystroke *key,
                                 void *user_data);

AccessibleNode* gok_spy_create_list (Accessible *anAccessible,
				     GokSpySearchType type,
                                     AccessibleRole anAccessiblerole);

static void gok_spy_find_children (AccessibleNode* pNode, GokSpySearchType type, AccessibleRole Role);

static gboolean gok_spy_has_child_aux (Accessible* accessible,
				       GokSpySearchType search_type,
                                       AccessibleRole role,
				       gint children,
                                       gint depth, gint cdepth,
                                       gboolean* found);

static AccessibleNode*  gok_spy_append_node (AccessibleNode* pnode,
                                             Accessible* pAccessible);

static void gok_spy_modifier_listener(const AccessibleEvent *event,
                                      void *user_data);

static gboolean gok_spy_worth_searching (Accessible* accessible, AccessibleRole role);

static void gok_spy_window_activate_listener (const AccessibleEvent *event,
                                              void *user_data);

static void gok_spy_window_deactivate_listener (const AccessibleEvent *event,
                                                void *user_data);
												
static gboolean gok_spy_is_visible (Accessible *accessible);
												

/*
 * gok-spy variables
 */

static GQueue* eventQueue;
static void* m_ptheAppChangeListener;
static void* m_ptheWindowChangeListener;
static void* m_ptheMouseButtonListener;

/* pointer to the current window */
static Accessible* m_ptheWindowAccessible;

/* pointer to the current accessible with editable text */
static Accessible* m_ptheAccessibleWithText;

static AccessibleEventListener* focusListener;
static AccessibleEventListener* defunctnessListener;
static AccessibleDeviceListener* mouseListener;
static AccessibleEventListener* modifierListener;
static AccessibleEventListener* windowActivateListener;
static AccessibleEventListener* windowDeactivateListener;
static gboolean m_gokSpyOpen = FALSE;
static gint explicitrefs;
static gint explicitunrefs;
static gint implicitrefs;
static gulong keyboardmods;
static gboolean spyshutdown = FALSE;/* this global could be removed if we have a
						way of removing our idle handler in gok_spy_stop */


/* 
 * for debugging
 */
void gok_spy_accessible_ref(Accessible* accessible)
{
	char *s;

	gok_log_enter ();

	if (accessible != NULL)
	{
		explicitrefs++;
		Accessible_ref (accessible);
#if defined(ENABLE_LOGGING_NORMAL)
		s = Accessible_getName (accessible);
		if (s == NULL)
		{
			gok_log ("%#x", accessible);
		}
		else
		{
			gok_log ("%#x %s", accessible, s);
			SPI_freeString (s);
		}
#endif
	}
	else
	{
		gok_log ("NULL");
	}

	gok_log_leave ();
}
void gok_spy_accessible_implicit_ref(Accessible* accessible)
{
	char *s;

	gok_log_enter ();

	if (accessible != NULL)
	{
		implicitrefs++;
#if defined(ENABLE_LOGGING_NORMAL)
		s = Accessible_getName (accessible);
		if (s == NULL)
		{
			gok_log ("%#x", accessible);
		}
		else
		{
			gok_log ("%#x %s", accessible, s);
			SPI_freeString (s);
		}
#endif
	}
	else
	{
		gok_log ("NULL");
	}

	gok_log_leave ();
}
void gok_spy_accessible_unref(Accessible* accessible)
{
	char *s;

	gok_log_enter ();

	if (accessible != NULL)
	{
#if defined(ENABLE_LOGGING_NORMAL)
		s = Accessible_getName (accessible);
		if (s == NULL)
		{
			gok_log ("%#x", accessible);
		}
		else
		{
			gok_log ("%#x %s", accessible, s);
			SPI_freeString (s);
		}
#endif
		explicitunrefs++;
		Accessible_unref (accessible);
	}
	else
	{
		gok_log ("NULL");
	}

	gok_log_leave ();
}

gboolean gok_spy_accessible_is_okay(Accessible* accessible)
{
	gboolean returncode = FALSE;
	gok_log_enter();
	if (accessible != NULL) 
	{
		/* FIXME: we should have a better way of doing this! */
		returncode = cspi_accessible_is_a (accessible, "IDL:Accessibility/Accessible:1.0");
		/*
		if (Accessible_isApplication(accessible) ||
		    Accessible_isComponent(accessible))
		{
			returncode = TRUE;
		}
		*/
			/*
	    	if (gok_spy_accessible_is_desktopChild(accessible) == FALSE)
	    	{
				accessible = NULL;
			}
			else
			{
				returncode = TRUE;
			}
			*/
	}
	gok_log("returning: %d",returncode);
	gok_log_leave();
	return returncode;
}

static gboolean
gok_spy_idle_handler (gpointer data)
{
	return gok_spy_check_queues ();
}


/**
 * gok_spy_open:
 *
 * Intializes gok spy.
 * Note: user must initialize the SPI prior to this call; call this only once.
 */
void gok_spy_open()
{
	gboolean success = FALSE;

	gok_log_enter ();
	if (m_gokSpyOpen != TRUE)
	{
		eventQueue = g_queue_new();
		explicitunrefs = 0;
		explicitrefs = 0;
		implicitrefs = 0;
		m_gokSpyOpen = TRUE;
		m_ptheMouseButtonListener =NULL;
		m_ptheAppChangeListener = NULL;
		m_ptheWindowChangeListener = NULL;
		m_ptheWindowAccessible = NULL;
		m_ptheAccessibleWithText = NULL;

		focusListener = SPI_createAccessibleEventListener (
			gok_spy_focus_listener, NULL);
		defunctnessListener = SPI_createAccessibleEventListener (
			gok_spy_defunctness_listener, NULL);
		mouseListener = SPI_createAccessibleDeviceListener (
			gok_spy_mouse_listener, NULL);
		modifierListener = SPI_createAccessibleEventListener (
			gok_spy_modifier_listener, NULL);
		windowActivateListener = SPI_createAccessibleEventListener (
			gok_spy_window_activate_listener, NULL);
		windowDeactivateListener = SPI_createAccessibleEventListener (
			gok_spy_window_deactivate_listener, NULL);

		success = SPI_registerGlobalEventListener (focusListener, "focus:"); 
		gok_log ("SPI_registerGlobalEventListener for focus events returned: %d" , success);
		success = SPI_registerGlobalEventListener (defunctnessListener, "object:state-changed:defunct");
		gok_log ("SPI_registerGlobalEventListener for defunct state events returned: %d" , success); 
		success = SPI_registerDeviceEventListener (mouseListener,
			SPI_BUTTON_PRESSED | SPI_BUTTON_RELEASED, NULL);
		gok_log ("SPI_registerDeviceEventListener for mouse events returned: %d" , success);

		success = SPI_registerGlobalEventListener (modifierListener,
			"keyboard:modifiers");
		gok_log ("SPI_registerGlobalEventListener for keyboard modifiers returned: %d", success);

		success = SPI_registerGlobalEventListener (
			windowActivateListener, "window:activate");
		gok_log ("SPI_registerGlobalEventListener for window:activate returned: %d", success);

		success = SPI_registerGlobalEventListener (
			windowDeactivateListener, "window:deactivate");
		gok_log ("SPI_registerGlobalEventListener for window:deactivate returned: %d", success);

		success = SPI_registerGlobalEventListener (
			windowDeactivateListener, "window:destroy");
		gok_log ("SPI_registerGlobalEventListener for window:destroy returned: %d", success);

	}
	gok_log_leave ();
}

/** 
 * gok_spy_stop:
 *
 * Call this to unhook all the gok_spy internal listeners. Do this as as the
 * first thing when shutting down the application.
 */
void gok_spy_stop (void)
{
	gboolean result = FALSE;	
	
	spyshutdown = TRUE;
	
	if (focusListener != NULL) 
	{
		result = SPI_deregisterGlobalEventListenerAll(focusListener);
		gok_log("deregistering focus listener returned: %d",result);
		AccessibleEventListener_unref(focusListener);
		focusListener = NULL;
	}
	if (defunctnessListener != NULL) 
	{
		result = SPI_deregisterGlobalEventListenerAll(defunctnessListener);
		gok_log("deregistering defunctness listener returned: %d",result);
		AccessibleEventListener_unref(defunctnessListener);
		defunctnessListener = NULL;
	}
	if (mouseListener != NULL)
	{
		result = SPI_deregisterDeviceEventListener (mouseListener, NULL);
		gok_log("deregistering mouse listener returned: %d",result);
		AccessibleDeviceListener_unref(mouseListener);
		mouseListener = NULL;
	}
	if (modifierListener != NULL)
	{
		result = SPI_deregisterGlobalEventListenerAll (modifierListener);
		gok_log("deregistering modifier listener returned: %d",	result);
		AccessibleEventListener_unref(modifierListener);
		modifierListener = NULL;
	}
	if (windowActivateListener != NULL)
	{
		result = SPI_deregisterGlobalEventListenerAll (windowActivateListener);
		gok_log("deregistering window activate listener returned: %d",result);
		AccessibleDeviceListener_unref(windowActivateListener);
		windowActivateListener = NULL;
	}
	if (windowDeactivateListener != NULL)
	{
		result= SPI_deregisterGlobalEventListenerAll (windowDeactivateListener);
		gok_log("deregistering window deactivate listener returned: %d",result);
		AccessibleEventListener_unref(windowDeactivateListener );
		windowDeactivateListener = NULL;
	}
}
	
/** 
 * gok_spy_close:
 *
 * Frees any allocated memory.
 */
void gok_spy_close(void)
{
	EventNode* en;
	
	gok_log_enter ();

	if (m_gokSpyOpen == TRUE)
	{
		m_gokSpyOpen = FALSE;
		
		gok_spy_stop ();
		
		if (m_ptheWindowAccessible != NULL) gok_spy_accessible_unref(m_ptheWindowAccessible);
		if (m_ptheAccessibleWithText != NULL) gok_spy_accessible_unref(m_ptheAccessibleWithText);

		gok_log("accessible reference balance is  [%d] (should be zero)",explicitrefs + implicitrefs - explicitunrefs);
		gok_log("explicit accessible references   [%d]",explicitrefs);
		gok_log("explicit accessible dereferences [%d]",explicitunrefs);
		gok_log("implicit accessible references   [%d]",implicitrefs);

		gok_log("eventQueue->length = %d", eventQueue->length);

		/* iterate over event queue and dereference accessibles */
		while (g_queue_is_empty(eventQueue) == FALSE)
		{
			en = (EventNode*)g_queue_pop_head(eventQueue);
			gok_spy_accessible_unref(en->paccessible);
			g_free(en);
		}
		g_queue_free(eventQueue);
		eventQueue = NULL;
	}
	
	gok_log_leave ();
}


/** 
* gok_spy_register_appchangelistener
* 
* @callback: the listener to register
**/ 
void gok_spy_register_appchangelistener(AppChangeListener* callback)
{
	gok_log_x("this function deprecated, use gok_spy_register_windowchangelistener instead");
	/* m_ptheAppChangeListener = callback; */
}

/** 
* gok_spy_deregister_appchangelistener
* 
* @callback: the listener to deregister
**/ 
void gok_spy_deregister_appchangelistener(AppChangeListener* callback)
{
	gok_log_x("this function deprecated, use gok_spy_deregister_windowchangelistener instead");
	/* m_ptheAppChangeListener = NULL; */
}

/** 
* gok_spy_register_windowchangelistener
* 
* @callback: the listener to register
**/ 
void gok_spy_register_windowchangelistener(AppChangeListener* callback)
{
	m_ptheWindowChangeListener = callback;
}

/** 
* gok_spy_deregister_windowchangelistener
* 
* @callback: the listener to deregister
**/ 
void gok_spy_deregister_windowchangelistener(AppChangeListener* callback)
{
	m_ptheWindowChangeListener = NULL;
}

/** 
* gok_spy_register_mousebuttonlistener
* 
* @callback: the listener to register
**/ 
void gok_spy_register_mousebuttonlistener(MouseButtonListener* callback)
{
	m_ptheMouseButtonListener = callback;
}

/** 
* gok_spy_deregister_mousebuttonlistener
* 
* @callback: the listener to deregister
**/ 
void gok_spy_deregister_mousebuttonlistener(MouseButtonListener* callback)
{
	m_ptheMouseButtonListener = NULL;
}

/** 
* gok_spy_get_list 
* 
* @paccessible: The parent accessible to the list
* @type: The type of list to return, i.e. the search criteria
* @role: The role of the list members
*
* User must call gok_spy_free when finished with this list.
* 
* Returns: pointer to the list or NULL
**/ 
AccessibleNode* gok_spy_get_list( Accessible* paccessible, GokSpySearchType type, AccessibleRole role )
{
	AccessibleNode* plist = NULL;
	gok_log_enter();
	plist = gok_spy_create_list( paccessible, type, role );
	gok_log_leave();
	return plist;
}

/** 
* gok_spy_refresh 
* 
* @plist: Pointer to the list to refresh
*
* not implemented.
*
* Returns: pointer to the refreshed list
**/ 
AccessibleNode* gok_spy_refresh( AccessibleNode* plist)
{
	gok_log_enter();
	gok_log_x(" this function not implemented ");
	gok_log_leave();
	return NULL;
}

/**
* gok_spy_free
*
* Frees the memory used by the given list. This must be called for every list
* that is created.
*
* @pNode: Pointer to the list that you want freed.
*
* Returns: void
**/
void gok_spy_free (AccessibleNode* pNode)
{
	AccessibleNode* pNodeToFree;
	pNodeToFree = NULL;
	
	gok_log_enter();
	while (pNode != NULL)
	{
		pNodeToFree = pNode;
		pNode = pNode->pnext;
		/* we gfree this since we've dup'd the SPI string */
		g_free (pNodeToFree->pname);
		gok_spy_accessible_unref (pNodeToFree->paccessible);
		g_free (pNodeToFree);
	}
	gok_log_leave();
}

/** 
* gok_spy_get_accessibleWithText
*
* accessor
**/
Accessible* gok_spy_get_accessibleWithText()
{
	return m_ptheAccessibleWithText;
}

/** 
* gok_spy_has_child
* 
* @accessible: Pointer to the accessible.
* @role: The role you are searching for. 
*
* If anything under this accessible* has the specified role, then returns true.
* 
* Returns: true if a child exists, false otherwise. 
**/ 
gboolean gok_spy_has_child (Accessible* accessible, GokSpySearchType type, AccessibleRole role) 
{
	gboolean bfound = FALSE;
	gok_log_enter();
	if (gok_spy_accessible_is_okay(accessible) == TRUE)
	{
		gint breadth;
		gint depth;
		GConfClient* client = gok_data_get_gconf_client ();
		gok_gconf_get_int ( client, GOK_GCONF_SPY_SEARCH_BREADTH, &breadth );
		gok_gconf_get_int ( client, GOK_GCONF_SPY_SEARCH_DEPTH, &depth );
		gok_spy_has_child_aux (accessible, type, role, breadth, depth, 
			0, &bfound);
	}
	else
	{
		gok_log("accessible is NULL");
	}
	gok_log_leave();
	return bfound;
}


/** 
* gok_spy_check_queues
*
* this should be called during idle time. 
**/ 
gboolean gok_spy_check_queues(void)
{
	EventNode* en = NULL;
	char *s;

	gok_log_enter();
	
	if (spyshutdown) {
		return FALSE;
	}
	
	while (eventQueue && g_queue_is_empty(eventQueue) == FALSE)
	{
		en = (EventNode*)g_queue_pop_head(eventQueue);

		gok_log ("eventQueue->length = %d", eventQueue->length);

#if defined(ENABLE_LOGGING_NORMAL)
		if (en->paccessible != NULL)
		{
			s = Accessible_getName (en->paccessible);
			if (s == NULL)
			{
				gok_log ("%#x", en->paccessible);
			}
			else
			{
				gok_log ("%#x %s", en->paccessible, s);
				SPI_freeString (s);
			}
		}
#endif

		switch (en->type)
		{
		case GOKSPY_FOCUS_EVENT:
			/* TODO: remove obsolete focus events from queue since process
			   accessible is expensive. */
			gok_log("GOKSPY_FOCUS_EVENT");
			gok_spy_process_accessible(en->paccessible);
			break;
		case GOKSPY_WINDOW_ACTIVATE_EVENT:
			gok_log("GOKSPY_WINDOW_ACTIVATE_EVENT");
			/* FIXME !! the ref needs to happen in the original event loop */
			if (en->paccessible != m_ptheWindowAccessible) {
				gok_spy_accessible_unref (m_ptheWindowAccessible);
				m_ptheWindowAccessible = en->paccessible;
				gok_spy_accessible_ref (m_ptheWindowAccessible);
				((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible);
			}
			break;
		case GOKSPY_DEFUNCT_EVENT:
			gok_log("GOKSPY_DEFUNCT_EVENT");
			/* note: currently only defunct events for the current application 
				are pushed on the queue */
			gok_spy_accessible_unref (m_ptheWindowAccessible);
			gok_spy_accessible_unref (m_ptheAccessibleWithText);
			m_ptheWindowAccessible = NULL;
			m_ptheAccessibleWithText = NULL;
			((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible);
			break;
		case GOKSPY_WINDOW_DEACTIVATE_EVENT:
			gok_log("GOKSPY_WINDOW_DEACTIVATE_EVENT");

			/*
			 * The test below is needed to workaround the case
			 * that a window:deactivate event may arrive
			 * after a window:activate has been emitted by
			 * another window. The test ignores window:deactivate
			 * events unless they come from what gok thinks
			 * is the current window.
			 */
			if (en->paccessible == m_ptheWindowAccessible) {
				gok_spy_accessible_unref (m_ptheWindowAccessible);
				gok_spy_accessible_unref (m_ptheAccessibleWithText);
				m_ptheWindowAccessible = NULL;
				m_ptheAccessibleWithText = NULL;
				((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible);
			}
			break;
		default:
			gok_log_x("unknown event type in internal gok event queue!");
			break;
		}
		gok_spy_accessible_unref (en->paccessible);
		g_free(en);
	}
	gok_log_leave();
	return FALSE; /* only return true if we left events in the queue! */
}

/** 
* gok_spy_check_window:
*
* @role: the role to check.
*
* This function decides if the role corresponds to something that looks like a window to the user.
*
* Returns: boolean success. 
*/ 
gboolean gok_spy_check_window(AccessibleRole role)
{
	/* TODO - improve efficiency here? Also, roles get added and we need
	   to maintain this...  maybe we need to go about this differently */
	if ((role == SPI_ROLE_WINDOW) ||
	    (role == SPI_ROLE_DIALOG) ||
	    (role == SPI_ROLE_FILE_CHOOSER) ||
	    (role == SPI_ROLE_FRAME) ||	
	    (role == SPI_ROLE_DESKTOP_FRAME) ||
		(role == SPI_ROLE_FONT_CHOOSER)	||
		(role == SPI_ROLE_COLOR_CHOOSER) ||
		(role == SPI_ROLE_APPLICATION) ||
		(role == SPI_ROLE_ALERT)
		)
	{
		return TRUE;
	}
	return FALSE;
}

/** 
 * gok_spy_worth_searching:
 *
 * @accessible: the accessible (possibly a parent) to examine for worthiness.
 * @role: the role we are searching for.
 *
 * This function decides if the accessible might have a subtree worth searching.
 *
 * Returns: boolean success. 
*/ 
gboolean gok_spy_worth_searching (Accessible* accessible, AccessibleRole role)
{
	gboolean worthiness;
	gboolean bmenu;
	AccessibleStateSet* ass = NULL;

	gok_log_enter();
	bmenu = gok_spy_is_menu_role (role);
	if (bmenu)
	{		
		switch (Accessible_getRole(accessible))
		{
			/* using this breaks mozilla menus:
			case SPI_ROLE_TOOL_BAR: */
			case SPI_ROLE_STATUS_BAR:
			case SPI_ROLE_COMBO_BOX:
				/* add cases here */
				gok_log("aborting menu search");
				gok_log_leave();
				return FALSE;
		}
	}
	else
	{
		switch (Accessible_getRole(accessible))
		{
			case SPI_ROLE_MENU_BAR:
			case SPI_ROLE_MENU:
			case SPI_ROLE_COMBO_BOX:
				/* add cases here */
				gok_log("aborting search");
				gok_log_leave();
				return FALSE;
		}
	}
	
	worthiness = TRUE;
	ass = Accessible_getStateSet(accessible);
	if (ass != NULL)
	{
		AccessibleRole role = Accessible_getRole (accessible);
		/* state heuristic: 
		 * - if this thing manages its own children, then it 
		 * is probably not worth traversing.
		 * - if it is a menu-like thing then it doesn't need to be "showing"
		 */
		if (AccessibleStateSet_contains( ass, SPI_STATE_MANAGES_DESCENDANTS ) ||
			(!bmenu && !AccessibleStateSet_contains( ass, SPI_STATE_SHOWING )) )
		{
			worthiness = FALSE;
		}
		AccessibleStateSet_unref(ass);
	}
	gok_log_leave();
	return worthiness;
}

	
/** 
 * gok_spy_has_child_aux:
 * @accessible: Pointer to the accessible.
 * @type: The search criteria to apply.
 * @role: The role you are searching for, if @type is GOK_SPY_SEARCH_ROLE.
 * @children: the breadth of the gui tree to search (for optimization).
 * @depth: how deep to look in the gui tree (for optimization).
 * @cdepth: recursive variable that keeps track of current depth
 * (pass in zero).
 * @found: out variable to hold the success boolean.
 * 
 * If anything under this accessible* has the specified role, then
 * returns true.
 *
 * Returns: true if a child exists, false otherwise.
 */
gboolean gok_spy_has_child_aux (Accessible* accessible, 
				GokSpySearchType type,
				AccessibleRole role, 
				gint children, gint depth, gint cdepth, gboolean* found) 
{ 
    Accessible* pAccessibleChild; 
    AccessibleStateSet *states;
    int child_count;
    int i; 
	gboolean nameless_menu = FALSE;

    if (accessible == NULL) return *found; 
	if (cdepth++ == depth) return *found;
	
	if (gok_spy_worth_searching(accessible, role) == FALSE) return *found;
	
    child_count = Accessible_getChildCount (accessible);
    
    if (child_count < 0)
    {
    	gok_log_x("exception occurred when trying to get the child count");
    	return *found;
    } 
    
    if (child_count > children) child_count = children;
    
    /* look through all the children of the given node */ 
	for (i=0; i<child_count; i++) 
	{ 
		pAccessibleChild = Accessible_getChildAtIndex (accessible, i);
		gok_spy_accessible_implicit_ref(pAccessibleChild); 
		if (pAccessibleChild != NULL) 
		{ 
		    /* does this child have the role we're looking for? */ 
		    if (type == GOK_SPY_SEARCH_ROLE) 
		    {
			    if (Accessible_getRole (pAccessibleChild) == role) {
				    gok_log ("gok_spy_has_child_aux: found child at depth %d", cdepth);
					*found = TRUE;
					gok_spy_accessible_unref(pAccessibleChild);
					return *found;
			    }
		    }
		    else if (type == GOK_SPY_SEARCH_MENU) {
			    AccessibleRole a_role = Accessible_getRole (pAccessibleChild);
			    if ((a_role == role) || 
				 (a_role == SPI_ROLE_MENU)) {
				    gchar *name = Accessible_getName (pAccessibleChild);
				    /* menu items must have either text or an image! */
				    if (name && (strlen (name) > 0)) { /* todo: what about icon-menus? */
					    gok_log ("gok_spy_has_child_aux: found [%s] at depth %d", name, cdepth);
					    gok_spy_accessible_unref(pAccessibleChild);
					    SPI_freeString (name);
					    *found = TRUE;					    
					    return *found;
				    }
					else {
						nameless_menu = TRUE;
					}
						
			    }
		    }
		    else if (type == GOK_SPY_SEARCH_EDITABLE_TEXT) {
			    if (Accessible_isEditableText(pAccessibleChild)) {
				    *found = TRUE;
				    gok_spy_accessible_unref(pAccessibleChild);
				    return *found;
			    }
		    }
			/* ignore children of a nameless menu, and abort if we have already
			   found one target */
			if (!*found && !nameless_menu )
		    	gok_spy_has_child_aux (pAccessibleChild, type, role, 
					children, depth, cdepth, found);
		    gok_spy_accessible_unref(pAccessibleChild);
		} 
	}
	
	return *found;
} 

static gboolean
gok_spy_is_visible (Accessible *accessible)
{
	gboolean retval = FALSE;
	if (accessible) {
		AccessibleStateSet *states = Accessible_getStateSet (accessible);
		if (AccessibleStateSet_contains (states, SPI_STATE_VISIBLE) &&
		    AccessibleStateSet_contains (states, SPI_STATE_SHOWING))
			retval = TRUE;
		AccessibleStateSet_unref (states);
	}
	return retval;
}

gboolean
gok_spy_is_menu_role (AccessibleRole role)
{
	return ((role  ==  SPI_ROLE_MENU_ITEM) ||
		(role ==  SPI_ROLE_CHECK_MENU_ITEM) ||
		(role ==  SPI_ROLE_RADIO_MENU_ITEM) ||
		(role ==  SPI_ROLE_MENU));
}


static gboolean
gok_spy_is_ui (Accessible *accessible, AccessibleRole role)
{
	gboolean is_interesting_ui_component = FALSE;
	switch (role) {
	case SPI_ROLE_PUSH_BUTTON:
	case SPI_ROLE_CHECK_BOX:
	case SPI_ROLE_COMBO_BOX:
	case SPI_ROLE_RADIO_BUTTON:
	case SPI_ROLE_PAGE_TAB:
	case SPI_ROLE_TOGGLE_BUTTON:
		if (gok_spy_is_visible (accessible))
			is_interesting_ui_component = TRUE;
	default:
		break;
	}
	/* EditableText and Hypertext components are always interesting */
	if (!is_interesting_ui_component) {
		if ((Accessible_isEditableText (accessible)) ||
		    (Accessible_isHypertext (accessible)))
			if (gok_spy_is_visible (accessible)) 
				is_interesting_ui_component = TRUE;
	}
	return is_interesting_ui_component;
}

/**
 * gok_spy_find_children:
 * @pNode: Before execution of this function it is the node that you
 * would like to search for children under.  After execution of this
 * function pNode->pnext is the start of the list of children that
 * satisfy the role requested.  Please note that this function may
 * also modify pNode->paccessible.
 * @role: The role you are searching for.
 *
 * Finds the children of the given node that have the given role.
 * These children are added to the list of children on the node,
 * beginning at pNode->pnext.  Please note that this function may also
 * modify pNode->paccessible.  This is a recursive function.
 */
void gok_spy_find_children (AccessibleNode* pNode, GokSpySearchType type, AccessibleRole role) 
{ 
	Accessible* pAccessibleChild;
	AccessibleRole childrole;
	Accessible* pAccessibleParent;
	long child_count;
	long i;

	g_assert(pNode != NULL);
	g_assert(pNode->paccessible != NULL); 

	pAccessibleParent = pNode->paccessible; 

	if (gok_spy_worth_searching(pAccessibleParent, role) == FALSE) return;

	child_count = Accessible_getChildCount (pAccessibleParent);

	if (child_count < 0)
	{
		gok_log_x("exception occurred trying to get the child count");
	}

	/* look through all the children of the given node */
	for (i=0; i<child_count; i++)
	{
		pAccessibleChild
			= Accessible_getChildAtIndex (pAccessibleParent, i);
		gok_spy_accessible_implicit_ref (pAccessibleChild);

		if (pAccessibleChild != NULL)
		{
			childrole = Accessible_getRole (pAccessibleChild);

			/* check role: special-casing some stuff REFACTOR THIS later TODO */
			if ((type == GOK_SPY_SEARCH_ALL) /* include everything */
			    ||
			    /* include all menu-ish when search type: GOK_SPY_SEARCH_MENU*/
			    ((type == GOK_SPY_SEARCH_MENU) && 
			     gok_spy_is_menu_role (childrole))
			    ||
			    /* the search role matches the child role */
			    ((type == GOK_SPY_SEARCH_ROLE) && (childrole == role))
			    ||
			    /* all ui */
			    ((type == GOK_SPY_SEARCH_UI) && 
			     (gok_spy_is_ui (pAccessibleChild, childrole)))
			    ||
				/* editable text */
			    ((type == GOK_SPY_SEARCH_EDITABLE_TEXT) &&
			     (Accessible_isEditableText(pAccessibleChild))))
			{
				gok_spy_append_node (pNode, pAccessibleChild);
				if (childrole == SPI_ROLE_PAGE_TAB) {
					pNode->paccessible = pAccessibleChild;
					gok_spy_find_children (pNode, type, role);
				}
			}
			else /* search through the children of this child */
			{
				pNode->paccessible = pAccessibleChild;
				gok_spy_find_children (pNode, type, role);
			}

		gok_spy_accessible_unref (pAccessibleChild);

		}
	}
}
 
/**
 * gok_spy_create_list:
 * @anAccessible: Pointer to parent accessible object.
 * @anAccessibleRole: The role you are searching for.
 *
 * Creates a list of accessible pointers of the given role.
 *
 * Returns: First node in the list of children. Will be NULL if no
 * children found.
 */
AccessibleNode* gok_spy_create_list (Accessible *anAccessible,
				     GokSpySearchType type,
                                     AccessibleRole anAccessibleRole)
{
	AccessibleNode nodeRoot;

	gok_log_enter();

	g_assert(anAccessible != NULL);

	nodeRoot.paccessible = anAccessible;
	nodeRoot.pnext = NULL;
	nodeRoot.pname = NULL;

	gok_spy_find_children (&nodeRoot, type, anAccessibleRole);

	gok_log_leave();
	return nodeRoot.pnext;
}

static gboolean
gok_spy_add_node (AccessibleNode *prev, Accessible *accessible, gint i, char *name)
{
	AccessibleNode *newnode = NULL;

	if (!prev) return FALSE;

	newnode = (AccessibleNode*)g_malloc(sizeof(AccessibleNode));
	
	if (newnode != NULL) {
		newnode->paccessible = accessible;
		gok_spy_accessible_ref (accessible);
		newnode->is_link = (i >= 0) ? TRUE : FALSE;
		newnode->link = MAX (0, i);
		newnode->pname = name;
		newnode->pnext = NULL;
		gok_log("newnode->pname = %s", newnode->pname);
		/* append this to end of list */
		while (prev->pnext)
			prev = prev->pnext;
		prev->pnext = newnode;	
		return TRUE;
	}
	return FALSE;
}

/**
 * gok_spy_append_node:
 * @pNode: An existing list of AccessibleNode to append to, must be
 * non-NULL.
 * @pAccessible: The Accessible to append to the list.
 *
 * Creates a new AccessibleNode for pAccessible and appends it to the
 * existing list of AccessibleNode at pNode.  Please note that this
 * function calls gok_spy_accessible_ref (pAccessible) when it
 * attaches it to the AccessibleNode created for it.  Please also note
 * that the list must exist already and pNode must be non-NULL.
 *
 * Returns: the address of the AccessibleNode created for pAccessible
 * or NULL if problems occur.
 */
AccessibleNode* gok_spy_append_node (AccessibleNode* pNode,
                                     Accessible* pAccessible)
{
	AccessibleNode* pLastNode = NULL;
	AccessibleNode* pNewNode = NULL;
	AccessibleRelation** relations = NULL;
	Accessible* targetAccessible = NULL;
	AccessibleRelationType type = -1;
	char* pName = NULL, *s;
	int i = 0;
	int j = 0;
	int ntargets = 0;

	g_assert (pNode != NULL);

	gok_log_enter();

        s = Accessible_getName (pAccessible);
	if (s && strlen (s)) 
	  pName = g_strdup (s);
	if (s)
	  SPI_freeString (s);

	/* Hypertext objects are treated differently; one node per link */
	if (Accessible_isHypertext (pAccessible)) {
		AccessibleHypertext *hypertext = Accessible_getHypertext (pAccessible);
		AccessibleText *text = Accessible_getText (pAccessible);
		gint i, nlinks = AccessibleHypertext_getNLinks (hypertext);
		for (i = 0; i < nlinks; ++i) {
			AccessibleHyperlink *link = AccessibleHypertext_getLink (hypertext, i);
			Accessible *link_object = AccessibleHyperlink_getObject (link, 0);
			gchar *linkname;
			long int start, end;
			/* TODO: handle multi-anchor links! */
			AccessibleHyperlink_getIndexRange (link, &start, &end);
			linkname = AccessibleText_getText (text, start, end);
			if (linkname) {
				/* create and append a node */
			  /* 
			   * FIXME: when AccessibleHyperlink objects get un-broken,
			   * we should pass the hyperlink object here instead of the 
			   * hypertext object, so we can just invoke the default action
			   * on the link!  But at the moment the link objects are not
			   * actionable, so we have to fake it.
			   */
				gok_spy_add_node (pNode, pAccessible, i, g_strdup (linkname));
				SPI_freeString (linkname);
			}
			Accessible_unref (link_object);
		}
		AccessibleText_unref (text);
		AccessibleHypertext_unref (hypertext);
	}

	/* if the name is empty then try LABELLED_BY, else assign "???" */ 
	else {
		/* look for a label */
		if (pName == NULL)
		{
			gok_log("no name, so looking at relations...");
			relations = Accessible_getRelationSet(pAccessible);
			/*
			if (relations == NULL)	{
				gok_log("no relationset for this object");
				gok_log_leave();
				return pNewNode;
			}
			*/
			if (relations != NULL)
			while (relations[i]) {
				type = AccessibleRelation_getRelationType(relations[i]);
				if (type == SPI_RELATION_LABELED_BY) {
					ntargets = AccessibleRelation_getNTargets(relations[i]);
					for (j=0; j < ntargets; j++) {
						targetAccessible = 
							AccessibleRelation_getTarget(relations[i], j);
						if (targetAccessible != NULL) {
							s = Accessible_getName (targetAccessible);
							
							if (s != NULL) {
								if ( strlen (s) != 0 )  {
									pName = g_strdup (s);
									SPI_freeString (s);
									break; /* from for loop*/
								}
								else {
									SPI_freeString(s);
								}
							}
						}
					}
					break; /* from while loop */
				}
				i++;
			}
		}
		
		if (pName == NULL) {
			/* one last try, we pull chars from description */
			s = Accessible_getDescription (pAccessible);
			if (s != NULL) {
				gint len = strlen (pName);
				if ( len != 0 )  {
					if (len > 18) 
					{
						gchar *tmp = g_strndup (s, 15);
						pName = g_strconcat (tmp, "...", NULL);
						g_free (tmp);
					}
					else 
					{
						pName = g_strndup (s, len);
					}
				}
				SPI_freeString (s);
			}
		}
		if (pName != NULL) {
			/* create a new node */
			gok_spy_add_node (pNode, pAccessible, -1, pName);
		}
		/* if we decide to allow access to nameless menus ...
		else if ( Accessible_getRole (pAccessible) == SPI_ROLE_MENU ) {
			pName = g_strdup (Accessible_getRoleName(pAccessible));
			gok_log ("Can't think of a good name so using role:[%s]",pName);
			gok_spy_add_node (pNode, pAccessible, -1, pName);
		}
		*/
		else if (Accessible_isText (pAccessible)) {
			/* 
			* sadly, many text objects are still nameless - yet we want 
			* to allow the user to get to them.
			* first tack is to get the starting text... if the object is
			* empty, then we can still "create" a name for it, though we 
			* can't assign it a unique meaningful name.
			*/
			AccessibleText *text = Accessible_getText (pAccessible);
			gint j = 0, len;
			gchar *word, *gs = NULL;
			if (text) {
				long int start, end = 0;
				do {
					gchar *sp = gs;
					word = AccessibleText_getTextAtOffset (text, end, 
									       SPI_TEXT_BOUNDARY_WORD_END,
									       &start, &end);
					if (word) {
						len = strlen (word);
					}
					else {
						len = 0;
					}
					if (len) {
						if (sp) {
							gs = g_strconcat (sp, word, NULL);
							g_free (sp);
						}
						else {
							gs = g_strdup (word);
						}
					}
					if (word) { 
						SPI_freeString (word);
					}
					++j;
				} while ((j < 3) && len);
				AccessibleText_unref (text);
			}
			if (gs && strlen (gs)) {
				pName = g_strconcat (gs, "...", NULL);
			}
			else {
				if (gs) {
					g_free (gs);
				}
				if (Accessible_isEditableText (pAccessible)) {
					pName = g_strdup ("Text Entry (empty)");
				}
			}
			if (pName != NULL) {
				gok_spy_add_node (pNode, pAccessible, -1, pName);
			}
		}
			
	}
	
	
	while (pNode->pnext) 
		pNode = pNode->pnext;

	gok_log_leave();

	return pNode;
}

/** 
* gok_spy_focus_listener 
*
* callback for focus events in the at-spi 
**/ 
void gok_spy_focus_listener (const AccessibleEvent *event, void *user_data) 
{ 
	EventNode* en; 
	gok_log_enter();
	gok_spy_accessible_ref(event->source);
	en = (EventNode*)g_malloc(sizeof(EventNode));
	en->paccessible = event->source;
	en->type = GOKSPY_FOCUS_EVENT;
	g_queue_push_tail( eventQueue, en );

	gok_spy_add_idle_handler ();
	gok_log_leave();
}

/** 
* gok_spy_defunctness_listener 
*
* callback for when objects go defunct in the at-spi 
**/ 
void gok_spy_defunctness_listener (const AccessibleEvent *event, void *user_data) 
{
	EventNode* en; 
	gok_log_enter();

	if (event->source == m_ptheWindowAccessible) {
		en = (EventNode*) g_malloc(sizeof(EventNode));
		en->paccessible = NULL; /* the source accessible is defunct! */
		en->type = GOKSPY_DEFUNCT_EVENT;
		g_queue_push_tail ( eventQueue, en );
		gok_spy_add_idle_handler ();
		gok_log ("Our target window is now defunct!");
	}
	
	gok_log_leave();
}

static void
gok_spy_window_activate_listener(const AccessibleEvent *event,
                                 void *user_data)
{
	EventNode *en;
	gok_log_enter ();

	gok_spy_accessible_ref (event->source);
	en = (EventNode*)g_malloc (sizeof (EventNode));
	en->paccessible = event->source;
	en->type = GOKSPY_WINDOW_ACTIVATE_EVENT;
	g_queue_push_tail (eventQueue, en);
	gok_spy_add_idle_handler ();

	gok_log_leave ();
}

static void
gok_spy_window_deactivate_listener(const AccessibleEvent *event,
                                   void *user_data)
{
	EventNode *en;
	gok_log_enter ();

	gok_spy_accessible_ref (event->source);
	en = (EventNode*)g_malloc (sizeof (EventNode));
	en->paccessible = event->source;
	en->type = GOKSPY_WINDOW_DEACTIVATE_EVENT;
	g_queue_push_tail (eventQueue, en);
	gok_spy_add_idle_handler ();

	gok_log_leave ();
}

/** 
* gok_spy_process_accessible 
* 
* @accessible: pointer to the accessible to process, assumed to have no references (see Accessible_ref in cspi)
* 
* side effects: sets global for foreground accessible container.  calls registered listener when
* the foreground window has changed.
**/ 
void gok_spy_process_accessible (Accessible* accessible) 
{ 
	Accessible* pnewWindowAccessible;
	Accessible* ptempAccessible;
	
	gok_log_enter();
	g_assert(accessible != NULL);
	
	pnewWindowAccessible = NULL;
	ptempAccessible = NULL;
	
	/* start by assuming we want to keep this accessible around */
	gok_spy_accessible_ref(accessible); 

	/*
	 *  check to see if focus is on a text object 
	 */
	if (Accessible_isText(accessible) == TRUE)
	{
		gok_log("this thing has a text interface");
		if (accessible != m_ptheAccessibleWithText)
		{
			gok_spy_accessible_ref(accessible); 
			gok_spy_accessible_unref(m_ptheAccessibleWithText);
			m_ptheAccessibleWithText = accessible;
			gok_log("found a (new) text interface"); 
		}
	}
	else
	{
		gok_log("no text interface on this thing");
		/* we might not want to do this - needs user testing */
		if (m_ptheAccessibleWithText != NULL)
		{
			gok_spy_accessible_unref(m_ptheAccessibleWithText);
		}
		m_ptheAccessibleWithText = NULL;
	}

	/* 
	   bug with gedit toolbar has to do with needing to carry on up the parent chain 
	   if we do this we might lose context - need to balance pros and cons...
	*/	

	/*
	 *  find the container for the focussed accessible thing 
	 */
	
	gok_log("Looking for container...");
	gok_log("accessible [%d]",accessible);
	pnewWindowAccessible = accessible;
	while (gok_spy_check_window(Accessible_getRole(pnewWindowAccessible)) != TRUE)
	{
		gok_log("*");
		ptempAccessible = Accessible_getParent(pnewWindowAccessible);
		gok_spy_accessible_implicit_ref(ptempAccessible);
		if (ptempAccessible == NULL)
		{
			break;
		} 
		gok_spy_accessible_unref(pnewWindowAccessible);
		pnewWindowAccessible = ptempAccessible;
	}
	
	/*
     * check to see if we have a new window 
	 */
	gok_log("Comparing this container with existing...");
	if (pnewWindowAccessible != m_ptheWindowAccessible)
	{
		gok_log("different.");
		gok_spy_accessible_unref (m_ptheWindowAccessible);
		m_ptheWindowAccessible = pnewWindowAccessible;
		((AppChangeListenerCB)(m_ptheWindowChangeListener))(m_ptheWindowAccessible); 
	}
	else
	{
		gok_log("same.");
		/* we have one ref too many (this isn't a different window) */
		gok_spy_accessible_unref(pnewWindowAccessible);
	}
	gok_log_leave();
} 

/**
 * gok_spy_button_is_switch_trigger:
 *
 * @button: an int representing logical mouse button number to query.
 *  
 * Returns: TRUE if the button is used by GOK as a switch, FALSE otherwise.
 **/
static gboolean
gok_spy_button_is_switch_trigger (int button)
{
        if (!strcmp (gok_data_get_name_accessmethod (), "directselection"))
	{
		if (gok_main_window_contains_pointer () == TRUE)
			return TRUE;
		else
			return FALSE;	
	}
	else if (gok_data_get_drive_corepointer() == TRUE) {
		gok_log("gok_data_get_drive_corepointer() is TRUE");
		return FALSE;
	}
	else {
		gok_log("gok_data_get_drive_corepointer() is FALSE");
		return gok_scanner_current_state_uses_core_mouse_button(button);
	}
}

/** 
* gok_spy_mouse_listener 
* 
* callback for mouse events in the at-spi 
* 
* @event: event structure
* @user_data: not used here
*
*/ 
SPIBoolean 
gok_spy_mouse_listener (const AccessibleDeviceEvent *event, void *user_data) 
{ 
	gint button = 0;
	gint state = 0;
	gboolean is_switch_trigger;
	gok_log_enter();

	/* 
	 * must check now and save for later, since a branch action 
	 * may change this by altering the window size, etc.
	 */
	button = event->keycode;
	is_switch_trigger = gok_spy_button_is_switch_trigger (button);

	if (m_ptheMouseButtonListener != NULL)
	{
		gok_log("mouse event %ld %x %x", 
			event->keyID, (unsigned) event->type, 
			(unsigned) event->modifiers);

			if (event->type == SPI_BUTTON_PRESSED)
			{
				state = 1;
			}
			((MouseButtonListenerCB)(m_ptheMouseButtonListener))(button, 
									     state, 
									     (long) event->modifiers,
									     event->timestamp);
	}
	gok_log_leave();
        return is_switch_trigger;
}

/**
 * gok_spy_get_modmask:
 **/
gulong gok_spy_get_modmask (void)
{
	return keyboardmods;
}


/** 
 * gok_spy_modifier_listener:
 * @event: the keyboard modifier event.
 * @user_data: not used.
 *
 * Callback for keyboard modifier events from the at-spi.
 */
void gok_spy_modifier_listener(const AccessibleEvent *event, void *user_data)
{
	const struct gok_modifier_keymask *m;
       
	keyboardmods = event->detail2;

	gok_log_enter();
	gok_log ("event->detail2 = %d", event->detail2);
	gok_keyboard_update_labels ();
	gok_modifier_update_modifier_keys (gok_main_get_current_keyboard ());
	gok_log_leave();
}

/** 
 * gok_spy_add_idle_handler:
 *
 * Adds the idle handler to deal with spy events
 */
void gok_spy_add_idle_handler (void)
{
	g_idle_add (gok_spy_idle_handler, NULL);
}

/* 
* gok_spy_get_active_frame
* 
* User must call gok_spy_free when finished with this list.
* 
* Returns: pointer to the active Accessible* or NULL
*/
Accessible* 
gok_spy_get_active_frame( )
{
	gint i;
	gint j;
	gint k;
	gint n_desktops;
	gint n_apps;
	gint n_frames;
	Accessible* desktop;
	Accessible* child;
	AccessibleStateSet* ass = NULL;

	gok_log_enter();

	n_desktops = SPI_getDesktopCount();
	for (i = 0; i < n_desktops; i++)
	{
		desktop = SPI_getDesktop(i);
		n_apps= Accessible_getChildCount(desktop);
		for (j = 0; j < n_apps; j++)
		{
			child = Accessible_getChildAtIndex(desktop, j);
			
			if (!child)
				continue;
			
			/* Applications are not active
			   so we must look for a child frame */
			n_frames = Accessible_getChildCount (child);
			for (k = 0; k < n_frames; k++) {
				Accessible *frame;		
				frame = Accessible_getChildAtIndex (child, k);

				if (!frame)
					continue;

				ass = Accessible_getStateSet (frame); /* implicit ref? */
				
				if (!ass) {
					Accessible_unref (frame);
					continue;
				}

				if (AccessibleStateSet_contains (ass, SPI_STATE_ACTIVE)) {
					gok_log ("Found active frame");
					Accessible_unref (desktop);
					AccessibleStateSet_unref (ass);
					Accessible_unref (child);
					return frame;
				}
				AccessibleStateSet_unref (ass);
				Accessible_unref (frame);
			}
			Accessible_unref (child);
		}
		Accessible_unref (desktop);
	}
	gok_log_leave();
	return NULL;
}

/*
gboolean gok_spy_is_desktop(Accessible* pAccessible)
{
	gint i;
	gint ndesktops;
	Accessible* desktop;

	gok_log_enter();

	ndesktops = SPI_getDesktopCount();
	for (i = 0; i < ndesktops; i++)
	{
		desktop = SPI_getDesktop(i);
		gok_spy_accessible_implicit_ref(desktop);
		if (pAccessible == desktop)
		{
			gok_spy_accessible_unref(desktop);
			gok_log_leave();
			return TRUE;
		}
	}
	gok_spy_accessible_unref(desktop);
	return FALSE;
}

gboolean gok_spy_accessible_is_desktopChild(Accessible* accessible)
{
	gint i;
	gint j;
	gint ndesktops;
	gint nchildren;
	Accessible* desktop;
	Accessible* child;
	gboolean returnCode = FALSE;
	
	gok_log_enter();

	ndesktops = SPI_getDesktopCount();
	for (i = 0; i < ndesktops; i++)
	{
		desktop = SPI_getDesktop(i);
		gok_spy_accessible_implicit_ref(desktop);
		if (desktop == NULL)
		{
			gok_log("desktop disappeared!?");
			break;
		}
		nchildren = Accessible_getChildCount(desktop);
		for (j = 0; j < nchildren; j++)
		{
			child = Accessible_getChildAtIndex(desktop, j);
			gok_spy_accessible_implicit_ref(child);
			if (child != NULL)
			{
				if (child == accessible)
				{
					gok_spy_accessible_unref(child);
					returnCode = TRUE;
					break;
				}
			}
		}
		if (returnCode == TRUE)
		{
			gok_spy_accessible_unref(desktop);
			break;
		}
		gok_spy_accessible_unref(desktop);
	}
	return returnCode;
	gok_log_leave();
}
*/
