/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;
using GLib;

class BaseView : Gtk.VBox {
	enum ViewType { Time, Space };
	protected SimulationData sd;
	protected ViewType       type;
	protected Histogram      hist;
	protected bool           hierarchical;
	Gtk.RadioButton rbTime;
	Gtk.RadioButton rbSpace;
	Gtk.TreeStore   treeStore;
    Gtk.TreeView    treeView;
    enum StoreFields : int { Name, Node };
    Gtk.Label       curLabel;
	Gtk.TreeViewColumn nameColumn;

	void AddToTree (Gtk.TreeIter parent, Node node)
	{
		Gtk.TreeIter thisIter;

		if (parent.Equals (Gtk.TreeIter.Zero))
            thisIter = this.treeStore.AppendNode();
		else 
            thisIter = this.treeStore.AppendNode (parent);

		this.treeStore.SetValue (thisIter, (int) StoreFields.Name, node.TreeElemName);
		this.treeStore.SetValue (thisIter, (int) StoreFields.Node, node);

		if (node.Children != null)
			foreach (Node n in node.Children)
				AddToTree (thisIter, n);
	}

	void BuildTreeStore ()
	{
		this.treeStore.Clear();

        Node root = (Node) this.hist.Root;
		AddToTree (Gtk.TreeIter.Zero, root);
        this.treeView.ExpandRow (FindNode (root), false);
	}

    Gtk.TreePath FindNode (Node node)
    {
        Gtk.TreePath path = null;
        this.treeStore.Foreach( delegate (Gtk.TreeModel model, Gtk.TreePath treePath, Gtk.TreeIter iter)
        {
            Node treeNode = (Node) this.treeStore.GetValue (iter, (int) StoreFields.Node);
            if (treeNode == node)
                path = treePath.Copy();
            return treeNode == node;
        } );
        return path;
    }

    public virtual Node UpdateView ()
    {
        Console.WriteLine ("ERROR: unimplemented UpdateView\n");
        return null;
    }

    void UpdateSettingsAndView()
    {
        this.type = this.rbTime.Active ? ViewType.Time : ViewType.Space;
        Node root = UpdateView();
		root.Accumulate();
        root.SortChildren();

        GotoResetHistory (root);
		BuildTreeStore();
    }

    void ConstructSettings ()
    {
        Gtk.HBox box;
        
        box = new Gtk.HBox (false, 6);
        
        rbTime = new Gtk.RadioButton("time");
        rbSpace = new Gtk.RadioButton(rbTime, "space");
        rbTime.Active = (type == ViewType.Time);
        rbTime.Toggled += delegate { UpdateSettingsAndView(); };
        rbSpace.Toggled += delegate { UpdateSettingsAndView(); };
        
        box.PackStart (rbTime, false, false, 0);
        box.PackStart (rbSpace, false, false, 0);
        
        Gtk.ToggleButton tg = new Gtk.CheckButton("view as tree");
        tg.Active = hierarchical;
        tg.Toggled += delegate {
            this.hierarchical = tg.Active;
            UpdateSettingsAndView ();
        };
        box.PackStart (tg, false, false, 0);
        
        box.ShowAll ();
        
        PackStart (box, false, false, 3);
    }
    
    // --- Tree View ---
	Gtk.Widget InitTreeView ()
	{
		Gtk.ScrolledWindow scrolledWindow = new Gtk.ScrolledWindow ();

		scrolledWindow.SetPolicy (Gtk.PolicyType.Automatic,
								  Gtk.PolicyType.Automatic);

		scrolledWindow.Add (this.treeView);

		this.treeView.Selection.Mode = Gtk.SelectionMode.Single;
		this.treeView.RulesHint = true;
        this.treeView.EnableSearch = true;
		
		// column for element name
		Gtk.CellRendererText rendererText = new Gtk.CellRendererText ();
		rendererText.Xalign = 0.0f;
		this.nameColumn = new Gtk.TreeViewColumn ("Name", rendererText, 
                                                  "text", 0);
		this.treeView.InsertColumn (this.nameColumn, 0);

		scrolledWindow.ShowAll();
		return scrolledWindow;
	}

	Gdk.Rectangle expandToPathScrollRect;
	uint expandToPathIdleId = 0;

	bool ExpandToPathScrollIdleCb ()
	{
		this.treeView.ScrollToPoint (expandToPathScrollRect.X, -1);
		expandToPathIdleId = 0;
		return false;
	}

    void ExpandToPath (Node node, bool select)
    {
        Gtk.TreePath path = FindNode (node);

        if (path == null)
            return;
        this.treeView.ExpandToPath (path);

        this.treeView.ScrollToCell (path, this.nameColumn, false, (float)0.5, (float)1.0);
		expandToPathScrollRect = this.treeView.GetCellArea (path, this.nameColumn);

		if (expandToPathIdleId != 0)
			GLib.Source.Remove (expandToPathIdleId);

		expandToPathIdleId = GLib.Idle.Add (ExpandToPathScrollIdleCb);

        this.treeView.Selection.UnselectAll ();
        this.treeView.Selection.SelectPath (path);
    }

	void OnRowActivated (object obj, Gtk.RowActivatedArgs args)
    {
        Gtk.TreeIter iter;
        if (this.treeStore.GetIter (out iter, args.Path)) {
            Node treeNode = (Node) this.treeStore.GetValue (iter, (int) StoreFields.Node);
            if (treeNode != null)
                GotoNode (treeNode);
        }
    }
    void OnCursorChanged(object obj, EventArgs e)
    {
        Gtk.TreeSelection selection = (obj as Gtk.TreeView).Selection;
        Gtk.TreeModel model;
        Gtk.TreeIter  iter;

        if (selection.GetSelected(out model, out iter))
        {
            Node treeNode = (Node) this.treeStore.GetValue (iter, (int) StoreFields.Node);
            Console.WriteLine ("Row selected " +
                               (treeNode != null ? treeNode.Label : "<null>"));
            if (treeNode != null && this.hist.Selected != treeNode) {
                Node root = (Node) this.hist.Root;
                if (!root.IsAncestor (treeNode)) {
                    Node newRoot = Node.CommonParent ((Node) this.hist.Selected,
                                                      treeNode);
                    if (newRoot == null)
                        Console.WriteLine ("Can't find common parent!");
                    else
                        GotoNode (newRoot);
                }
                this.hist.Selected = treeNode;
            }
        }
    }

	bool skipSetProportion;
	protected override void OnSizeAllocated (Gdk.Rectangle allocation)
	{
		base.OnSizeAllocated (allocation);

//        Console.WriteLine ("OnSizeAllocated " + allocation.Width + " x " +
//                           allocation.Height + " " + skipSetProportion);
		if (this.skipSetProportion)
			return;

		this.skipSetProportion = true;
		this.treeView.SetSizeRequest ((int) (this.Allocation.Width * 0.25), -1);
	}

    // --- Navigation ---
    List<Node> history;
    int        curHistory;
    Gtk.Button up, back, forward; 
    void UpdateNavigation()
    {
        Node cur = (Node) this.hist.Root;
        if (cur == null)
            return;

        this.up.Sensitive = cur.NonEmptyParent != null;
        this.back.Sensitive = curHistory > 0;
        this.forward.Sensitive = curHistory < history.Count - 1;
    }
    Gtk.HBox CreateNavigation()
    {
        Gtk.HBox hbox = new Gtk.HBox (false, 0);
        this.up = new Gtk.Button (Gtk.Stock.GoUp);
        this.back = new Gtk.Button (Gtk.Stock.GoBack);
        this.forward = new Gtk.Button (Gtk.Stock.GoForward);
        hbox.PackStart (this.up, false, false, 0);
        hbox.PackStart (this.back, false, false, 0);
        hbox.PackStart (this.forward, false, false, 0);
        this.up.Clicked += delegate {
            GotoNode (((Node) this.hist.Root).NonEmptyParent);
        };
        this.back.Clicked += delegate { GotoDirection (-1); };
        this.forward.Clicked += delegate { GotoDirection (+1); };
        hbox.ShowAll();
        return hbox;
    }
    void GotoDirection (int offset)
    {
        curHistory += offset;
        Node newNode = history[curHistory];
        hist.Root = newNode;
        ExpandToPath (newNode, false);
        UpdateNavigation();
    }
    void GotoNode (Node newNode)
    {
        if (newNode == null) {
            Console.WriteLine ("Error - avoid null nodes\n");
            return;
        }
        if (history == null) {
            history = new List<Node>();
            curHistory = -1;
        }

        if (curHistory < history.Count - 1)
            history.RemoveRange (curHistory + 1, history.Count - curHistory - 1);
        history.Add (newNode);
        GotoDirection (+1);

        curLabel.Text = newNode.ShortLabel;
    }
    void GotoResetHistory (Node newNode)
    {
        history = null;
        GotoNode (newNode);
    }
	void HistSelectionEvent (Histogram hist, bool zoom,
                             IHistNode selected, Cairo.Rectangle range)
    {
        if (zoom)
            GotoNode ((Node) selected);
        else {
            hist.Select (selected, range);
            ExpandToPath ((Node) selected, true);
        }
    }

    Pango.AttrList bold;

	public BaseView (SimulationData sd, IHistColorModel colorModel,
                     HistogramSettings settings)
	{
		this.type = ViewType.Time;
		this.sd = sd;
		this.hierarchical = true;
		this.treeStore = new Gtk.TreeStore (typeof (string), typeof (Node));
        this.treeView = new Gtk.TreeView (this.treeStore);
        this.treeView.RowActivated += OnRowActivated;
        this.treeView.CursorChanged += OnCursorChanged;
		this.skipSetProportion = false;

		ConstructSettings ();

		Gtk.HPaned pane = new Gtk.HPaned();
		pane.Show();
		PackStart (pane, true, true, 0);

		this.hist = new Histogram (settings, null, colorModel);
        this.hist.SelectionEvent += HistSelectionEvent;
		this.hist.Show();

        Gtk.VBox vbox = new Gtk.VBox (false, 0);
        Gtk.HBox hbox = CreateNavigation();
        vbox.PackStart (hbox, false, true, 2);

        this.curLabel = new Gtk.Label ("");
// FIXME: this totally hoses gtk-sharp / pango ... - why !?
//        bold = new Pango.AttrList ();
//        bold.Insert (new Pango.AttrWeight (Pango.Weight.Bold));
//        this.curLabel.Attributes = bold;
        this.curLabel.Show();
        hbox.PackStart (this.curLabel, false, true, 8);

        vbox.PackEnd (this.hist, true, true, 0);
        vbox.ShowAll();

		UpdateSettingsAndView ();
		pane.Add1 (InitTreeView ());
		pane.Add2 (vbox);
	}
}
