/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using Cairo;
using System.Collections.Generic;

public struct PosSpan {
    public long start;
    public long end;
    public PosSpan( long a, long b )
    { start = a; end = b; }
	public override string ToString()
	{
		return String.Format ("{0} -> {1}", start, end);
	}
}

public struct LayoutSettings {
    public double PixelsPerTimeUnit;
	public double PixelsPerVMPage;
	public bool   Bands;
}

public class Layout {
	public Layout (SimulationData sd, LayoutSettings settings)
	{
		if (!(settings.PixelsPerTimeUnit > 0.0))
			throw new ArgumentOutOfRangeException ("settings.PixelsPerTimeUnit",
												   settings.PixelsPerTimeUnit,
												   "must be greater than 0.0");

		if (!(settings.PixelsPerVMPage > 0.0))
			throw new ArgumentOutOfRangeException ("settings.PixelsPerTimeUnit",
												   settings.PixelsPerVMPage,
												   "must be greater than 0.0");

		this.settings = settings;
        LoadFromSimulator (sd);
	}

	void LoadFromSimulator (SimulationData sd)
	{
		if (sd == null)
			throw new ArgumentNullException ("simulator");

        /* Build address layout */
		if (settings.Bands)
			vlayout = new BandLayout((int)sd.PageSize, sd);
		else
			vlayout = new AddrLayout((int)sd.PageSize, sd);

        this.AddressRange.start = 0;
        this.AddressRange.end = vlayout.Extent;
        this.TimeRange.start = 0;
        this.TimeRange.end = sd.Elapsed;

		this.ViewAddressRange.start = 0;
		this.ViewAddressRange.end = 0;

		this.ViewTimeRange.start = 0;
		this.ViewTimeRange.end = 0;

		MapPalette = new GUtils.Palette();
		SetupEventPalette ();
		LayoutMaps (sd);
		LayoutEvents (sd);
	}

	void LayoutMaps (SimulationData sd)
	{
		int i = 0;

		Maps = new Map[sd.allMaps.Count];

        foreach (SimMap simMap in sd.allMaps)
        {
            Trace.MapEvent mape = simMap.Map;
			PosSpan unscaledVM;
			PosSpan unscaledTime;

			unscaledVM = vlayout.GetPos (mape);
			unscaledTime = simMap.MapDuration;

			Maps[i].vm.start = (long) (unscaledVM.start * settings.PixelsPerVMPage); /* round down */
			Maps[i].vm.end = (long) (unscaledVM.end * settings.PixelsPerVMPage + 1); /* round up */

			Maps[i].tm.start = (long) (unscaledTime.start * settings.PixelsPerTimeUnit); /* round down */
			Maps[i].tm.end = (long) (unscaledTime.end * settings.PixelsPerTimeUnit + 1); /* round up */

			Maps[i].Idx = i;
            Maps[i].SimMap = simMap;

			if (this.ViewAddressRange.end < Maps[i].vm.end)
				this.ViewAddressRange.end = Maps[i].vm.end;

			if (this.ViewTimeRange.end < Maps[i].tm.end)
				this.ViewTimeRange.end = Maps[i].tm.end;

            i++;
		}
	}
	void LayoutEvents (SimulationData sd)
	{
		int i = 0;

		Events = new Event[sd.allEvents.Count];

        foreach (SimEvent e in sd.allEvents) {

            if (!e.IsPageTouch || e.GetAsPageTouch().Start == 0)
                continue;

			PosSpan unscaledVM;
			PosSpan unscaledTime;
			long vmEnd;

			unscaledVM = vlayout.GetPos ((MemRegion) e.Detail);
			unscaledTime = e.Times;

            Events[i].ev = e;
			Events[i].vm.start = (long) (unscaledVM.start * settings.PixelsPerVMPage); /* round down */
			vmEnd = System.Math.Max(unscaledVM.start + sd.PageSize, unscaledVM.end);
			Events[i].vm.end = (long) (vmEnd * settings.PixelsPerVMPage + 1); /* round up */

			Events[i].tm.start = (long) (unscaledTime.start * settings.PixelsPerTimeUnit); /* round down */
			Events[i].tm.end = (long) (unscaledTime.end * settings.PixelsPerTimeUnit + 1); /* round up */

			if (this.ViewAddressRange.end < Events[i].vm.end)
				this.ViewAddressRange.end = Events[i].vm.end;

			if (this.ViewTimeRange.end < Events[i].tm.end)
				this.ViewTimeRange.end = Events[i].tm.end;

            i++;
		}
	}

	enum EventColors {
		LOAD,
		STORE
	}

	void SetupEventPalette ()
	{
		/* Keep this in sync with enum EventColors */

		EventPalette = new Cairo.Color[2];

		EventPalette[(int) EventColors.LOAD].R = 0.0;
		EventPalette[(int) EventColors.LOAD].G = 0.0;
		EventPalette[(int) EventColors.LOAD].B = 0.0;
		EventPalette[(int) EventColors.LOAD].A = 1.0;

		EventPalette[(int) EventColors.STORE].R = 0.0;
		EventPalette[(int) EventColors.STORE].G = 0.0;
		EventPalette[(int) EventColors.STORE].B = 0.0;
		EventPalette[(int) EventColors.STORE].A = 1.0;
	}

	public bool GetMapAt (long tPixels, long aPixels, ref Map outMap)
	{
		//		Console.WriteLine ("Get Map at " + tPixels + ", " + aPixels);
		foreach (Map m in Maps) {
			//			Console.WriteLine ("cf " + m.tm + " vm " + m.vm);
			if (tPixels >= m.tm.start && tPixels < m.tm.end &&
				aPixels >= m.vm.start && aPixels < m.vm.end) {
				outMap = m;
				return true;
			}
		}
		return false;
	}

	// Lame ass cut/paste evilness ...
	public bool GetEventAt (long time, long addr, out Event outEv)
	{
		foreach (Event m in Events) {
			if (time >= m.tm.start && time < m.tm.end &&
				addr >= m.vm.start && addr < m.vm.end) {
				outEv = m;
				return true;
			}
		}
        outEv = new Layout.Event();
		return false;
	}

    IVerticalLayout vlayout;
	public long GetAddressAtOffset (long offset)
	{
		return vlayout.GetAddressAtOffset (offset);
	}
    public PosSpan GetOffsetAtAddress (MemRegion span)
    {
        return vlayout.GetPos (span);
    }

	public Cairo.Color GetColor (Event ev)
	{ // FIXME: should hook type from base event pointer on ev.
		return this.EventPalette[(int)EventColors.LOAD];
	}

	public Cairo.Color GetColor (Map m)
	{
		return this.MapPalette.GetColor(m.Idx);
	}

	public GUtils.Palette MapPalette;
	public Cairo.Color[] EventPalette;

	public Map[]   Maps;
	public Event[] Events;

	/* Logical extents */
    public PosSpan AddressRange;
    public PosSpan TimeRange;

	/* View extents */
	public PosSpan ViewAddressRange;
	public PosSpan ViewTimeRange;

	public LayoutSettings settings;

	/* For the following structs, all dimensions are in pixels, and all the "end" fields
	 * are open intervals:
	 *
	 *   [start = 0, end = 5)  means it uses pixels { 0, 1, 2, 3, 4 }
	 */

	public struct Event {
        /* underlying event */
        public SimEvent ev;

		/* VM addresses (in pixels) */
        public PosSpan vm;

		/* Duration of the event ("tm" = "time") (in pixels) */
        public PosSpan tm;
	}

	public struct Map {
		/* VM addresses (in pixels) */
        public PosSpan vm;

		/* Lifespan of the map ("tm" = "time") (in pixels) */
        public PosSpan tm;

		/* Color for the map; index into the MapPalette */
		public int Idx;

        public SimMap SimMap;
		public string Label {
			// Nurgh ...
			get {
				return SimMap.FileName + " " + Utils.Percentage (SimMap.Percentage);
			}
		}
	}
}
