/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;
using FS;

public enum FSType { INode, Indirect, File, SymLink, Directory };
public interface FSColorModel {
    Cairo.Color GetColor (FSType type, bool selected);
}

public class FSRenderer {
    Cairo.Context cr;
    long topBlock, bottomBlock;         // total range to render in blocks
    long widthInBlocks;                 // block count across the screen
    long clipTopBlock, clipBottomBlock; // current view, block clipping boundary
    int border;                         // blank surround size around the boundary
    int width, height;                  // width & height of display window
    long totalHeight;                   // total pixel height of complete file-system
    long curOffset;                     // current pixel offset of view
    double blockSize;                   // size in pixels of each block
    bool isSelected;                    // render blocks as selected
    bool simpleRender;                  // render in minature for overview
    FSColorModel colors;
    bool renderingAsync;

    public FSRenderer (FSColorModel colors, bool simpleRender)
    {
        this.colors = colors;
        this.border = 8;
        this.simpleRender = simpleRender;
    }

    public void SetContext (Cairo.Context cr)
    {
        this.cr = cr;
    }

    // Scroll the render state to a new pixel offset 
    public void SetOffset (long pixOffset)
    {
        this.curOffset = pixOffset;
        this.clipTopBlock = (long) TotalPixToBlkY (pixOffset);
        this.clipBottomBlock = (long) TotalPixToBlkY (pixOffset + this.height);

        this.clipTopBlock -= this.clipTopBlock % this.widthInBlocks;
        this.clipBottomBlock -= this.clipBottomBlock % this.widthInBlocks;
        this.clipBottomBlock += this.widthInBlocks;
//        Console.WriteLine ("SetBlockBounds top: " + this.clipTopBlock + " bot: " + this.clipBottomBlock);
    }
    public long FirstVisibleBlock { get { return this.clipTopBlock; } }
    public long LastVisibleBlock  { get { return this.clipBottomBlock; } }

    public double TotalPixHeight {
        get { return this.totalHeight; }
    }

    // Set the rendering parameters, if blockSize < 0 we will calculate
    // a suitable block-size to render everything into width x height
    public void SetRegion (long topBlock, long bottomBlock, int width,
                           int height, double blockSize)
    {
        this.topBlock = this.clipTopBlock = topBlock;
        this.bottomBlock = this.clipBottomBlock = bottomBlock;
        this.width = width;
        this.height = height;

        int realWidth = this.width - this.border * 2;
        long totalBlocks = this.bottomBlock - this.topBlock;

        Console.WriteLine ("Set Region " + width + "x" + height +
                           " spix " + blockSize);

        if (blockSize < 0) {
            // Fit everything in as best we can ...
            int realHeight = this.height - this.border * 2;
            long totalPixels = (long)realWidth * realHeight;

            double widthPerBlock = Math.Sqrt ((double)totalPixels / totalBlocks);
            double widthInBlocks = Math.Max ((long) (realWidth / widthPerBlock), 1);
 
            this.blockSize = Math.Min ((double) realWidth / widthInBlocks,
                                       (double) realHeight / (totalBlocks / widthInBlocks));
        } else
            this.blockSize = blockSize;

        this.simpleRender = (this.blockSize < 1); // not enough pixels to see gradients
        this.widthInBlocks = Math.Max ((long) (realWidth / this.blockSize), 1);
        this.totalHeight = (long)(((totalBlocks + this.widthInBlocks - 1) / this.widthInBlocks) * this.blockSize) + this.border * 2;

/*        Console.WriteLine ("pix" + totalPixels + " blocks " +
                           totalBlocks + " widthInBlocks " +
                           this.widthInBlocks + " width per block " +
                           widthPerBlock + " blk size " + this.blockSize); */
    }

    // ie. blk to total pixels in a virtual whole file-system space
    public double BlkToPixY (long blk)
    {
        long start = blk / this.widthInBlocks;
        return ((double)start * this.blockSize) + this.border;
    }

    // ie. pixels in a virtual whole file-system space
    public long TotalPixToBlkY (double pix)
    {
        pix -= this.border;
        pix /= this.blockSize;
        pix *= this.widthInBlocks;
        return (long) pix;
    }

    // return top-left of blk on-screen 
    public void BlkToScreenPix (long blk, out double x, out double y)
    {
        long hblk = blk % this.widthInBlocks;
        long vblk = blk / this.widthInBlocks;

        x = this.blockSize * hblk + this.border;
        y = this.blockSize * vblk + this.border - this.curOffset;
    }

    // FIXME: unfortunately, cairo doesn't do this for us at all well [!]
    public void ClipToScreen (ref double x1, ref double y1,
                              ref double x2, ref double y2)
    {
        double grad = (y2 - y1) / (x1 - x2);
        if (y1 < 0) {
            x1 -= grad / y1;
            y1 = 0;
//            Console.WriteLine ("clip top to " + x1 + ", " + y1);
        }
        if (y2 > this.height) {
            x2 += grad / (y2 - this.height);
            y2 = this.height;
//            Console.WriteLine ("clip bot to " + x2 + ", " + y2);
        }
    }

    public long BlockAt (double x, double y)
    {
        long blk = TotalPixToBlkY (this.curOffset + y);
        blk -= blk % this.widthInBlocks;
        blk += (long)(x / this.blockSize);
        if (blk < this.topBlock || blk > this.bottomBlock)
            return -1;
        else
            return blk;
    }

    public void DrawSpan (File fi, long startBlk, long afterEndBlk, Cairo.Color color)
    {
        if (afterEndBlk < this.clipTopBlock ||
            startBlk > this.clipBottomBlock)
            return; // not visible

        this.cr.Color = color;

        long startX = startBlk % this.widthInBlocks;
        long afterEndX = afterEndBlk % this.widthInBlocks;
        long startY = startBlk / this.widthInBlocks;
        long afterEndY = afterEndBlk / this.widthInBlocks;

        if (afterEndY == startY) {
            double x = this.border + startX * this.blockSize;
            double y = this.border + startY * this.blockSize - this.curOffset;
            double width = (1 + afterEndX - startX) * this.blockSize;
            double height = this.blockSize;
            this.cr.Rectangle (x, y, width, height);
            this.cr.Fill ();
            if (!this.simpleRender && fi != null &&
                width > 64 && height > 16) {
                
                double txtBorder = 4;
                cr.Color = new Cairo.Color (1, 1, 1, 1);
                
                Cairo.TextExtents extents;
                cr.SetFontSize (Math.Min (this.blockSize / 2, 12.0));
                extents = cr.TextExtents (fi.Name);
                if (extents.Height < height - txtBorder &&
                    extents.Width < width - txtBorder) {
                    
//                    Console.WriteLine ("render " + fi.Name + " at " + startY);

                    cr.MoveTo (x + width/2 - extents.Width/2,
                               y + height/2 + extents.Height/2);
                    cr.ShowText (fi.Name);
                }
            }
        } else {
            DrawSpan (fi, startBlk, (startBlk - startX) + this.widthInBlocks - 1, color);
            for (startY++; startY < afterEndY - 1; startY++) {
                long start = startY * this.widthInBlocks;
                DrawSpan (fi, start, start + this.widthInBlocks - 1, color);
            }
            DrawSpan (fi, startY * this.widthInBlocks, afterEndBlk, color);
        }
    }

    class RenderStackItem {
        public int Offset;
        public List<File> Children;
        public RenderStackItem (int offset, List<File> children)
        {
            this.Offset = offset;
            this.Children = children;
        }
    }
    class RenderStack : List<RenderStackItem>
    {
        public RenderStack (File fi)
        {
            List<File> li = new List<File>(1);
            li.Add (fi);
            Push (li);
        }
        public bool Empty {
            get { return Count == 0; }
        }
        RenderStackItem Pop()
        {
            RemoveRange (Count - 1, 1);
//            Console.WriteLine ("Pop " + Count);
            return Cur();
        }
        RenderStackItem Cur()
        {
            if (Count <= 0)
                return null;
            return this[ Count - 1 ];
        }
        public File Next()
        {
            RenderStackItem cur = Cur();
            if (cur == null)
                return null;

            while (cur.Offset >= cur.Children.Count) {
                cur = Pop();
                if (cur == null)
                    return null;
            }

            File fi = cur.Children[cur.Offset++];
            return fi;
        }
        public void Push (List<File> children)
        {
            Add (new RenderStackItem (0, children));
        }
    }

    void RenderFile (RenderStack stk, File fi)
    {
//        Console.WriteLine ("Render " + fi.Name);

        // inode
        DrawSpan (fi, fi.INodeBlock, fi.INodeBlock + 1,
                  colors.GetColor (FSType.INode, this.isSelected));

        // blocks
        Cairo.Color mainColor = this.colors.GetColor (FSType.File, this.isSelected);
        if (fi is SymLink)
            mainColor = this.colors.GetColor (FSType.SymLink, this.isSelected);
        else if (fi is Directory)
            mainColor = this.colors.GetColor (FSType.Directory, this.isSelected);

        if (fi.Blocks != null) {
            double sat = 0.3;
            double satincr = 0.5 / fi.Blocks.Count;
            foreach (Block blk in fi.Blocks) {
                Cairo.Color color;
                if (blk.IsIndirect) 
                    color = colors.GetColor (FSType.Indirect, this.isSelected);
                else
                    color = mainColor;
                if (!this.simpleRender) {
                    color = GUtils.LightenColor (color, sat);
                    sat += satincr;
                }
                DrawSpan (fi, blk.Start, blk.End, color);
            }
        }

        // children
        if (fi is Directory && 
            (fi as Directory).Children != null) {
            // clip node & children to visible range ...
            if (! ((fi as Directory).MaxBlock <= this.clipTopBlock ||
                   (fi as Directory).MinBlock >= this.clipBottomBlock) )
                stk.Push ((fi as Directory).Children);

//            Directory dir = (Directory) fi;
//            Console.WriteLine ("Not ! cropped " + dir.Name + " " +
//                               dir.MinBlock + " -> " + dir.MaxBlock + " " +
//                               "top " + this.clipTopBlock + " bottom " + this.clipBottomBlock);
        }
    }

    public void ClearBackground()
    {
		this.cr.Color = new Cairo.Color (1, 1, 1, 1);
        this.cr.Rectangle (0, 0, this.width, this.height);
		this.cr.Fill ();
    }

    public void RenderFile (File fi, bool isSelected)
    {
        this.isSelected = isSelected;
        RenderStack stk = new RenderStack (fi);
        do {
            File it = stk.Next();
            if (it != null)
                RenderFile (stk, it);
        } while (!stk.Empty);
	}

    public void RenderScribble (List<SimBlk> blks, Cairo.Color col,
                                double lineWidth)
    {
        // if begin & end points are external - don't render ...
        bool first = true;
        long lastEnd = 0;

        this.cr.LineWidth = lineWidth;
        this.cr.Color = col;
        foreach (SimBlk blk in blks) {
            if (blk.IsSequence)
                continue;
            long end = blk.FsBlkStart + blk.FsBlkLen;
            DrawSpan (null, blk.FsBlkStart, end, col);
            if (!first) {
                // Draw line from lastEnd -> our start
                // ... translate to real world ...
                double x1, y1;
                double x2, y2;
                BlkToScreenPix (lastEnd, out x1, out y1);
                BlkToScreenPix (blk.FsBlkStart, out x2, out y2);

                if ((y1 >= 0 && y1 < this.height) ||
                    (y2 >= 0 && y2 < this.height) ||
                    (y1 < 0 && y2 > this.height))
                {
                    // Some of the line is on the screen:
//                    Console.WriteLine ("Line from " + x1 + ", " + y1 +
//                                       " -> " + x2 + ", " + y2);
                    ClipToScreen (ref x1, ref y1, ref x2, ref y2);
                    this.cr.MoveTo (x1, y1);
                    this.cr.LineTo (x2, y2);
                    this.cr.Stroke ();
                }
            }
            first = false;
            lastEnd = end;
        }
    }

    public delegate void StatusCallback ();

    uint asyncHandler;
    public void RenderAsync (File fi, StatusCallback lumpFn,
                             StatusCallback doneFn)
    {
        AsyncRenderStop();

        RenderStack stk = new RenderStack (fi);
        this.renderingAsync = true;
        this.asyncHandler = GLib.Idle.Add (
            delegate {
                for (int i = 0; i < 16 * 1024 && !stk.Empty; i++)
                {
                    File it = stk.Next();
                    if (it != null)
                        RenderFile (stk, it);
                }
                lumpFn();
                if (stk.Empty) {
                    this.renderingAsync = false;
                    doneFn();
                }
                if (stk.Empty)
                    this.asyncHandler = 0;
                return !stk.Empty;
            }
        );
    }
    public void AsyncRenderStop()
    {
        if (this.asyncHandler != 0)
            GLib.Source.Remove (this.asyncHandler);
        this.asyncHandler = 0;
    }
    public bool InAsyncRender {
        get { return this.renderingAsync; }
    }

    public void Dispose()
    {
        AsyncRenderStop();
    }
}

public class FSOverview : Gtk.DrawingArea, FSColorModel
{
    FSData fsData;
    FSRenderer renderer;
    Gdk.Pixmap overviewPixmap;
    int pixmapWidth, pixmapHeight;
    List<File> selection;
    List<SimBlk> blkSelection;

    long viewBlockStart, viewBlockEnd;

    bool IsDegenerate {
        get { return this.pixmapWidth < 4 || this.pixmapHeight < 4; }
    }

    public Cairo.Color GetColor (FSType type, bool isSelected)
    {
        if (isSelected)
            return new Cairo.Color (0, 1, 0, 1);
        switch (type) {
        case FSType.INode:
        case FSType.Indirect:
            return new Cairo.Color (1, 0, 0, 1);
        case FSType.Directory:
            return new Cairo.Color (0, 0.3, 0.3, 1);
        case FSType.File:
        case FSType.SymLink:
        default:
            return new Cairo.Color (0, 0, 0, 1);
        }
    }

    public FSOverview (FSData fsData, List<File> selection)
    {
        this.fsData = fsData;
        this.renderer = new FSRenderer (this, true);
        this.selection = selection;
    }

    void RenderContent ()
    {
		if (!IsMapped || IsDegenerate)
			return;

        if (Allocation.Width == this.pixmapWidth &&
            Allocation.Height == this.pixmapHeight &&
            this.overviewPixmap != null)
            return;

        this.overviewPixmap = new Gdk.Pixmap (GdkWindow,
                                              this.pixmapWidth,
                                              this.pixmapHeight);

        Cairo.Context cr;
        cr = Gdk.CairoHelper.Create (this.overviewPixmap);
        this.renderer.SetContext (cr);
        this.renderer.SetRegion (0, this.fsData.MaxBlock,
                                 this.pixmapWidth,
                                 this.pixmapHeight,
                                 -1.0);
        this.renderer.ClearBackground ();

        this.renderer.RenderAsync (this.fsData.Root,
                                   delegate { QueueDraw(); },
                                   delegate { (cr as IDisposable).Dispose(); });
    }

    public override void Dispose()
    {
        this.renderer.Dispose();
        base.Dispose();
    }

    protected override bool OnMapEvent (Gdk.Event evt)
    {
        base.OnMapEvent (evt);
        RenderContent();
        return false;
    }

    protected override void OnSizeAllocated (Gdk.Rectangle allocation)
    {
        base.OnSizeAllocated (allocation);

        if (allocation.Width == this.pixmapWidth &&
            allocation.Height == this.pixmapHeight &&
            this.overviewPixmap != null)
            return;

        this.pixmapWidth = allocation.Width;
        this.pixmapHeight = allocation.Height;
        if (this.overviewPixmap != null) {
            this.overviewPixmap.Dispose();
            this.overviewPixmap = null;
        }

        RenderContent();
    }

    protected override bool OnExposeEvent (Gdk.EventExpose evt)
	{
        if (!IsDegenerate)
            foreach (Gdk.Rectangle r in evt.Region.GetRectangles())
                GdkWindow.DrawDrawable (Style.BlackGC,
                                        this.overviewPixmap,
                                        r.X, r.Y,
                                        r.X, r.Y,
                                        r.Width, r.Height);
        if (this.viewBlockEnd > this.viewBlockStart) {
            Cairo.Context cr;
            using (cr = Gdk.CairoHelper.Create (GdkWindow)) {
                double top = this.renderer.BlkToPixY (this.viewBlockStart);
                double bottom = this.renderer.BlkToPixY (this.viewBlockEnd);
                cr.Color = new Cairo.Color (0.2, 0.2, 1.0, 0.5);
                cr.Rectangle (0, top - 1.0, this.Allocation.Width, (bottom - top) + 2.0);
                cr.Fill();

                // If still doing async rendering, don't mess with the
                // render context
                if (this.renderer.InAsyncRender)
                    return false;
                this.renderer.SetContext (cr);

                if (this.selection.Count > 0) {
                    long min = this.fsData.MaxBlock;
                    long max = 0;
                    // over-render selection
                    foreach (File fi in this.selection) {
                        long tmin, tmax;
                        fi.GetMinMax (out tmin, out tmax);
                        min = Math.Min (min, tmin);
                        max = Math.Max (max, tmax);
                        if (fi is Directory &&
                            ((fi as Directory).BlockExtent >
                             this.fsData.MaxBlock / 10))
                            continue; // > 10% of the file-system - too slow to render.
                        // FIXME: we prolly want to do some idle selection
                        // rendering too ... (to an alpha pixbuf to over-render?)
                        this.renderer.RenderFile (fi, true);
                    }
                    
                    double seltop = this.renderer.BlkToPixY (min);
                    double selbot = this.renderer.BlkToPixY (max);
                    cr.Color = GetColor (FSType.Directory, true);
                    double barwidth = 3.0;
                    // border each side
                    cr.Rectangle (0, seltop - 1.0, barwidth, (selbot - seltop));
                    cr.Fill();
                    cr.Rectangle (this.Allocation.Width - barwidth,
                                  seltop - 1.0, barwidth, (selbot - seltop));
                    cr.Fill();
                }
                if (this.blkSelection != null &&
                    this.blkSelection.Count > 0) {
                    Cairo.Color red = new Cairo.Color (1, 0, 0, 1);
                    this.renderer.RenderScribble (this.blkSelection, red, 1.0);
                }
            }
        }
		return false;
	}

    public void UpdateViewPos (long startBlk, long endBlk)
    {
        if (this.viewBlockStart != startBlk ||
            this.viewBlockEnd != endBlk) {
            this.viewBlockStart = startBlk;
            this.viewBlockEnd = endBlk;
            QueueDraw();
        }
    }
    public void UpdateBlkSelection (List<SimBlk> blkSelection)
    {
        this.blkSelection = blkSelection;
    }
}

public abstract class FSBase : Gtk.Table, FSColorModel {

    double blockSizePix;
    protected internal FSData fsData;

    Gtk.Widget leftView;
    public FSRenderer renderer;
    public FSOverview overview;
    Gtk.Statusbar statusBar;
    Gtk.DrawingArea drawingArea;
    protected internal Gtk.Adjustment vAdjustment;
    protected internal Gtk.Label selectionDisplay;

    protected internal List<File> selection;

    public Cairo.Color GetColor (FSType type, bool isSelected)
    {
        if (isSelected)
            return new Cairo.Color (0, 1, 0, 1);
        switch (type) {
        case FSType.INode:
            return new Cairo.Color (1, 0, 0, 1);
        case FSType.Indirect:
            return new Cairo.Color (1, 0, 1, 1);
        case FSType.Directory:
            return new Cairo.Color (0, 0.5, 0.5, 1);
        case FSType.File:
            return new Cairo.Color (1, 1, 1, 1);
        case FSType.SymLink:
            return new Cairo.Color (1, 1, 0, 1);
        default:
            return new Cairo.Color (1, 1, 1, 1);
        }
    }

	public FSBase (FSData fsData) :
		base (4, 3, false)
	{
		this.fsData = fsData;
        this.blockSizePix = 24.0;
        this.selection = new List<File>();

        this.renderer = new FSRenderer (this, false);

        this.skipSetProportion = false;
		Gtk.HPaned pane = new Gtk.HPaned();
		pane.Show();
        Attach (pane, 0, 2, 1, 2,
                Gtk.AttachOptions.Fill | Gtk.AttachOptions.Expand,
                Gtk.AttachOptions.Fill | Gtk.AttachOptions.Expand,
                0, 0);

        this.leftView = CreateTree();
		pane.Add1 (this.leftView);
		pane.Add2 (CreateMainPane());

        CreateOverview();
        if (this.overview != null)
            Attach (this.overview, 2, 3, 1, 2, Gtk.AttachOptions.Fill,
                    Gtk.AttachOptions.Fill, 0, 0);

        CreateStatusBar();
	}

    // FIXME: should really build up a list here cf. inodes etc.
    File findBlock (File fi, long blkId) 
    {
        if (fi.INodeBlock == blkId)
            return fi;

        if (fi.Blocks != null) {
            foreach (Block blk in fi.Blocks) {
                if (blk.Contains (blkId))
                    return fi;
            }
        }
        if (fi is Directory) {
            Directory dir = (Directory) fi;

            // prune tree via block spans
            if (dir.Children == null || blkId < dir.MinBlock
                || blkId > dir.MaxBlock)
                return null;

            foreach (File child in dir.Children) {
                File hit = findBlock (child, blkId);
                if (hit != null)
                    return hit;
            }
        }
        return null;
    }

    protected internal void UpdateSelectionDisplay()
    {
        if (this.selection.Count <= 0) {
            this.selectionDisplay.LabelProp = "";
            return;
        }
        File fi = this.selection[0];
        this.selectionDisplay.LabelProp = String.Format
            ("{0}: '{1}' size: 0x{2:x}",
             fi is Directory ? "dir" : "file",
             fi.GetPath(), fi.Size);
    }

    protected internal void MoveToBlock (long blk)
    {
        Gtk.Adjustment adj = this.vAdjustment;
        adj.Value = (this.renderer.BlkToPixY (blk) -
                     this.Allocation.Height / 2);
    }

    protected internal void MoveToSelection (File fi)
    {
        long firstBlock = fi.INodeBlock;
        if (fi.Blocks != null)
            firstBlock = fi.Blocks[0].Start;
        MoveToBlock (firstBlock);
    }

	void AreaButtonPress (object obj, Gtk.ButtonPressEventArgs args)
	{
        args.RetVal = true;

        Gdk.EventButton ev = args.Event;
        long blkSelected = this.renderer.BlockAt (ev.X, ev.Y);

        if (blkSelected < 0)
            return; // border / non-block

        File fi = findBlock (this.fsData.Root, blkSelected);
        if (fi == null)
            return; // un-allocated space
        selection.Clear();
        selection.Add (fi);
        UpdateSelectionDisplay();
        // FIXME - do we want to expand the tree & show this item too ... ?
        QueueDraw();
    }

	Gtk.Widget CreateDrawingArea ()
	{
        Gtk.Widget widget;

        if (this.fsData != null &&
            this.fsData.MaxBlock > 0) {
            this.drawingArea = new Gtk.DrawingArea();
            this.drawingArea.AddEvents ((int) Gdk.EventMask.ButtonPressMask);
            this.drawingArea.ButtonPressEvent += AreaButtonPress;

            Gtk.HBox box = new Gtk.HBox (false, 0);
             
            box.PackStart (this.drawingArea, true, true, 0);
            this.vAdjustment = new Gtk.Adjustment (0.0, 0.0, 1.0, 0, 0, 0);
            box.PackStart (new Gtk.VScrollbar (this.vAdjustment), false, false, 0);
            widget = box;
            
            this.vAdjustment.ValueChanged += delegate { this.drawingArea.QueueDraw(); };
            this.drawingArea.SizeAllocated += DrawingAreaSizeAllocated;

            this.drawingArea.ExposeEvent += DrawAreaExposeEvent;
            widget.ShowAll ();
        } else {
            widget = new Gtk.Label ("No file system data");
            widget.Show();
        }

        return widget;
	}

    public void Zoom (double factor)
    {
        double newBlockSize = this.blockSizePix * factor;
        if (newBlockSize < 4 || newBlockSize > 256)
            return;
        this.blockSizePix = newBlockSize;
        SetupRegion (this.drawingArea.Allocation);
        QueueDraw();
    }

    public abstract Gtk.Widget CreateTree ();

    Gtk.Widget CreateControls ()
    {
        Gtk.HBox hbox = new Gtk.HBox (false, 4);
		GUtils.AddButton (Gtk.Stock.ZoomOut, hbox, delegate { Zoom (0.75); });
		GUtils.AddButton (Gtk.Stock.ZoomIn,  hbox, delegate { Zoom (1/0.75); });
        this.selectionDisplay = new Gtk.Label ("");
        hbox.PackStart (this.selectionDisplay, false, false, 0);
        hbox.ShowAll();
        return hbox;
    }

	Gtk.Widget CreateMainPane ()
    {
        Gtk.VBox vbox = new Gtk.VBox (false, 0);
        vbox.PackStart (CreateControls(), false, false, 2);
        vbox.PackStart (CreateDrawingArea(), true, true, 0);
        vbox.Show();
        return vbox;
    }

    void CreateOverview()
    {
        // Create widget to render image ...
        if (this.fsData == null ||
            this.fsData.MaxBlock <= 0)
            return;

        this.overview = new FSOverview (this.fsData, this.selection);
        this.overview.Show();
    }

    // Get initial proportions in the toplevel right ...
	bool skipSetProportion;
	protected override void OnSizeAllocated (Gdk.Rectangle allocation)
	{
		base.OnSizeAllocated (allocation);

		if (this.skipSetProportion)
			return;

		this.skipSetProportion = true;
        // FIXME: this is fugly & breaks sizing badly.
		this.leftView.SetSizeRequest ((int) (this.Allocation.Width * 0.2), -1);
        if (this.overview != null)
            this.overview.SetSizeRequest ((int) (this.Allocation.Width * 0.15), -1);
	}

	void CreateStatusBar ()
	{
		this.statusBar = new Gtk.Statusbar();
		Attach (this.statusBar,
				0, 2, 2, 3,
				Gtk.AttachOptions.Fill,
				Gtk.AttachOptions.Fill,
				0, 0);
		statusBar.Push (0, "File System: " + this.fsData.ImageName);
		this.statusBar.Show();
	}

    void SetupRegion (Gdk.Rectangle alloc)
    {
        long curBlkOffset = this.renderer.TotalPixToBlkY (this.vAdjustment.Value +
                                                          this.Allocation.Height / 2);

        // Set the parameters
        this.renderer.SetRegion (0, this.fsData.MaxBlock,
                                 alloc.Width, alloc.Height,
                                 this.blockSizePix);
//        Console.WriteLine ("Total pix height " + this.renderer.TotalPixHeight);
        // Get the height that equates to & upd. the adjustment
        this.vAdjustment.SetBounds (0,
                                    (double)this.renderer.TotalPixHeight,
                                    (double)alloc.Height / 16,
                                    (double)alloc.Height * 0.75,
                                    (double)alloc.Height);
//        Console.WriteLine ("Set size " + width + " x " + height);

        this.vAdjustment.Value = (this.renderer.BlkToPixY (curBlkOffset) -
                                  this.Allocation.Height / 2);
    }

    // Ensure the proportions for rendering are right
	void DrawingAreaSizeAllocated (object obj, Gtk.SizeAllocatedArgs args)
    {
        SetupRegion (args.Allocation);
    }

	protected internal virtual void RenderContent ()
    {
        this.renderer.RenderFile (this.fsData.Root, false);
        
        // over-render selection
        foreach (File fi in this.selection)
            this.renderer.RenderFile (fi, true);
    }

	protected internal virtual void RenderVisibleContent ()
	{
		Cairo.Context cr;

		if (!this.drawingArea.IsMapped)
			return;

//        Console.WriteLine ("Render vis content " + topy + " + " + bottomy);
        this.renderer.SetOffset ((long) this.vAdjustment.Value);
        this.overview.UpdateViewPos (this.renderer.FirstVisibleBlock,
                                     this.renderer.LastVisibleBlock);

        // render blocks
        using (cr = Gdk.CairoHelper.Create (this.drawingArea.GdkWindow)) {
            this.renderer.SetContext (cr);
            this.renderer.ClearBackground ();
            RenderContent();
            this.renderer.SetContext (null);
        }
        
	}

	void DrawAreaExposeEvent (object obj, Gtk.ExposeEventArgs args)
	{
        RenderVisibleContent ();
		args.RetVal = false;
	}
}

public class FSView : FSBase
{
    enum StoreFields : int { Name, File };
    Gtk.TreeView    treeView;
	Gtk.TreeStore   treeStore;

    void AddFileToTree (Gtk.TreeIter parent, File fi, int toDepth)
    {
		Gtk.TreeIter thisIter;

		if (parent.Equals (Gtk.TreeIter.Zero))
            thisIter = this.treeStore.AppendNode();
		else 
            thisIter = this.treeStore.AppendNode (parent);

		this.treeStore.SetValue (thisIter, (int) StoreFields.Name, fi.Name);
		this.treeStore.SetValue (thisIter, (int) StoreFields.File, fi);

        if (fi is Directory) {
            Directory dir = (Directory) fi;
            if (dir.Children != null && toDepth > 1) {
                foreach (File child in dir.Children)
                    AddFileToTree (thisIter, child, toDepth - 1);
            }
        }
    }

    void TreeSelectionChanged(object obj, System.EventArgs args)
    {
        this.selection.Clear();
        this.treeView.Selection.SelectedForeach (
            delegate (Gtk.TreeModel model, Gtk.TreePath path, Gtk.TreeIter iter)
            {
                this.selection.Add (this.treeStore.GetValue (iter, (int) StoreFields.File) as File);
            }
            );
        UpdateSelectionDisplay();
        if (this.selection.Count > 0)
            MoveToSelection (this.selection[0]);
        QueueDraw();
    }

    public override Gtk.Widget CreateTree ()
    {
		this.treeStore = new Gtk.TreeStore (typeof (string), typeof (File));
        this.treeView = new Gtk.TreeView (this.treeStore);
		this.treeView.Selection.Mode = Gtk.SelectionMode.Multiple;
		this.treeView.RulesHint = true;
        this.treeView.EnableSearch = true;
        this.treeView.Selection.Changed += TreeSelectionChanged;

		// column for element name
        Gtk.TreeViewColumn nameColumn;
		Gtk.CellRendererText rendererText = new Gtk.CellRendererText ();
		rendererText.Xalign = 0.0f;
		nameColumn = new Gtk.TreeViewColumn ("Name", rendererText, 
											 "text", 0);
		this.treeView.InsertColumn (nameColumn, 0);

		Gtk.ScrolledWindow scrolledWindow = new Gtk.ScrolledWindow ();

		scrolledWindow.SetPolicy (Gtk.PolicyType.Automatic,
								  Gtk.PolicyType.Automatic);

		scrolledWindow.Add (this.treeView);
        scrolledWindow.ShowAll();

        // populate store ...
        AddFileToTree (Gtk.TreeIter.Zero, fsData.Root, 3);
        this.treeView.RowExpanded += TreeIdleRowExpand;

        Gtk.TreeIter root;
        if (this.treeStore.GetIterFirst (out root))
            this.treeView.ExpandRow (this.treeStore.GetPath (root), false);

        return scrolledWindow;
    }

    void TreeIdleRowExpand (object o, Gtk.RowExpandedArgs args)
    {
        Gtk.TreeIter iter = args.Iter;
        Gtk.TreeIter child;
        if (this.treeStore.IterChildren (out child, iter)) {
//            Console.WriteLine ("Expand sub children ...");
            do {
                File fi = this.treeStore.GetValue (child, (int) StoreFields.File) as File;
                Gtk.TreeIter subChildren;
                if (!this.treeStore.IterChildren (out subChildren, child) &&
                    fi is Directory && (fi as Directory).Children != null) {
                    // build some more tree ...
                    foreach (File childFile in (fi as Directory).Children)
                        AddFileToTree (child, childFile, 1);
                }
            } while (this.treeStore.IterNext (ref child));
        }
    }

    public FSView (FSData fsData)
        : base (fsData)
    {
    }
}
