/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using Gdk;
using Gtk;
using Pango;
using Cairo;
using System.Collections.Generic;
using System.Diagnostics;
using System.Runtime.InteropServices;

interface IHistNode {
	List<IHistNode> Children { get; }
	string Label      { get; }
	string ShortLabel { get; }
	double Self { get; }
	double Weight { get; }
}

class Node : IHistNode {
    Node            parent;
    List<IHistNode> children;
    string label;
	double self;
    double weight;

	public List<IHistNode> Children { get { return children; } }
	public virtual string Label        { get { return label; } }
	public virtual string ShortLabel   { get { return Label; } }
    public virtual string TreeElemName { get { return Label; } }
	public double Self   { get { return self; } }
	public double Weight {
		get { return weight; }
		set { weight = value; }
	}

    public Node NonEmptyParent {
        get {
            Node cur = this.parent, last = null;
            while (cur != null && cur.weight <= weight) {
                last = cur;
                cur = cur.parent;
            }
            cur = cur != null ? cur : last;
//            System.Console.WriteLine ("Parent of " + this.Label + " is "
//                                      + (cur != null ? cur.Label : "<null>"));
            return cur;
        }
    }

    public Node (Node parent, string label, double weight)
    {
        this.label = label;
        this.self = weight;
        if (parent != null)
            parent.AddChild (this);
    }
    public void AddChild(Node child)
    {
        if (this.children == null)
            this.children = new List<IHistNode>();
        children.Add (child);
        child.parent = this;
    }
    public void SortChildren()
    {
		if (children == null)
			return;

		children.Sort( delegate (IHistNode a, IHistNode b)
						{ return b.Weight.CompareTo(a.Weight); } );
        foreach (IHistNode n in children)
            ((Node)n).SortChildren();
    }
	public double Accumulate ()
	{
		double sum = 0;
		if (children != null)
			foreach (IHistNode ni in children)
			{
				Node n = (Node)ni;
				n.weight = n.Accumulate ();
				sum += n.weight;
			}
		sum += self;
		weight = sum;
		return sum;
	}
    List<Node> GetParents() {
        if (parent == null)
            return null;
        List<Node> parents = new List<Node>();
        for (Node cur = parent; cur != null; cur = cur.parent)
            parents.Add (cur);
        parents.Reverse();
        return parents;
    }
    public static Node CommonParent (Node a, Node b)
    {
        if (a == b)
            return a;
        List<Node> aPars = a.GetParents();
        List<Node> bPars = b.GetParents();
        if (aPars == null)
            return a;
        if (bPars == null)
            return b;
        int i, count = Math.Min (aPars.Count, bPars.Count);
        for (i = 0; i < count && aPars[i] == bPars[i]; i++);
        if (i == 0)
            return null;
        return aPars[i-1];
    }
    public bool IsAncestor (Node a)
    {
        Node cur;
        for (cur = a; cur != null && cur != this; cur = cur.parent);
        return cur != null && cur == this;
    }
}

struct HistogramSettings {
    public bool ElideEmpty;
    public HistogramSettings (bool elideEmpty)
    {
        this.ElideEmpty = elideEmpty;
    }
}

class HistogramLayout {

    HistogramSettings settings;
    public HistogramLayout (HistogramSettings settings)
    {
        this.settings = settings;
    }

	public virtual bool VisitRegion (IHistNode       node,
									 Cairo.Rectangle total, // complete area
									 Cairo.Rectangle inner, // minus borders
									 Cairo.Rectangle item)  // item itself
	{ // FIXME: make me abstract ...
		return true;
	}

	bool IsDegenerate (Cairo.Rectangle rect)
	{
		if (rect.Width < 10 || rect.Height < 10 ||
			rect.Width * rect.Height < 250)
			return true;
		else
			return false;
	}

	protected void Partition (Cairo.Rectangle     range,
							  double              proportion,
							  ref Cairo.Rectangle item,
							  out Cairo.Rectangle remainder)
	{
		// Cairo.Rectangle is highly unusable ...
		Cairo.Rectangle rect;
		int dx = 0, dy = 0;

		rect = new Cairo.Rectangle (range.X, range.Y, range.Width, range.Height);
		bool vert = rect.Height > rect.Width;
		if (vert)
			dy = (int) (rect.Height * proportion + 0.5);
		else
			dx = (int) (rect.Width * proportion + 0.5);

		item = new Cairo.Rectangle (rect.X, rect.Y,
								 	vert ? rect.Width : dx,
									vert ? dy : rect.Height);
		remainder = new Cairo.Rectangle (rect.X + dx, rect.Y + dy,
										 rect.Width - dx, rect.Height - dy);

//		Console.WriteLine ("Partition {0:##}% of " + range +
//						   " into " + item + " remainder " + remainder,
//						   proportion * 100.0);

	}

	protected Cairo.Rectangle ShrinkBorder (Cairo.Rectangle range)
	{
		return CairoRect.ShrinkBorder (range, 2);
	}

	void LayoutChildren (List<IHistNode> children,
						 Cairo.Rectangle rect, double totalWeight)
	{
		Cairo.Rectangle item = new Cairo.Rectangle();

		foreach (IHistNode n in children) {
			Partition (rect, n.Weight / totalWeight,
					   ref item, out rect);
			totalWeight -= n.Weight;
			LayoutItem (n, item);
		}
	}

    public void LayoutItem (IHistNode node, Cairo.Rectangle rect)
	{
		if (node == null)
			return;

		Cairo.Rectangle item = new Cairo.Rectangle();
		Cairo.Rectangle remainder;
		Cairo.Rectangle inner = ShrinkBorder (rect);

//		Console.WriteLine ("Item " + node.Label + " weight " + node.Weight +
//						   " Self " + node.Self + " rect " + rect);

        if (!settings.ElideEmpty || node.Self != 0)
        {
            Partition (inner, node.Self / node.Weight,
                       ref item, out remainder);

            if (!VisitRegion (node, rect, inner, item))
                return;
        } else {
            remainder = new Cairo.Rectangle();
            remainder = rect;
        }

		if (node.Children != null && !IsDegenerate (remainder))
			LayoutChildren (node.Children, remainder, node.Weight - node.Self);
	}
}

class CairoRect {
	static public bool Contains (Cairo.Rectangle rect, double x, double y)
	{
		return ( x > rect.X && x < rect.X + rect.Width &&
				 y > rect.Y && y < rect.Y + rect.Height);
	}
	static public Cairo.Rectangle Union (Cairo.Rectangle a, Cairo.Rectangle b)
	{
		double x1, y1;
		double x2, y2;
		x1 = Math.Min (a.X, b.X);
		y1 = Math.Min (a.Y, b.Y);
		x2 = Math.Max (a.X + a.Width,  b.X + b.Width);
		y2 = Math.Max (a.Y + a.Height, b.Y + b.Height);
		return new Cairo.Rectangle (x1, y1, x2 - x1, y2 - y1);
	}
	static public bool Intersects (Cairo.Rectangle a, Cairo.Rectangle b)
	{
		bool noXOverlap = a.X + a.Width  <= b.X || b.X + b.Width  <= a.X;
		bool noYOverlap = a.Y + a.Height <= b.Y || b.Y + b.Height <= a.Y;
		return !(noXOverlap && noYOverlap);
	}
	static public Cairo.Rectangle FromGdk (Gdk.Rectangle r)
	{
		return new Cairo.Rectangle (r.X, r.Y, r.Width, r.Height);
	}
	static public Gdk.Rectangle ToGdk (Cairo.Rectangle r)
	{
		Gdk.Rectangle ret = new Gdk.Rectangle();

		// FIXME: rounding ... (?)
		ret.X = (int)r.X;
		ret.Y = (int)r.Y;
		ret.Width = (int)r.Width;
		ret.Height = (int)r.Height;

		return ret;
	}
	static public Cairo.Rectangle ShrinkBorder (Cairo.Rectangle range, double border)
	{
		return new Cairo.Rectangle (range.X + border, range.Y + border,
									Math.Max (range.Width - border*2, 0),
									Math.Max (range.Height - border*2, 0));
	}
}

interface IHistColorModel {
    Cairo.Color GetColor (IHistNode node);
};

class HistogramRenderer : HistogramLayout {
	const double LeftRadians = -Math.PI / 2;

	IHistNode        selected;
	IHistNode        prelighted;
	IHistColorModel  colorModel;
	Cairo.Rectangle  selectedRegion;
	Cairo.Rectangle  renderRegion;
	
	public Cairo.Context Cr;
	
	public HistogramRenderer(Cairo.Context cr,
							 IHistNode selected,
							 IHistNode prelighted,
							 Cairo.Rectangle   renderRegion,
							 IHistColorModel   colorModel,
                             HistogramSettings settings)
        : base (settings)
	{
		this.Cr = cr;
        this.Cr.SetFontSize (16.0);
		this.colorModel = colorModel;
		this.selected = selected;
		this.prelighted = prelighted;
		this.selectedRegion = new Cairo.Rectangle (0,0,0,0);
		this.renderRegion = renderRegion;
	}

	[DllImport ("pangocairo-1.0")]
	static extern IntPtr pango_cairo_create_layout (IntPtr cr);

	[DllImport ("pangocairo-1.0")]
	static extern void pango_cairo_show_layout (IntPtr cr, IntPtr layout);

	/* This is lame.  layout.Raw is protected, so we can't access it from the
	 * outside.  So, we need a derived class just to access that member.
	 */

	class PangoCairoLayout : Pango.Layout {
		public PangoCairoLayout (Cairo.Context cr) :
			base (pango_cairo_create_layout (cr.Handle))
		{
		}

		public void Show (Cairo.Context cr)
		{
			pango_cairo_show_layout (cr.Handle, this.Raw);
		}
	}

    static bool DrawTextFit (Cairo.Context gr, string str,
                             Cairo.Rectangle region)
    {
		Pango.FontDescription fd;
		PangoCairoLayout layout;
		Pango.Rectangle log_rect, ink_rect;

		layout = new PangoCairoLayout (gr);

		fd = new Pango.FontDescription ();
		fd.Family = "sans";
		fd.Size = Pango.Units.FromPixels (8);

		layout.FontDescription = fd;
		layout.SetText (str);

		layout.GetPixelExtents (out log_rect, out ink_rect);

        gr.Color = new Cairo.Color(0,0,0,1);

		double ink_w, ink_h;
		double avail_w, avail_h;

		ink_w = ink_rect.Width;
		ink_h = ink_rect.Height;
		avail_w = region.Width;
		avail_h = region.Height;

		if (ink_w <= avail_w && ink_h <= avail_h) {
			gr.Save ();
			gr.MoveTo (new PointD (region.X + (avail_w - ink_w) / 2.0,
								   region.Y + (avail_h - ink_h) / 2.0));
		} else if (ink_w <= avail_h && ink_h <= avail_w) {
			gr.Save ();
			gr.MoveTo (new PointD (region.X + (avail_w - ink_h) / 2.0,
								   region.Y + ink_w + (avail_h - ink_w) / 2.0));
			gr.Rotate (LeftRadians);
		} else
			return false;

		layout.Show (gr);
		gr.Restore ();

        return true;
    }

	public override bool VisitRegion (IHistNode node,
									  Cairo.Rectangle total,
									  Cairo.Rectangle inner,
									  Cairo.Rectangle item)
	{
//		Console.WriteLine ("Visit to render " + node.Label);
		bool isSelection = node == this.selected;
		bool isPrelighted = node == this.prelighted;

		if (isSelection)
			this.selectedRegion = total;

		// Draw our backgrounds ...
		Cairo.Color c = colorModel.GetColor (node);

		bool selected = CairoRect.Contains (this.selectedRegion, inner.X, inner.Y);
		if (selected)
            c = GUtils.SaturateColor (c, 2);

		Cairo.Color light, dark, black;
		GUtils.SpreadColor (c, out light, out dark);
        black = new Cairo.Color (0, 0, 0);

		if (isSelection || isPrelighted) {
			Cairo.Rectangle border = CairoRect.ShrinkBorder (total, 0.5);
			Cr.Rectangle (border);
			Cr.LineWidth = 1.0;
			Cr.Color = black;
			Cr.Stroke ();
		} else {
			/* Top left */

			Cr.MoveTo (total.X, total.Y);
			Cr.LineTo (total.X + total.Width, total.Y);
			Cr.LineTo (total.X + total.Width - 1, total.Y + 1);
			Cr.LineTo (total.X + 1, total.Y + 1);
			Cr.LineTo (total.X + 1, total.Y + total.Height - 1);
			Cr.LineTo (total.X, total.Y + total.Height);
			Cr.ClosePath ();
			Cr.Color = light;
			Cr.Fill ();

			/* Bottom right */

			Cr.MoveTo (total.X + total.Width, total.Y);
			Cr.LineTo (total.X + total.Width, total.Y + total.Height);
			Cr.LineTo (total.X, total.Y + total.Height);
			Cr.LineTo (total.X + 1, total.Y + total.Height - 1);
			Cr.LineTo (total.X + total.Width - 1, total.Y + total.Height - 1);
			Cr.LineTo (total.X + total.Width - 1, total.Y + 1);
			Cr.ClosePath ();
			Cr.Color = dark;
			Cr.Fill ();
		}

		/* Inside */

		Cr.Color = c;
		Cr.Rectangle (inner);
		Cr.Fill ();

		if (!DrawTextFit (Cr, node.Label, item))
            DrawTextFit (Cr, node.ShortLabel, item);

		return CairoRect.Intersects (total, this.renderRegion);
	}
}

class HistogramHitTest : HistogramLayout {
	bool keepSearching;
	double x, y;
    public Cairo.Rectangle FoundRect;
	public IHistNode Found;

	public HistogramHitTest (HistogramSettings settings, double x, double y)
        : base (settings)
    {
		this.keepSearching = true;
		this.x = x;
		this.y = y;
	}

	public override bool VisitRegion (IHistNode node,
									  Cairo.Rectangle total,
									  Cairo.Rectangle inner,
									  Cairo.Rectangle item)
	{
		if (!keepSearching)
			return false;

		if (CairoRect.Contains (item, this.x, this.y) ||   // the item itself
			(CairoRect.Contains (total, this.x, this.y) && // or it's border
			 ! CairoRect.Contains (inner, this.x, this.y))) {
			Found = node;
			FoundRect = total;
			keepSearching = false;
		}

		return CairoRect.Contains (total, this.x, this.y);
	}
}

class Histogram : Gtk.DrawingArea {
    IHistNode         root;
	IHistNode         selected;
	IHistNode         prelighted;
	IHistColorModel   colorModel;
	Cairo.Rectangle   selectedArea;
	Cairo.Rectangle   prelightedArea;
    HistogramSettings settings;
	
    public Histogram (HistogramSettings settings,
                      IHistNode root, IHistColorModel colorModel)
    {
		AddEvents ((int) EventMask.ButtonPressMask | (int) EventMask.PointerMotionMask);
		this.root = root;
		this.colorModel = colorModel;
        this.settings = settings;
		ExposeEvent += DrawAreaExposeEventCallback;
		ButtonPressEvent += ButtonPressEventCallback;
		MotionNotifyEvent += DrawAreaMotionNotifyEventCallback;
    }

	public IHistNode Root {
		get { return root; }
		set { if (root != value) {root = value; QueueDraw(); } }
	}

	public IHistNode Selected {
		get { return selected; }
		set { if (selected != value) { selected = value; QueueDraw(); } }
	}

	Cairo.Rectangle GetAllocation ()
	{
		return new Cairo.Rectangle
			(0, 0, Allocation.Width, Allocation.Height);
	}

	IHistNode GetNodeAt (double x, double y, ref Cairo.Rectangle rect)
	{
		HistogramHitTest hit = new HistogramHitTest (this.settings, x, y);
		hit.LayoutItem (root, GetAllocation());
//		if (hit.Found != null)
//			Console.WriteLine ("Hit " + hit.Found.Label);
		rect = hit.FoundRect;
		return hit.Found;
	}

	// FIXME: add a 'EnterSelect' delegage ...
	// use FileView.cs to update the root on that ...

	void DrawAreaExposeEventCallback (object obj, ExposeEventArgs args)
	{
		Gdk.Rectangle[] rectangles = args.Event.Region.GetRectangles ();

		// FIXME: we don't get much win from this pwrt. our button
		// press use-case - we should intersect later
		Cairo.Rectangle rect = CairoRect.FromGdk (rectangles[0]);
		for (int i = 1; i < rectangles.Length; i++)
			rect = CairoRect.Union (rect, CairoRect.FromGdk (rectangles[i]));

		using (Cairo.Context cr = Gdk.CairoHelper.Create (GdkWindow)) {
			HistogramRenderer rend = new HistogramRenderer (cr, selected, prelighted, rect,
															colorModel, this.settings);
			rend.LayoutItem (root, GetAllocation());
		}
		args.RetVal = false;
	}

	void DrawAreaMotionNotifyEventCallback (object obj, MotionNotifyEventArgs args)
	{
		IHistNode node;
		Cairo.Rectangle area;

		int x = (int) args.Event.X;
		int y = (int) args.Event.Y;

		area = new Cairo.Rectangle ();

		node = GetNodeAt (x, y, ref area);
		Prelight (node, area);

		args.RetVal = true;
	}

	void QueueDrawRect (Cairo.Rectangle range)
	{
		QueueDrawArea ((int)range.X, (int)range.Y,
					   (int)range.Width, (int)range.Height);
	}

	public delegate void SelectionEventHandler (Histogram hist, bool zoom,
                                                IHistNode selected,
                                                Cairo.Rectangle range);
    SelectionEventHandler handler;
    public event SelectionEventHandler SelectionEvent {
        add    { this.handler = value; }
        remove { this.handler = null; }
    }

	void ButtonPressEventCallback (object obj, ButtonPressEventArgs args)
	{
        Gdk.EventButton ev = args.Event;
		Cairo.Rectangle range = new Cairo.Rectangle();
		selected = GetNodeAt (ev.X, ev.Y, ref range);

        if (this.handler != null && this.selected != null)
        {
            this.handler (this, ev.Type == Gdk.EventType.TwoButtonPress,
                          selected, range);
            args.RetVal = true;
        }
        args.RetVal = false;
    }

    public void Select (IHistNode node, Cairo.Rectangle range)
    {
        QueueDrawRect (selectedArea);
        QueueDrawRect (range);
		selectedArea = range;
    }

	public void Prelight (IHistNode node, Cairo.Rectangle area)
	{
        if (node == prelighted)
            return;

		if (prelighted != null)
			QueueDrawRect (prelightedArea);

		QueueDrawRect (area);

		prelighted = node;
		prelightedArea = area;
	}
}
