#include <stdio.h>
#include <malloc.h>
#include "interface.h"

typedef struct {
    IDiskSim    *ids;
    double       cur_time;
    unsigned int completed : 1;

    double           next_callback_time;
    IDiskSimCallback callback;
    void            *callback_closure;
} SimData;

static void
schedule_callback (IDiskSim        *ids,
                   IDiskSimCallback callback,
                   void            *callback_closure,
                   IDiskSimTime     time_to_call,
                   void            *closure)
{
    SimData *sim = closure;

    sim->callback = callback;
    sim->callback_closure = callback_closure;
    sim->next_callback_time = time_to_call;
}

static void
deschedule_callback (IDiskSim        *ids,
                     IDiskSimCallback callback,
                     void            *closure)
{
    SimData *sim = closure;

    sim->next_callback_time = -1.0;
    sim->callback = NULL;
    sim->callback_closure = NULL;
}

static void
done_io_notify (IDiskSim        *ids,
                IDiskSimRequest *req,
                IDiskSimTime     time_completed,
                void            *closure)
{
    SimData *sim = closure;

    sim->completed = 1;
    sim->cur_time = time_completed;
}


static SimData *
create_simulation (const char *param_fname)
{
    SimData *sim;
    IDiskSim *ids;

    if (!(sim = calloc (1, sizeof (SimData))))
        return NULL;

    if (!(ids = idisksim_init (param_fname,
                               schedule_callback,
                               deschedule_callback,
                               done_io_notify,
                               sim))) {
        free (sim);
        return NULL;
    }

    fprintf (stderr, "block count %d size %d\n",
             idisksim_get_blk_count (ids),
             idisksim_get_blk_size (ids));

    sim->ids = ids;
    sim->cur_time = 0.0;
    sim->completed = 0;

    sim->next_callback_time = -1.0;
    sim->callback = NULL;
    sim->callback_closure = NULL;

    return sim;
}

static void
sim_free (SimData *sim)
{
    if (!sim)
        return;
    idisksim_free (sim->ids);
    free (sim);
}

static void
sim_callback (SimData *sim)
{
    IDiskSimCallback callback = sim->callback;
    void *closure = sim->callback_closure;

    /* the callback may register another callback
       so de-register it first */
    sim->next_callback_time = -1;
    sim->callback = NULL;
    sim->callback_closure = NULL;

    callback (sim->ids, closure, sim->cur_time);
}

static void
do_read (SimData *sim, int blk, int cnt)
{
    IDiskSimRequest *req = idisksim_request_new (IDISKSIM_READ, blk, cnt);
    double before;
        
    before = sim->cur_time;
    idisksim_submit (sim->ids, req, sim->cur_time);

    /* Wait for block to complete */
    while (sim->next_callback_time >= 0) {
        /* advance the clock */
        sim->cur_time = sim->next_callback_time;
        /* call our callback */
        sim_callback (sim);
    }
    fprintf (stderr, "\telapsed %.2g (ms)\n", (sim->cur_time - before)*1000);
    
    idisksim_request_free (req);
}

int main (int argc, char **argv)
{
    int i;
    int read_sects = (4096/512)*16;
    SimData *sim;
    static int seeks[] = { 0x2f7299, 0x2f72a9, 
                           0x2f6a3a, 0x2f6a4a,
                           0x000001, 0x100000 };

    /* single arg. is parameter filename */
    sim = create_simulation (argv[argc-1]);

    fprintf (stderr, "512b reads\n");
    for (i = 0; i < 10; i++)
        do_read (sim, i, 1);

    fprintf (stderr, "64k reads\n");
    for (i = 0; i < 25; i++)
        do_read (sim, i * read_sects + 0x17b8648, read_sects);

    fprintf (stderr, "seeks\n");
    for (i = 0; i < 5; i++)
        do_read (sim, seeks[i], read_sects);

    fprintf (stderr, "completed in %g seconds\n", sim->cur_time);
    sim_free (sim);
    
    return 0;
}
