/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003-2004 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003      Johan Dahlin <johan@gnome.org>
 * Copyright (C) 2003      Anders Carlsson <andersca@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <math.h>
#include <time.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <gconf/gconf-client.h>
#include "main-window.h"
#include "song-list-view.h"
#include "smart-playlist-dialog.h"
#include "play-list-view.h"
#include "player.h"
#include "playlist.h"
#include "playlist-xml.h"
#include "id3-tag.h"
#include "cursors.h"
#include "stock-icons.h"
#include "utils.h"
#include "volume-button.h"

/*#define CRACK 1*/

static void     main_window_class_init                 (MainWindowClass  *klass);
static void     main_window_init                       (MainWindow       *window);
static void     main_window_finalize                   (GObject          *object);
static void     genre_selection_changed_cb             (GtkTreeSelection *selection,
							MainWindow       *window);
static void     artist_selection_changed_cb            (GtkTreeSelection *selection,
							MainWindow       *window);
static void     album_selection_changed_cb             (GtkTreeSelection *selection,
							MainWindow       *window);
static void     volume_changed_cb                      (GtkWidget        *button,
							int               volume,
							MainWindow       *window);
static void     seek_scale_value_changed_cb            (GtkWidget        *widget,
							MainWindow       *window);
static gboolean seek_scale_button_press_cb             (GtkWidget        *widget,
							GdkEventButton   *event,
							MainWindow       *window);
static gboolean seek_scale_button_release_cb           (GtkWidget        *widget,
							GdkEventButton   *event,
							MainWindow       *window);
static gboolean seek_scale_motion_notify_cb (GtkWidget      *widget,
			     GdkEventMotion *event,
					     MainWindow     *window);

static void     search_entry_activate_cb               (GtkEntry         *entry,
							MainWindow       *window);
static void     update_playing_info                    (MainWindow       *window);
static void     player_end_of_stream_cb                (Player           *player,
							Song             *song,
							MainWindow       *window);
static void     player_error_cb                        (Player           *player,
							char             *error,
							MainWindow       *window);
static void     player_tick_cb                         (Player           *player,
							long              secs,
							MainWindow       *window);
static void     player_state_changed_cb                (Player           *player,
							PlayerState       state,
							MainWindow       *window);
static void     load_glade_gui                         (MainWindow       *window);
static void     update_edit_smart_playlist_sensitivity (MainWindow       *window);
static void     playlist_selection_changed_cb          (GtkTreeSelection *selection,
							MainWindow       *window);
static void     setup_playlist_tree                    (MainWindow       *window);
static void     new_playlist_cb                        (MainWindow       *window);
static void     new_smart_playlist_cb                  (MainWindow       *window);
static void     edit_smart_playlist_cb                 (MainWindow       *window);
static void     add_folder_cb                          (MainWindow       *window);
static void     quit_cb                                (MainWindow       *window);
static void     actions_play_cb                        (MainWindow       *window);
static void     actions_pause_cb                       (MainWindow       *window);
static void     actions_stop_cb                        (MainWindow       *window);
static void     actions_prev_cb                        (MainWindow       *window);
static void     actions_next_cb                        (MainWindow       *window);
static void     actions_song_columns_cb                (MainWindow       *window);
static void     help_about_cb                          (MainWindow       *window);
static void     add_paths                              (MainWindow       *window,
							GList            *paths);
static char *   item_factory_trans_cb                  (const char       *path,
							gpointer          data);
static GList *  query_update_func                      (Query            *query,
							gboolean          first,
							GList            *current,
							gpointer          user_data);
static void     start_query                            (MainWindow       *window,
							Query            *query);
static void     gconf_notify_cb                        (GConfClient      *client,
							guint             cnxn_id,
							GConfEntry       *entry,
							gpointer          user_data);


#ifdef CRACK
static void       generate_m3u_cb               (MainWindow       *window);
#endif



struct _MainWindowPriv
{
  SongDB        *db;
  Player        *player;
  GList         *playlists;

  gboolean       seeking;
  guint          seeking_idle_id;
  
  /* Current query. */
  Query         *query;
  
  gboolean       browse_mode;

  GHashTable    *artist_hash;
  GHashTable    *album_hash;

  guint          search_timeout_id;
  
  GtkListStore  *playlist_store;
  GtkListStore  *genre_store;
  GtkListStore  *artist_store;
  GtkListStore  *album_store;
  SongListModel *song_model;

  GtkWidget     *search_entry;
  GtkTreeView   *playlist_tree;
  GtkTreeView   *genre_tree;
  GtkTreeView   *artist_tree;
  GtkTreeView   *album_tree;
  SongListView  *song_tree;
  GtkWidget     *playlist_sw;
  
  GtkWidget     *browse_pane;
  
  GtkWidget     *play_button;
  GtkWidget     *volume_button;
  GtkWidget     *info_label;
  GtkWidget     *shuffle_button;
  GtkWidget     *next_button;
  GtkWidget     *prev_button;
  GtkWidget     *menu_next_button;
  GtkWidget     *menu_prev_button;
  GtkWidget     *edit_smart_menu_item;
  GtkWidget     *playing_label;
  GtkWidget     *playing_song_label;
  GtkWidget     *playing_artist_label;
  GtkWidget     *playing_info_hbox;
 
  GtkWidget     *tick_label;
  GtkWidget     *length_label;

  GtkWidget     *seek_scale;
  
  GtkWidget     *top_vbox;

  int            width;
  int            height;
  
  GdkCursor     *watch_arrow_cursor;

  int            num_songs;
  guint          total_length;
  guint64        total_size;
};


#define FIRST_BURST    1500
#define BURST          500

#define FIXUP_STOCK_ICON(widget,icon) \
	gtk_image_set_from_stock (GTK_IMAGE (glade_xml_get_widget (glade, widget)), \
                                  icon, GTK_ICON_SIZE_LARGE_TOOLBAR)

static GObjectClass *parent_class;
static MainWindow *glob_window;

static int genre_indices[] = {
  0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, -1
};


enum {
  GENRE_COL_NAME,
  GENRE_COL_NUM,
  GENRE_NUM_COLS
};

enum {
  ARTIST_COL_NAME,
  ARTIST_COL_KEY,
  ARTIST_COL_ALL,
  ARTIST_NUM_COLS
};

enum {
  ALBUM_COL_NAME,
  ALBUM_COL_KEY,
  ALBUM_COL_ALL,
  ALBUM_NUM_COLS
};

enum {
  TARGET_STRING,
  TARGET_URI_LIST,
  TARGET_ICON_LIST,
};

enum {
  ACTION_PREV = 1,
  ACTION_NEXT,
  ACTION_EDIT
};

static const GtkTargetEntry songs_target_types[] = {
  { "STRING",     0, TARGET_STRING },
  { "text/plain", 0, TARGET_STRING },
  { "text/uri-list", 0, TARGET_URI_LIST },
  { "x-special/gnome-icon-list", 0, TARGET_ICON_LIST }
};

static GtkItemFactoryEntry menu_items[] = {
  /* File */
  { N_("/_File"),                       NULL,          NULL,             0, "<Branch>"                       },
  { N_("/File/_New Playlist"),          "<control>N",  new_playlist_cb,       0, "<StockItem>", GTK_STOCK_NEW     },  
  { N_("/File/_New Smart Playlist..."), "<control>M",  new_smart_playlist_cb, 0, "<StockItem>", GTK_STOCK_NEW },  
  { N_("/File/_Edit Smart Playlist..."),NULL,          edit_smart_playlist_cb, ACTION_EDIT, "<Item>"                  },
  { N_("/File/sep1"),                   NULL,          NULL,             0, "<Separator>"                    },
  { N_("/File/_Add folder"),            NULL,          add_folder_cb,    0, "<StockItem>", GTK_STOCK_ADD     },  
  { N_("/File/sep2"),                   NULL,          NULL,             0, "<Separator>"                    },
  { N_("/File/_Quit"),                  "<control>Q",  quit_cb,          0, "<StockItem>", GTK_STOCK_QUIT    },
  
  /* Actions */
  { N_("/_Actions"),                    NULL,          NULL,             0, "<Branch>"                         },
  { N_("/Actions/_Play"),               NULL,          actions_play_cb, 0, "<StockItem>", JAMBOREE_STOCK_PLAY },
  { N_("/Actions/_Stop") ,              NULL,          actions_stop_cb, 0, "<StockItem>", JAMBOREE_STOCK_STOP },
  { "/Actions/sep1",                    NULL,          NULL,            0, "<Separator>" },
  { N_("/Actions/_Previous"),           NULL,          actions_prev_cb, ACTION_PREV, "<StockItem>", JAMBOREE_STOCK_PREV },
  { N_("/Actions/_Next"),               NULL,          actions_next_cb, ACTION_NEXT, "<StockItem>", JAMBOREE_STOCK_NEXT },
  { "/Actions/sep2",                    NULL,          NULL,            0, "<Separator>" },
  { N_("/Actions/_Edit Song Columns..."),NULL,         actions_song_columns_cb, 0, NULL },

#ifdef CRACK
  { "/Actions/sepcrack",                NULL,          NULL,            0, "<Separator>" },
  { N_("/Actions/_Export playlist"),    NULL,          generate_m3u_cb, 0, NULL },
#endif
  
  /* Help */
  { N_("/_Help"),                       NULL,          NULL,             0, "<Branch>"                       },
  { N_("/_Help/_About"),                NULL,          help_about_cb,    0, "<StockItem>", GNOME_STOCK_ABOUT }
};


GType
main_window_get_type (void)
{
  static GType type = 0;
	
  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (MainWindowClass),
	  NULL,           /* base_init */
	  NULL,           /* base_finalize */
	  (GClassInitFunc) main_window_class_init,
	  NULL,           /* class_finalize */
	  NULL,           /* class_data */
	  sizeof (MainWindow),
	  0,
	  (GInstanceInitFunc) main_window_init,
	};

      type = g_type_register_static (GTK_TYPE_WINDOW,
				     "MainWindow",
				     &info, 0);
    }

  return type;
}

static void
main_window_class_init (MainWindowClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = main_window_finalize;
}

static void
grab_key_cb (XF86Key key, gpointer user_data)
{
  MainWindow *window = user_data;
  
  switch (key)
    {
    case XF86AUDIO_PLAY:
      actions_play_cb (window);
      break;
    case XF86AUDIO_PAUSE:
      actions_pause_cb (window);
      break;
    case XF86AUDIO_STOP:
      actions_stop_cb (window);
      break;
    case XF86AUDIO_PREV:
      actions_prev_cb (window);
      break;
    case XF86AUDIO_NEXT:
      actions_next_cb (window);
      break;
      
    default:
      break;
    }
}

static gboolean
window_configure_event_cb (GtkWidget         *window,
			   GdkEventConfigure *event,
			   gpointer           user_data)
{
  MainWindow *main_window = MAIN_WINDOW (window);
  MainWindowPriv *priv = main_window->priv;
  
  if (event->width == priv->width && event->height == priv->height)
    return FALSE;

  gconf_client_set_int (gconf_client, "/apps/jamboree/ui/width", event->width, NULL);
  gconf_client_set_int (gconf_client, "/apps/jamboree/ui/height", event->height, NULL);

  return FALSE;
}

static void
main_window_init (MainWindow *window)
{
  GtkAccelGroup *accel_group;      
  GtkItemFactory *item_factory;
  GtkWidget *menubar;
  GdkPixmap *cursor, *mask;
  MainWindowPriv *priv;
  GList *list;
  GdkPixbuf *pixbuf;
  int volume;
  
  priv = g_new0 (MainWindowPriv, 1);
  window->priv = priv;

  grab_keys (grab_key_cb, window);
  
  pixbuf = gdk_pixbuf_new_from_file (DATADIR "/jamboree/master-out.png", NULL);
  list = g_list_append (NULL, pixbuf);
  gtk_window_set_default_icon_list (list);
  g_list_free (list);
  g_object_unref (pixbuf);

  priv->artist_hash = g_hash_table_new (g_str_hash, g_str_equal);
  priv->album_hash = g_hash_table_new (g_str_hash, g_str_equal);

  priv->browse_mode = TRUE;

  priv->player = player_new ();
  
  g_signal_connect (priv->player,
		    "end_of_stream",
		    G_CALLBACK (player_end_of_stream_cb),
		    window);
  
  g_signal_connect (priv->player,
		    "tick",
		    G_CALLBACK (player_tick_cb),
		    window);

  g_signal_connect (priv->player,
		    "state_changed",
		    G_CALLBACK (player_state_changed_cb),
		    window);
  
  g_signal_connect (priv->player,
		    "error", 
		    G_CALLBACK (player_error_cb),
		    window);

  load_glade_gui (window);

  priv->width = gconf_client_get_int (gconf_client, "/apps/jamboree/ui/width", NULL);
  priv->height = gconf_client_get_int (gconf_client, "/apps/jamboree/ui/height", NULL);

  if (priv->width > 0 && priv->height > 0)
    gtk_window_set_default_size (GTK_WINDOW (window), priv->width, priv->height);
  
  g_signal_connect (window,
		    "configure_event",
		    G_CALLBACK (window_configure_event_cb),
		    NULL);
  
  accel_group = gtk_accel_group_new ();
  gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
  g_object_unref (accel_group);
  
  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", accel_group);
  
  /* Set up item factory to go away with the window. */
  g_object_ref (item_factory);
  gtk_object_sink (GTK_OBJECT (item_factory));
  g_object_set_data_full (G_OBJECT (window),
			  "<main>",
			  item_factory,
			  (GDestroyNotify) g_object_unref);
  
  gtk_item_factory_set_translate_func (item_factory,
				       item_factory_trans_cb,
				       NULL,
				       NULL);

  gtk_item_factory_create_items (item_factory,
				 G_N_ELEMENTS (menu_items),
				 menu_items,
				 window);

  menubar = gtk_item_factory_get_widget (item_factory, "<main>");
  priv->menu_prev_button = gtk_item_factory_get_widget_by_action (item_factory, ACTION_PREV);
  priv->menu_next_button = gtk_item_factory_get_widget_by_action (item_factory, ACTION_NEXT);
  priv->edit_smart_menu_item = gtk_item_factory_get_widget_by_action (item_factory, ACTION_EDIT);
  
  gtk_box_pack_start (GTK_BOX (priv->top_vbox), menubar, FALSE, FALSE, 0);
  gtk_box_reorder_child (GTK_BOX (priv->top_vbox), menubar, 0);
  gtk_widget_show_all (menubar);

  cursor = gdk_bitmap_create_from_data (NULL, cursor_bits, 32, 32);
  mask = gdk_bitmap_create_from_data (NULL, cursor_mask_bits, 32, 32);
  priv->watch_arrow_cursor =  gdk_cursor_new_from_pixmap (cursor, mask,
							  &GTK_WIDGET (window)->style->black,
							  &GTK_WIDGET (window)->style->white,
							  2, 2);
  g_object_unref (cursor);
  g_object_unref (mask);

  gconf_client_notify_add (gconf_client, GCONF_PATH,
			   gconf_notify_cb,
			   window,
			   NULL,
			   NULL);

  volume = gconf_client_get_int (gconf_client,
				 GCONF_PATH "/volume",
				 NULL);

  volume_button_set_volume (VOLUME_BUTTON (priv->volume_button), volume);
}

static void
main_window_finalize (GObject *object)
{
  MainWindow     *window = MAIN_WINDOW (object);
  MainWindowPriv *priv = window->priv;

  g_object_unref (priv->player);
  g_object_unref (priv->db);

  g_list_foreach (priv->playlists, (GFunc) playlist_free, NULL); 
  g_list_free (priv->playlists);

  gdk_cursor_unref (priv->watch_arrow_cursor);

  g_hash_table_destroy (priv->artist_hash);
  g_hash_table_destroy (priv->album_hash);
  
  g_free (priv);
  
  if (G_OBJECT_CLASS (parent_class)->finalize)
    G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
add_song (MainWindow *window, Song *song)
{
  MainWindowPriv *priv;
  GtkTreeIter iter;
  const char *str;

  priv = window->priv;
 
  priv->num_songs++;
  priv->total_size += song->filesize;
  priv->total_length += song->length;
  
  song_list_model_add (priv->song_model, song);

  if (!priv->browse_mode)
    return;

  str = string_entry_get_str (song->artist);
  
  if (!g_hash_table_lookup (priv->artist_hash, str))
    {
      g_hash_table_insert (priv->artist_hash, (char*)str, GINT_TO_POINTER (TRUE));
      
      gtk_list_store_append (priv->artist_store, &iter);
      gtk_list_store_set (priv->artist_store, &iter,
			  ARTIST_COL_NAME, str,
			  ARTIST_COL_KEY, string_entry_get_collated (song->artist),
			  -1);
    }

  str = string_entry_get_str (song->album);

  if (!g_hash_table_lookup (priv->album_hash, str))
    {
      g_hash_table_insert (priv->album_hash, (char*)str, GINT_TO_POINTER (TRUE));
      
      gtk_list_store_append (priv->album_store, &iter);
      gtk_list_store_set (priv->album_store, &iter,
			  ALBUM_COL_NAME, str,
			  ALBUM_COL_KEY, string_entry_get_collated (song->album),
			  -1);
    }
}

static GtkListStore *
create_genre_store (void)
{
  GtkListStore *store;
  GtkTreeIter iter;
  int i = 0;
  char *str;

  store = gtk_list_store_new (GENRE_NUM_COLS,
			      G_TYPE_STRING,
			      G_TYPE_INT);

  while (genre_indices[i] != -1)
    {
      gtk_list_store_append (store, &iter);
      gtk_list_store_set (store, &iter,
			  GENRE_COL_NUM, genre_indices[i],
			  GENRE_COL_NAME, id3_genres[genre_indices[i]],
			  -1);
      
      i++;
    };

  if (i == 1)
    str = g_strdup (_("All (1 genre)"));
  else
    str = g_strdup_printf (_("All (%d genres)"), i);
    
  gtk_list_store_prepend (store, &iter);
  gtk_list_store_set (store, &iter,
		      GENRE_COL_NUM, -1,
		      GENRE_COL_NAME, str,
		      -1);
  g_free (str);
  
  return store;
}

static gboolean
foreach_remove_cb (gpointer key, gpointer data, gpointer user_data)
{
  return TRUE;
}

static void
update_songs_info (MainWindow *window)
{
  MainWindowPriv *priv;
  GString *str;
  char *tmp;

  priv = window->priv;
  
  str = g_string_new (NULL);
  
  if (priv->num_songs == 1)
    g_string_append (str, _("1 song"));
  else if (priv->num_songs == 0)
    g_string_append (str, _("0 songs"));
  else
    g_string_append_printf (str, _("%d songs"), priv->num_songs);
  
  g_string_append (str, ", ");
  
  tmp = format_length_for_display (priv->total_length);
  g_string_append (str, tmp);
  g_free (tmp);
  
#if 0
  g_string_append (str, ", ");
  
  tmp = format_file_size_for_display (priv->total_size);
  g_string_append (str, tmp);
  g_free (tmp);
#endif
  
  tmp = g_string_free (str, FALSE);
  gtk_label_set_text (GTK_LABEL (priv->info_label), tmp);
  g_free (tmp);
}

static void
setup_album_view (MainWindow *window)
{
  MainWindowPriv *priv;
  GtkTreeView *view;
  GtkListStore *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreePath *path;

  priv = window->priv;

  view = GTK_TREE_VIEW (priv->album_tree);
  model = GTK_LIST_STORE (gtk_tree_view_get_model (view));
  
  selection = gtk_tree_view_get_selection (view);

  g_signal_handlers_block_by_func (selection, album_selection_changed_cb, window);
  
  gtk_list_store_clear (model);

  if (GTK_WIDGET_REALIZED (priv->album_tree))
    gtk_tree_view_scroll_to_point (GTK_TREE_VIEW (priv->album_tree), 0, 0);

  if (priv->browse_mode)
    {
      gtk_list_store_append (model, &iter);
      gtk_list_store_set (model, &iter,
			  ALBUM_COL_NAME, _("All (... albums)"), /*  */
			  ALBUM_COL_ALL, TRUE,
			  -1);

      path = gtk_tree_path_new_first ();
      gtk_tree_selection_select_path (selection, path);
      gtk_tree_path_free (path);
    }
  
  g_signal_handlers_unblock_by_func (selection,
				     album_selection_changed_cb,
				     window);
}

static void
setup_artist_view (MainWindow *window)
{
  MainWindowPriv *priv;
  GtkTreeView *view;
  GtkListStore *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreePath *path;

  priv = window->priv;

  view = GTK_TREE_VIEW (priv->artist_tree);
  model = GTK_LIST_STORE (gtk_tree_view_get_model (view));
  
  selection = gtk_tree_view_get_selection (view);
  g_signal_handlers_block_by_func (selection, artist_selection_changed_cb, window);
  
  gtk_list_store_clear (model);

  if (GTK_WIDGET_REALIZED (priv->artist_tree))
    gtk_tree_view_scroll_to_point (GTK_TREE_VIEW (priv->artist_tree), 0, 0);
  
  if (priv->browse_mode)
    {
      gtk_list_store_append (model, &iter);
      gtk_list_store_set (model, &iter,
			  ARTIST_COL_NAME, _("All (... artists)"),
			  ARTIST_COL_ALL, TRUE,
			  -1);
      
      path = gtk_tree_path_new_first ();
      gtk_tree_selection_select_path (selection, path);
      gtk_tree_path_free (path);
    }
  
  g_signal_handlers_unblock_by_func (selection,
				     artist_selection_changed_cb,
				     window);
}

static Expr *
create_search_expr (const char *str)
{
  Expr *v, *c;
  Expr *e, *tmp;
  
  v = expr_variable_new (VARIABLE_TITLE);
  c = expr_constant_new (constant_string_new (str));
  e = expr_binary_new (EXPR_OP_CONTAINS, v, c);
  
  v = expr_variable_new (VARIABLE_ARTIST);
  c = expr_constant_new (constant_string_new (str));
  tmp = expr_binary_new (EXPR_OP_CONTAINS, v, c);

  e = expr_binary_new (EXPR_OP_OR, e, tmp);
  
  v = expr_variable_new (VARIABLE_ALBUM);
  c = expr_constant_new (constant_string_new (str));
  tmp = expr_binary_new (EXPR_OP_CONTAINS, v, c);
      
  e = expr_binary_new (EXPR_OP_OR, e, tmp);

  return e;
}

static void
start_browse_query (MainWindow *window,
		    gboolean    search_artists,
		    gboolean    search_albums)
{
  MainWindowPriv *priv;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GList *paths;
  GtkTreeIter iter;
  GList *l;
  Expr *tmp, *e_artists, *e_albums;
  Expr *e_search;
  Expr *v, *c;
  char *str;
  const char *search_text;
  gboolean all;
  Query *query;
  
  priv = window->priv;

  search_text = gtk_entry_get_text (GTK_ENTRY (priv->search_entry));
  if (search_text[0] == '\0')
    search_text = NULL;
  
  if (!search_artists)
    g_hash_table_foreach_remove (priv->artist_hash, foreach_remove_cb, NULL);

  if (!search_albums)
    g_hash_table_foreach_remove (priv->album_hash, foreach_remove_cb, NULL);

  e_artists = NULL;
  e_albums = NULL;
  
  if (search_artists)
    {
      selection = gtk_tree_view_get_selection (priv->artist_tree);
      paths = gtk_tree_selection_get_selected_rows (selection, &model);
      for (l = paths; l; l = l->next)
	{
	  if (!gtk_tree_model_get_iter (model, &iter, l->data))
	    continue;
	  
	  gtk_tree_model_get (model, &iter,
			      ARTIST_COL_NAME, &str,
			      ARTIST_COL_ALL, &all,
			      -1);
	  
	  if (all)
	    {
	      g_free (str);
	      continue;
	    }
	  
	  v = expr_variable_new (VARIABLE_ARTIST);
	  c = expr_constant_new (constant_string_new (str));
  
	  tmp = expr_binary_new (EXPR_OP_EQ, v, c);

	  if (e_artists)
	    e_artists = expr_binary_new (EXPR_OP_OR, e_artists, tmp);
	  else
	    e_artists = tmp;
	  
	  g_free (str);
	}
	  
      g_list_foreach (paths, (GFunc) gtk_tree_path_free, NULL);
      g_list_free (paths);
    }

  if (search_albums)
    {
      selection = gtk_tree_view_get_selection (priv->album_tree);
      paths = gtk_tree_selection_get_selected_rows (selection, &model);

      for (l = paths; l; l = l->next)
	{
	  if (!gtk_tree_model_get_iter (model, &iter, l->data))
	    continue;
	  
	  gtk_tree_model_get (model, &iter,
			      ALBUM_COL_NAME, &str,
			      ALBUM_COL_ALL, &all,
			      -1);

	  if (all)
	    {
	      g_free (str);
	      continue;
	    }

	  v = expr_variable_new (VARIABLE_ALBUM);
	  c = expr_constant_new (constant_string_new (str));
	  
	  tmp = expr_binary_new (EXPR_OP_EQ, v, c);
	  
	  if (e_albums)
	    e_albums = expr_binary_new (EXPR_OP_OR, e_albums, tmp);
	  else
	    e_albums = tmp;

	  g_free (str);
	}	  
      g_list_foreach (paths, (GFunc) gtk_tree_path_free, NULL);
      g_list_free (paths);
    }

  if (e_artists && e_albums)
    tmp = expr_binary_new (EXPR_OP_AND, e_artists, e_albums);
  else if (e_artists)
    tmp = e_artists;
  else if (e_albums)
    tmp = e_albums;
  else
    tmp = NULL;

  if (search_text)
    {
      e_search = create_search_expr (search_text);

      if (tmp)
	tmp = expr_binary_new (EXPR_OP_AND, tmp, e_search);
      else
	tmp = e_search;
    }
  
  priv->browse_mode = TRUE; 

  query = query_new (tmp);
  start_query (window, query);
  query_unref (query);
}

static void
genre_selection_changed_cb (GtkTreeSelection *selection,
			    MainWindow       *window)
{
  start_browse_query (window, FALSE, FALSE);
}

static void
artist_selection_changed_cb (GtkTreeSelection *selection,
			     MainWindow       *window)
{
  start_browse_query (window, TRUE, FALSE);
}

static void
album_selection_changed_cb (GtkTreeSelection *selection,
			    MainWindow       *window)
{
  start_browse_query (window, TRUE, TRUE);
}

static void
update_playing_info (MainWindow *window)
{
  MainWindowPriv *priv;
  Song *song = NULL;
  PlayerState state;
  GtkTreeIter iter;
  GtkTreePath *path;
  char *str, *tmp;
  const char *stock_icon;
  int sec, min;
  gboolean bool;
  
  priv = window->priv;

  state = player_get_state (priv->player);
  switch (state)
    {
    case PLAYER_STATE_PAUSED:
      stock_icon = JAMBOREE_STOCK_PLAY;
      song = player_get_song (priv->player);
      break;
    case PLAYER_STATE_PLAYING:
      stock_icon = JAMBOREE_STOCK_PAUSE;
      song = player_get_song (priv->player);
      break;
    case PLAYER_STATE_STOPPED:
    default:
      stock_icon = JAMBOREE_STOCK_PLAY;
      break;
    }

  if (!song)
    song = song_list_model_get_current (priv->song_model);

  if (!song)
    song = song_list_model_first (priv->song_model);
  
  if (!song)
    {
      gtk_widget_hide (priv->playing_info_hbox);
      return;
    }
  
  if (song_list_model_song_get_iter (priv->song_model, song, &iter))
    {
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (priv->song_model), &iter);
      gtk_tree_model_row_changed (GTK_TREE_MODEL (priv->song_model), path, &iter);

      gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (priv->song_tree),
				    path, NULL,
				    TRUE, 0.2, 0);
      
      gtk_tree_path_free (path);
    }

  gtk_image_set_from_stock (GTK_IMAGE (GTK_BIN (priv->play_button)->child),
			    stock_icon, GTK_ICON_SIZE_LARGE_TOOLBAR);
  
  tmp = g_markup_escape_text (string_entry_get_str (song->title), -1);

  str = g_strdup_printf ("<b>%s</b>", tmp[0] ? tmp : _("Unknown"));

  gtk_label_set_markup (GTK_LABEL (priv->playing_song_label), str);
  g_free (tmp);
  g_free (str);

  str = g_strdup_printf ("%s - Jamboree", string_entry_get_str (song->title));
  
  gtk_window_set_title (GTK_WINDOW (window), str);
  g_free (str);

  tmp = g_markup_escape_text (string_entry_get_str (song->artist), -1);
  str = g_strdup_printf ("<b>%s</b>", tmp[0] ? tmp : _("Unknown")); 
  gtk_label_set_markup (GTK_LABEL (priv->playing_artist_label), str);
  g_free (tmp);
  g_free (str);
  
  sec = song->length / 1000;  
  min = sec / 60;
  sec -= min * 60;

  if (song->length > 0)
    {
      str = g_strdup_printf ("%d:%02d", min, sec);
      gtk_widget_set_sensitive (priv->seek_scale, TRUE);
    }
  else
    {
      str = g_strdup_printf ("?:??");
      gtk_widget_set_sensitive (priv->seek_scale, FALSE);
    }
  
  gtk_label_set_text (GTK_LABEL (priv->length_label), str);
  g_free (str);

  if (state == PLAYER_STATE_STOPPED)
    gtk_label_set_text (GTK_LABEL (priv->tick_label), _("0:00"));
  
  bool = song_list_model_has_next (priv->song_model);
  gtk_widget_set_sensitive (priv->next_button, bool);
  gtk_widget_set_sensitive (priv->menu_next_button, bool);

  bool = player_get_state (priv->player) == PLAYER_STATE_PLAYING ||
    song_list_model_has_prev (priv->song_model);
  gtk_widget_set_sensitive (priv->prev_button, bool);
  gtk_widget_set_sensitive (priv->menu_prev_button, bool);  

  gtk_widget_show (priv->playing_info_hbox);
}

static void
player_end_of_stream_cb (Player     *player,
			 Song       *song,
			 MainWindow *window)
{
  MainWindowPriv *priv;
  Expr *expr;
  
  priv = window->priv;

  if (song)
    {
      song->play_count++;
      song->last_played = time (NULL);

#if 0
      if (song->length == 0)
	{
	  song->length = player_tell (priv->player) / 10000;
	}
#endif

      song_db_update_song (priv->db, song);

      if (priv->query)
	{
	  expr = query_get_expr (priv->query);
	  if (expr && !expr_evaluate (expr, song))
	    song_list_model_remove (priv->song_model, song);
	}

      /* If a song is being played, and we switch the view, then switch back,
       * continue after the playing one instead of starting from the top.
       */
      if (song_list_model_song_get_iter (priv->song_model, song, NULL) &&
	  !song_list_model_get_current (priv->song_model))
	song_list_model_set_current (priv->song_model, song);
      
      song = song_list_model_next (priv->song_model);
      if (song)
	player_play_song (priv->player, song);
      else
	player_stop (priv->player);
    }
  
  update_playing_info (window);
}

static void
set_elapsed_time_label (MainWindow *window, int sec)
{
  MainWindowPriv *priv;
  char *str;
  
  priv = window->priv;

  str = g_strdup_printf ("%d:%02d", sec / 60, sec % 60);
  gtk_label_set_text (GTK_LABEL (priv->tick_label), str);
  g_free (str);
}

static void
set_elapsed_time_scale (MainWindow *window, int sec)
{
  MainWindowPriv *priv;
  Song *song;
  double value;
  
  priv = window->priv;

  song = player_get_song (priv->player);
  if (song)
    {
      if (sec == 0 || song->length == 0)
	value = 0;
      else
	value = sec * 1000.0 * 100.0 / song->length;
      
      g_signal_handlers_block_by_func (priv->seek_scale, seek_scale_value_changed_cb, window);
      gtk_range_set_value (GTK_RANGE (priv->seek_scale), value);
      g_signal_handlers_unblock_by_func (priv->seek_scale, seek_scale_value_changed_cb, window);
    }
}

static void
player_tick_cb (Player     *player,
		long        sec,
		MainWindow *window)
{
  if (!window->priv->seeking)
    {
      set_elapsed_time_scale (window, sec);
      set_elapsed_time_label (window, sec);
    }
}

static void
player_error_cb (Player     *player,
		 char       *error,
		 MainWindow *window)
{
  eel_show_error_dialog_with_details (_("Could not play song"),
				      _("Sound Error"),
				      error,
				      GTK_WINDOW (window));
}

static void
player_state_changed_cb (Player      *player,
			 PlayerState  state,
			 MainWindow  *window)
{
  update_playing_info (window);
}

static void
shuffle_button_toggled_cb (GtkToggleButton *button,
			   MainWindow      *window)
{
  MainWindowPriv *priv;
  gboolean active;

  priv = window->priv;
  
  active = gtk_toggle_button_get_active (button);
  
  song_list_model_set_shuffle (priv->song_model, active);

  gconf_client_set_bool (gconf_client,
			 "/apps/jamboree/control/shuffle",
			 active,
			 NULL);

  if (player_get_state (priv->player) == PLAYER_STATE_STOPPED)
    main_window_handle_next (window);
}

static void
song_tree_size_allocate_cb (GtkTreeView   *tree,
			    GtkAllocation *alloc,
			    MainWindow    *window)
{
  GList *columns, *l;
  static int widths[] = { 7, 24, 24, 24, 7, 7, 3, 0, 0, 0, 0 };
  int width;
  int i;

  if (alloc->width <= 0 || alloc->height <= 0)
    return;
    
  columns = gtk_tree_view_get_columns (tree);
  for (l = columns, i = 0; l; l = l->next, i++)
    {
      if (widths[i] > 0)
	{      
	  width = floor (0.5 + widths[i] * alloc->width / 100.0);
	  width = MAX (20, width);
	  
	  gtk_tree_view_column_set_min_width (l->data, width);
	}
    }

  g_signal_handlers_disconnect_by_func (tree, song_tree_size_allocate_cb, window);
}

static void
song_tree_row_activated_cb (GtkTreeView       *tree,
			    GtkTreePath       *path,
			    GtkTreeViewColumn *column,
			    MainWindow        *window)
{
  MainWindowPriv *priv;
  GtkTreeIter iter;
  Song *song;

  priv = window->priv;

  gtk_tree_model_get_iter (GTK_TREE_MODEL (priv->song_model), &iter, path);
  song = song_list_model_get_song (priv->song_model, &iter);

  player_stop (priv->player);
  
  song_list_model_set_current (priv->song_model, song);
  main_window_handle_play (window);
}

static GtkWidget *
append_context_item (GtkWidget *menu,
		     char      *stock_item,
		     gpointer   function,
		     gpointer   data)
{
  GtkWidget *menuitem;
    
  menuitem = gtk_image_menu_item_new_from_stock (stock_item, NULL);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
  gtk_widget_show (menuitem);

  if (function)
    g_signal_connect (menuitem,
		      "activate",
		      G_CALLBACK (function),
		      data);
  return menuitem;
}


static void
context_play_cb (GtkWidget       *menu_item,
		 MainWindow      *window)
{
  MainWindowPriv *priv;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GList *rows;
  Song *song;

  priv = window->priv;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->song_tree));
  rows = gtk_tree_selection_get_selected_rows (selection, NULL);
  if (!rows)
    return;

  /* Play the first selected song. */
  gtk_tree_model_get_iter (GTK_TREE_MODEL (priv->song_model), &iter, rows->data);
  song = song_list_model_get_song (SONG_LIST_MODEL (priv->song_model), &iter);
  
  if (song != player_get_song (priv->player))
    player_play_song (priv->player, song);
  else
    player_play (priv->player);
  
}

static void
context_pause_cb (GtkWidget       *menu_item,
		  MainWindow      *window)
{
  player_pause (window->priv->player);
}

#if 0
static void
context_properties_cb (GtkWidget       *menu_item,
		       MainWindow      *window)
{
  /* FIXME: Show song properties. */
}
#endif

static void
context_remove_cb (GtkWidget  *menu_item,
		   MainWindow *window)
{
  MainWindowPriv *priv;
  GtkTreeSelection *selection;
  GList *rows, *songs = NULL, *l, *m;
  GtkTreeIter iter;
  Playlist *playlist;
  Song *song;

  priv = window->priv;
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->song_tree));
  rows = gtk_tree_selection_get_selected_rows (selection, NULL);
  for (l = rows; l; l = l->next)
    {
      gtk_tree_model_get_iter (GTK_TREE_MODEL (priv->song_model), &iter, l->data);
      song = song_list_model_get_song (SONG_LIST_MODEL (priv->song_model), &iter);

      songs = g_list_prepend (songs, song);
    }

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (priv->playlist_tree));

  for (l = songs; l; l = l->next)
    {
      song = l->data;
      
      /* FIXME: Should we stop the player after removing a song?
       * player_remove_song (priv->player, song);
       */

      song_list_model_remove (SONG_LIST_MODEL (priv->song_model), song);
      
      if (playlist->type == PLAYLIST_TYPE_LIBRARY)
	song_db_remove_song (priv->db, song);
      else if (playlist->type == PLAYLIST_TYPE_REGULAR)
	for (m = song->playlists; m; m = m->next)
	  {
	    if (GPOINTER_TO_INT (m->data) == playlist->id)
	      {
		song->playlists = g_list_delete_link (song->playlists, m);
		song_db_update_song (priv->db, song);
		break;
	      }
	  }
    }
  
  g_list_foreach (rows, (GFunc) gtk_tree_path_free, NULL);
  g_list_free (rows);
  g_list_free (songs);
}

static gboolean
song_tree_row_button_press_event_cb (GtkWidget      *widget,
				     GdkEventButton *event,
				     MainWindow     *window)
{
  MainWindowPriv *priv;
  GtkWidget *menu;
  GtkTreePath *path;
  GtkTreeIter iter;
  gboolean has_cell;
  Song *song;
  GtkTreeSelection *selection;

  if (event->button != 3)
    return FALSE;

  priv = window->priv;
  
  has_cell = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (priv->song_tree),
					    event->x,
					    event->y,
					    &path,
					    NULL, NULL, NULL);
  if (!has_cell)
    return FALSE;
  
  gtk_tree_model_get_iter (GTK_TREE_MODEL (priv->song_model), &iter, path);
  song = song_list_model_get_song (SONG_LIST_MODEL (priv->song_model), &iter);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->song_tree));
  if (!gtk_tree_selection_iter_is_selected (selection, &iter))
    {
      gtk_tree_selection_unselect_all (selection);
      gtk_tree_selection_select_iter (selection, &iter);
    }
  menu = gtk_menu_new ();

  if (player_is_playing (priv->player, song))
    append_context_item (menu, JAMBOREE_STOCK_PAUSE, context_pause_cb, window);
  else
    append_context_item (menu, JAMBOREE_STOCK_PLAY, context_play_cb, window);    
  
  append_context_item (menu, GTK_STOCK_REMOVE, context_remove_cb, window);
  
  /* FIXME: Add a dialog */
#if 0  
  append_context_item (menu, GTK_STOCK_PROPERTIES, context_properties_cb, window);
#endif      
  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 3, event->time);
  
  return TRUE;
}
				     
static void
song_tree_drag_data_received_cb (GtkWidget        *widget,
				 GdkDragContext   *context,
				 int               x,
				 int               y,
				 GtkSelectionData *data,
				 guint             info,
				 guint             time,
				 MainWindow       *window)
{
  MainWindowPriv *priv;

  priv = window->priv;
  
  if (data->length >= 0 && data->format == 8)
    {
      const char *ptr = data->data;
      char url[1024];
      char *path;
      GList *paths = NULL, *l;
      
      while (sscanf (ptr, "%s\r\n", (char *) &url) > 0)
	{
	  path = gnome_vfs_get_local_path_from_uri (url);
	  paths = g_list_append (paths, (char *) path);
	  
	  ptr += strlen (url) + 2;
	} 
      
      gtk_drag_finish (context, TRUE, FALSE, time);

      add_paths (window, paths);

      for (l = paths; l; l = l->next) {
	g_free (l->data);
      }
      g_list_free (paths);
    }
  else
    {
      g_message ("Don't know how to handle format %d", data->format);
      gtk_drag_finish (context, FALSE, FALSE, time);      
    }

  g_signal_stop_emission_by_name (widget, "drag_data_received");
}

/* Don't look here. Ugly ugly. */
static void
playlist_tree_drag_data_received_cb (GtkWidget        *widget,
				     GdkDragContext   *context,
				     int               x,
				     int               y,
				     GtkSelectionData *data,
				     guint             info,
				     guint             time,
				     MainWindow       *window)
{
  MainWindowPriv *priv;
  gboolean found;
  GtkTreePath *path;
  GtkTreeModel *model;
  GtkTreeIter iter;
  Playlist *playlist;
  
  priv = window->priv;

  found = gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW (widget), x, y, &path, NULL);
  if (!found)
    return;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (widget));
  
  if (gtk_tree_model_get_iter (model, &iter, path))
    gtk_tree_model_get (model, &iter,
			0, &playlist,
			-1);

  gtk_tree_path_free (path);

  if (data->length >= 0 && data->format == 8)
    {
      const char *ptr = data->data;
      char **strv;
      gpointer p;
      Song *song;
      int i = 0;
      int id;

      id = playlist_get_id (playlist);
      
      strv = g_strsplit (ptr, "\r\n", -1);

      
      while (strv[i])
	{
	  if (sscanf (strv[i], "%p", &p) > 0)
	    {
	      song = p;

	      if (!g_list_find (song->playlists, GINT_TO_POINTER (id)))
		{
		  song->playlists = g_list_prepend (song->playlists, GINT_TO_POINTER (id));
		  song_db_update_song (priv->db, song);
		}
	    }
	  	  
	  i++;
	}

      g_strfreev (strv);
      
      gtk_drag_finish (context, TRUE, FALSE, time);
    }
  else
    {
      g_message ("Don't know how to handle format %d", data->format);
      gtk_drag_finish (context, FALSE, FALSE, time);      
    }

  g_signal_stop_emission_by_name (widget, "drag_data_received");
}

static void
song_playing_func (GtkTreeViewColumn *tree_column,
		   GtkCellRenderer   *cell,
		   GtkTreeModel      *tree_model,
		   GtkTreeIter       *iter,
		   gpointer           data)
{
  MainWindow *window = data;
  MainWindowPriv *priv;
  static GdkPixbuf *playing_pixbuf = NULL;
  static GdkPixbuf *paused_pixbuf = NULL;
  static GdkPixbuf *blank_pixbuf = NULL;
  Song *song, *current_song;
  GdkPixbuf *p;
  PlayerState state;

  priv = window->priv;

  if (!playing_pixbuf)
    {
      playing_pixbuf = gdk_pixbuf_new_from_file (DATADIR "/jamboree/jamboree-note.png", NULL);
      paused_pixbuf = gdk_pixbuf_new_from_file (DATADIR "/jamboree/jamboree-small-pause.png", NULL);
      blank_pixbuf = gdk_pixbuf_new_from_file (DATADIR "/jamboree/jamboree-small-blank.png", NULL);
    }
  
  song = song_list_model_get_song ((SongListModel *) tree_model, iter);

  p = blank_pixbuf;

  state = player_get_state (priv->player);

  current_song = song_list_model_get_current ((SongListModel *) tree_model);
  
  if (state == PLAYER_STATE_PLAYING && song == player_get_song (priv->player))
    p = playing_pixbuf;
  else if (state != PLAYER_STATE_PLAYING && song == current_song)
    p = paused_pixbuf;

  g_object_set (cell, "pixbuf", p, NULL);
}

static int
album_sort_func (GtkTreeModel *model,
		 GtkTreeIter  *a,
		 GtkTreeIter  *b,
		 gpointer      user_data)
{
  char *val1, *val2;
  gboolean all1, all2;
  int ret;
  
  gtk_tree_model_get (model, a,
		      ARTIST_COL_KEY, &val1,
		      ARTIST_COL_ALL, &all1,
		      -1);
  gtk_tree_model_get (model, b,
		      ARTIST_COL_KEY, &val2,
		      ARTIST_COL_ALL, &all2,
		      -1);

  if (all1)
    ret = -1;
  else if (all2)
    ret = 1;
  else
    ret = strcasecmp (val1, val2);

  g_free (val1);
  g_free (val2);

  return ret;
}

static int
artist_sort_func (GtkTreeModel *model,
		  GtkTreeIter  *a,
		  GtkTreeIter  *b,
		  gpointer      user_data)
{
  char *val1, *val2;
  gboolean all1, all2;
  int ret;
  
  gtk_tree_model_get (model, a,
		      ARTIST_COL_KEY, &val1,
		      ARTIST_COL_ALL, &all1,
		      -1);
  gtk_tree_model_get (model, b,
		      ARTIST_COL_KEY, &val2,
		      ARTIST_COL_ALL, &all2,
		      -1);

  if (all1)
    ret = -1;
  else if (all2)
    ret = 1;
  else
    ret = strcasecmp (val1, val2);

  g_free (val1);
  g_free (val2);

  return ret;
}

static void
artist_data_func (GtkTreeViewColumn *tree_column,
		  GtkCellRenderer   *cell,
		  GtkTreeModel      *tree_model,
		  GtkTreeIter       *iter,
		  gpointer           data)
{
  char *artist = NULL;
	
  gtk_tree_model_get (tree_model, iter,
		      ARTIST_COL_NAME, &artist,
		      -1);

  g_object_set (cell, "text", artist, NULL);

  g_free (artist);
}

static void
volume_changed_cb (GtkWidget  *button,
		   int         volume,
		   MainWindow *window)
{
  player_set_volume (window->priv->player, volume);
}

static gboolean
seeking_done_cb (MainWindow *window)
{
  window->priv->seeking = FALSE;
  window->priv->seeking_idle_id = 0;
  return FALSE;
}

static void
seek_scale_value_changed_cb (GtkWidget  *widget,
			     MainWindow *window)
{
  MainWindowPriv *priv = window->priv;
  GtkRange *range = GTK_RANGE (widget);
  Song *song;
  double value;
  guint64 offset;
  
  if (priv->seeking_idle_id)
    return;
  
  priv->seeking = TRUE;

  value = gtk_range_get_value (range);

  song = player_get_song (priv->player);

  if (!song)
    return;
  
  offset = value * (song->length - 500) / 100.0;
  
  player_seek (priv->player, offset);

  priv->seeking_idle_id = g_idle_add ((GSourceFunc)seeking_done_cb, window);
}

static gboolean
seek_scale_button_press_cb (GtkWidget      *widget,
			    GdkEventButton *event,
			    MainWindow     *window)
{
  MainWindowPriv *priv = window->priv;

  if (priv->seeking_idle_id)
    return FALSE;
    
  priv->seeking = TRUE;

  return FALSE;
}

static gboolean
seek_scale_button_release_cb (GtkWidget      *widget,
			      GdkEventButton *event,
			      MainWindow     *window)
{
  MainWindowPriv *priv = window->priv;
  GtkRange *range = GTK_RANGE (widget);
  Song *song;
  GtkAdjustment *adj;
  double value;
  guint64 offset;

  if (priv->seeking_idle_id)
    return FALSE;
  
  song = player_get_song (priv->player);
  if (!song)
    return FALSE;

  adj = gtk_range_get_adjustment (range);
  value = gtk_adjustment_get_value (adj);

  offset = value * (song->length - 500) / 100.0;

  player_seek (priv->player, offset);

  priv->seeking_idle_id = g_idle_add ((GSourceFunc)seeking_done_cb, window);

  return FALSE;
}

static gboolean
seek_scale_motion_notify_cb (GtkWidget      *widget,
			     GdkEventMotion *event,
			     MainWindow     *window)
{
  MainWindowPriv *priv = window->priv;
  GtkRange *range = GTK_RANGE (widget);
  Song *song;
  GtkAdjustment *adj;
  int sec;
  double value;

  if (!priv->seeking)
    return FALSE;

  song = player_get_song (priv->player);
  if (!song)
    return FALSE;

  adj = gtk_range_get_adjustment (range);
  value = gtk_adjustment_get_value (adj);

  sec = value * song->length / (100.0 * 1000);

  set_elapsed_time_label (window, sec);
  
  return FALSE;
}

static gboolean
search_entry_timeout_cb (MainWindow *window)
{
  MainWindowPriv *priv;
    
  priv = window->priv;
  
  priv->search_timeout_id = 0;

  start_browse_query (window, TRUE, TRUE);
  
  return FALSE;
}

static void
search_entry_activate_cb (GtkEntry   *entry,
			  MainWindow *window)
{
  search_entry_timeout_cb (window);
}

static gboolean
search_entry_key_press_event_cb (GtkEntry    *entry,
				 GdkEventKey *event,
				 MainWindow  *window)
{
  MainWindowPriv *priv;

  priv = window->priv;

  if (priv->search_timeout_id)
    g_source_remove (priv->search_timeout_id);
    
  priv->search_timeout_id = g_timeout_add (500, (GSourceFunc) search_entry_timeout_cb, window);

  return FALSE;
}

static void
load_glade_gui (MainWindow *window)
{
  MainWindowPriv *priv;
  GladeXML *glade;
  GtkCellRenderer *cell;
  GtkTreeViewColumn *column;
  GtkTreeSelection *selection;
  GtkSizeGroup *size_group;
  GtkWidget *w;
  gboolean shuffle;
  
  priv = window->priv;

  glade = glade_xml_new (DATADIR "/jamboree/jamboree.glade", "top_vbox", NULL);

  priv->top_vbox = glade_xml_get_widget (glade, "top_vbox");
  gtk_container_add (GTK_CONTAINER (window), priv->top_vbox);
  
  priv->playing_song_label = glade_xml_get_widget (glade, "playing_song_label");
  priv->playing_artist_label = glade_xml_get_widget (glade, "playing_artist_label");
  priv->playing_info_hbox = glade_xml_get_widget (glade, "playing_info_hbox");
  priv->tick_label = glade_xml_get_widget (glade, "tick_label");
  priv->length_label = glade_xml_get_widget (glade, "length_label");

  priv->seek_scale = glade_xml_get_widget (glade, "seek_hscale");

  g_signal_connect (priv->seek_scale,
		    "value_changed",
		    G_CALLBACK (seek_scale_value_changed_cb),
		    window);
  g_signal_connect (priv->seek_scale,
		    "button_press_event",
		    G_CALLBACK (seek_scale_button_press_cb),
		    window);
  g_signal_connect (priv->seek_scale,
		    "button_release_event",
		    G_CALLBACK (seek_scale_button_release_cb),
		    window);
  g_signal_connect (priv->seek_scale,
		    "motion_notify_event",
		    G_CALLBACK (seek_scale_motion_notify_cb),
		    window);
 
  w = glade_xml_get_widget (glade, "volume_container");
  priv->volume_button = volume_button_new ();
  gtk_widget_show (priv->volume_button);
  gtk_box_pack_start (GTK_BOX (w), priv->volume_button, FALSE, TRUE, 0);

  g_signal_connect (priv->volume_button,
		    "volume_changed",
		    G_CALLBACK (volume_changed_cb),
		    window);
  
  priv->search_entry = glade_xml_get_widget (glade, "search_entry");

  g_signal_connect (priv->search_entry,
		    "activate",
		    G_CALLBACK (search_entry_activate_cb),
		    window);
  
  g_signal_connect (priv->search_entry,
		    "key_press_event",
		    G_CALLBACK (search_entry_key_press_event_cb),
		    window);

  /* Workaround for broken glade: You're not allowed to set non-existent stock
   * icons in glade.
   */
  FIXUP_STOCK_ICON ("image_prev", "jamboree-prev");
  FIXUP_STOCK_ICON ("image_play", "jamboree-play");
  FIXUP_STOCK_ICON ("image_next", "jamboree-next");
  FIXUP_STOCK_ICON ("image_stop", "jamboree-stop");
  FIXUP_STOCK_ICON ("image_shuffle", "jamboree-shuffle");

  /* Genre */
  priv->genre_tree = GTK_TREE_VIEW (glade_xml_get_widget (glade, "genre_tree"));
  priv->genre_store = create_genre_store ();
  gtk_tree_view_set_model (priv->genre_tree, GTK_TREE_MODEL (priv->genre_store));

  selection = gtk_tree_view_get_selection (priv->genre_tree);
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
  g_signal_connect (selection, "changed",
		    G_CALLBACK (genre_selection_changed_cb),
		    window);

  /* Hide for now. */
  gtk_widget_hide (glade_xml_get_widget (glade, "genre_scrolledwindow"));
  
  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Genre"), 
						     cell, 
						     "text", 0, 
						     NULL);
  gtk_tree_view_append_column (priv->genre_tree, column);
  gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);

  priv->artist_tree = GTK_TREE_VIEW (glade_xml_get_widget (glade, "artist_tree"));
  priv->artist_store = gtk_list_store_new (ARTIST_NUM_COLS,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_BOOLEAN);
  gtk_tree_view_set_model (priv->artist_tree, GTK_TREE_MODEL (priv->artist_store));

  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (priv->artist_store),
				   ARTIST_COL_NAME,
				   artist_sort_func,
				   NULL,
				   NULL);
  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (priv->artist_store),
					ARTIST_COL_NAME,
					GTK_SORT_ASCENDING);
  
  selection = gtk_tree_view_get_selection (priv->artist_tree);
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
  g_signal_connect (selection, "changed",
		    G_CALLBACK (artist_selection_changed_cb),
		    window);
  
  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Artist"), 
						     cell, 
						     "text", ARTIST_COL_NAME, 
						     NULL);
  gtk_tree_view_column_set_cell_data_func (column,
                                           cell,
					   artist_data_func,
					   NULL,
					   NULL);
  gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_append_column (priv->artist_tree, column);

  priv->album_tree = GTK_TREE_VIEW (glade_xml_get_widget (glade, "album_tree"));
  priv->album_store = gtk_list_store_new (ALBUM_NUM_COLS,
					  G_TYPE_STRING,
					  G_TYPE_STRING,
					  G_TYPE_BOOLEAN);
  gtk_tree_view_set_model (priv->album_tree, GTK_TREE_MODEL (priv->album_store));

  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (priv->album_store),
				   ALBUM_COL_NAME,
				   album_sort_func,
				   NULL,
				   NULL);
  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (priv->album_store),
					ALBUM_COL_NAME,
					GTK_SORT_ASCENDING);
  
  selection = gtk_tree_view_get_selection (priv->album_tree);
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
  g_signal_connect (selection, "changed",
		    G_CALLBACK (album_selection_changed_cb),
		    window);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Album"), 
						     cell, 
						     "text", ARTIST_COL_NAME, 
						     NULL);
  gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_append_column (priv->album_tree, column);

  priv->playlist_sw = glade_xml_get_widget (glade, "playlist_scrolledwindow");

  w = glade_xml_get_widget (glade, "song_scrolledwindow");
  priv->song_tree = SONG_LIST_VIEW (song_list_view_new ());
  song_list_view_setup_columns (priv->song_tree);
  
  priv->song_model = SONG_LIST_MODEL (gtk_tree_view_get_model (GTK_TREE_VIEW (priv->song_tree)));

  gtk_container_add (GTK_CONTAINER (w), GTK_WIDGET (priv->song_tree));
  gtk_widget_show (GTK_WIDGET (priv->song_tree));

  g_signal_connect (priv->song_tree,
		    "size_allocate",
		    G_CALLBACK (song_tree_size_allocate_cb),
		    window);

  g_signal_connect (priv->song_tree,
		    "row_activated",
		    G_CALLBACK (song_tree_row_activated_cb),
		    window);

  g_signal_connect (priv->song_tree,
		    "button_press_event",
		    G_CALLBACK (song_tree_row_button_press_event_cb),
		    window);

  /* Setup drag-n-drop. */
  gtk_drag_dest_set (GTK_WIDGET (priv->song_tree),
		     GTK_DEST_DEFAULT_ALL,
		     songs_target_types,
		     G_N_ELEMENTS (songs_target_types),
		     GDK_ACTION_COPY);

  g_signal_connect (priv->song_tree,
		    "drag_data_received",
		    G_CALLBACK (song_tree_drag_data_received_cb),
		    window);

  gtk_tree_view_column_set_cell_data_func (priv->song_tree->play_column,
					   priv->song_tree->play_cell,
					   song_playing_func,
					   window,
					   NULL);
  
  priv->shuffle_button = glade_xml_get_widget (glade, "shuffle_button");

  shuffle = gconf_client_get_bool (gconf_client,
				   "/apps/jamboree/control/shuffle",
				   NULL);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->shuffle_button), shuffle);
  song_list_model_set_shuffle (priv->song_model, shuffle);
  
  g_signal_connect (priv->shuffle_button, "toggled",
		    G_CALLBACK (shuffle_button_toggled_cb),
		    window);
  
  priv->browse_pane = glade_xml_get_widget (glade, "browse_pane");

  size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
  gtk_size_group_add_widget (size_group, GTK_WIDGET (priv->genre_tree));
  gtk_size_group_add_widget (size_group, GTK_WIDGET (priv->artist_tree));
  gtk_size_group_add_widget (size_group, GTK_WIDGET (priv->album_tree));
  g_object_unref (size_group);

  priv->play_button = glade_xml_get_widget (glade, "play_button");
  g_signal_connect_swapped (priv->play_button,
			    "clicked",
			    G_CALLBACK (actions_play_cb),
			    window);
  
  w = glade_xml_get_widget (glade, "stop_button");
  g_signal_connect_swapped (w,
			    "clicked",
			    G_CALLBACK (actions_stop_cb),
			    window);

  priv->next_button = glade_xml_get_widget (glade, "next_button");
  g_signal_connect_swapped (priv->next_button,
			    "clicked",
			    G_CALLBACK (actions_next_cb),
			    window);

  priv->prev_button = glade_xml_get_widget (glade, "prev_button");
  g_signal_connect_swapped (priv->prev_button,
			    "clicked",
			    G_CALLBACK (actions_prev_cb),
			    window);
  
  priv->info_label = glade_xml_get_widget (glade, "info_label");

  g_object_unref (glade);
}

static void
setup_playlists (MainWindow *window)
{
  MainWindowPriv *priv;
  GtkTreeSelection *selection;
  GList *l;
  GtkTreeIter iter;
  Playlist *playlist;
  
  priv = window->priv;

  priv->playlist_tree = GTK_TREE_VIEW (play_list_view_new (priv->db));
  setup_playlist_tree (window);
  
  gtk_container_add (GTK_CONTAINER (priv->playlist_sw), GTK_WIDGET (priv->playlist_tree));
  gtk_widget_show (GTK_WIDGET (priv->playlist_tree));

  g_signal_connect (priv->playlist_tree,
		    "drag_data_received",
		    G_CALLBACK (playlist_tree_drag_data_received_cb),
		    window);
  
  selection = gtk_tree_view_get_selection (priv->playlist_tree);
  g_signal_handlers_block_by_func (selection, playlist_selection_changed_cb, window);

  priv->playlists = playlist_xml_load_playlists ();
  for (l = priv->playlists; l; l = l->next)
    {
      gtk_list_store_append (priv->playlist_store, &iter);
      gtk_list_store_set (priv->playlist_store, &iter, 0, l->data, -1);
    }

  play_list_view_restore_prefs (PLAY_LIST_VIEW (priv->playlist_tree));

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (priv->playlist_tree));

  if (playlist && playlist->type != PLAYLIST_TYPE_LIBRARY)
    gtk_widget_hide (GTK_PANED (priv->browse_pane)->child1);
  
  g_signal_handlers_unblock_by_func (selection, playlist_selection_changed_cb, window);

  update_edit_smart_playlist_sensitivity (window);
}

static gboolean
first_browse_func (MainWindow *window)
{
  MainWindowPriv *priv;
  Playlist *playlist;

  priv = window->priv;

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (priv->playlist_tree));
  if (playlist->type == PLAYLIST_TYPE_LIBRARY)
    start_browse_query (window, FALSE, FALSE);
  else
    {
      priv->browse_mode = FALSE;
      start_query (window, playlist_get_query (playlist));
    }
  
  return FALSE;
}

GtkWidget *
main_window_new (SongDB *db)
{
  MainWindow *window;

  g_assert (glob_window == NULL);
  
  window = g_object_new (TYPE_MAIN_WINDOW, NULL);

  window->priv->db = g_object_ref (db);

  setup_playlists (window);

  g_idle_add ((GSourceFunc) first_browse_func, window);

  glob_window = window;
  
  return GTK_WIDGET (window);
}

static void
playlist_cell_edited_cb (GtkCellRendererText *cell,
			 const char          *path_string,
			 const char          *new_text,
			 gpointer             data)
{
  MainWindow *window = data;
  MainWindowPriv *priv = window->priv;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter iter;
  Playlist *playlist;

  gtk_tree_model_get_iter (GTK_TREE_MODEL (window->priv->playlist_store), &iter, path);

  if (new_text[0] != '\0')
    {
      gtk_tree_model_get (GTK_TREE_MODEL (window->priv->playlist_store),
			  &iter, 0, &playlist,
			  -1);

      playlist_set_name (playlist, new_text);
      playlist_xml_save_playlists (priv->playlists);
    }
  
  gtk_tree_path_free (path);
}
  
static void
update_edit_smart_playlist_sensitivity (MainWindow *window)
{
  MainWindowPriv *priv;
  Playlist *playlist;

  priv = window->priv;

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (priv->playlist_tree));
  if (!playlist)
    return;
  
  switch (playlist->type)
    {
    case PLAYLIST_TYPE_LIBRARY:
      gtk_widget_set_sensitive (priv->edit_smart_menu_item, FALSE);
      break;

    case PLAYLIST_TYPE_REGULAR:
      gtk_widget_set_sensitive (priv->edit_smart_menu_item, FALSE);
      break;

    case PLAYLIST_TYPE_SMART:
      gtk_widget_set_sensitive (priv->edit_smart_menu_item, TRUE);
      break;

    case PLAYLIST_TYPE_IRADIO:
      gtk_widget_set_sensitive (priv->edit_smart_menu_item, FALSE);
      break;
    }
}

static void
playlist_selection_changed_cb (GtkTreeSelection *selection,
			       MainWindow       *window)
{
  MainWindowPriv *priv;
  Playlist *playlist;

  priv = window->priv;

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (priv->playlist_tree));
  if (!playlist)
    return;

  update_edit_smart_playlist_sensitivity (window);

  play_list_view_store_prefs (PLAY_LIST_VIEW (priv->playlist_tree));

  switch (playlist->type)
    {
    case PLAYLIST_TYPE_LIBRARY:
      gtk_widget_show (GTK_PANED (priv->browse_pane)->child1);

      start_browse_query (window, TRUE, TRUE);
      break;

    case PLAYLIST_TYPE_REGULAR:
      gtk_widget_hide (GTK_PANED (priv->browse_pane)->child1);

      priv->browse_mode = FALSE;
      start_query (window, playlist_get_query (playlist));
      break;

    case PLAYLIST_TYPE_SMART:
      gtk_widget_hide (GTK_PANED (priv->browse_pane)->child1);

      priv->browse_mode = FALSE;
      start_query (window, playlist_get_query (playlist));
      break;

    case PLAYLIST_TYPE_IRADIO:
      gtk_widget_hide (GTK_PANED (priv->browse_pane)->child1);
     
      song_list_model_clear (SONG_LIST_MODEL (priv->song_model));
      break;
    }
}

static void
setup_playlist_tree (MainWindow *window)
{
  MainWindowPriv *priv = window->priv;
  PlayListView *view = PLAY_LIST_VIEW (priv->playlist_tree);

  priv->playlist_store = GTK_LIST_STORE (view->model);
    
  g_signal_connect (view->cell,
		    "edited",
		    G_CALLBACK (playlist_cell_edited_cb),
		    window);
  
  g_signal_connect (view->selection,
		    "changed",
		    G_CALLBACK (playlist_selection_changed_cb),
		    window);
}

static void
new_playlist_cb (MainWindow *window)
{
  MainWindowPriv *priv;
  Playlist *playlist;
  GtkTreeIter iter;
  GtkTreePath *path;
  GtkTreeViewColumn *col;
  
  priv = window->priv;

  playlist = playlist_new (PLAYLIST_TYPE_REGULAR, _("New Playlist"));

  gtk_list_store_append (priv->playlist_store, &iter);
  gtk_list_store_set (priv->playlist_store, 
		      &iter,
		      0, playlist,
		      -1);
  
  priv->playlists = g_list_append (priv->playlists, playlist);
  playlist_xml_save_playlists (priv->playlists);

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (priv->playlist_store), &iter);

  col = gtk_tree_view_get_column (priv->playlist_tree, 0);
  gtk_tree_view_set_cursor (priv->playlist_tree, path, col, TRUE);
}

static void
new_smart_playlist_cb (MainWindow *window)
{
  MainWindowPriv *priv;
  Playlist *playlist;
  GtkTreeIter iter;
  GtkTreePath *path;
  GtkTreeViewColumn *col;

  priv = window->priv;
  
  playlist = smart_playlist_dialog_run (GTK_WINDOW (window), NULL);
  if (!playlist)
    return;
  
  gtk_list_store_append (priv->playlist_store, &iter);
  gtk_list_store_set (priv->playlist_store, &iter, 0, playlist, -1);
  priv->playlists = g_list_append (priv->playlists, playlist);
  playlist_xml_save_playlists (priv->playlists);

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (priv->playlist_store), &iter);

  col = gtk_tree_view_get_column (priv->playlist_tree, 0);
  gtk_tree_view_set_cursor (priv->playlist_tree, path, col, FALSE);

  gtk_tree_path_free (path);
}

static void
edit_smart_playlist_cb (MainWindow *window)
{
  MainWindowPriv *priv;
  GtkTreeIter iter;
  GtkTreePath *path;
  Playlist *playlist, *new_playlist;
  GtkTreeSelection *selection;
  
  priv = window->priv;

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (priv->playlist_tree));
  g_assert (playlist->type == PLAYLIST_TYPE_SMART);

  new_playlist = smart_playlist_dialog_run (GTK_WINDOW (window), playlist);
  if (!new_playlist)
    return;

  selection = gtk_tree_view_get_selection (priv->playlist_tree);
  if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      g_assert_not_reached ();
      return;
    }

  /* Update the old playlist with the values from the new and save */
  playlist_set_query (playlist, playlist_get_query (new_playlist));
  playlist_set_id (playlist, playlist_get_id (new_playlist));
  playlist_set_name (playlist, playlist_get_name (new_playlist));
  playlist_xml_save_playlists (priv->playlists);

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (priv->playlist_store), &iter);
  gtk_tree_model_row_changed (GTK_TREE_MODEL (priv->playlist_store), path, &iter);

  start_query (window, playlist_get_query (playlist));
}

static void
add_folder_cb (MainWindow *window)
{
  GtkWidget *dialog;
  const char *dir;
  char *start_dir;

  dialog = gtk_file_chooser_dialog_new (_("Add Music Folder"),
					GTK_WINDOW (window),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_ADD, GTK_RESPONSE_ACCEPT,
					NULL);

  gtk_file_chooser_set_folder_mode (GTK_FILE_CHOOSER (dialog), TRUE);

  start_dir = gconf_client_get_string (gconf_client,
				       GCONF_PATH "/default_add_folder",
				       NULL);
  if (start_dir)
    {
      if (strcmp (start_dir, "~") == 0)
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), g_get_home_dir ());
      else
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), start_dir);
      
      g_free (start_dir);
    }
  
  gtk_window_set_default_size (GTK_WINDOW (dialog), 600, 400);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      dir = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

      gtk_widget_hide (dialog);

      if (dir)
	{
	  GList *list;
	  
	  gconf_client_set_string (gconf_client,
				   GCONF_PATH "/default_add_folder",
				   dir,
				   NULL);

	  list = g_list_append (NULL, (gchar*)dir);
	  add_paths (window, list);
	  g_list_free (list);
	}
    }
  
  gtk_widget_destroy (dialog);
}

static void
quit_cb (MainWindow *window)
{
  gtk_widget_destroy (GTK_WIDGET (window));
}

static void
actions_play_cb (MainWindow *window)
{
  main_window_handle_play (window);
}

static void
actions_pause_cb (MainWindow *window)
{
  player_pause (window->priv->player);
}

static void
actions_stop_cb (MainWindow *window)
{
  main_window_handle_stop (window);
}

static void
actions_prev_cb (MainWindow *window)
{
  main_window_handle_prev (window);
}

static void
actions_next_cb (MainWindow *window)
{
  main_window_handle_next (window);
}

static void
actions_song_columns_cb (MainWindow *window)
{
  song_list_view_column_chooser (window->priv->song_tree, GTK_WINDOW (window));
}

static void
help_about_cb (MainWindow *window)
{
  static GtkWidget *about = NULL;

  const char *authors[] = {
    "Richard Hult <richard@imendio.com>",
    "Anders Carlsson <andersca@gnome.org>",
    "Johan Dahlin <johan@gnome.org>",
    NULL
  };
  const char *documenters [] = {
    NULL
  };
  const char *translator_credits = "translator_credits";

  if (about)
    {
      gtk_window_present (GTK_WINDOW (about));
      return;
    }
  
  if (strcmp (_(translator_credits), "translator_credits") != 0)
    translator_credits = _(translator_credits);
  else
    translator_credits = NULL;
  
  about = gnome_about_new ("Jamboree", VERSION,
			   _("Yes, there is a Swedish conspiracy"),
                           _("A music player for GNOME"),
                           (const char **) authors,
                           (const char **) documenters,
			   translator_credits,
			   NULL);
  
  gtk_window_set_transient_for (GTK_WINDOW (about), GTK_WINDOW (window));
  g_object_add_weak_pointer (G_OBJECT (about), (gpointer) &about);

  gtk_widget_show (about);  
}

typedef struct {
  MainWindow *window;
  GtkWidget  *dialog;
  GtkWidget  *progress;
  GtkWidget  *label;

  gboolean    cancel;
  guint       timeout_id;
  char       *dirname;
} ProgressData;

static gboolean
add_progress_show_cb (ProgressData *data)
{
  gtk_widget_show_all (data->dialog);

  data->timeout_id = 0;
  
  return FALSE;
}

static void
add_progress_response_cb (GtkWidget    *dialog,
			  int           response,
			  ProgressData *data)
{
  data->cancel = TRUE;
  data->dialog = NULL;

  gtk_widget_destroy (dialog);
}

static gboolean
add_progress_callback (SongDB       *db,
		       const char   *path,
		       ProgressData *data)
{
  char *dirname;
  char *tmp;

  while (gtk_events_pending ())
    gtk_main_iteration ();
  
  if (data->cancel)
    return FALSE;

  tmp = g_path_get_dirname (path);
  dirname = g_path_get_basename (tmp);
  g_free (tmp);
  
  if (!data->dirname || strcmp (dirname, data->dirname) != 0)
    {
      g_free (data->dirname);
      data->dirname = g_strdup (dirname);
      
      if (!g_utf8_validate (dirname, -1, NULL))
	{
	  tmp = g_filename_to_utf8 (dirname, -1, NULL, NULL, NULL);
	  if (!tmp)
	    tmp = g_strdup ("???");
	  
	  g_free (dirname);
	  dirname = tmp;
	}
      
      if (g_utf8_strlen (dirname, -1) > 28)
	{
	  tmp = g_utf8_offset_to_pointer (dirname, 25);
	  
	  tmp[0] = '.';
	  tmp[1] = '.';
	  tmp[2] = '.';
	  tmp[3] = '\0';
	}
	  
      gtk_label_set_text (GTK_LABEL (data->label), dirname);
    }

  g_free (dirname);
  
  return TRUE;
}

static void
add_paths (MainWindow *window, GList *paths)
{
  MainWindowPriv *priv;
  GladeXML *glade;
  ProgressData data;
  GtkTreeSelection *selection;
  
  priv = window->priv;

  data.window = window;
  data.cancel = FALSE;
  data.dirname = NULL;

  glade = glade_xml_new (DATADIR "/jamboree/jamboree.glade", "progress_dialog", NULL);

  data.dialog = glade_xml_get_widget (glade, "progress_dialog");
  data.label = glade_xml_get_widget (glade, "folder_label");

  g_signal_connect (data.dialog,
		    "response",
		    G_CALLBACK (add_progress_response_cb),
		    &data);
  
  data.timeout_id = g_timeout_add (1500, (GSourceFunc) add_progress_show_cb, &data);

  song_db_add_dir (priv->db, paths->data, (SongDBAddProgressFunc) add_progress_callback, &data);

  if (data.timeout_id)
    g_source_remove (data.timeout_id);
  
  if (data.dialog)
    gtk_widget_destroy (data.dialog);
  
  g_free (data.dirname);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->playlist_tree));
  playlist_selection_changed_cb (selection, window);

  g_object_unref (glade);
}

static char *
item_factory_trans_cb (const char *path,
		       gpointer    data)
{
	return _((char*) path);
}

static GList *
query_update_func (Query    *query,
		   gboolean  first,
		   GList    *current,
		   gpointer  user_data)
{
  MainWindow *window = user_data;
  MainWindowPriv *priv;
  GList *l;
  int i;
  char *str;
  GtkTreeIter iter;
  int burst;

  priv = window->priv;

  /* The first burst, set things up. */
  if (first)
    {
      priv->num_songs = 0;
      priv->total_size = 0;
      priv->total_length = 0;
      
      song_list_model_remove_delta (priv->song_model, current);

      if (GTK_WIDGET (window)->window)
	gdk_window_set_cursor (GTK_WIDGET (window)->window, priv->watch_arrow_cursor);
      
      if (g_hash_table_size (priv->artist_hash) == 0)
	setup_artist_view (window);
      
      if (g_hash_table_size (priv->album_hash) == 0)
	setup_album_view (window);

      burst = FIRST_BURST;
    }
  else
    burst = BURST;

  if (GTK_WIDGET_REALIZED (priv->song_tree))
    gtk_tree_view_scroll_to_point (GTK_TREE_VIEW (priv->song_tree), 0, 0);
  
  for (i = 0, l = current; i < burst && l; i++, l = l->next)
    add_song (window, l->data);
  
  /* The last is done, finish up. */
  if (l == NULL)
    {
      if (priv->browse_mode)
	{
	  i = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (priv->artist_store), NULL) - 1;
	  
	  if (i == 1)
	    str = g_strdup (_("All (1 artist)"));
	  else
	    str = g_strdup_printf (_("All (%d artists)"), i);
	  
	  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->artist_store), &iter))
	    gtk_list_store_set (priv->artist_store, &iter,
				ARTIST_COL_NAME, str,
				-1);
	  g_free (str);
	  
	  i = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (priv->album_store), NULL) - 1;
	  
	  if (i == 1)
	    str = g_strdup (_("All (1 album)"));
	  else
	    str = g_strdup_printf (_("All (%d albums)"), i);
	  
	  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->album_store), &iter))
	    gtk_list_store_set (priv->album_store, &iter,
				ALBUM_COL_NAME, str,
				-1);
	  g_free (str);
	}
      
      if (GTK_WIDGET (window)->window)
	gdk_window_set_cursor (GTK_WIDGET (window)->window, NULL);

      /* FIXME: rework this to not free/null the query until we change the
       * query.
       */
      priv->query = NULL;
      
      if (player_get_state (priv->player) != PLAYER_STATE_STOPPED)
	song_list_model_set_current (priv->song_model, NULL);
      else 
	song_list_model_first (priv->song_model);

      update_songs_info (window);
      update_playing_info (window);
    }

  return l;
}

static void
start_query (MainWindow *window, Query *query)
{
  MainWindowPriv *priv;

  priv = window->priv;

  if (query == priv->query)
    return;
  
  query_ref (query);

  if (priv->query)
    {
      if (!query_cancel (priv->query))
	query_unref (priv->query);
    }  

  priv->query = query;
  query_run (priv->query, priv->db->songs, query_update_func, window);

  query_unref (query);
}

static void
gconf_notify_cb (GConfClient *client,
		 guint        cnxn_id,
		 GConfEntry  *entry,
		 gpointer     user_data)
{

}

#ifdef CRACK
static void 
generate_m3u_cb (MainWindow *window)
{
  MainWindowPriv *priv;
  GList *l;
  Song *song;
  char *filename;
  FILE *file;
  char *str, *tmp;

  priv = window->priv;

  filename = g_build_filename (g_get_home_dir (), "jamboree.m3u", NULL);

  file = fopen (filename, "w");

  fwrite ("#EXTM3U\n", strlen ("#EXTM3U\n"), 1, file);
  
  for (l = priv->song_model->songs; l; l = l->next)
    {
      song = l->data;

      tmp = g_convert (song->title, -1,
		       "ISO-8859-1", "UTF-8",
		       NULL, NULL, NULL);
      
      str = g_strdup_printf ("#EXTINF:%d,%s\n%s\n",
			     song->length / 1000,
			     tmp,
			     song->filename);

      fwrite (str, strlen (str), 1, file);

      g_free (tmp);
      g_free (str);
    }
  
  fclose (file);
  g_free (filename);
}
#endif

void
main_window_handle_next (MainWindow *window)
{
  MainWindowPriv *priv;
  Song *song;
  
  priv = window->priv;
  
  song = song_list_model_next (priv->song_model);

  if (!song)
    song = song_list_model_first (priv->song_model);

  if (player_set_song (priv->player, song))
    update_playing_info (window);
}

void
main_window_handle_prev (MainWindow *window)
{
  MainWindowPriv *priv;
  Song *song;
  int sec;
  
  priv = window->priv;

  if (player_get_state (priv->player) == PLAYER_STATE_PLAYING)
    {
      sec = player_tell (priv->player) / 1000;

      if (sec > 3)
	{
	  player_stop (priv->player);
	  player_play (priv->player);
	  update_playing_info (window);
	  return;
	}
    }
  
  song = song_list_model_prev (priv->song_model);

  if (!song)
    song = song_list_model_first (priv->song_model);

  if (player_set_song (priv->player, song))
    update_playing_info (window);
}

void
main_window_handle_stop (MainWindow *window)
{
  player_stop (window->priv->player);
}

void
main_window_handle_play (MainWindow *window)
{
  MainWindowPriv *priv;
  Song *song;
  
  priv = window->priv;

  switch (player_get_state (priv->player))
    {
    case PLAYER_STATE_PLAYING:
      player_pause (priv->player);
      break;
    case PLAYER_STATE_PAUSED:
      player_play (priv->player);
      break;
    default:
      song = song_list_model_get_current (priv->song_model);
      if (!song)
	song = song_list_model_first (priv->song_model);

      player_play_song (priv->player, song);
      break;
    }
  
  update_playing_info (window);
}

void
main_window_push_mute (MainWindow *window)
{
  MainWindowPriv *priv;
  
  priv = window->priv;

  player_push_mute (priv->player);
}

void
main_window_pop_mute (MainWindow *window)
{
  MainWindowPriv *priv;
  
  priv = window->priv;

  player_pop_mute (priv->player);
}

/* Note: window and player should just be singletons. */
MainWindow *
main_window_get (void)
{
  return glob_window;
}

Song *
main_window_get_current_song (MainWindow *window)
{
  MainWindowPriv *priv = window->priv;
  PlayerState state;
  Song *song = NULL;

  state = player_get_state (priv->player);
  switch (state)
    {
    case PLAYER_STATE_PAUSED:
      song = player_get_song (priv->player);
      break;
    case PLAYER_STATE_PLAYING:
      song = player_get_song (priv->player);
      break;
    default:
      break;
    }

  if (!song)
    song = song_list_model_get_current (priv->song_model);

  if (!song)
    song = song_list_model_first (priv->song_model);

  return song;
}

Player *
main_window_get_player (MainWindow *window)
{
  return window->priv->player;
}
