/* testhashset.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>




void test_hashset_add (void);
void test_hashset_clear (void);
void test_hashset_contains (void);
void test_hashset_remove (void);
void test_hashset_size (void);
void test_hashset_iterator (void);
void test_hashset_empty (void);
void test_hashset_add_all (void);
void test_hashset_contains_all (void);
void test_hashset_remove_all (void);
void test_hashset_retain_all (void);
static void _test_hashset_add_gcallback (void);
static void _test_hashset_clear_gcallback (void);
static void _test_hashset_contains_gcallback (void);
static void _test_hashset_remove_gcallback (void);
static void _test_hashset_size_gcallback (void);
static void _test_hashset_empty_gcallback (void);
static void _test_hashset_add_all_gcallback (void);
static void _test_hashset_contains_all_gcallback (void);
static void _test_hashset_remove_all_gcallback (void);
static void _test_hashset_retain_all_gcallback (void);
static void _test_hashset_iterator_gcallback (void);
void _main (char** args, int args_length1);



void test_hashset_add (void) {
	GeeHashSet* hashset;
	GeeHashSet* hashsetInt;
	GeeHashSet* hashsetObject;
	GObject* fooObject;
	GObject* fooObject2;
	/* Check adding of strings*/
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "42");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, "42"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 1);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "43");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, "42"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, "43"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	/* Check add same element*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "43");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, "42"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, "43"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	/* Check adding of ints*/
	hashsetInt = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (42));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (42)));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 1);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (42)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	/* Check add same element*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (42)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	/* Check adding of objects*/
	hashsetObject = gee_hash_set_new (G_TYPE_OBJECT, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal);
	fooObject = g_object_new (G_TYPE_OBJECT, NULL);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetObject, fooObject);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetObject, fooObject));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 1);
	fooObject2 = g_object_new (G_TYPE_OBJECT, NULL);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetObject, fooObject2);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetObject, fooObject));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetObject, fooObject2));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 2);
	/* Check add same element*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 2);
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetObject, fooObject2);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetObject, fooObject));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetObject, fooObject2));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 2);
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
	(hashsetInt == NULL) ? NULL : (hashsetInt = (g_object_unref (hashsetInt), NULL));
	(hashsetObject == NULL) ? NULL : (hashsetObject = (g_object_unref (hashsetObject), NULL));
	(fooObject == NULL) ? NULL : (fooObject = (g_object_unref (fooObject), NULL));
	(fooObject2 == NULL) ? NULL : (fooObject2 = (g_object_unref (fooObject2), NULL));
}


void test_hashset_clear (void) {
	GeeHashSet* hashset;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check clear on empty set*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check clear one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check clear multiple items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "3");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
}


void test_hashset_contains (void) {
	GeeHashSet* hashsetString;
	GeeHashSet* hashsetInt;
	hashsetString = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check on empty set*/
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "1"));
	/* Check items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "10");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "10"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "20"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "30"));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "20");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "10"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "20"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "30"));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "30");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "10"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "20"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "30"));
	/* Clear and recheck*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashsetString);
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "10"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "20"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "30"));
	hashsetInt = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* Check items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (10));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (30)));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (20));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (30)));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (30));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (30)));
	/* Clear and recheck*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashsetInt);
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (30)));
	(hashsetString == NULL) ? NULL : (hashsetString = (g_object_unref (hashsetString), NULL));
	(hashsetInt == NULL) ? NULL : (hashsetInt = (g_object_unref (hashsetInt), NULL));
}


void test_hashset_remove (void) {
	GeeHashSet* hashsetString;
	GeeHashSet* hashsetInt;
	hashsetString = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check remove if list is empty*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetString, "42");
	/* Add 5 different elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "43");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "44");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "45");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 5);
	/* Check remove first*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 4);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "43"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "44"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "45"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "46"));
	/* Check remove last*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 3);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "43"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "44"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "45"));
	/* Check remove in between*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetString, "44");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 2);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "43"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetString, "45"));
	/* Check removing of int element*/
	hashsetInt = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* Add 5 different elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (42));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (44));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (45));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 5);
	/* Remove first*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (42));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 4);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (44)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (45)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (46)));
	/* Remove last*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 3);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (44)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (45)));
	/* Remove in between*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (44));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashsetInt, GINT_TO_POINTER (45)));
	(hashsetString == NULL) ? NULL : (hashsetString = (g_object_unref (hashsetString), NULL));
	(hashsetInt == NULL) ? NULL : (hashsetInt = (g_object_unref (hashsetInt), NULL));
}


void test_hashset_size (void) {
	GeeHashSet* hashset;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check empty list*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check when one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 1);
	/* Check when more items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "2");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	/* Check when items cleared*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
}


void test_hashset_iterator (void) {
	GeeHashSet* hashset;
	GeeIterator* iterator;
	GeeIterator* _tmp0_;
	char* firstString;
	char* secondString;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check iterate empty list*/
	iterator = gee_abstract_collection_iterator ((GeeAbstractCollection*) hashset);
	g_assert (!gee_iterator_next (iterator));
	/* Check iterate list*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "43");
	_tmp0_ = NULL;
	iterator = (_tmp0_ = gee_abstract_collection_iterator ((GeeAbstractCollection*) hashset), (iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL)), _tmp0_);
	/* A set is usually not ordered, so this is not a requirement */
	g_assert (gee_iterator_next (iterator));
	firstString = (char*) gee_iterator_get (iterator);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, firstString));
	g_assert (gee_iterator_next (iterator));
	secondString = (char*) gee_iterator_get (iterator);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset, secondString));
	g_assert (!g_str_equal (firstString, secondString));
	/* they can not be identical neither equal*/
	g_assert (!gee_iterator_next (iterator));
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
	(iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL));
	firstString = (g_free (firstString), NULL);
	secondString = (g_free (secondString), NULL);
}


void test_hashset_empty (void) {
	GeeHashSet* hashset;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Case 1: Check empty set*/
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset));
	/* Case 2: Check when one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "1");
	g_assert (!gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset));
	/* Case 3: Check when more items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset, "2");
	g_assert (!gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset));
	/* Case 4: Check when items cleared*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset);
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset));
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
}


void test_hashset_add_all (void) {
	GeeHashSet* hashset1;
	GeeHashSet* hashset2;
	hashset1 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	hashset2 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Case 1: Check set empty*/
	gee_abstract_collection_add_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset1));
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset2));
	/* Case 2: Check hashset1 not empty, hashset2 is empty*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 1);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "1"));
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 3: Check hashset1 empty, hashset2 contains 1 element*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "1");
	gee_abstract_collection_add_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 1);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "1"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 1);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "1"));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 4: Check more elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "0");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "5");
	gee_abstract_collection_add_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 6);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "0"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "1"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "2"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "3"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "4"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "5"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "3"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "4"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "5"));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 5: Check intersecting elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "0");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "5");
	gee_abstract_collection_add_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 6);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "0"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "1"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "2"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "3"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "4"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "5"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 4);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "2"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "3"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "4"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, "5"));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 6: Add large collections*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "0");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	{
		gint i;
		i = 3;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				char* _tmp1_;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < 103)) {
					break;
				}
				_tmp1_ = NULL;
				gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, _tmp1_ = g_strdup_printf ("%d", i));
				_tmp1_ = (g_free (_tmp1_), NULL);
			}
		}
	}
	gee_abstract_collection_add_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 103);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "0"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "1"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "2"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "3"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "4"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "5"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "99"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "100"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "101"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, "102"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 100);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	(hashset1 == NULL) ? NULL : (hashset1 = (g_object_unref (hashset1), NULL));
	(hashset2 == NULL) ? NULL : (hashset2 = (g_object_unref (hashset2), NULL));
}


void test_hashset_contains_all (void) {
	GeeHashSet* hashset1;
	GeeHashSet* hashset2;
	hashset1 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	hashset2 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Case 1: Check empty*/
	g_assert (gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	/* Case 2: Hashset1 has elements, hashset2 is empty*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	g_assert (gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 3: Hashset1 is empty, hashset2 has elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "1");
	g_assert (!gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 4: Hashset1 and hashset2 are the same*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	g_assert (gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 5: Hashset1 and hashset2 are not the same*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "2");
	g_assert (!gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 6: Hashset1 has a subset of hashset2*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "5");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "6");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "6");
	g_assert (gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 7: Hashset1 has a subset of hashset2 in all but one element hashset2*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "5");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "6");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "6");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "7");
	g_assert (!gee_abstract_collection_contains_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	(hashset1 == NULL) ? NULL : (hashset1 = (g_object_unref (hashset1), NULL));
	(hashset2 == NULL) ? NULL : (hashset2 = (g_object_unref (hashset2), NULL));
}


void test_hashset_remove_all (void) {
	GeeHashSet* hashset1;
	GeeHashSet* hashset2;
	hashset1 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	hashset2 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Case 1: Check empty*/
	gee_abstract_collection_remove_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset1));
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset2));
	/* Case 2: Hashset1 and hashset2 have no common elements -> nothing is removed in hashset1*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "5");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "6");
	gee_abstract_collection_remove_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 3);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 3: Hashset1 and hashset2 have all elements the same -> everything is removed in hashset1 but not hashset2*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, "3");
	gee_abstract_collection_remove_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset1));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	(hashset1 == NULL) ? NULL : (hashset1 = (g_object_unref (hashset1), NULL));
	(hashset2 == NULL) ? NULL : (hashset2 = (g_object_unref (hashset2), NULL));
}


void test_hashset_retain_all (void) {
	GeeHashSet* hashset1;
	GeeHashSet* hashset2;
	hashset1 = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	hashset2 = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* Case 1: Check empty*/
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset1));
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset2));
	gee_abstract_collection_retain_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset1));
	g_assert (gee_abstract_collection_get_is_empty ((GeeAbstractCollection*) hashset2));
	/* Case 2: Hashset1 has elements, hashset2 is empty -> everything in hashset1 is removed*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (1));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (3));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 3);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 0);
	gee_abstract_collection_retain_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 0);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 0);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 3: Hashset1 is empty and hashset2 has elements -> nothing changes*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (4));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (5));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (6));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 0);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_retain_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 0);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 4: Hashset1 and hashset2 have no common elements -> everything is removed in hashset1*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (1));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (3));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (4));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (5));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (6));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 3);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_retain_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 0);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 5: Hashset1 and hashset2 have all elements the same -> nothing is removed in hashset1*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (1));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (3));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (1));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (3));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 3);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_retain_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 3);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	/* Case 6: Hashset1 and hashset2 have 2 common elements but each also has his own elements -> hashset1 only retains what is in hashset2*/
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (1));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (3));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (4));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (5));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (0));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (3));
	gee_abstract_collection_add ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (7));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 5);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 4);
	gee_abstract_collection_retain_all ((GeeAbstractCollection*) hashset1, (GeeCollection*) hashset2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset1) == 2);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset2) == 4);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset1, GINT_TO_POINTER (2)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) hashset2, GINT_TO_POINTER (3)));
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) hashset2);
	(hashset1 == NULL) ? NULL : (hashset1 = (g_object_unref (hashset1), NULL));
	(hashset2 == NULL) ? NULL : (hashset2 = (g_object_unref (hashset2), NULL));
}


static void _test_hashset_add_gcallback (void) {
	test_hashset_add ();
}


static void _test_hashset_clear_gcallback (void) {
	test_hashset_clear ();
}


static void _test_hashset_contains_gcallback (void) {
	test_hashset_contains ();
}


static void _test_hashset_remove_gcallback (void) {
	test_hashset_remove ();
}


static void _test_hashset_size_gcallback (void) {
	test_hashset_size ();
}


static void _test_hashset_empty_gcallback (void) {
	test_hashset_empty ();
}


static void _test_hashset_add_all_gcallback (void) {
	test_hashset_add_all ();
}


static void _test_hashset_contains_all_gcallback (void) {
	test_hashset_contains_all ();
}


static void _test_hashset_remove_all_gcallback (void) {
	test_hashset_remove_all ();
}


static void _test_hashset_retain_all_gcallback (void) {
	test_hashset_retain_all ();
}


static void _test_hashset_iterator_gcallback (void) {
	test_hashset_iterator ();
}


void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	/* Methods of Collection interface*/
	g_test_add_func ("/HashSet/Collection/add", (GCallback) _test_hashset_add_gcallback);
	g_test_add_func ("/HashSet/Collection/clear", (GCallback) _test_hashset_clear_gcallback);
	g_test_add_func ("/HashSet/Collection/contains", (GCallback) _test_hashset_contains_gcallback);
	g_test_add_func ("/HashSet/Collection/remove", (GCallback) _test_hashset_remove_gcallback);
	g_test_add_func ("/HashSet/Collection/size", (GCallback) _test_hashset_size_gcallback);
	g_test_add_func ("/HashSet/Collection/empty", (GCallback) _test_hashset_empty_gcallback);
	g_test_add_func ("/HashSet/Collection/add_all", (GCallback) _test_hashset_add_all_gcallback);
	g_test_add_func ("/HashSet/Collection/contains_all", (GCallback) _test_hashset_contains_all_gcallback);
	g_test_add_func ("/HashSet/Collection/remove_all", (GCallback) _test_hashset_remove_all_gcallback);
	g_test_add_func ("/HashSet/Collection/retain_all", (GCallback) _test_hashset_retain_all_gcallback);
	g_test_add_func ("/HashSet/Iterable/iterator", (GCallback) _test_hashset_iterator_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}




