/* GNOME DB library
 * Copyright (C) 1999 - 2005 The GNOME Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <libgda/gda-client.h>
#include <libgda/gda-config.h>
#include <libgda/gda-parameter-list.h>
#include <libgda/gda-parameter.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnomeui/gnome-druid-page-edge.h>
#include <libgnomeui/gnome-druid-page-standard.h>
#include <libgnomedb/gnome-db-dsn-config-druid.h>
#include <libgnomedb/gnome-db-dsn-spec.h>
#include <libgnomedb/gnome-db-provider-selector.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-basic-form.h>
#include <glib/gi18n-lib.h>

#define PARENT_TYPE GNOME_TYPE_DRUID

struct _GnomeDbDsnConfigDruidPrivate {
	GdaClient         *client;
	GdaDataSourceInfo *dsn_info;
	GdaParameterList  *paramlist;

	GdkPixbuf *logo;

	/* widgets */
	GtkWidget *start_page;

	GtkWidget *general_page;
	GtkWidget *general_name;
	GtkWidget *general_provider;
	GtkWidget *general_description;
	GtkWidget *general_username;
	GtkWidget *general_password;
	GtkWidget *general_is_global;
	gchar     *create_db_spec;

	GtkWidget *choose_page;
	GtkWidget *choose_toggle;

	GtkWidget *newdb_page;
	GtkWidget *newdb_box;
	GtkWidget *newdb_params;

	GtkWidget *provider_page;
	GtkWidget *provider_container;
	GtkWidget *provider_detail;

	GtkWidget  *end_page;
};

static void gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass);
static void gnome_db_dsn_config_druid_init       (GnomeDbDsnConfigDruid *druid,
						  GnomeDbDsnConfigDruidClass *klass);
static void gnome_db_dsn_config_druid_finalize   (GObject *object);

enum {
	FINISHED,
	LAST_SIGNAL
};

static guint config_druid_signals[LAST_SIGNAL] = { 0, };
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static gchar *
get_specs_database_creation (GnomeDbDsnConfigDruid *druid)
{
	if (! druid->priv->create_db_spec) {
		if (!druid->priv->client)
			druid->priv->client = gda_client_new ();
		
		druid->priv->create_db_spec = gda_client_get_provider_specs (druid->priv->client, 
								    	     gnome_db_provider_selector_get_selected_provider (
								             GNOME_DB_PROVIDER_SELECTOR (druid->priv->general_provider)),
								             GDA_CLIENT_SPECS_CREATE_DATABASE);
	}

	return druid->priv->create_db_spec;
}

static void
druid_cancelled_cb (GnomeDruidPage *druid_page, GtkWidget *druid_widget, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));
	g_signal_emit_by_name (G_OBJECT (druid), "finished", TRUE);
}

static void
druid_finished_cb (GnomeDruidPage *druid_page, GtkWidget *druid_widget, GnomeDbDsnConfigDruid *druid)
{
	gboolean allok = TRUE;
	GString *cnc_string = NULL;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* set the internal dsn_info with the values of the widgets */
	if (druid->priv->dsn_info) {
		gda_data_source_info_free (druid->priv->dsn_info);
		druid->priv->dsn_info = NULL;
	}

	/* New database creation first */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (druid->priv->choose_toggle))) {
		if (!gda_parameter_list_is_valid (druid->priv->paramlist)) {
			gnome_db_show_error (NULL, _("Missing mandatory informmation, to create database"));
			gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->newdb_page));
			return;
		}
		else {
			GSList *params;	
			GdaProviderInfo *prov_info;
			GSList *dsn_params;
			GError *error = NULL;

			params = druid->priv->paramlist->parameters;
			while (params) {
				gda_object_set_name (GDA_OBJECT (params->data), gda_object_get_id (GDA_OBJECT (params->data)));
				params = g_slist_next (params);
			}
			
			allok = gda_client_create_database (druid->priv->client, 
							    gnome_db_provider_selector_get_selected_provider (
								    GNOME_DB_PROVIDER_SELECTOR (druid->priv->general_provider)),
							    druid->priv->paramlist, &error);
			if (!allok) {
				gchar *str;
				str = g_strdup_printf (_("Error creating database: %s"), 
						       error && error->message ? error->message : _("Unknown error"));
				gnome_db_show_error (NULL, str);
				g_free (str);
				
				gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->newdb_page));
				return;
			}
			
			/* make the connection string for the data source */
			prov_info = gda_config_get_provider_by_name (gnome_db_provider_selector_get_selected_provider (
									     GNOME_DB_PROVIDER_SELECTOR (druid->priv->general_provider)));
			g_return_if_fail (prov_info);
			dsn_params = prov_info->gda_params->parameters;
			while (dsn_params) {
				GdaParameter *param = GDA_PARAMETER (dsn_params->data);				
				const GdaValue *value;
				
				value = gda_parameter_get_value (param);
				if (value && !gda_value_is_null ((GdaValue *) value)) {
					gchar *str;

					if (dsn_params == prov_info->gda_params->parameters)
						cnc_string = g_string_new ("");
					else
						g_string_append (cnc_string, "; ");
					str = gda_value_stringify ((GdaValue *) value);
					g_string_append_printf (cnc_string, "%s=%s",
								gda_object_get_name (GDA_OBJECT (param)), str);
					g_free (str);
				}
				
				dsn_params = g_slist_next (dsn_params);
			}
		}
	}

	/* Data source declaration */
	if (allok) {
		druid->priv->dsn_info = g_new0 (GdaDataSourceInfo, 1);
		druid->priv->dsn_info->name = g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_name)));
		druid->priv->dsn_info->provider = g_strdup (gnome_db_provider_selector_get_selected_provider (
								    GNOME_DB_PROVIDER_SELECTOR (druid->priv->general_provider)));
		if (cnc_string) {
			druid->priv->dsn_info->cnc_string = cnc_string->str;
			g_string_free (cnc_string, FALSE);
		}
		else
			druid->priv->dsn_info->cnc_string = gnome_db_dsn_spec_get_specs 
				(GNOME_DB_DSN_SPEC (druid->priv->provider_detail));
		druid->priv->dsn_info->description =
			g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_description)));
		druid->priv->dsn_info->username =
			g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_username)));
		druid->priv->dsn_info->password =
			g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_password)));
		if (gda_config_can_modify_global_config ())
			druid->priv->dsn_info->is_global = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (druid->priv->general_is_global));
		else
			druid->priv->dsn_info->is_global = FALSE;

		allok = gda_config_save_data_source_info (druid->priv->dsn_info);
	}

	/* notify listeners */
	g_signal_emit (G_OBJECT (druid), config_druid_signals[FINISHED], 0, !allok);
}

static void
provider_changed_cb (GtkWidget *combo, GnomeDbDsnConfigDruid *druid)
{
	/* clean any previous Provider specific stufv */
	if (druid->priv->newdb_params) {
		gtk_widget_destroy (druid->priv->newdb_params);
		druid->priv->newdb_params = NULL;
	}

	/* is the database creation supported by the chosen provider? */
	if (druid->priv->create_db_spec) {
		g_free (druid->priv->create_db_spec);
		druid->priv->create_db_spec = NULL;
	}
}

static gboolean
general_next_pressed_cb (GnomeDruidPage *druid_page,
			 GtkWidget *druid_widget,
			 GnomeDbDsnConfigDruid *druid)
{
	GdaDataSourceInfo *dsn_info;
	GdaProviderInfo *prov_info;
	gint i;
	const gchar *name;
	const gchar *provider;
	gchar *str;

	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), TRUE);

	/* check required fields have values */
	name = gtk_entry_get_text (GTK_ENTRY (druid->priv->general_name));
	if (!name || strlen (name) < 1) {
		gnome_db_show_error (NULL, _("You must enter a name for the new data source"));
		gtk_widget_grab_focus (druid->priv->general_name);
		return TRUE;
	}

	for (i = 0; i < strlen (name); i++) {
		if (!g_ascii_isalnum (name[i])) {
			gnome_db_show_error (NULL, _("Data source names can only contain alphanumeric characters"));
			gtk_widget_grab_focus (druid->priv->general_name);
			return TRUE;
		}
	}

	dsn_info = gda_config_find_data_source (name);
	if (dsn_info != NULL) {
		gda_data_source_info_free (dsn_info);
		gnome_db_show_error (NULL, _("An existing data source is already registered under the same name"));
		gtk_widget_grab_focus (druid->priv->general_name);
		return TRUE;
	}

	for (i = 0; i < strlen (name); i++) {
		if (g_ascii_isspace (name[i])) {
			gnome_db_show_error (NULL, _("Data source names cannot contain spaces"));
			gtk_widget_grab_focus (druid->priv->general_name);
			return TRUE;
		} else if (!g_ascii_isalnum (name[i])) {
			gnome_db_show_error (NULL, _("Data source names can only contain ASCII letters and numbers"));
			gtk_widget_grab_focus (druid->priv->general_name);
			return TRUE;
		}
	}

	provider =  gnome_db_provider_selector_get_selected_provider (
		GNOME_DB_PROVIDER_SELECTOR (druid->priv->general_provider));
	if (!provider) {
		gnome_db_show_error (NULL, _("You must select a provider for the new data source"));
		gtk_widget_grab_focus (druid->priv->general_provider);
		return TRUE;
	}

	/* set up widgets for this specific provider */
	prov_info = gda_config_get_provider_by_name (provider);
	if (!prov_info) {
		str = g_strdup_printf (_("Could not find provider %s in the current setup"),
				       provider);
		gnome_db_show_error (NULL, str);
		g_free (str);
		gtk_widget_grab_focus (druid->priv->general_provider);
		return TRUE;
	}

	if (!druid->priv->provider_detail) {
		druid->priv->provider_detail = gnome_db_dsn_spec_new (provider);
		gtk_box_pack_start (GTK_BOX (druid->priv->provider_container),
				    druid->priv->provider_detail, TRUE, TRUE, 0);
		gtk_widget_show (druid->priv->provider_detail);
	}
	else
		gnome_db_dsn_spec_set_provider (GNOME_DB_DSN_SPEC (druid->priv->provider_detail), provider);


	str = get_specs_database_creation (druid);
	if (str) 
		return FALSE;
	else {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (druid->priv->choose_toggle), FALSE);
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->provider_page));
		return TRUE;
	}	
}

static gboolean
choose_next_pressed_cb (GnomeDruidPage *druid_page,
			GtkWidget *druid_widget,
			GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), FALSE);

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (druid->priv->choose_toggle))) {
		/* prepare a new data set for database creation parameters */
		gchar *str;

		if (! druid->priv->newdb_params) {
			if (!druid->priv->client)
				druid->priv->client = gda_client_new ();
			
			str = gda_client_get_provider_specs (druid->priv->client, 
						             gnome_db_provider_selector_get_selected_provider (
						             GNOME_DB_PROVIDER_SELECTOR (druid->priv->general_provider)),
						             GDA_CLIENT_SPECS_CREATE_DATABASE);
			if (str) {
				GdaParameterList *dset;
				GError *error = NULL;
				
				dset = GDA_PARAMETER_LIST (gda_parameter_list_new_from_spec (NULL, str, &error));
				if (dset) {
					druid->priv->paramlist = dset;
					druid->priv->newdb_params = gnome_db_basic_form_new (dset);
					gnome_db_basic_form_show_entries_actions (GNOME_DB_BASIC_FORM (druid->priv->newdb_params),
										  FALSE);
				}
				else {
					gchar *str2;
					str2 = g_strdup_printf ("Provider internal error: %s", 
								error && error->message ? error->message : "???" );
					druid->priv->newdb_params = gtk_label_new (str2);
					g_free (str2);
				}
				
				gtk_widget_show (druid->priv->newdb_params);
				gtk_box_pack_start (GTK_BOX (druid->priv->newdb_box), druid->priv->newdb_params,
						    TRUE, TRUE, 0);
				g_free (str);
			}
		}
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->newdb_page));
	}
	else
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->provider_page));

	return TRUE;
}

static gboolean
newdb_next_pressed_cb (GnomeDruidPage *druid_page,
		       GtkWidget *druid_widget,
		       GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), FALSE);

	gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->end_page));

	return TRUE;
}

static gboolean
provider_back_pressed_cb (GnomeDruidPage *druid_page,
			  GtkWidget *druid_widget,
			  GnomeDbDsnConfigDruid *druid)
{
	gchar *str;
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), TRUE);


	/*
	 * is the database creation supported by the chosen provider?
	 */
	str = get_specs_database_creation (druid);
	if (str) {
		/* database creation is supported */
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->choose_page));
	}
	else 
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->general_page));

	return TRUE;
}

static gboolean
end_back_pressed_cb (GnomeDruidPage *druid_page,
			  GtkWidget *druid_widget,
			  GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), TRUE);
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (druid->priv->choose_toggle)))
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->newdb_page));
	else
		gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druid->priv->provider_page));

	return TRUE;
}

/*
 * GnomeDbDsnConfigDruid class implementation
 */

static void
gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	config_druid_signals[FINISHED] =
		g_signal_new ("finished",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbDsnConfigDruidClass, finished),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);

	klass->finished = NULL;
	object_class->finalize = gnome_db_dsn_config_druid_finalize;
}

static void
gnome_db_dsn_config_druid_init (GnomeDbDsnConfigDruid *druid,
				GnomeDbDsnConfigDruidClass *klass)
{
	GtkWidget *label;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* create private structure */
	druid->priv = g_new0 (GnomeDbDsnConfigDruidPrivate, 1);
	druid->priv->dsn_info = g_new0 (GdaDataSourceInfo, 1);
	druid->priv->provider_detail = NULL;
	druid->priv->create_db_spec = NULL;

	/* load icons */
	druid->priv->logo = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);

	/* set up widgets */
	gnome_druid_set_show_help (GNOME_DRUID (druid), FALSE);

	/* 
	 * start page
	 */
	druid->priv->start_page = gnome_druid_page_edge_new_with_vals (
		GNOME_EDGE_START, TRUE, _("Add a new data source..."),
		_("This druid will guide you through the process of\n"
		  "creating a new data source for your GNOME-DB\n"
		  "installation. Just follow the steps!"),
		druid->priv->logo, NULL, NULL);
	gtk_widget_show (druid->priv->start_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->start_page));

	g_signal_connect (G_OBJECT (druid->priv->start_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);

	/* 
	 * general info page 
	 */
	druid->priv->general_page = gnome_druid_page_standard_new_with_vals (
		_("General Information"), druid->priv->logo, NULL);

	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
			      _("The following fields represent the basic information "
				"items for your new data source. Some of them (description, "
				"username) are optional, whereas the others are mandatory. "
				"If you don't know which provider to use, just select "
				"the default one."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page), NULL, label, NULL);

	druid->priv->general_name = gnome_db_new_entry_widget (0, TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("Data source _name:"), druid->priv->general_name, NULL);

	if (gda_config_can_modify_global_config ()) {
		druid->priv->general_is_global = gtk_check_button_new ();
		gnome_druid_page_standard_append_item (
			GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
			_("System wide data source:"), druid->priv->general_is_global, NULL);
	}
	else
		druid->priv->general_is_global = NULL;

	druid->priv->general_provider = gnome_db_provider_selector_new ();
	gtk_widget_show (druid->priv->general_provider);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("_Provider:"), druid->priv->general_provider, NULL);
	g_signal_connect (G_OBJECT (druid->priv->general_provider), "changed",
			  G_CALLBACK (provider_changed_cb), druid);

	druid->priv->general_description = gnome_db_new_entry_widget (0, TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("_Description:"), druid->priv->general_description, NULL);

	druid->priv->general_username = gnome_db_new_entry_widget (0, TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("_Username:"), druid->priv->general_username, NULL);

	druid->priv->general_password = gnome_db_new_entry_widget (0, TRUE);
	gtk_entry_set_visibility (GTK_ENTRY (druid->priv->general_password), FALSE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("P_assword:"), druid->priv->general_password, NULL);

	gtk_widget_show (druid->priv->general_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->general_page));

	g_signal_connect (G_OBJECT (druid->priv->general_page), "cancel",
			  G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->general_page), "next",
			  G_CALLBACK (general_next_pressed_cb), druid);

	/*
	 * Choose between existing database or create a new one
	 */
	druid->priv->choose_page = gnome_druid_page_standard_new_with_vals (
		_("Use existing database"), druid->priv->logo, NULL);
	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
			      _("This page lets you choose between using an existing database "
				"or to create a new database to use with this new data source"));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);

	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->choose_page), NULL, label, NULL);

	druid->priv->choose_toggle = gtk_check_button_new_with_label (_("Create a new database"));
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->choose_page), NULL,
		druid->priv->choose_toggle, NULL);

	gtk_widget_show (druid->priv->choose_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->choose_page));

	g_signal_connect (G_OBJECT (druid->priv->choose_page), "next",
			  G_CALLBACK (choose_next_pressed_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->choose_page), "cancel", 
			  G_CALLBACK (druid_cancelled_cb), druid);

	/*
	 * New database page
	 */
	druid->priv->newdb_page = gnome_druid_page_standard_new_with_vals (
		_("New database creation"), druid->priv->logo, NULL);
	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
			      _("The following fields represent the information needed by the "
				"provider you selected in the previous page to create a new database "
				"(mandatory fields are marked in bold text)."
				"This information is provider-specific, so check "
				"the manual for more information."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->newdb_page), NULL, label, NULL);

	druid->priv->newdb_box = gtk_vbox_new (FALSE, 0);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->newdb_page), NULL,
		druid->priv->newdb_box, NULL);
	druid->priv->newdb_params = NULL;
	
	g_signal_connect (G_OBJECT (druid->priv->newdb_page), "cancel", 
			  G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->newdb_page), "next", 
			  G_CALLBACK (newdb_next_pressed_cb), druid);

	gtk_widget_show (druid->priv->newdb_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->newdb_page));

	/* 
	 * provider parameters page 
	 */
	druid->priv->provider_page = gnome_druid_page_standard_new_with_vals (
		_("Provider Parameters"), druid->priv->logo, NULL);

	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
			      _("The following fields represent the information needed by the "
				"provider you selected in the previous page to open a given "
				"connection. "
				"This information is provider-specific, so check "
				"the manual for more information."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->provider_page), NULL, label, NULL);

	druid->priv->provider_container = gnome_db_new_vbox_widget (FALSE, 0);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->provider_page), NULL,
		druid->priv->provider_container, NULL);

	gtk_widget_show (druid->priv->provider_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->provider_page));

	g_signal_connect (G_OBJECT (druid->priv->provider_page), "cancel",
			  G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->provider_page), "back",
			  G_CALLBACK (provider_back_pressed_cb), druid);

	/* 
	 * end page 
	 */
	druid->priv->end_page = gnome_druid_page_edge_new_with_vals (
		GNOME_EDGE_FINISH, TRUE, _("All information retrieved"),
		_("All information needed to create a new data source\n"
		  "has been retrieved. Now, press 'Apply' to close\n"
		  "this dialog. To open your newly created data source,\n"
		  "just select it when asked for a data source."),
		  druid->priv->logo, NULL, NULL);
	gtk_widget_show (druid->priv->end_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->end_page));

	g_signal_connect (G_OBJECT (druid->priv->end_page), "back", G_CALLBACK (end_back_pressed_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->end_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->end_page), "finish", G_CALLBACK (druid_finished_cb), druid);
}

static void
gnome_db_dsn_config_druid_finalize (GObject *object)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* free memory */
	if (druid->priv->client)
		g_object_unref (druid->priv->client);

	gdk_pixbuf_unref (druid->priv->logo);
	if (druid->priv->dsn_info)
		gda_data_source_info_free (druid->priv->dsn_info);

	if (druid->priv->create_db_spec)
		g_free (druid->priv->create_db_spec);

	if (druid->priv->paramlist)
		g_object_unref (druid->priv->paramlist);

	g_free (druid->priv);
	druid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_dsn_config_druid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDsnConfigDruidClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dsn_config_druid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDsnConfigDruid),
			0,
			(GInstanceInitFunc) gnome_db_dsn_config_druid_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbDsnConfigDruid",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_dsn_config_druid_new
 *
 *
 *
 * Returns:
 */
GtkWidget *
gnome_db_dsn_config_druid_new (void)
{
	GnomeDbDsnConfigDruid *druid;

	druid = g_object_new (GNOME_DB_TYPE_DSN_CONFIG_DRUID, NULL);
	return GTK_WIDGET (druid);
}

/**
 * gnome_db_dsn_config_druid_get_dsn
 * @druid:
 *
 *
 *
 * Returns:
 */
const GdaDataSourceInfo *
gnome_db_dsn_config_druid_get_dsn (GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), NULL);
	return (const GdaDataSourceInfo *) druid->priv->dsn_info;
}
