/* gnome-db-util.c
 *
 * Copyright (C) 2005 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include "gnome-db-util.h"
#include "gnome-db-data-entry.h"
#include "data-entries/gnome-db-entry-boolean.h"
#include "data-entries/gnome-db-entry-string.h"
#include "data-entries/gnome-db-entry-time.h"
#include "data-entries/gnome-db-entry-none.h"
#include "data-entries/gnome-db-data-cell-renderer-textual.h"
#include "data-entries/gnome-db-data-cell-renderer-boolean.h"
#include <libgda/libgda.h>

/**
 * gnome_db_util_new_data_entry
 * @type: a #GdaValueType
 * @entry_name: the name of an entry plugin, or %NULL
 *
 * Creates a new #GtkCellRenderer object which is suitable to use in
 * a #GtkTreeView widget.
 * 
 * Returns: a new #GtkCellRenderer object, _NEVER_ %NULL
 */
GnomeDbDataEntry *
gnome_db_util_new_data_entry (GdaValueType type, const gchar *entry_name)
{
	GdaDataHandler *dh;
	g_return_val_if_fail (type < GDA_VALUE_TYPE_UNKNOWN, NULL);

	switch (type) {
	case GDA_VALUE_TYPE_NULL:
		return (GnomeDbDataEntry *) gnome_db_entry_none_new (GDA_VALUE_TYPE_NULL);

	case GDA_VALUE_TYPE_BIGINT:
	case GDA_VALUE_TYPE_BIGUINT:
	case GDA_VALUE_TYPE_BINARY:
	case GDA_VALUE_TYPE_BLOB:
	case GDA_VALUE_TYPE_DOUBLE:
	case GDA_VALUE_TYPE_INTEGER:
	case GDA_VALUE_TYPE_NUMERIC:
	case GDA_VALUE_TYPE_SINGLE:
	case GDA_VALUE_TYPE_SMALLINT:
	case GDA_VALUE_TYPE_SMALLUINT:
	case GDA_VALUE_TYPE_STRING:
	case GDA_VALUE_TYPE_TINYINT:
	case GDA_VALUE_TYPE_TINYUINT:
	case GDA_VALUE_TYPE_TYPE:
	case GDA_VALUE_TYPE_UINTEGER:
		dh = default_handlers [type];
		return (GnomeDbDataEntry *) gnome_db_entry_string_new (dh, type);

	case GDA_VALUE_TYPE_BOOLEAN:
		dh = default_handlers [GDA_VALUE_TYPE_BOOLEAN];
		return (GnomeDbDataEntry *) gnome_db_entry_boolean_new (dh, GDA_VALUE_TYPE_BOOLEAN);

	case GDA_VALUE_TYPE_GEOMETRIC_POINT:
	case GDA_VALUE_TYPE_GOBJECT:
	case GDA_VALUE_TYPE_LIST:
	case GDA_VALUE_TYPE_MONEY:
		return (GnomeDbDataEntry *) gnome_db_entry_none_new (type);

	case GDA_VALUE_TYPE_TIME:
	case GDA_VALUE_TYPE_TIMESTAMP:
	case GDA_VALUE_TYPE_DATE:
		dh = default_handlers [type];
		return (GnomeDbDataEntry *) gnome_db_entry_time_new (dh, type);
	default:
		g_assert_not_reached ();
	}
}

/**
 * gnome_db_util_new_cell_renderer
 * @type: a #GdaValueType
 * @entry_name: the name of an entry plugin, or %NULL
 *
 * Creates a new #GtkCellRenderer object which is suitable to use in
 * a #GtkTreeView widget.
 * 
 * Returns: a new #GtkCellRenderer object, _NEVER_ %NULL
 */
GtkCellRenderer *
gnome_db_util_new_cell_renderer (GdaValueType type, const gchar *entry_name)
{
	GdaDataHandler *dh;
	g_return_val_if_fail (type < GDA_VALUE_TYPE_UNKNOWN, NULL);

	switch (type) {
	case GDA_VALUE_TYPE_NULL:
		return gnome_db_data_cell_renderer_textual_new (NULL, GDA_VALUE_TYPE_NULL);
	case GDA_VALUE_TYPE_BOOLEAN:
		dh = default_handlers [GDA_VALUE_TYPE_BOOLEAN];
		return gnome_db_data_cell_renderer_boolean_new (dh, GDA_VALUE_TYPE_BOOLEAN);
	default:
		dh = default_handlers [type];
		return gnome_db_data_cell_renderer_textual_new (dh, type);
	}
}

