/* gnome-db-entity.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-entity.h"
#include "gnome-db-base.h"
#include "gnome-db-data-set.h"
#include "marshal.h"


/* signals */
enum
{
	FIELD_ADDED,
	FIELD_REMOVED,
	FIELD_UPDATED,
	FIELDS_ORDER_CHANGED,
	LAST_SIGNAL
};

static gint gnome_db_entity_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };

static void gnome_db_entity_iface_init (gpointer g_class);

GType
gnome_db_entity_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbEntityIface),
			(GBaseInitFunc) gnome_db_entity_iface_init,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) NULL,
			NULL,
			NULL,
			0,
			0,
			(GInstanceInitFunc) NULL
		};
		
		type = g_type_register_static (G_TYPE_INTERFACE, "GnomeDbEntity", &info, 0);
		g_type_interface_add_prerequisite (type, GNOME_DB_TYPE_BASE);
	}
	return type;
}


static void
gnome_db_entity_iface_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

	if (! initialized) {
		gnome_db_entity_signals[FIELD_ADDED] =
			g_signal_new ("field_added",
				      GNOME_DB_TYPE_ENTITY,
				      G_SIGNAL_RUN_FIRST,
				      G_STRUCT_OFFSET (GnomeDbEntityIface, field_added),
				      NULL, NULL,
				      marshal_VOID__POINTER, G_TYPE_NONE,
				      1, G_TYPE_POINTER);
		gnome_db_entity_signals[FIELD_REMOVED] =
			g_signal_new ("field_removed",
				      GNOME_DB_TYPE_ENTITY,
				      G_SIGNAL_RUN_FIRST,
				      G_STRUCT_OFFSET (GnomeDbEntityIface, field_removed),
				      NULL, NULL,
				      marshal_VOID__POINTER, G_TYPE_NONE,
				      1, G_TYPE_POINTER);
		gnome_db_entity_signals[FIELD_UPDATED] =
			g_signal_new ("field_updated",
				      GNOME_DB_TYPE_ENTITY,
				      G_SIGNAL_RUN_FIRST,
				      G_STRUCT_OFFSET (GnomeDbEntityIface, field_updated),
				      NULL, NULL,
				      marshal_VOID__POINTER, G_TYPE_NONE,
				      1, G_TYPE_POINTER);
		gnome_db_entity_signals[FIELDS_ORDER_CHANGED] =
			g_signal_new ("fields_order_changed",
				      GNOME_DB_TYPE_ENTITY,
				      G_SIGNAL_RUN_FIRST,
				      G_STRUCT_OFFSET (GnomeDbEntityIface, fields_order_changed),
				      NULL, NULL,
				      marshal_VOID__VOID, G_TYPE_NONE,
				      0);		
		initialized = TRUE;
	}
}

/**
 * gnome_db_entity_has_field
 * @iface: an object implementing the #GnomeDbEntity interface
 * @field: an object implementing the #GnomeDbField interface
 *
 * Tells if @field belongs to the @iface entity
 *
 * Returns: TRUE if @field belongs to the @iface entity
 */
gboolean
gnome_db_entity_has_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), FALSE);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->has_field)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->has_field) (iface, field);
	else
		return FALSE;
}

/**
 * gnome_db_entity_get_fields
 * @iface: an object implementing the #GnomeDbEntity interface
 *
 * Get a new list containing all the #GnomeDbField objects held within the object
 * implementing the #GnomeDbEntity interface.
 *
 * The returned list nodes are in the order in which the fields are within the entity.
 *
 * Returns: the new list.
 */
GSList *
gnome_db_entity_get_fields (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), NULL);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->get_fields)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->get_fields) (iface);
	
	return NULL;
}

/**
 * gnome_db_entity_get_n_fields
 * @iface: an object implementing the #GnomeDbEntity interface
 *
 * Get the number of fields in @iface
 *
 * Returns: the number of fields, or -1 if an error occured
 */
gint
gnome_db_entity_get_n_fields (GnomeDbEntity *iface)
{
	GSList *list;
	gint retval;

	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), -1);
	list = gnome_db_entity_get_fields (iface);
	retval = g_list_length (list);
	g_slist_free (list);
	return retval;
}

/**
 * gnome_db_entity_get_field_by_name
 * @iface: an object implementing the #GnomeDbEntity interface
 * @name:
 *
 * Get a #GnomeDbField using its name. The notion of "field name" is the
 * string returned by gnome_db_field_get_name() on each of the fields composing @iface.
 * However, if that definition does not return any field, then each particular
 * implementation of @iface may try to give an extra definition to the notion of 
 * "field name".
 *
 * For instance, in the case of the #GnomeDbQuery object, the  gnome_db_field_get_name() is used
 * as a first try to find a field, and if that fails, then the object tries to find
 * fields from their SQL naming.
 *
 * In the case where there can be more than one field with the same name (depending on
 * @iface's implementation), then the returned value is %NULL.
 *
 * Returns: the requested #GnomeDbField or %NULL if the field cannot be found, or if
 *          more than one field has been found.
 */
GnomeDbField *
gnome_db_entity_get_field_by_name (GnomeDbEntity *iface, const gchar *name)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), NULL);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_by_name)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_by_name) (iface, name);
	
	return NULL;
}


/**
 * gnome_db_entity_get_field_by_xml_id
 * @iface: an object implementing the #GnomeDbEntity interface
 * @xml_id:
 *
 *
 * Returns: the requested GnomeDbField
 */
GnomeDbField *
gnome_db_entity_get_field_by_xml_id (GnomeDbEntity *iface, const gchar *xml_id)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), NULL);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_by_xml_id)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_by_xml_id) (iface, xml_id);
	
	return NULL;
}

/**
 * gnome_db_entity_get_field_by_index
 * @iface: an object implementing the #GnomeDbEntity interface
 * @index:
 *
 *
 * Returns: the requested GnomeDbField or NULL if the index is out of bounds
 */
GnomeDbField *
gnome_db_entity_get_field_by_index (GnomeDbEntity *iface, gint index)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), NULL);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_by_index)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_by_index) (iface, index);
	
	return NULL;
}

/**
 * gnome_db_entity_get_field_index
 * @iface: an object implementing the #GnomeDbEntity interface
 * @field: an object implementing the #GnomeDbField interface
 *
 * Get the position of the field in the given entity. Positions start at 0.
 * @field MUST be a visible field.
 *
 * Returns: the position or -1 if the field is not in the entity or is not visible
 */
gint
gnome_db_entity_get_field_index (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), FALSE);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_index)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->get_field_index) (iface, field);	

	return FALSE;
}

/**
 * gnome_db_entity_add_field
 * @iface: an object implementing the #GnomeDbEntity interface
 * @field: an object implementing the #GnomeDbField interface to add
 *
 * Add @field to @iface's fields (at the end of the list)
 */
void
gnome_db_entity_add_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_if_fail (iface && IS_GNOME_DB_ENTITY (iface));

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->add_field)
		(GNOME_DB_ENTITY_GET_IFACE (iface)->add_field) (iface, field);
}

/**
 * gnome_db_entity_add_field_before
 * @iface: an object implementing the #GnomeDbEntity interface
 * @field: an object implementing the #GnomeDbField interface to add
 * @field_before: an object implementing the #GnomeDbField interface before which @field will be added, or %NULL
 *
 * Add @field to @iface's fields, before @field_before if it is not %NULL, 
 * or at the end if @field_before is %NULL.
 */
void
gnome_db_entity_add_field_before (GnomeDbEntity *iface, GnomeDbField *field, GnomeDbField *field_before)
{
	g_return_if_fail (iface && IS_GNOME_DB_ENTITY (iface));

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->add_field_before)
		(GNOME_DB_ENTITY_GET_IFACE (iface)->add_field_before) (iface, field, field_before);
}

/**
 * gnome_db_entity_swap_fields
 * @iface: an object implementing the #GnomeDbEntity interface
 * @field1: an object implementing the #GnomeDbField interface
 * @field2: an object implementing the #GnomeDbField interface
 */
void
gnome_db_entity_swap_fields (GnomeDbEntity *iface, GnomeDbField *field1, GnomeDbField *field2)
{
	g_return_if_fail (iface && IS_GNOME_DB_ENTITY (iface));

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->swap_fields)
		(GNOME_DB_ENTITY_GET_IFACE (iface)->swap_fields) (iface, field1, field2);
}

/**
 * gnome_db_entity_remove_field
 * @iface: an object implementing the #GnomeDbEntity interface
 * @field: an object implementing the #GnomeDbField interface to remove
 */
void
gnome_db_entity_remove_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_if_fail (iface && IS_GNOME_DB_ENTITY (iface));

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->remove_field)
		(GNOME_DB_ENTITY_GET_IFACE (iface)->remove_field) (iface, field);
}

/**
 * gnome_db_entity_is_writable
 * @iface: an object implementing the #GnomeDbEntity interface
 *
 * Tells if the real entity (the corresponding DBMS object) represented by @iface can be written to.
 *
 * Returns: TRUE if it is possible to write to @iface
 */
gboolean
gnome_db_entity_is_writable (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), FALSE);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->is_writable)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->is_writable) (iface);	

	return FALSE;
}

/**
 * gnome_db_entity_get_parameters
 * @iface: an object implementing the #GnomeDbEntity interface
 *
 * Get a list of parameters required before @iface can be used. Usualy this function is used with queries
 * to grab the required parameters before executing a query.
 *
 * This function is usually not used directly, but rather through gnome_db_entity_get_exec_dataset().
 *
 * Returns: a new list of #GnomeDbParameter objects. @iface DOES NOT hold any reference to any of the listed parameter.
 */
GSList *
gnome_db_entity_get_parameters (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), NULL);

	if (GNOME_DB_ENTITY_GET_IFACE (iface)->get_parameters)
		return (GNOME_DB_ENTITY_GET_IFACE (iface)->get_parameters) (iface);	

	return FALSE;
}



/**
 * gnome_db_entity_get_exec_dataset
 * @iface: an object implementing the #GnomeDbEntity interface
 *
 * Creates a new #GnomeDbDataSet object which contains all the parameters required to use @iface.
 * This function is similar to gnome_db_entity_get_parameters() except that it also creates a
 * #GnomeDbDataSet object to help manage all the parameters.
 *
 * Returns: a new #GnomeDbDataSet object
 */
GnomeDbDataSet *
gnome_db_entity_get_exec_dataset (GnomeDbEntity *iface)
{
	GnomeDbDataSet *dataset;
	GSList *list, *params;
	GnomeDbDict *dict;

	g_return_val_if_fail (iface && IS_GNOME_DB_ENTITY (iface), NULL);	

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (iface));
	params = gnome_db_entity_get_parameters (iface);
	dataset = GNOME_DB_DATA_SET (gnome_db_data_set_new (dict, params));
 
	/* get rid of the params list since we don't use them anymore */
	list = params;
	while (list) {
		g_object_unref (G_OBJECT (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (params);

	return dataset;
}
