/* gnome-db-data-cell-renderer-textual.c
 *
 * Copyright (C) 2000  Red Hat, Inc.,  Jonathan Blandford <jrb@redhat.com>
 * Copyright (C) 2003 - 2005 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <libgda/libgda.h>
#include "gnome-db-data-cell-renderer-textual.h"
#include "marshal.h"
#include "gnome-db-dict.h"
#include "gnome-db-server.h"
#include "gnome-db-data-entry.h"
#include "gnome-db-data-handler.h"
#include "handlers/gnome-db-entry-string.h"

static void gnome_db_data_cell_renderer_textual_init       (GnomeDbDataCellRendererTextual      *celltext);
static void gnome_db_data_cell_renderer_textual_class_init (GnomeDbDataCellRendererTextualClass *class);
static void gnome_db_data_cell_renderer_textual_dispose    (GObject                  *object);
static void gnome_db_data_cell_renderer_textual_finalize   (GObject                  *object);

static void gnome_db_data_cell_renderer_textual_get_property  (GObject                  *object,
							       guint                     param_id,
							       GValue                   *value,
							       GParamSpec               *pspec);
static void gnome_db_data_cell_renderer_textual_set_property  (GObject                  *object,
							       guint                     param_id,
							       const GValue             *value,
							       GParamSpec               *pspec);
static void gnome_db_data_cell_renderer_textual_get_size   (GtkCellRenderer          *cell,
						      GtkWidget                *widget,
						      GdkRectangle             *cell_area,
						      gint                     *x_offset,
						      gint                     *y_offset,
						      gint                     *width,
						      gint                     *height);
static void gnome_db_data_cell_renderer_textual_render     (GtkCellRenderer          *cell,
						      GdkWindow                *window,
						      GtkWidget                *widget,
						      GdkRectangle             *background_area,
						      GdkRectangle             *cell_area,
						      GdkRectangle             *expose_area,
						      GtkCellRendererState      flags);

static GtkCellEditable *gnome_db_data_cell_renderer_textual_start_editing (GtkCellRenderer      *cell,
								     GdkEvent             *event,
								     GtkWidget            *widget,
								     const gchar          *path,
								     GdkRectangle         *background_area,
								     GdkRectangle         *cell_area,
								     GtkCellRendererState  flags);

enum {
	CHANGED,
	LAST_SIGNAL
};

enum {
	PROP_0,

	PROP_VALUE,
	PROP_VALUE_ATTRIBUTES,
	PROP_TO_BE_DELETED
};

struct _GnomeDbDataCellRendererTextualPrivate
{
	GnomeDbDataHandler   *dh;
	GdaValueType          type;
	GdaValue             *value;
	gboolean              to_be_deleted;
};

typedef struct 
{
	/* text renderer */
	gulong focus_out_id;
} GnomeDbDataCellRendererTextualInfo;
#define GNOME_DB_DATA_CELL_RENDERER_TEXTUAL_INFO_KEY "gnome_db_data_cell_renderer_textual_info_key"



static GObjectClass *parent_class = NULL;
static guint text_cell_renderer_textual_signals [LAST_SIGNAL];

#define GNOME_DB_DATA_CELL_RENDERER_TEXTUAL_PATH "gnome_db_data_cell_renderer_textual_path"

GType
gnome_db_data_cell_renderer_textual_get_type (void)
{
	static GType cell_text_type = 0;

	if (!cell_text_type) {
		static const GTypeInfo cell_text_info =	{
			sizeof (GnomeDbDataCellRendererTextualClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) gnome_db_data_cell_renderer_textual_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (GnomeDbDataCellRendererTextual),
			0,              /* n_preallocs */
			(GInstanceInitFunc) gnome_db_data_cell_renderer_textual_init,
		};
		
		cell_text_type =
			g_type_register_static (GTK_TYPE_CELL_RENDERER_TEXT, "GnomeDbDataCellRendererTextual",
						&cell_text_info, 0);
	}

	return cell_text_type;
}

static void
gnome_db_data_cell_renderer_textual_init (GnomeDbDataCellRendererTextual *datacell)
{
	datacell->priv = g_new0 (GnomeDbDataCellRendererTextualPrivate, 1);
	datacell->priv->dh = NULL;
	datacell->priv->type = GDA_VALUE_TYPE_UNKNOWN;
	datacell->priv->value = NULL;

	GTK_CELL_RENDERER (datacell)->xalign = 0.0;
	GTK_CELL_RENDERER (datacell)->yalign = 0.5;
	GTK_CELL_RENDERER (datacell)->xpad = 2;
	GTK_CELL_RENDERER (datacell)->ypad = 2;
}

static void
gnome_db_data_cell_renderer_textual_class_init (GnomeDbDataCellRendererTextualClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

    	object_class->dispose = gnome_db_data_cell_renderer_textual_dispose;
	object_class->finalize = gnome_db_data_cell_renderer_textual_finalize;

	object_class->get_property = gnome_db_data_cell_renderer_textual_get_property;
	object_class->set_property = gnome_db_data_cell_renderer_textual_set_property;

	cell_class->get_size = gnome_db_data_cell_renderer_textual_get_size;
	cell_class->render = gnome_db_data_cell_renderer_textual_render;
	cell_class->start_editing = gnome_db_data_cell_renderer_textual_start_editing;
  
	g_object_class_install_property (object_class,
					 PROP_VALUE,
					 g_param_spec_pointer ("value",
							       _("Value"),
							       _("GdaValue to render"),
							       G_PARAM_READWRITE));
  
	g_object_class_install_property (object_class,
					 PROP_VALUE_ATTRIBUTES,
					 g_param_spec_uint ("value_attributes", NULL, NULL,
                                                            0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_TO_BE_DELETED,
					 g_param_spec_boolean ("to_be_deleted", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	  
	text_cell_renderer_textual_signals [CHANGED] =
		g_signal_new ("changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbDataCellRendererTextualClass, changed),
			      NULL, NULL,
			      marshal_VOID__STRING_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_POINTER);

}

static void
gnome_db_data_cell_renderer_textual_dispose (GObject *object)
{
	GnomeDbDataCellRendererTextual *datacell = GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (object);

	if (datacell->priv->dh) {
		g_object_unref (G_OBJECT (datacell->priv->dh));
		datacell->priv->dh = NULL;
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_data_cell_renderer_textual_finalize (GObject *object)
{
	GnomeDbDataCellRendererTextual *datacell = GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (object);

	if (datacell->priv) {
		g_free (datacell->priv);
		datacell->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void
gnome_db_data_cell_renderer_textual_get_property (GObject        *object,
						  guint           param_id,
						  GValue         *value,
						  GParamSpec     *pspec)
{
	GnomeDbDataCellRendererTextual *datacell = GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (object);

	switch (param_id) {
	case PROP_VALUE:
		g_value_set_pointer (value, datacell->priv->value);
		break;
	case PROP_VALUE_ATTRIBUTES:
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}




static void
gnome_db_data_cell_renderer_textual_set_property (GObject      *object,
						  guint         param_id,
						  const GValue *value,
						  GParamSpec   *pspec)
{
	GnomeDbDataCellRendererTextual *datacell = GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (object);

	switch (param_id) {
	case PROP_VALUE:
		if (datacell->priv->value) {
			gda_value_free (datacell->priv->value);
			datacell->priv->value = NULL;
		}

		if (value) {			
			GdaValue *gval = g_value_get_pointer (value);
			if (gval && !gda_value_is_null (gval)) {
				gchar *str;
				
				g_return_if_fail (gda_value_get_type (gval) == datacell->priv->type);

				datacell->priv->value = gda_value_copy (gval);
				str = gnome_db_data_handler_get_str_from_value (datacell->priv->dh, gval);
				g_object_set (G_OBJECT (object), "text", str, NULL);
				g_free (str);
			}
			else
				g_object_set (G_OBJECT (object), "text", "", NULL);
		}
		else
			g_object_set (G_OBJECT (object), "text", "", NULL);
			      
		g_object_notify (object, "value");
		break;
	case PROP_VALUE_ATTRIBUTES:
		break;
	case PROP_TO_BE_DELETED:
		datacell->priv->to_be_deleted = g_value_get_boolean (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

/**
 * gnome_db_data_cell_renderer_textual_new
 * @dh: a #GnomeDbDataHandler object
 * @type: the #GdaValueType being edited
 * 
 * Creates a new #GnomeDbDataCellRendererTextual. Adjust how text is drawn using
 * object properties. Object properties can be
 * set globally (with g_object_set()). Also, with #GtkTreeViewColumn,
 * you can bind a property to a value in a #GtkTreeModel. For example,
 * you can bind the "text" property on the cell renderer to a string
 * value in the model, thus rendering a different string in each row
 * of the #GtkTreeView
 * 
 * Return value: the new cell renderer
 **/
GtkCellRenderer *
gnome_db_data_cell_renderer_textual_new (GnomeDbDataHandler *dh, GdaValueType type)
{
	GObject *obj;
	GnomeDbDataCellRendererTextual *datacell;

	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	obj = g_object_new (GNOME_DB_TYPE_DATA_CELL_RENDERER_TEXTUAL, NULL);

	datacell = GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (obj);
	datacell->priv->dh = dh;
	g_object_ref (G_OBJECT (dh));
	datacell->priv->type = type;

	return GTK_CELL_RENDERER (obj);
}


static void
gnome_db_data_cell_renderer_textual_get_size (GtkCellRenderer *cell,
					      GtkWidget       *widget,
					      GdkRectangle    *cell_area,
					      gint            *x_offset,
					      gint            *y_offset,
					      gint            *width,
					      gint            *height)
{
	GtkCellRendererClass *text_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TEXT);
	(text_class->get_size) (cell, widget, cell_area, x_offset, y_offset, width, height);
}

static void
gnome_db_data_cell_renderer_textual_render (GtkCellRenderer      *cell,
					    GdkWindow            *window,
					    GtkWidget            *widget,
					    GdkRectangle         *background_area,
					    GdkRectangle         *cell_area,
					    GdkRectangle         *expose_area,
					    GtkCellRendererState  flags)
	
{
	GtkCellRendererClass *text_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TEXT);
	(text_class->render) (cell, window, widget, background_area, cell_area, expose_area, flags);

	if (GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (cell)->priv->to_be_deleted)
		gtk_paint_hline (widget->style,
				 window, GTK_STATE_SELECTED,
				 cell_area, 
				 widget,
				 "hline",
				 cell_area->x + cell->xpad, cell_area->x + cell_area->width - cell->xpad,
				 cell_area->y + cell_area->height / 2.);
}

static void
gnome_db_data_cell_renderer_textual_editing_done (GtkCellEditable *entry,
						  gpointer         data)
{
	const gchar *path;
	GnomeDbDataCellRendererTextualInfo *info;
	GdaValue *value;

	info = g_object_get_data (G_OBJECT (data),
				  GNOME_DB_DATA_CELL_RENDERER_TEXTUAL_INFO_KEY);

	if (info->focus_out_id > 0) {
		g_signal_handler_disconnect (entry, info->focus_out_id);
		info->focus_out_id = 0;
	}

	if (g_object_class_find_property (G_OBJECT_GET_CLASS (entry), "editing_cancelled")) {
		gboolean editing_cancelled;

		g_object_get (G_OBJECT (entry), "editing_cancelled", &editing_cancelled, NULL);
		if (editing_cancelled)
			return;
	}

	path = g_object_get_data (G_OBJECT (entry), GNOME_DB_DATA_CELL_RENDERER_TEXTUAL_PATH);
	
	value = gnome_db_data_entry_get_value (GNOME_DB_DATA_ENTRY (entry));
	g_signal_emit (data, text_cell_renderer_textual_signals[CHANGED], 0, path, value);
	gda_value_free (value);
}

static gboolean
gnome_db_data_cell_renderer_textual_focus_out_event (GtkWidget *entry,
						     GdkEvent  *event,
						     gpointer   data)
{
	gnome_db_data_cell_renderer_textual_editing_done (GTK_CELL_EDITABLE (entry), data);

	/* entry needs focus-out-event */
	return FALSE;
}

static GtkCellEditable *
gnome_db_data_cell_renderer_textual_start_editing (GtkCellRenderer      *cell,
						   GdkEvent             *event,
						   GtkWidget            *widget,
						   const gchar          *path,
						   GdkRectangle         *background_area,
						   GdkRectangle         *cell_area,
						   GtkCellRendererState  flags)
{
	GnomeDbDataCellRendererTextual *datacell;
	GtkWidget *entry;
	GnomeDbDataCellRendererTextualInfo *info;
	gboolean editable;

	datacell = GNOME_DB_DATA_CELL_RENDERER_TEXTUAL (cell);

	/* If the cell isn't editable we return NULL. */
	g_object_get (G_OBJECT (cell), "editable", &editable, NULL);
	if (!editable)
		return NULL;

	entry = GTK_WIDGET (gnome_db_data_handler_get_entry_from_value (datacell->priv->dh, 
									datacell->priv->value, datacell->priv->type));
	if (! GTK_IS_CELL_EDITABLE (entry)) {
		g_warning ("The %s class does not implement the GtkCellEditable interface, "
			   "editing disabled", G_OBJECT_TYPE_NAME (entry));
		gtk_widget_destroy (entry);
		return NULL;
	}

	g_object_set (G_OBJECT (entry), "is_cell_renderer", TRUE, "actions", FALSE, NULL);
	gnome_db_data_entry_set_value_orig (GNOME_DB_DATA_ENTRY (entry), datacell->priv->value);
	
	info = g_new0 (GnomeDbDataCellRendererTextualInfo, 1);
 	g_object_set_data_full (G_OBJECT (entry), GNOME_DB_DATA_CELL_RENDERER_TEXTUAL_PATH, g_strdup (path), g_free); 
	g_object_set_data_full (G_OBJECT (cell), GNOME_DB_DATA_CELL_RENDERER_TEXTUAL_INFO_KEY, info, g_free);
  
	g_signal_connect (entry, "editing_done",
			  G_CALLBACK (gnome_db_data_cell_renderer_textual_editing_done), datacell);
	info->focus_out_id = g_signal_connect (entry, "focus_out_event",
					       G_CALLBACK (gnome_db_data_cell_renderer_textual_focus_out_event),
					       datacell);
	gtk_widget_show (entry);
	return GTK_CELL_EDITABLE (entry);
}

