/* gnome-db-goo-field.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-goo.h"
#include "gnome-db-goo-field.h"
#include "gnome-db-goo-entity.h"

static void gnome_db_goo_field_class_init (GnomeDbGooFieldClass * class);
static void gnome_db_goo_field_init       (GnomeDbGooField * drag);
static void gnome_db_goo_field_dispose   (GObject   * object);

static void gnome_db_goo_field_set_property (GObject *object,
					  guint param_id,
					  const GValue *value,
					  GParamSpec *pspec);
static void gnome_db_goo_field_get_property (GObject *object,
					  guint param_id,
					  GValue *value,
					  GParamSpec *pspec);

static void gnome_db_goo_field_extra_event  (GnomeDbGooItem * citem, GdkEventType event_type);
enum
{
	PROP_0,
	PROP_FIELD,
};

struct _GnomeDbGooFieldPrivate
{
	GdaEntityField *field;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *field_parent_class = NULL;

GType
gnome_db_goo_field_get_type (void)
{
	static GType type = 0;

        if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGooFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_goo_field_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGooField),
			0,
			(GInstanceInitFunc) gnome_db_goo_field_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_GOO_TEXT, "GnomeDbGooField", &info, 0);
	}

	return type;
}

	

static void
gnome_db_goo_field_class_init (GnomeDbGooFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	field_parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_goo_field_dispose;
	GNOME_DB_GOO_ITEM_CLASS (class)->extra_event = gnome_db_goo_field_extra_event;

	/* Properties */
	object_class->set_property = gnome_db_goo_field_set_property;
	object_class->get_property = gnome_db_goo_field_get_property;

	g_object_class_install_property
                (object_class, PROP_FIELD,
                 g_param_spec_object ("field", NULL, NULL, GDA_TYPE_ENTITY_FIELD, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_goo_field_init (GnomeDbGooField * field)
{
	field->priv = g_new0 (GnomeDbGooFieldPrivate, 1);
	field->priv->field = NULL;
}

static void
field_destroyed_cb (GdaEntityField *field, GnomeDbGooField *cf)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (field),
					      G_CALLBACK (field_destroyed_cb), cf);
	g_object_unref (cf->priv->field);
	cf->priv->field = NULL;
	goo_canvas_item_remove (GOO_CANVAS_ITEM (cf));
}

static void
gnome_db_goo_field_dispose (GObject   * object)
{
	GnomeDbGooField *cf;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_FIELD (object));

	cf = GNOME_DB_GOO_FIELD (object);
	if (cf->priv) {
		if (cf->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (cf->priv->field),
							      G_CALLBACK (field_destroyed_cb), cf);

			g_object_unref (cf->priv->field);
			cf->priv->field = NULL;
		}
		g_free (cf->priv);
		cf->priv = NULL;
	}

	/* for the parent class */
	field_parent_class->dispose (object);
}

static void 
gnome_db_goo_field_set_property (GObject *object,
				 guint param_id,
				 const GValue *value,
				 GParamSpec *pspec)
{
	GnomeDbGooField *cf = NULL;
	GObject* propobject = NULL;
	GString *string = NULL;
	GdaDictType *dtype = NULL;
	GdaEntity *ent = NULL;

	cf = GNOME_DB_GOO_FIELD (object);

	switch (param_id) {
	case PROP_FIELD:
		propobject = g_value_get_object (value);
		g_return_if_fail (GDA_IS_ENTITY_FIELD (propobject));

		/* signals management */
		if (cf->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (cf->priv->field),
							      G_CALLBACK (field_destroyed_cb), cf);
			g_object_unref (cf->priv->field);
			cf->priv->field = NULL;
		}

		cf->priv->field = GDA_ENTITY_FIELD (propobject);
		g_object_ref (cf->priv->field);
		gda_object_connect_destroy (cf->priv->field, G_CALLBACK (field_destroyed_cb), cf);
		
		/* attributes setting */
		string = g_string_new ("");
		dtype = gda_entity_field_get_dict_type (cf->priv->field);
		if (dtype)
			g_string_append_printf (string, _("Type: %s"), gda_object_get_name (GDA_OBJECT (dtype)));
		
		g_object_set (object, "gda_object", cf->priv->field, NULL);
		ent = gda_entity_field_get_entity (cf->priv->field);
		if (GDA_IS_DICT_TABLE (ent)) {
			gboolean nullok = gda_dict_field_is_null_allowed (GDA_DICT_FIELD (cf->priv->field));
			gboolean pkpart = gda_dict_field_is_pkey_part (GDA_DICT_FIELD (cf->priv->field));
			g_object_set (object, 
				      "highlight_color", GNOME_DB_GOO_DB_TABLE_COLOR, 
				      "text_underline", !nullok,
				      "text_bold", pkpart,
				      NULL);
			if (gda_dict_field_is_pkey_alone (GDA_DICT_FIELD (cf->priv->field)))
				g_string_append (string, _("\nPrimary key"));
			else
				if (pkpart)
					g_string_append (string, _("\nPart of primary key"));
				
		}
		if (GDA_IS_QUERY (ent))
			g_object_set (object, "highlight_color", GNOME_DB_GOO_QUERY_COLOR, NULL);
		
		if (*string->str)
			g_object_set (object, "tip_text", string->str, NULL);
		else
			g_object_set (object, "tip_text", NULL, NULL);
		g_string_free (string, TRUE);
		break;
	}
}

static void 
gnome_db_goo_field_get_property    (GObject *object,
				 guint param_id,
				 GValue *value,
				 GParamSpec *pspec)
{
	GnomeDbGooField *cf;

	cf = GNOME_DB_GOO_FIELD (object);

	switch (param_id) {
	case PROP_FIELD:
		g_value_set_object (value, cf->priv->field);
		break;
	}
}

static void
gnome_db_goo_field_extra_event  (GnomeDbGooItem *citem, GdkEventType event_type)
{
	if (event_type == GDK_LEAVE_NOTIFY)
		gnome_db_goo_text_set_highlight (GNOME_DB_GOO_TEXT (citem), FALSE);
}

/**
 * gnome_db_goo_field_new
 * @parent: the parent item, or %NULL
 * @field: the represented entity's field
 * @x: the x coordinate of the text
 * @y: the y coordinate of the text
 * @...: optional pairs of property names and values, and a terminating %NULL.
 *
 * Creates a new #GnomeDbGooField object
 */
GooCanvasItem*
gnome_db_goo_field_new (GooCanvasItem *parent, GdaEntityField *field,
			gdouble x, gdouble y, ...)
{
	GooCanvasItem *item;
	GnomeDbGooField *goofield;
	const char *first_property;
	va_list var_args;

	item = g_object_new (GNOME_DB_TYPE_GOO_FIELD, NULL);
	goofield = (GnomeDbGooField*) item;

	if (parent) {
		goo_canvas_item_add_child (parent, item, -1);
		g_object_unref (item);
	}

	va_start (var_args, y);
	first_property = va_arg (var_args, char*);
	if (first_property)
		g_object_set_valist ((GObject*) item, first_property, var_args);
	va_end (var_args);

	g_object_set (G_OBJECT (item), "field", field, NULL);
	goo_canvas_item_translate (item, x, y);
	
	return item;
}


/**
 * gnome_db_goo_field_get_field
 * @cfield: a #GnomeDbGooField object
 *
 * Get the #GdaEntityField which @cfield represents
 *
 * Returns: the object implementing the #GdaEntityField interface
 */
GdaEntityField *
gnome_db_goo_field_get_field (GnomeDbGooField *cfield)
{
	g_return_val_if_fail (GNOME_DB_IS_GOO_FIELD (cfield), NULL);
	g_return_val_if_fail (cfield->priv, NULL);

	return cfield->priv->field;
}


/**
 * gnome_db_goo_field_get_parent_item
 * @cfield: a #GnomeDbGooField object
 *
 * Get the #GnomeDbGooEntity in which @cfield is
 *
 * Returns: the #GnomeDbGooEntity in which @cfield is, or %NULL
 */
GnomeDbGooEntity *
gnome_db_goo_field_get_parent_item (GnomeDbGooField *cfield)
{
	GooCanvasItem *ci;

	g_return_val_if_fail (GNOME_DB_IS_GOO_FIELD (cfield), NULL);
	ci = goo_canvas_item_get_parent (GOO_CANVAS_ITEM (cfield));

	while (ci && !GNOME_DB_IS_GOO_ENTITY (ci))
		ci = goo_canvas_item_get_parent (ci);
	return (GnomeDbGooEntity *) ci;
}
