/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "examplewindow.h" 

#include <libgdamm.h>
#include <iostream>

ExampleWindow::ExampleWindow(const Glib::RefPtr<Gnome::Gda::Dict>& dict)
{    
  m_box.set_border_width(6);
  m_box.pack_start(m_label, Gtk::PACK_SHRINK);

  // Select relevant order fields from database
  Glib::RefPtr<Gnome::Gda::Query> query = Gnome::Gda::Query::create(dict);
  query->set_sql_text("SELECT id, customer, creation_date FROM orders");
  m_model = Gnome::Gda::DataModelQuery::create(query);

  // This selects fields from the customers table to restrict the customer
  // field in the datamodel above to the values that are in the customers
  // table. We also select name to show the user the name of the customer
  // in the grid instead of just its ID (as it would appear in the orders
  // table).
  Glib::RefPtr<Gnome::Gda::Query> restr_query = Gnome::Gda::Query::create(dict);
  restr_query->set_sql_text("SELECT id, name FROM customers");
  Glib::RefPtr<Gnome::Gda::DataModel> restr = Gnome::Gda::DataModelQuery::create(restr_query);

  try
  {
    m_model->set_modification_query(
                   "UPDATE orders set "
                   "customer=##/*name:'+1' type:gint*/ "
                   "WHERE id=##/*name:'-0' type:gint*/");

    m_model->set_modification_query(
                   "DELETE FROM orders WHERE id=##/*name:'-0' type:gint*/");

    m_model->set_modification_query(
                   "INSERT INTO orders (customer) "
                   "VALUES (##/*name:'+1' type:gint*/)");
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception caught: " << err.what() << std::endl;
    exit(1);
  }

  // Lookup the Gda::Parameter object that corresponds to the customer field
  // in the order table. We find it by its field name "customer".
  m_grid.reset(new Gnome::Db::Grid(m_model));
  Glib::RefPtr<Gnome::Gda::DataModelIter> iter = m_grid->get_raw_grid()->get_current_data();
  Glib::RefPtr<Gnome::Gda::Parameter> param = iter->find_param("customer");

  // Restrict values to those in the first column of the customers table.
  // Here is also some more gda magic involved so that the grid actually
  // shows the name of the customer instead of the ID. I am not absolutely
  // sure how this works.
  param->restrict_values(restr, 0);

  m_box.pack_start(*m_grid);
  add(m_box);
  set_default_size(400, 400);
  show_all();
}
