/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "examplewindow.h" 

#include <libgdamm/query.h>
#include <libgdamm/datamodelquery.h>
#include <libgdamm/datamodel.h>
#include <iostream>

#ifdef GLIBMM_EXCEPTIONS_ENABLED
void ExampleWindow::create_model(const Glib::RefPtr<Gnome::Gda::Dict>& dict)
#else
void ExampleWindow::create_model(const Glib::RefPtr<Gnome::Gda::Dict>& dict, std::auto_ptr<Glib::Error>& error)
#endif // GLIBMM_EXCEPTIONS_ENABLED
{
  Glib::RefPtr<Gnome::Gda::Query> query = Gnome::Gda::Query::create(dict);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  query->set_sql_text("SELECT id, name, country, city FROM warehouses");
#else
  query->set_sql_text("SELECT id, name, country, city FROM warehouses", error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED
	
  // Create the demo widget:	
  m_model = Gnome::Gda::DataModelQuery::create(query);

  // TODO: Explain what these SQL queries do:
  const Glib::ustring update_query = "UPDATE warehouses set "
                                     "id=##/*name:'+0' type:gint*/, "
                                     "name=##/*name:'+1' type:gchararray*/,"
                                     "country=##/*name:'+2' type:gchararray*/, "
                                     "city=##/*name:'+3' type:gchararray*/ "
                                     "WHERE id=##/*name:'-0' type:gint*/";

  const Glib::ustring delete_query = "DELETE FROM warehouses WHERE id=##/*name:'-0' type:gint*/";

  const Glib::ustring insert_query = "INSERT INTO warehouses (id, name, country, city) "
                                     "VALUES (##/*name:'+0' type:gint*/, "
                                     "##/*name:'+1' type:gchararray*/, "
                                     "##/*name:'+2' type:gchararray*/, "
                                     "##/*name:'+3' type:gchararray*/)";

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  m_model->set_modification_query(update_query);
  m_model->set_modification_query(delete_query);
  m_model->set_modification_query(insert_query);
#else
  m_model->set_modification_query(update_query, error);
  m_model->set_modification_query(delete_query, error);
  m_model->set_modification_query(insert_query, error);
#endif // GLIBMM_EXCEPTIONS_ENABLED
}

ExampleWindow::ExampleWindow(Glib::RefPtr<Gnome::Gda::Dict>& dict) :
  m_label("The following Gnome::Db::Form show how fields can become mandatory. Try to add a\n"
          "new dataset using the '+' button. You should get an error message if you omit a field that\n"
          "has been marked mandatory with a '*'"),
  m_form(0)
{		
  m_box.set_border_width(6);
  m_box.pack_start(m_label, Gtk::PACK_SHRINK);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    create_model(dict);
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception Caught: " << err.what() << std::endl;
    exit(1);
  }
#else
  std::auto_ptr<Glib::Error> error;
  create_model(dict, error);
  if(error.get() != NULL)
  {
    std::cerr << "Exception Caught: " << error->what() << std::endl;
    exit(1);
  }
#endif // GLIBMM_EXCEPTIONS_ENABLED

  // Configure param to be mandatory
  // Note that this has to be done before the form is created
  m_form = Gtk::manage(new Gnome::Db::Form(m_model));
  Glib::RefPtr<Gnome::Gda::DataModelIter> iter = m_form->get_raw_form()->get_current_data();
  Glib::RefPtr<Gnome::Gda::Parameter> id = iter->find_param("id");
  Glib::RefPtr<Gnome::Gda::Parameter> name = iter->find_param("name");
  Glib::RefPtr<Gnome::Gda::Parameter> country = iter->find_param("country");  
  Glib::RefPtr<Gnome::Gda::Parameter> city = iter->find_param("city");
  if (name && city && country && id)
  {
    // This seems to have absolutely no effect!
    id->set_not_null(false);
    name->set_not_null(true);
    country->set_not_null(false);
    city->set_not_null(false);
  }
  else
  {
    std::cerr << "Param not found" << std::endl;
    exit(1);
  }

  m_box.pack_start(*m_form);
  add(m_box);
  show_all();
}
