
#include "demo.h"

#include <iostream>
#include <stdexcept>

// Examples
#include "readonly-grid.h"
#include "readwrite-grid.h"
#include "picture-grid.h"
#include "readonly-form.h"
#include "readwrite-form.h"
#include "picture-form.h"
#include "same-data.h"

namespace
{
  std::string find_file(const std::string& base)
  {
    if(Glib::file_test(base, Glib::FILE_TEST_EXISTS))
    {
      return base;
    }
    else
    {
      std::string filename = Glib::build_filename(LIBGNOMEDB_DATADIR, base);
      if(!Glib::file_test(filename, Glib::FILE_TEST_EXISTS))
        throw std::runtime_error("Cannot find demo data file \"" + base + "\"");

      return filename;
    }
  }
}

int main(int argc, char* argv[])
{
  Gtk::Main kit(argc, argv);
  Gnome::Gda::init("gnomedbmm-demo", "3.0", argc, argv);
  Gnome::Db::init("gnomedbmm-demo", "3.0", argc, argv);

  Glib::RefPtr<Gnome::Gda::Dict> demo_dict = Gnome::Gda::Dict::create();
  gda_dict_extend_with_functions(demo_dict->gobj());
  // Not yet wrapped.
  //demo_dict->extend_with_functions();
  std::string xml_filename = find_file("demo_dict.xml");

  Glib::RefPtr<Gnome::Gda::Client> client = Gnome::Gda::Client::create();
  std::string db_filename = find_file("demo_db.db");
  std::string dirname = Glib::path_get_dirname(db_filename);
  std::string cncstring = "DB_DIR=" + dirname + ";DB_NAME=demo_db";

  Glib::RefPtr<Gnome::Gda::Connection> cnc;

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    demo_dict->load_xml_file(xml_filename);
    cnc = client->open_connection_from_string("SQLite", cncstring, "" /* username */, "" /* password */);
  }
  catch(const Glib::Error& e)
  {
    std::cerr << e.what() << std::endl;
    return 1;
  }
#else
  std::auto_ptr<Glib::Error> error;
  
  demo_dict->load_xml_file(xml_filename, error);
  cnc = client->open_connection_from_string("SQLite", cncstring, "" /* username */, "" /* password */, Gnome::Gda::ConnectionOptions(0), error);

  if(error.get() != NULL)
  {
    std::cerr << error->what() << std::endl;
    return 1;
  }
#endif // GLIBMM_EXCEPTIONS_ENABLED

  demo_dict->set_connection(cnc);

  DemoBrowser browser(demo_dict);
  browser.add_demo<ReadOnlyGridDemo>("Grids", "Read-only grid", "The Gnome::Db::Grid widget displays data stored in a Gnome::Gda::DataModel.");
  browser.add_demo<ReadWriteGridDemo>("Grids", "Read-write grid", "The Gnome::Db::Grid widget displays data stored in a Gnome::Gda::DataModel.");
  browser.add_demo<ReadOnlyFormDemo>("Forms", "Read-only form", "The Gnome::Db::Form widget displays data stored in a Gnome::Gda::DataModel.");
  browser.add_demo<ReadWriteFormDemo>("Forms", "Read-write form", "The Gnome::Db::Form widget displays data stored in a Gnome::Gda::DataModel.");
  browser.add_demo<PictureFormDemo>("Forms", "Using the picture plugin", "A Gnome::Db::Form widget where the 'picture' plugin is used to display an image internally stored as a BLOB.");
  browser.add_demo<PictureGridDemo>("Grids", "Using the picture plugin", "A Gnome::Db::Grid widget where the 'picture' plugin is used to display an image internally stored as a BLOB.");
  browser.add_demo<SameDataDemo>("Data widgets linking", "Same data", "Shows a form and a grid synchronized to display the same data and optionally to have the same selected row.");
  browser.show();

  kit.run(browser);
  return 0;
}
