/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-fontmap.c: fontmap implementation
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Lauris Kaplinski <lauris@ximian.com>
 *    Chema Celorio <chema@celorio.com>
 *    Tambet Ingo <tambet@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc.
 *
 */

/* TODO: Recycle font entries, if they are identical for different maps */

#include "config.h"
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <fontconfig/fontconfig.h>
#include "gnome-fontmap.h"

static GPFontMap *gp_fontmap_load (void);

static void gp_fontmap_ref (GPFontMap * map);
static void gp_fontmap_unref (GPFontMap * map);
static void gp_family_entry_ref (GPFamilyEntry * entry);
static void gp_family_entry_unref (GPFamilyEntry * entry);

static gint gp_fe_sortname (gconstpointer a, gconstpointer b);
static gint gp_fe_sortspecies (gconstpointer a, gconstpointer b);
static gint gp_familyentry_sortname (gconstpointer a, gconstpointer b);

/* Fontlist -> FontMap */
static GHashTable * fontlist2map = NULL;
/* Familylist -> FontMap */
static GHashTable * familylist2map = NULL;

GPFontMap *
gp_fontmap_get (void)
{
	static GPFontMap *map = NULL;
	static time_t lastaccess = 0;

	if (map) {
		/* If > 1 sec is passed from last query, check timestamps */
		if ((time (NULL) > lastaccess)) {
			/* Any directory is changed, so force rereading of map */
			gp_fontmap_release (map);
			map = NULL;
		}
	}

	if (!map) {
		map = gp_fontmap_load ();
	}

	/* Save acess time */
	lastaccess = time (NULL);

	map->refcount++;

	return map;
}

void
gp_fontmap_release (GPFontMap * map)
{
	gp_fontmap_unref (map);
}

static void
gp_fontmap_ref (GPFontMap * map)
{
	g_return_if_fail (map != NULL);

	map->refcount++;
}

static void
gp_fontmap_unref (GPFontMap * map)
{
	g_return_if_fail (map != NULL);

	if (--map->refcount < 1) {
		if (map->familydict) g_hash_table_destroy (map->familydict);
		if (map->fontdict) g_hash_table_destroy (map->fontdict);
		if (map->familylist) {
			g_hash_table_remove (familylist2map, map->familylist);
			g_list_free (map->familylist);
		}
		if (map->fontlist) {
			g_hash_table_remove (fontlist2map, map->fontlist);
			g_list_free (map->fontlist);
		}
		while (map->families) {
			gp_family_entry_unref ((GPFamilyEntry *) map->families->data);
			map->families = g_slist_remove (map->families, map->families->data);
		}
		while (map->fonts) {
			gp_font_entry_unref ((GPFontEntry *) map->fonts->data);
			map->fonts = g_slist_remove (map->fonts, map->fonts->data);
		}
		while (map->defaults) {
			GSList *l;
			l = map->defaults->data;
			map->defaults = g_slist_remove (map->defaults, l);
			while (l) {
				g_free (l->data);
				l = g_slist_remove (l, l->data);
			}
		}
		if (map->defaultsdict) g_hash_table_destroy (map->defaultsdict);
		g_free (map);
	}
}

static void
gp_family_entry_ref (GPFamilyEntry * entry)
{
	entry->refcount++;
}

static void
gp_family_entry_unref (GPFamilyEntry * entry)
{
	if (--entry->refcount < 1) {
		if (entry->name) g_free (entry->name);
		if (entry->fonts) g_slist_free (entry->fonts);
		g_free (entry);
	}
}

static GnomeFontWeight
convert_fc_weight (gint i)
{
	if (i < FC_WEIGHT_LIGHT)
		return GNOME_FONT_LIGHTEST;
	if (i < (FC_WEIGHT_LIGHT + FC_WEIGHT_MEDIUM) / 2)
		return GNOME_FONT_LIGHT;
	if (i < (FC_WEIGHT_MEDIUM + FC_WEIGHT_DEMIBOLD) / 2)
		return GNOME_FONT_REGULAR;
	if (i < (FC_WEIGHT_DEMIBOLD + FC_WEIGHT_BOLD) / 2)
		return GNOME_FONT_DEMI;
	if (i < (FC_WEIGHT_BOLD + FC_WEIGHT_BLACK) / 2)
		return GNOME_FONT_BOLD;

	return GNOME_FONT_HEAVIEST;
}

static GPFontEntry *
fcpattern_to_gp_font_entry (FcPattern *font)
{
	GPFontEntryType font_type = GP_FONT_ENTRY_UNKNOWN;
	GPFontEntryTT *tt;
	GPFontEntryT1 *t1;
	GPFontEntry *e;
	FcResult result;
	FcChar8 *fc_family, *fc_style, *fc_file;
	int italic_angle, weight;
	gchar *c;

	/* Get Family, Style & Filename */
	result = FcPatternGetString (font, FC_FAMILY, 0, &fc_family);
	if (result != FcResultMatch || fc_family == NULL) {
		g_warning ("Can't create GPFontEntry because we could not get a FC_FAMILY for the font\n");
		return NULL;
	}
	result = FcPatternGetString (font, FC_STYLE, 0, &fc_style);
	if (result != FcResultMatch || fc_style == NULL) {
		g_warning ("Can't create GPFontEntry for %s because we could not get its FC_STYLE\n",
			   fc_family);
		return NULL;
	}
	result = FcPatternGetString (font, FC_FILE, 0, &fc_file);
	if (result != FcResultMatch || fc_file == NULL) {
		g_warning ("Can't create GPFontEntry for %s-%s because we could not get its FC_FILE\n",
			   fc_family, fc_style);
		return NULL;
	}
	italic_angle = 0;
	FcPatternGetInteger (font, FC_SLANT, 0, &italic_angle);

	/* Determine what type of font we've got */
	c = fc_file + strlen (fc_file) - 4;
	if (strcasecmp (c, ".pfb") == 0)
		font_type = GP_FONT_ENTRY_TYPE1;
	else if (strcasecmp (c, ".pfa") == 0)
		font_type = GP_FONT_ENTRY_TYPE1;
	else if (strcasecmp (c, ".ttf") == 0)
		font_type = GP_FONT_ENTRY_TRUETYPE;
	else {
		return NULL;
		g_warning ("Can't create GPFontEntry for %s-%s because the extension could not be recognized (%s)\n",
			   fc_family, fc_style, c);
		return NULL;
	}

	switch (font_type) {
	case GP_FONT_ENTRY_TRUETYPE:
		tt = g_new0 (GPFontEntryTT, 1);
		e = (GPFontEntry *) tt;
		e->type = GP_FONT_ENTRY_TRUETYPE;
		tt->ttf.name = g_strdup (fc_file);
		tt->facenum = 0; /* FIMXE: Is this correct? (Chema) */
		break;
	case GP_FONT_ENTRY_TYPE1:
		t1 = g_new0 (GPFontEntryT1, 1);
		e = (GPFontEntry *) t1;
		e->type = GP_FONT_ENTRY_TYPE1;
		t1->pfb.name = g_strdup (fc_file);
		t1->afm.name = NULL;
		break;
	default:
		g_warning ("Can't create GPFontEntry, invalid font_type)\n");
		return NULL;
	}

	e->refcount = 1;
	e->face = NULL;
	e->speciesname = g_strdup (fc_style);
	e->weight = g_strdup (fc_style);
	e->familyname = g_strdup (fc_family);
	e->version = g_strdup ("1.0"); /* FIMXE: Is this correct? (Chema) */
	e->name    = g_strdup_printf ("%s %s", e->familyname, e->weight);
	e->italic_angle = italic_angle;
	
	result = FcPatternGetInteger (font, FC_WEIGHT, 0, &weight);
	if (result == FcResultMatch)
		e->Weight = convert_fc_weight (weight);
	else
		e->Weight = GNOME_FONT_REGULAR;

	e->psname  = g_strdup_printf ("%s-%s", e->familyname, e->weight);
	c = e->psname;
	while (*c != '\0') {
		if (*c == ' ')
			*c = '_';
		c++;
	}

	return e;
}

static GPFontEntry *
fcpattern_to_gp_font_entry_alias (FcPattern *font, const gchar *family, const gchar *style)
{
	GPFontEntry *e;
	gchar *c;

	e = fcpattern_to_gp_font_entry (font);
	if (e == NULL)
		return NULL;

	/* Override font name and style */
	if (e->name)
		g_free (e->name);
	if (e->psname)
		g_free (e->psname);

	e->name = g_strdup_printf ("%s %s", family, style);
	e->psname = g_strdup_printf ("%s-%s", family, style);

	c = e->psname;
	while (*c != '\0') {
		if (*c == ' ')
			*c = '_';
		c++;
	}

	return e;
}

/* Add well-known aliases Sans Sans-Serif and Monospace */
static void
gp_fontmap_add_aliases (GPFontMap *map)
{
	FcPattern *match_pattern, *result_pattern;
	GPFontEntry *e;
	gchar *aliases[] = { "Sans", "Serif", "Monospace", NULL };
	gchar *styles[] = { "Regular", "Bold", "Italic", "Bold Italic", NULL };
	FcResult res;
	gint i, j;

	/* If no fonts could be loaded, we can't create aliases */
	if (map == NULL)
		return;

	for (i = 0; aliases[i]; i++) {
		for (j = 0; styles[j]; j++) {
			match_pattern = FcPatternBuild (NULL,
							FC_FAMILY, FcTypeString, aliases[i],
							FC_STYLE,  FcTypeString, styles[j],
							NULL);

			FcConfigSubstitute (NULL, match_pattern, FcMatchPattern);
			FcDefaultSubstitute (match_pattern);

			g_assert (match_pattern);

			result_pattern = FcFontMatch (NULL, match_pattern, &res);
			if (result_pattern) {
				e = fcpattern_to_gp_font_entry_alias (result_pattern, aliases[i], styles[j]);
				if (e) {
					g_hash_table_insert (map->fontdict, e->name, e);
					map->num_fonts++;
					map->fonts = g_slist_prepend (map->fonts, e);
				}

				FcPatternDestroy (result_pattern);
			}

			FcPatternDestroy (match_pattern);
		}
	}
}

/**
 * gp_fontmap_check_duplicates:
 * @map: 
 * 
 * This function makes sure that the e->psname is unique for each font
 * it appends a dash + 3 digits if it find a duplicate. It grows exponentially which
 * might be a problem for system with a lot of fonts.
 **/
static void
gp_fontmap_check_duplicates (GPFontMap *map)
{
	GSList *list_1, *list_2;
	static gint i;

	list_1 = map->fonts;
	while (list_1) {
		GPFontEntry *e1 = list_1->data;
		list_2 = map->fonts;
		while (list_2) {
			GPFontEntry *e2 = list_2->data;
			if (list_2 != list_1) {
				if (strcmp (e1->psname, e2->psname) == 0) {
					gchar *new;
					new = g_strdup_printf ("%s-%.3d", e2->psname, i++);
					g_free (e2->psname);
					e2->psname = new;
				}
			}
			list_2 = list_2->next;
		}
		list_1 = list_1->next;
	}
}

static void
gp_fontmap_load_fontconfig (GPFontMap *map)
{
	FcFontSet *fontset;
	FcPattern *font;
	GPFontEntry *e;
	int i;

	fontset = FcConfigGetFonts (NULL, FcSetSystem);
	if (fontset == NULL) {
		return;
	}

	for (i = 0; i < fontset->nfont; i++) {
		font = fontset->fonts[i];

		e = fcpattern_to_gp_font_entry (font);
		if (e) {
			g_hash_table_insert (map->fontdict, e->name, e);
			map->num_fonts++;
			map->fonts = g_slist_prepend (map->fonts, e);
		}
	}

	gp_fontmap_add_aliases (map);
	gp_fontmap_check_duplicates (map);
}

static GPFontMap *
gp_fontmap_load (void)
{
	GPFontMap *map;
	GSList * l;

	map = g_new (GPFontMap, 1);
	/* We always hold private ref to fontmap, this is released if directories change */
	map->refcount = 1;
	map->num_fonts = 0;
	/* Clear timestamps */
	map->mtime_static = 0;
	map->mtime_dynamic = 0;
	map->mtime_user = 0;
	map->fontdict = g_hash_table_new (g_str_hash, g_str_equal);
	map->familydict = g_hash_table_new (g_str_hash, g_str_equal);
	map->fonts = NULL;
	map->families = NULL;
	map->fontlist = NULL;
	map->familylist = NULL;
	map->defaults = NULL;
	map->defaultsdict = g_hash_table_new (g_str_hash, g_str_equal);

	gp_fontmap_load_fontconfig (map);

	/* Sort fonts alphabetically */
	map->fonts = g_slist_sort (map->fonts, gp_fe_sortname);

	/* Sort fonts into familia */
	for (l = map->fonts; l != NULL; l = l->next) {
		GPFontEntry * e;
		GPFamilyEntry * f;
		e = (GPFontEntry *) l->data;
		f = g_hash_table_lookup (map->familydict, e->familyname);
		if (!f) {
			f = g_new0 (GPFamilyEntry, 1);
			gp_family_entry_ref (f);
			f->name = g_strdup (e->familyname);
			f->fonts = g_slist_prepend (f->fonts, e);
			g_hash_table_insert (map->familydict, f->name, f);
			map->families = g_slist_prepend (map->families, f);
		} else {
			f->fonts = g_slist_prepend (f->fonts, e);
		}
	}

	/* Sort familia alphabetically */
	map->families = g_slist_sort (map->families, gp_familyentry_sortname);

	/* Sort fonts inside familia */
	for (l = map->families; l != NULL; l = l->next) {
		GPFamilyEntry * f;
		f = (GPFamilyEntry *) l->data;
		f->fonts = g_slist_sort (f->fonts, gp_fe_sortspecies);
	}

	/* Compose defaultsdict */
	map->defaults = g_slist_reverse (map->defaults);
	while (map->defaults) {
		GSList *l;
		guchar *locales, *fontname;
		GPFontEntry *entry;
		l = map->defaults->data;
		map->defaults = g_slist_remove (map->defaults, l);
		locales = l->data;
		fontname = l->next->data;
		g_slist_free (l);
		entry = g_hash_table_lookup (map->fontdict, fontname);
		if (!entry) {
			GPFamilyEntry *fe;
			/* Try family */
			fe = g_hash_table_lookup (map->familydict, fontname);
			if (fe && fe->fonts) {
				entry = (GPFontEntry *) fe->fonts->data;
				l = fe->fonts;
				while (l) {
					GPFontEntry *e;
					e = (GPFontEntry *) l->data;
					if (!strcasecmp (e->speciesname, "regular") ||
					    !strcasecmp (e->speciesname, "roman") ||
					    !strcasecmp (e->speciesname, "normal")) {
						entry = e;
						break;
					}
					l = l->next;
				}
			}
		}
		if (entry) {
			guchar *l;
			l = locales;
			while (l) {
				guchar *e;
				e = strchr (l, ',');
				if (e) {
					*e = '\0';
					e += 1;
				}
				if (!g_hash_table_lookup (map->defaultsdict, l)) {
					GQuark q;
					q = g_quark_from_string (l);
					g_hash_table_insert (map->defaultsdict, (gpointer) g_quark_to_string (q), entry);
				}
				l = e;
			}
		}
		g_free (locales);
		g_free (fontname);
	}

	return map;
}

/*
 * Font Entry stuff
 *
 * If face is created, it has to reference entry
 */

void
gp_font_entry_ref (GPFontEntry * entry)
{
	g_return_if_fail (entry != NULL);
	/* refcount can be 1 or 2 at moment */
	g_return_if_fail (entry->refcount > 0);
	g_return_if_fail (entry->refcount < 2);

	entry->refcount++;
}

void
gp_font_entry_unref (GPFontEntry * entry)
{
	g_return_if_fail (entry != NULL);
	/* refcount can be 1 or 2 at moment */
	g_return_if_fail (entry->refcount > 0);
	g_return_if_fail (entry->refcount < 3);

	if (--entry->refcount < 1) {
		GPFontEntryT1 *t1;
		GPFontEntryT1Alias *t1a;
		GPFontEntrySpecial *s;
		GPFontEntryTT *tt;

		g_return_if_fail (entry->face == NULL);

		if (entry->name)
			g_free (entry->name);
		if (entry->version)
			g_free (entry->version);
		if (entry->familyname)
			g_free (entry->familyname);
		if (entry->speciesname)
			g_free (entry->speciesname);
		if (entry->psname)
			g_free (entry->psname);
		if (entry->weight)
			g_free (entry->weight);

		switch (entry->type) {
		case GP_FONT_ENTRY_TYPE1_ALIAS:
			t1a = (GPFontEntryT1Alias *) entry;
			if (t1a->alias)
				g_free (t1a->alias);
		case GP_FONT_ENTRY_TYPE1:
			t1 = (GPFontEntryT1 *) entry;
			if (t1->afm.name)
				g_free (t1->afm.name);
			if (t1->pfb.name)
				g_free (t1->pfb.name);
			break;
		case GP_FONT_ENTRY_ALIAS:
			break;
		case GP_FONT_ENTRY_TRUETYPE:
			tt = (GPFontEntryTT *) entry;
			if (tt->ttf.name)
				g_free (tt->ttf.name);
			break;
		case GP_FONT_ENTRY_SPECIAL:
			s = (GPFontEntrySpecial *) entry;
			if (s->file.name)
				g_free (s->file.name);
			while (s->additional) {
				g_free (s->additional->data);
				s->additional = g_slist_remove (s->additional, s->additional->data);
			}
			break;
		default:
			g_assert_not_reached ();
			break;
		}
		g_free (entry);
	}
}

/*
 * Font list stuff
 *
 * We use Hack'O'Hacks here:
 * Getting list saves list->fontmap mapping and refs fontmap
 * Freeing list releases mapping and frees fontmap
 */

GList *
gnome_font_list ()
{
	GPFontMap * map;
	GSList * l;

	map = gp_fontmap_get ();

	if (!map->fontlist) {
		for (l = map->fonts; l != NULL; l = l->next) {
			GPFontEntry * e;
			e = (GPFontEntry *) l->data;
			map->fontlist = g_list_prepend (map->fontlist, e->name);
		}
		map->fontlist = g_list_reverse (map->fontlist);
		if (!fontlist2map)
			fontlist2map = g_hash_table_new (NULL, NULL);
		g_hash_table_insert (fontlist2map, map->fontlist, map);
	}

	return map->fontlist;
}

void
gnome_font_list_free (GList * fontlist)
{
	GPFontMap * map;

	g_return_if_fail (fontlist != NULL);

	map = g_hash_table_lookup (fontlist2map, fontlist);
	g_return_if_fail (map != NULL);

	gp_fontmap_unref (map);
}

GList *
gnome_font_family_list ()
{
	GPFontMap * map;
	GSList * l;

	map = gp_fontmap_get ();

	if (!map->familylist) {
		for (l = map->families; l != NULL; l = l->next) {
			GPFamilyEntry * f;
			f = (GPFamilyEntry *) l->data;
			map->familylist = g_list_prepend (map->familylist, f->name);
		}
		map->familylist = g_list_reverse (map->familylist);
		if (!familylist2map) familylist2map = g_hash_table_new (NULL, NULL);
		g_hash_table_insert (familylist2map, map->familylist, map);
	}

	gp_fontmap_ref (map);

	gp_fontmap_release (map);

	return map->familylist;
}

void
gnome_font_family_list_free (GList * fontlist)
{
	GPFontMap * map;

	g_return_if_fail (fontlist != NULL);

	map = g_hash_table_lookup (familylist2map, fontlist);
	g_return_if_fail (map != NULL);

	gp_fontmap_unref (map);
}

static gint
gp_fe_sortname (gconstpointer a, gconstpointer b)
{
	return strcasecmp (((GPFontEntry *) a)->name, ((GPFontEntry *) b)->name);
}

static gint
gp_fe_sortspecies (gconstpointer a, gconstpointer b)
{
	if (((GPFontEntry *)a)->speciesname == NULL)
		return -1;
	if (((GPFontEntry *)b)->speciesname == NULL)
		return 1;
	
	return strcasecmp (((GPFontEntry *) a)->speciesname, ((GPFontEntry *) b)->speciesname);
}

static gint
gp_familyentry_sortname (gconstpointer a, gconstpointer b)
{
	return strcasecmp (((GPFamilyEntry *) a)->name, ((GPFamilyEntry *) b)->name);
}

GnomeFontWeight
gp_fontmap_lookup_weight (const gchar * weight)
{
	static GHashTable * weights = NULL;
	GnomeFontWeight wcode;

	if (!weights) {
		weights = g_hash_table_new (g_str_hash, g_str_equal);

		g_hash_table_insert (weights, "Extra Light", GINT_TO_POINTER (GNOME_FONT_EXTRA_LIGHT));
		g_hash_table_insert (weights, "Extralight", GINT_TO_POINTER (GNOME_FONT_EXTRA_LIGHT));

		g_hash_table_insert (weights, "Thin", GINT_TO_POINTER (GNOME_FONT_THIN));

		g_hash_table_insert (weights, "Light", GINT_TO_POINTER (GNOME_FONT_LIGHT));

		g_hash_table_insert (weights, "Book", GINT_TO_POINTER (GNOME_FONT_BOOK));
		g_hash_table_insert (weights, "Roman", GINT_TO_POINTER (GNOME_FONT_BOOK));
		g_hash_table_insert (weights, "Regular", GINT_TO_POINTER (GNOME_FONT_BOOK));

		g_hash_table_insert (weights, "Medium", GINT_TO_POINTER (GNOME_FONT_MEDIUM));

		g_hash_table_insert (weights, "Semi", GINT_TO_POINTER (GNOME_FONT_SEMI));
		g_hash_table_insert (weights, "Semibold", GINT_TO_POINTER (GNOME_FONT_SEMI));
		g_hash_table_insert (weights, "Demi", GINT_TO_POINTER (GNOME_FONT_SEMI));
		g_hash_table_insert (weights, "Demibold", GINT_TO_POINTER (GNOME_FONT_SEMI));

		g_hash_table_insert (weights, "Bold", GINT_TO_POINTER (GNOME_FONT_BOLD));

		g_hash_table_insert (weights, "Heavy", GINT_TO_POINTER (GNOME_FONT_HEAVY));
 
		g_hash_table_insert (weights, "Extra", GINT_TO_POINTER (GNOME_FONT_EXTRABOLD));
		g_hash_table_insert (weights, "Extra Bold", GINT_TO_POINTER (GNOME_FONT_EXTRABOLD));

		g_hash_table_insert (weights, "Black", GINT_TO_POINTER (GNOME_FONT_BLACK));

		g_hash_table_insert (weights, "Extra Black", GINT_TO_POINTER (GNOME_FONT_EXTRABLACK));
		g_hash_table_insert (weights, "Extrablack", GINT_TO_POINTER (GNOME_FONT_EXTRABLACK));
		g_hash_table_insert (weights, "Ultra Bold", GINT_TO_POINTER (GNOME_FONT_EXTRABLACK));
	};

	wcode = GPOINTER_TO_INT (g_hash_table_lookup (weights, weight));

	return wcode;
}
