/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-print-job.h: A print job interface
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Michael Zucchi <notzed@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian Inc.
 */

#ifndef __GNOME_PRINT_JOB_H__
#define __GNOME_PRINT_JOB_H__

#include <glib.h>

G_BEGIN_DECLS

/*
 * GnomePrintJob is frontend to GnomePrintContext creation and
 * printing. Unlike contexts themselves, job cares about layout,
 * number of copies and collate values. So it is usually safe bet to
 * use it always.
 *
 * Notice, that page number(s) used by job are already adjusted
 * to layout. I.e. if you print 2 pages per sheet, and invoke
 * beginpage/showpage 4 times, gnome_print_job_get_pages gives
 * you value 2.
 *
 * Also, if you wonder, what page dimensions you should be using in
 * your application, use gnome_print_job_get_page_size_from_config.
 *
 */

#define GNOME_TYPE_PRINT_JOB (gnome_print_job_get_type ())
#define GNOME_PRINT_JOB(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), GNOME_TYPE_PRINT_JOB, GnomePrintJob))
#define GNOME_PRINT_JOB_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), GNOME_TYPE_PRINT_JOB, GnomePrintJobClass))
#define GNOME_IS_PRINT_JOB(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), GNOME_TYPE_PRINT_JOB))
#define GNOME_IS_PRINT_JOB_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), GNOME_TYPE_PRINT_JOB))
#define GNOME_PRINT_JOB_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), GNOME_TYPE_PRINT_JOB, GnomePrintJobClass))

typedef struct _GnomePrintJob      GnomePrintJob;
typedef struct _GnomePrintJobClass GnomePrintJobClass;

#include <libgnomeprint/gnome-print.h>

GType gnome_print_job_get_type (void);

/* Creates new GnomePrintJob */

GnomePrintJob *gnome_print_job_new (GnomePrintConfig *config);

/* Get REFERENCED configuration */
GnomePrintConfig *gnome_print_job_get_config (GnomePrintJob *job);

/* Get REFERENCED context for actual printing */
GnomePrintContext *gnome_print_job_get_context (GnomePrintJob *job);

/* Signal GnomePrintJob, that application has finished job */
/* You should not close context given you by job, but use this instead */
gint gnome_print_job_close (GnomePrintJob *job);

/* Send printout to printer, as defined by config */
gint gnome_print_job_print (GnomePrintJob *job);
/* Renders printout to specified context (with layout, ignoring copies) */
gint gnome_print_job_render (GnomePrintJob *job, GnomePrintContext *ctx);
/* Renders specified page (with layout) */
gint gnome_print_job_render_page (GnomePrintJob *job, GnomePrintContext *ctx, gint page, gboolean pageops);

/* Get number of real output pages (0 if job is not closed) */
gint gnome_print_job_get_pages (GnomePrintJob *job);

/* Get imaging area size available to application (TRUE sucess, FALSE error) */
/* This takes config for argument, as you may not want to create job yet */
/* Sizes are given in PS points (GNOME_PRINT_PS_UNIT) */
gboolean gnome_print_job_get_page_size_from_config (GnomePrintConfig *config, gdouble *width, gdouble *height);


/* This are convenience functions for common stuff that could be accomplished with gnome_print_config
 * calls but will require deeper knowledge when GnomePrintConfig trees get complex */
gint gnome_print_job_print_to_file (GnomePrintJob *job, gchar *output);
 
G_END_DECLS

#endif /* __GNOME_PRINT_JOB_H__ */
