#define __GNOME_PRINT_MASTER_PREVIEW_C__

/*
 *  Copyright (C) 2000-2001 Ximian Inc.
 *
 *  Authors:
 *    Michael Zucchi <notzed@helixcode.com>
 *    Miguel de Icaza <miguel@gnu.org>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  A system print preview window.  Based on print-preview.c
 *  from gnumeric.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if 0
#include <string.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkaccelgroup.h>
#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklabel.h>
#include <libgnomeprint/gnome-print-meta.h>

#include "gnome-print-i18n.h"

#include "gnome-print-master-preview.h"
#if 0
#include "gnome-print-preview-icons.h"
#endif

#define TOOLBAR_BUTTON_BASE 5
#define MOVE_INDEX 5

typedef enum {
	GPMP_MODE_MOVE,
	GPMP_MODE_MOVE_DRAGGING,
	GPMP_MODE_ZOOM_IN,
	GPMP_MODE_ZOOM_OUT
} GPMPPreviewMode;

struct _GnomePrintMasterPreview {
	GtkWindow window;

	gpointer priv;
};

struct _GnomePrintMasterPreviewClass {
	GtkWindowClass parent_class;
};

typedef struct _GPMPPrivate GPMPPrivate;

struct _GPMPPrivate {
	GnomePrintMaster *master;

	GnomePrintContext *preview;

	GtkWidget *page_entry;
	GtkWidget *scrolled_window;
	GtkWidget *last;
	GnomeCanvas *canvas;

	gint current_page;
	gint pagecount;

	/* Used for dragging the sheet */
	gint drag_anchor_x, drag_anchor_y;
	gint drag_ofs_x, drag_ofs_y;

	GPMPPreviewMode mode;

	gboolean landscape;
	gdouble width, height;
#if 0



	int first_page_rendered;

	GnomeUIInfo *toolbar;


#endif
};

/*
 * Padding in points around the simulated page
 */

#define PAGE_PAD 4

static gint
render_page (GnomePrintMasterPreview *pmp, gint page)
{
	GPMPPrivate *priv;
	GnomePrintConfig *config;
	ArtDRect region;
	gdouble transform[6];
	GnomePrintContext *meta;
	const guchar *data;
	gint len;

	priv = (GPMPPrivate *) pmp->priv;

	if (priv->preview) {
		g_object_unref (G_OBJECT (priv->preview));
		priv->preview = NULL;
	}

	/* Set page transformation */
	if (priv->landscape) {
		transform[0] =  0.0;
		transform[1] = -1.0;
		transform[2] = -1.0;
		transform[3] =  0.0;
		transform[4] =  priv->height;
		transform[5] =  priv->width;
	} else {
		transform[0] =  1.0;
		transform[1] =  0.0;
		transform[2] =  0.0;
		transform[3] = -1.0;
		transform[4] =  0.0;
		transform[5] =  priv->height;
	}

	/* Reset scrolling region always */
	region.x0 = region.y0 = 0.0 - PAGE_PAD;
	region.x1 = priv->width + PAGE_PAD;
	region.y1 = priv->height + PAGE_PAD;

	config = gnome_print_master_get_config (priv->master);
	priv->preview = gnome_print_preview_new_full (config, priv->canvas, transform, &region);
	gnome_print_config_unref (config);

#if 0
	meta = gnome_print_master_get_context (priv->master);
	data = gnome_print_meta_get_buffer (GNOME_PRINT_META (meta));
	len = gnome_print_meta_get_length (GNOME_PRINT_META (meta));
	g_object_unref (G_OBJECT (meta));

	gnome_print_meta_render_data_page (priv->preview, data, len, page, TRUE);
#else
	gnome_print_master_render_page (priv->master, priv->preview, page, TRUE);
#endif

	return GNOME_PRINT_OK;
}

static gint
goto_page (GnomePrintMasterPreview *pmp, gint page)
{
	GPMPPrivate *priv;
	guchar c[32];

	priv = (GPMPPrivate *) pmp->priv;

	g_snprintf (c, 32, "%d", page + 1);
	gtk_entry_set_text (GTK_ENTRY (priv->page_entry), c);
	
	if (page != priv->current_page) {
		priv->current_page = page;
		return render_page (pmp, page);
	}

	return GNOME_PRINT_OK;
}

static gint
change_page_cmd (GtkEntry *entry, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	const gchar *text;
	gint page;

	priv = (GPMPPrivate *) pmp->priv;

	text = gtk_entry_get_text (entry);

	page = CLAMP (atoi (text), 1, priv->pagecount) - 1;

	return goto_page (pmp, page);
}

static void
do_zoom (GnomePrintMasterPreview *pmp, gint factor)
{
	GPMPPrivate *priv;
	gdouble zoom;

	priv = (GPMPPrivate *) pmp->priv;
	
	if (factor > 0) {
		zoom = priv->canvas->pixels_per_unit * 1.25;
	} else if (factor < 0) {
		zoom = priv->canvas->pixels_per_unit / 1.25;
	} else {
		zoom = 1.0;
	}

	zoom = CLAMP (zoom, 0.1, 10.0);

	gnome_canvas_set_pixels_per_unit (priv->canvas, zoom);
}

/* Button press handler for the print preview canvas */

static gint
preview_canvas_button_press (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;
	gint retval;

	pmp = GNOME_PRINT_MASTER_PREVIEW (data);
	priv = (GPMPPrivate *) pmp->priv;

	retval = FALSE;

	switch (priv->mode) {
	case GPMP_MODE_MOVE:
		if (event->button != 1) break;
		priv->mode = GPMP_MODE_MOVE_DRAGGING;
		priv->drag_anchor_x = event->x;
		priv->drag_anchor_y = event->y;
		gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget), &priv->drag_ofs_x, &priv->drag_ofs_y);

		gdk_pointer_grab (widget->window, FALSE,
				  (GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON_RELEASE_MASK),
				  NULL, NULL, /*cursor_get (GNUMERIC_CURSOR_HAND_CLOSED),*/
				  event->time);

		retval = TRUE;
		break;
	default:
		break;
	}

	return retval;
}

/* Drags the print preview canvas to the specified position */

static void
drag_to (GnomePrintMasterPreview *pmp, gint x, gint y)
{
	GPMPPrivate *priv;
	gint dx, dy;

	priv = (GPMPPrivate *) pmp->priv;

	dx = priv->drag_anchor_x - x;
	dy = priv->drag_anchor_y - y;

	/*
	 * Right now this will suck for diagonal movement.  GtkLayout does not
	 * have a way to scroll itself diagonally, i.e. you have to change the
	 * vertical and horizontal adjustments independently, leading to ugly
	 * visual results.  The canvas freezes and thaws the layout in case of
	 * diagonal movement, forcing it to repaint everything.
	 *
	 * This will be resolved in the canvas.
	 */

	gnome_canvas_scroll_to (priv->canvas, priv->drag_ofs_x + dx, priv->drag_ofs_y + dy);
}

/* Button release handler for the print preview canvas */

static gint
preview_canvas_button_release (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;
	gint retval;

	pmp = GNOME_PRINT_MASTER_PREVIEW (data);
	priv = (GPMPPrivate *) pmp->priv;

	switch (priv->mode) {
	case GPMP_MODE_MOVE_DRAGGING:
		if (event->button != 1) break;
		drag_to (pmp, event->x, event->y);
		priv->mode = GPMP_MODE_MOVE;
		gdk_pointer_ungrab (event->time);
		retval = TRUE;
		break;
	default:
		break;
	}
	return retval;
}

/* Motion notify handler for the print preview canvas */

static gint
preview_canvas_motion (GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;
	gint retval;
	gint x, y;
	GdkModifierType mods;

	pmp = GNOME_PRINT_MASTER_PREVIEW (data);
	priv = (GPMPPrivate *) pmp->priv;

	retval = FALSE;

	switch (priv->mode) {
	case GPMP_MODE_MOVE_DRAGGING:
		if (event->is_hint)
			gdk_window_get_pointer (widget->window, &x, &y, &mods);
		else {
			x = event->x;
			y = event->y;
		}
		drag_to (pmp, x, y);
		retval = TRUE;
		break;
	default:
		break;
	}

	return retval;
}


static void
preview_close_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	g_object_unref (G_OBJECT (pmp));
}

static void
preview_file_print_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;
	
	gnome_print_master_print (priv->master);

	/* fixme: should we clean ourselves up now? */
}

static void
preview_first_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	goto_page (pmp, 0);
}

static void
preview_next_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;

	if (priv->current_page < (priv->pagecount - 1)) {
		goto_page (pmp, priv->current_page + 1);
	}
}

static void
preview_prev_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;

	if (priv->current_page > 0) {
		goto_page (pmp, priv->current_page - 1);
	}
}

static void
preview_last_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;
	
	goto_page (pmp, priv->pagecount - 1);
}

static void
preview_zoom_in_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	do_zoom (pmp, 1);
}

static void
preview_zoom_out_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	do_zoom (pmp, -1);
}

static void
preview_zoom_fit_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	gdouble zoomx, zoomy;
	gint width, height;

	priv = (GPMPPrivate *) pmp->priv;

	width = GTK_WIDGET (priv->canvas)->allocation.width;
	height = GTK_WIDGET (priv->canvas)->allocation.height;
	
	zoomx = width / (priv->width + 5.0 + PAGE_PAD);
	zoomy = height / (priv->height + 5.0 + PAGE_PAD);
	
	gnome_canvas_set_pixels_per_unit (priv->canvas, MIN (zoomx, zoomy));
}

static void
preview_zoom_fit_wide_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	gdouble zoom;
	gint width;

	priv = (GPMPPrivate *) pmp->priv;

	width = GTK_WIDGET (priv->canvas)->allocation.width;

	zoom = width / (priv->width + 5.0 + PAGE_PAD);
	gnome_canvas_set_pixels_per_unit (priv->canvas, zoom);
}

static void
preview_zoom_fit_tall_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	gdouble zoom;
	gint height;

	priv = (GPMPPrivate *) pmp->priv;

	height = GTK_WIDGET (priv->canvas)->allocation.height;

	zoom = height / (priv->height + 5.0 + PAGE_PAD);
	gnome_canvas_set_pixels_per_unit (priv->canvas, zoom);
}

static gint
preview_canvas_key (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;
	gint x,y;
	gint height, width;
	gint domove = 0;

	pmp = (GnomePrintMasterPreview *) data;

	priv = (GPMPPrivate *) pmp->priv;

	gnome_canvas_get_scroll_offsets (priv->canvas, &x, &y);
	height = GTK_WIDGET (priv->canvas)->allocation.height;
	width = GTK_WIDGET (priv->canvas)->allocation.width;

	switch (event->keyval) {
	case '1':
		preview_zoom_fit_cmd (0, pmp);
		break;
	case '2':		/* as good a key as any? */
		preview_zoom_fit_wide_cmd (0, pmp);
		break;
	case '3':
		preview_zoom_fit_tall_cmd (0, pmp);
		break;
	case '+':
	case '=':
	case GDK_KP_Add:
		do_zoom (pmp, 1);
		break;
	case '-':
	case '_':
	case GDK_KP_Subtract:
		do_zoom (pmp, -1);
		break;
	case GDK_KP_Right:
	case GDK_Right:
		if (event->state & GDK_SHIFT_MASK)
			x+=width;
		else
			x+=10;
		domove=1;
		break;
	case GDK_KP_Left:
	case GDK_Left:
		if (event->state & GDK_SHIFT_MASK)
			x-=width;
		else
			x-=10;
		domove=1;
		break;
	case GDK_KP_Up:
	case GDK_Up:
		if (event->state & GDK_SHIFT_MASK)
			goto page_up;
		y-=10;
		domove=1;
		break;
	case GDK_KP_Down:
	case GDK_Down:
		if (event->state & GDK_SHIFT_MASK)
			goto page_down;
		y+=10;
		domove=1;
		break;
	case GDK_KP_Page_Up:
	case GDK_Page_Up:
	case GDK_Delete:
	case GDK_KP_Delete:
	case GDK_BackSpace:
	page_up:
		if (y<=0) {
			if (priv->current_page > 0) {
				goto_page (pmp, priv->current_page - 1);
				y = GTK_LAYOUT (priv->canvas)->height - height;
			}
		} else {
			y -= height;
		}
		domove=1;
		break;
	case GDK_KP_Page_Down:
	case GDK_Page_Down:
	case ' ':
	page_down:
		if (y >= GTK_LAYOUT (priv->canvas)->height - height) {
			if (priv->current_page < priv->pagecount - 1) {
				goto_page (pmp, priv->current_page + 1);
				y = 0;
			}
		} else {
			y += height;
		}
		domove=1;
		break;
	case GDK_KP_Home:
	case GDK_Home:
		goto_page (pmp, 0);
		y=0;
		domove=1;
		break;
	case GDK_KP_End:
	case GDK_End:
		goto_page (pmp, priv->pagecount-1);
		y=0;
		domove=1;
		break;
	case GDK_Escape:
		g_object_unref (G_OBJECT (pmp));
		return TRUE;
	default:
		return FALSE;
	}

	if (domove)
		gnome_canvas_scroll_to (priv->canvas, x, y);

	gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
	return TRUE;
}

static void
create_preview_canvas (GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	GtkWidget *box, *status;
	GnomeCanvasItem *item;
	GnomePrintConfig *config;

	priv = (GPMPPrivate *) pmp->priv;

	priv->scrolled_window = gtk_scrolled_window_new (NULL, NULL);

	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	priv->canvas = GNOME_CANVAS (gnome_canvas_new_aa ());
	gtk_widget_pop_colormap ();

	gnome_canvas_set_pixels_per_unit (priv->canvas, 1.0);

	gtk_signal_connect (GTK_OBJECT (priv->canvas), "button_press_event",
			    GTK_SIGNAL_FUNC (preview_canvas_button_press), pmp);
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "button_release_event",
			    GTK_SIGNAL_FUNC (preview_canvas_button_release), pmp);
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "motion_notify_event",
			    GTK_SIGNAL_FUNC (preview_canvas_motion), pmp);
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "key_press_event",
			    GTK_SIGNAL_FUNC (preview_canvas_key), pmp);

	gtk_container_add (GTK_CONTAINER (priv->scrolled_window), GTK_WIDGET (priv->canvas));

	config = gnome_print_master_get_config (priv->master);
	priv->preview = gnome_print_preview_new (config, priv->canvas);
	gnome_print_config_unref (config);

	/*
	 * Now add some padding above and below and put a simulated
	 * page on the background
	 */

	item = gnome_canvas_item_new (gnome_canvas_root (priv->canvas),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", 0.0, "y1", 0.0, "x2", (gdouble) priv->width, "y2", (gdouble) priv->height,
				      "fill_color", "white",
				      "outline_color", "black",
				      "width_pixels", 1, NULL);
	gnome_canvas_item_lower_to_bottom (item);
	item = gnome_canvas_item_new (gnome_canvas_root (priv->canvas),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", 3.0, "y1", 3.0, "x2", (gdouble) priv->width + 3, "y2", (gdouble) priv->height + 3,
				      "fill_color", "black", NULL);
	gnome_canvas_item_lower_to_bottom (item);
	gnome_canvas_set_scroll_region (priv->canvas, 0 - PAGE_PAD, 0 - PAGE_PAD, priv->width + PAGE_PAD, priv->height + PAGE_PAD);

	box = gtk_vbox_new (FALSE, 0);
	status = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (status), gtk_label_new (_ ("Page: ")), FALSE, FALSE, 0);
	priv->page_entry = gtk_entry_new ();
	gtk_widget_set_usize (priv->page_entry, 40, 0);
	gtk_signal_connect (GTK_OBJECT (priv->page_entry), "activate", GTK_SIGNAL_FUNC (change_page_cmd), pmp);
	gtk_box_pack_start (GTK_BOX (status), priv->page_entry, FALSE, FALSE, 0);
	priv->last = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (status), priv->last, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (box), status, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (box), priv->scrolled_window, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (pmp), box);
	gtk_widget_show_all (box);

	gtk_widget_grab_focus (GTK_WIDGET (priv->canvas));
}


#if 0
static GnomeUIInfo preview_file_menu [] = {
	GNOMEUIINFO_MENU_PRINT_ITEM (preview_file_print_cmd, NULL),
	GNOMEUIINFO_MENU_CLOSE_ITEM (preview_close_cmd, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo preview_view_menu [] = {
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("_First page"), N_ ("Shows the first page"),
		preview_first_page_cmd, GNOME_STOCK_PIXMAP_FIRST),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("_Previous page"), N_ ("Shows the previous page"),
		preview_prev_page_cmd, GNOME_STOCK_PIXMAP_BACK),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("_Next page"), N_ ("Shows the next page"),
		preview_next_page_cmd, GNOME_STOCK_PIXMAP_FORWARD),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("_Last page"), N_ ("Shows the last page"),
		preview_last_page_cmd, GNOME_STOCK_PIXMAP_LAST),

	GNOMEUIINFO_SEPARATOR,

	{ GNOME_APP_UI_ITEM, N_ ("Zoom _in"), N_ ("Zooms in"), preview_zoom_in_cmd },
	{ GNOME_APP_UI_ITEM, N_ ("Zoom _out"), N_ ("Zooms out"), preview_zoom_out_cmd },

	GNOMEUIINFO_END
};

static GnomeUIInfo top_menu [] = {
	GNOMEUIINFO_MENU_FILE_TREE (preview_file_menu),
	{ GNOME_APP_UI_SUBTREE, N_ ("_View"), NULL, preview_view_menu },
	GNOMEUIINFO_END
};

static GnomeUIInfo toolbar [] = {
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("Print"), N_ ("Prints the current file"),
		preview_file_print_cmd, GNOME_STOCK_PIXMAP_PRINT),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("First"), N_ ("Shows the first page"),
		preview_first_page_cmd, GNOME_STOCK_PIXMAP_FIRST),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("Back"), N_ ("Shows the previous page"),
		preview_prev_page_cmd, GNOME_STOCK_PIXMAP_BACK),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("Next"), N_ ("Shows the next page"),
		preview_next_page_cmd, GNOME_STOCK_PIXMAP_FORWARD),
	GNOMEUIINFO_ITEM_STOCK (
		N_ ("Last"), N_ ("Shows the last page"),
		preview_last_page_cmd, GNOME_STOCK_PIXMAP_LAST),

	GNOMEUIINFO_ITEM (N_ ("Zoom in"), N_ ("Zooms the page in"), preview_zoom_in_cmd, stock_zoom_in_xpm),
	GNOMEUIINFO_ITEM (N_ ("Zoom out"), N_ ("Zooms the page out"), preview_zoom_out_cmd, stock_zoom_out_xpm),
	GNOMEUIINFO_ITEM (N_ ("Fit"), N_ ("Zooms to fit the whole page"), preview_zoom_fit_cmd, stock_zoom_fit_xpm),
	GNOMEUIINFO_ITEM (N_ ("Fit Wide"), N_ ("Zooms to fit the width of the page"), preview_zoom_fit_wide_cmd, stock_zoom_fit_wide_xpm),
	GNOMEUIINFO_ITEM (N_ ("Fit Tall"), N_ ("Zooms to fit the height of the page"), preview_zoom_fit_tall_cmd, stock_zoom_fit_tall_xpm),
	GNOMEUIINFO_END
};
#endif

static void
create_toplevel (GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	GnomePrintConfig *config;
	gint width, height;
	const GnomePrintUnit *unit;
	gdouble val;

	priv = (GPMPPrivate *) pmp->priv;

	config = gnome_print_master_get_config (priv->master);

	val = 21.0 * 72.0 / 2.54;
	if (gnome_print_config_get_length (config, GNOME_PRINT_KEY_PAPER_WIDTH, &val, &unit)) {
		g_print ("len %g %s\n", val, unit->abbr);
		gnome_print_convert_distance (&val, unit, GNOME_PRINT_PS_UNIT);
		g_print ("abs %g\n", val);
	}
	width = val;
	val = 29.7 * 72.0 / 2.54;
	if (gnome_print_config_get_length (config, GNOME_PRINT_KEY_PAPER_HEIGHT, &val, &unit)) {
		gnome_print_convert_distance (&val, unit, GNOME_PRINT_PS_UNIT);
	}
	height = val;

	if (priv->landscape) {
		priv->height = width;
		priv->width = height;
	} else {
		priv->width = width;
		priv->height = height;
	}

	width  = priv->width + PAGE_PAD * 3;
	height = priv->height + PAGE_PAD * 3;

	if (width > gdk_screen_width () - 40) {
		width = gdk_screen_width () - 40;
	}

	if (height > gdk_screen_height () - 100) {
		height = gdk_screen_height () - 100;
	}

	gtk_widget_set_usize (GTK_WIDGET (pmp), width, height);
	gtk_window_set_policy (GTK_WINDOW (pmp), TRUE, TRUE, FALSE);

#if 0
	char *old_msg_locale;
	const GnomePaper *paper;

	old_msg_locale = g_strdup (textdomain (NULL));
	textdomain (PACKAGE);
	gnome_app_create_menus_with_data (GNOME_APP (pmp), top_menu, pmp);

	pp->toolbar = g_malloc (sizeof (toolbar));
	memcpy (pp->toolbar, toolbar, sizeof (toolbar));

	gnome_app_create_toolbar_with_data (GNOME_APP (pmp), pp->toolbar, pmp);
	textdomain (old_msg_locale);
	g_free (old_msg_locale);
#endif
}

/*
  The GnomePrintMasterPreview object
*/

static void gnome_print_master_preview_class_init (GnomePrintMasterPreviewClass *klass);
static void gnome_print_master_preview_init (GnomePrintMasterPreview *pmp);

static GtkWindowClass *parent_class;

GtkType
gnome_print_master_preview_get_type (void)
{
	static GtkType print_master_preview_type = 0;
	
	if (!print_master_preview_type) {
		GtkTypeInfo print_master_preview_info = {
			"GnomePrintMasterPreview",
			sizeof (GnomePrintMasterPreview),
			sizeof (GnomePrintMasterPreviewClass),
			(GtkClassInitFunc) gnome_print_master_preview_class_init,
			(GtkObjectInitFunc) gnome_print_master_preview_init,
			NULL, NULL, NULL
		};
		
		print_master_preview_type = gtk_type_unique (GTK_TYPE_WINDOW, &print_master_preview_info);
	}
	
	return print_master_preview_type;
}

static void
gnome_print_master_preview_finalize (GObject *object)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;

	pmp = GNOME_PRINT_MASTER_PREVIEW (object);
	priv = pmp->priv;

	if (priv && priv->preview != NULL) {
		g_object_unref (G_OBJECT (priv->preview));
		priv->preview = NULL;
	}

	if (priv && priv->master != NULL) {
		g_object_unref (G_OBJECT (priv->master));
		priv->master = NULL;
	}

	if (priv) {
		g_free (priv);
		pmp->priv = NULL;
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_print_master_preview_class_init (GnomePrintMasterPreviewClass *klass)
{
	GObjectClass *object_class;
	
	object_class = (GObjectClass *) klass;

	parent_class = gtk_type_class (GTK_TYPE_WINDOW);

	object_class->finalize = gnome_print_master_preview_finalize;
}

static void
gnome_print_master_preview_init (GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = g_new0 (GPMPPrivate, 1);
	priv->current_page = -1;

	pmp->priv = priv;
}

GtkWidget *
gnome_print_master_preview_new (GnomePrintMaster *gpm, const guchar *title)
{
	GnomePrintMasterPreview *gpmp;
	GPMPPrivate *priv;
	GnomePrintConfig *config;
	gint orientation;
	gchar *text;

	g_return_val_if_fail (gpm != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), NULL);

	gpmp = gtk_type_new (GNOME_TYPE_PRINT_MASTER_PREVIEW);

	if (!title) title = _("Gnome Print Preview");
	gtk_window_set_title (GTK_WINDOW (gpmp), title);

	priv = (GPMPPrivate *) gpmp->priv;

	config = gnome_print_master_get_config (gpm);

	orientation = 0;
	gnome_print_config_get_int (config, GNOME_PRINT_KEY_ORIENTATION, &orientation);
	priv->landscape = ((orientation + 44) / 90) & 0x1;

	config = gnome_print_config_unref (config);

	priv->master = gpm;
	g_object_ref (G_OBJECT (gpm));

	create_toplevel (gpmp);
	create_preview_canvas (gpmp);

	/* this zooms to fit, once we know how big the window actually is */
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "realize",
			    GTK_SIGNAL_FUNC (preview_zoom_fit_cmd), gpmp);

	priv->pagecount = gnome_print_master_get_pages (gpm);

	goto_page (gpmp, 0);
	
	text = g_strdup_printf ("/%d", priv->pagecount);
	gtk_label_set_text (GTK_LABEL (priv->last), text);
	g_free (text);

	return (GtkWidget *) gpmp;
}

#if 0
/**
 * gnome_print_master_preview_new:
 * @gpm: A GnomePrintMaster which has been printed to and completed.
 * @title: Window title for the preview window.
 * 
 * Create a new preview window widget.  The preview has inbuilt
 * buttons for zooming and panning, and menu's to print.
 * 
 * Return value: A newly created GnomePrintMasterPreview widget.
 **/
GnomePrintMasterPreview *
gnome_print_master_preview_new (GnomePrintMaster *gpm, const guchar *title)
{
	return gnome_print_master_preview_new_with_orientation (gpm, title, FALSE);
}

/**
 * gnome_print_master_preview_new_with_orientation:
 * @gpm: A GnomePrintMaster which has been printed to and completed.
 * @title: Window title for the preview window.
 * @landscape: TRUE to rotate paper.
 * 
 * Create a new preview window widget.  The preview has inbuilt
 * buttons for zooming and panning, and menu's to print.
 * 
 * Return value: A newly created GnomePrintMasterPreview widget.
 **/
GnomePrintMasterPreview *
gnome_print_master_preview_new_with_orientation (GnomePrintMaster *gpm,
						 const char *title,
						 gboolean landscape)
{
	GnomePrintMasterPreview *pmp;
	Private *pp;
	char text[4 * sizeof (int)];

	pmp = GNOME_PRINT_MASTER_PREVIEW (gtk_type_new (gnome_print_master_preview_get_type ()));

	gnome_app_construct (GNOME_APP (pmp), "preview_window", title);

	pp = pmp->priv;
	pp->landscape = landscape;

	pp->master = gpm;
	gtk_object_ref (GTK_OBJECT (gpm));

	create_toplevel (pmp);
	create_preview_canvas (pmp);

	/* this zooms to fit, once we know how big the window actually is */
	gtk_signal_connect (GTK_OBJECT (pp->canvas), "realize",
			   GTK_SIGNAL_FUNC (preview_zoom_fit_cmd), pmp);

	pp->pagecount = gnome_print_master_get_pages (gpm);

	goto_page (pmp, 0);
	
	g_snprintf (text, 4 * sizeof (int), "/%d", pp->pagecount);
	gtk_label_set_text (GTK_LABEL (pp->last), text);

	return pmp;
}
#endif

