'''
Defines the base class for all L{AccessEngine} events (L{AEEvent}s).

@author: Peter Parente
@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

from LSRInterfaces import *
import Constants

class AccessEngineEvent(object):
  '''
  Most base class for all L{AccessEngine} events. Supports the concept of a 
  priority which may be used to prioritize event execution.

  @ivar priority: Priority of this event
  @type priority: integer
  @ivar focused: Was the source of the event focused at the time of event
    creation?
  @type focused: boolean
  @ivar por: Point of regard associated with this event
  @type por: L{POR}
  '''
  def __init__(self, priority=Constants.NORMAL, focused=False):
    '''
    Stores the event priority and whether the event source was focused or not.

    @param priority: Priority of this event, defaults to L{Constants.NORMAL}
    @type priority: integer    
    @param focused: Was the source of the event focused at the time of event 
      creation?
    @type focused: boolean
    '''
    self.priority = priority
    self.por = None
    if focused:
      self.layer = Constants.FOCUS_LAYER
    else:
      self.layer = Constants.BACKGROUND_LAYER
    
  def __str__(self):
    '''
    Returns the name of this event.
    
    @return: Class name of this AEEvent
    @rtype: string
    '''
    if self.layer == Constants.FOCUS_LAYER:
      level = 'focus'
    elif self.layer == Constants.TIER_LAYER:
      level = 'tier'
    else:
      level = 'background'
    return '%s (%s)' % (self.__class__.__name__, level)
    
  def execute(self, **kwargs):
    '''
    Executes the logic that will handle this event. The L{EventManager} can 
    provide whatever arguments are necessary for an L{AEEvent} subclass to 
    execute as keyword arguments. It is up to the subclass to unpack and use 
    the arguments it needs by name.
    
    See L{EventManager.EventManager._executeEvent} for the arguments provided.
    
    @param kwargs: Named references to various managers
    @type kwargs: dictionary
    @return: Was this class able to execute successfully? Always True here.
    @rtype: boolean
    '''
    return True
    
  def getPriority(self):
    '''
    @return: Current priority value
    @rtype: integer
    '''
    return self.priority
    
  def setPriority(self, priority):
    '''
    @param priority: New priority value
    @type priority: integer
    '''
    self.priority = priority
    
  def setLayer(self, layer):
    '''
    @param layer: Layer of this event (focus, view, tier, background)
    @type layer: integer
    '''
    self.layer = layer
    
  def getLayer(self):
    '''
    Gets the layer for this event. If the L{POR} is marked as imcomplete, builds
    a complete L{POR} and then determines if the proper L{POR} is in a different
    layer than the one originally computed.
    
    @return: Gets the layer for this event
    @rtype: integer
    '''
    if self.por is not None and self.por.incomplete:
      if IAccessibleInfo(self.getPOR()).hasAccState('focused'):
        self.layer = Constants.FOCUS_LAYER
    return self.layer

  def getDataForTask(self):
    '''
    Fetches data out of an L{AccessEngineEvent} for use by a L{Task}. This 
    method must be implemented by any L{AEEvent} that will be handled by 
    L{TierManager.TierManager.manageEvent} or
    L{TierManager.TierManager.manageGesture}.
    
    @return: Dictionary of parameters to be passed to a L{Task}.
    @rtype: dictionary
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
  
  def getPOR(self):
    '''
    @return: Point of regard associated with this event
    @rtype: L{POR}
    '''
    if self.por is not None and self.por.incomplete:
      self.por = IPORFactory(self.por).create()
    return self.por
  
  def getAppID(self):
    '''
    @return: Unique application ID identifying the top most container for the
      source of this event (i.e. the application)
    @rtype: opaque object
    '''
    if self.por is None:
      return None
    else:
      return IAccessibleInfo(self.por).getAccAppID()