'''
Defines an abstract base class for all input devices that serve double duty as 
sources of input for LSR and for the rest of the operating system.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Base, Constants

class SystemInput(Base.AEInput):
  '''
  Abstract base class for input devices that are used by both LSR and the OS.
  Provides an interface for registering modifier actions that can be used to
  indicate the start of commands intended for LSR. Also provides an interface
  for registering filtered L{GestureList}s that indicate which combination of
  L{Gesture}s should not be passed to other applications. The definition of what
  constitues a valid L{Gesture} and L{GestureList} is left to a subclass. The
  decision as to when registered L{Gesture}s are filtered or not is determined
  by a mode managed in this class via L{getFilterMode} and L{setFilterMode}.
  
  This class is abstract as some of the methods inherited from 
  L{Base.AEInput} are not overridden and raise NotImplementedError.
  
  @ivar modifiers: Dictionary of action codes that are considered modifiers
  @type modifiers: dictionary
  @ivar filter_mode: Mode that determines which L{Gesture}s are filtered, one of 
      L{Constants.FILTER_NONE}, L{Constants.FILTER_HANDLED}, or 
      L{Constants.FILTER_ALL}
  @type filter_mode: integer
  '''
  def __init__(self):
    '''
    Initializes the filters to an empty list, the modifiers to an empty
    dictionary, and the filtering mode to L{Constants.FILTER_NONE}.
    '''
    Base.AEInput.__init__(self)
    self.modifiers = {}
    self.filter_mode = Constants.FILTER_NONE

  def getFilterMode(self):
    '''
    Gets the current filter mode.
    
    @return: Current filter mode, one of L{Constants.FILTER_ALL}, 
        L{Constants.FILTER_HANDLED}, or L{Constants.FILTER_NONE}
    @rtype: integer
    '''
    return self.filter_mode
    
  def setFilterMode(self, mode):
    '''
    Stores the current filter mode.
    
    @param mode: Current filter mode, one of L{Constants.FILTER_ALL}, 
        L{Constants.FILTER_HANDLED}, or L{Constants.FILTER_NONE}
    @type mode: integer
    '''
    self.filter_mode = mode
    
  def addModifier(self, code):
    ''' 
    Adds an action code to the modifiers dictionary to identify it as a modifier
    for other actions.
    
    @param code: Action code to register as a modifier
    @type code: integer
    '''
    self.modifiers[code] = None
    
  def removeModifier(self, code):
    '''
    Removes an action code from the modifiers dictionary.
    
    @param code: Action code to unregister as a modifier
    @type code: integer
    @raise KeyError: When the action code to remove is not registered as a 
      modifier
    '''
    del self.modifiers[code]
    
  def clearModifiers(self):
    '''
    Removes all modifiers by destroying the modifiers dictionary and recreating
    it.
    '''
    self.modifiers = {}
    
  def resetState(self):
    '''
    Virtual method. Resets the state of the input device.

    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError