'''
Defines default L{AccAdapt.Adapter}s for the L{LSRInterfaces.IAccessibleInfo} 
interface on L{pyLinAcc.Accessibility.Accessible} objects.

@author: Peter Parente
@author: Pete Brunet
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import pyLinAcc
from POR import POR
from AccAdapt import PORAdapter
from LSRInterfaces import *
from pyLinAcc import Constants, Interfaces

class DefaultAccInfoAdapter(PORAdapter):
  '''
  Adapts all AT-SPI accessibles to the L{IAccessibleNav} interface. No
  condition for adaption is given implying that this adapter is used as a 
  default by L{AccAdapt} when no better adapter is available. Expects the 
  subject to be a L{POR}.
  
  @cvar VISIBLE_STATES: States that an accessible must have to be considered 
    visible
  @type VISIBLE_STATES: tuple
  @cvar TRIVIAL_STATES: States that an accessible must not have to be considered
    non-trivial
  @type TRIVIAL_STATES: tuple
  @cvar TRIVIAL_ROLES: Roles that an accessible must not have to be considered
    non-trivial
  @type TRIVIAL_ROLES: tuple
  '''
  provides = [IAccessibleInfo]
  
  VISIBLE_STATES = (Constants.STATE_VISIBLE,
                    Constants.STATE_SHOWING)
  TRIVIAL_STATES = (Constants.STATE_DEFUNCT,
                    Constants.STATE_INVALID,
                    Constants.STATE_INDETERMINATE)
  TRIVIAL_ROLES = (Constants.ROLE_INVALID,
                   Constants.ROLE_GLASS_PANE,
                   Constants.ROLE_FILLER,
                   Constants.ROLE_VIEWPORT,
                   Constants.ROLE_MENU_BAR,
                   Constants.ROLE_TOOL_BAR,
                   Constants.ROLE_SCROLL_BAR,
                   Constants.ROLE_PANEL,
                   Constants.ROLE_SPLIT_PANE,
                   Constants.ROLE_SCROLL_PANE,
                   Constants.ROLE_PAGE_TAB_LIST,
                   Constants.ROLE_SEPARATOR)
  
  @pyLinAcc.errorToLookupError
  def getAccTextAttr(self, name):
    '''
    Gets the value of a given attribute name.
    
    @param name: Name of the attribute (eg. fgcolor, bgcolor)
    @type name: string
    @return: Value of the accessible attribute
    @rtype: string
    @raise LookupError: When the accessible object is dead
    @raise NotImplementedError: When this accessible does not support the text
      interface
    @todo: DBC: implement optimal and non-deprecated getAttributeValue()
    '''
    attrs = self.getAccAllTextAttrs()
    for s in attrs:
      n, v = s.split(':')
      if n == name:
        return v
    return ''
  
  @pyLinAcc.errorToLookupError
  def getAccAllTextAttrs(self):
    '''
    Gets all the attributes of a given accessible.
    
    @return: Value of the accessible attribute
    @rtype: list of name:value strings
    @raise LookupError: When the accessible object is dead
    @raise NotImplementedError: When this accessible does not support the text
      interface
    '''
    text = Interfaces.IText(self.accessible)  
    # use item_offset of zero if None
    if self.item_offset is None:
      self.item_offset = 0
    # get the attributes at the given item_offset (start of a line) plus 
    # char_offset
    val, start, end = text.getAttributes(self.item_offset + self.char_offset)
    if val is '':
      return ''
    attrs = val.split('; ')
    return attrs
  
  @pyLinAcc.errorToLookupError
  def getAccSelection(self):
    '''  
    Gets a list L{POR}s referring to the selected child acessibles within the
    subject accessible.
    
    @return: Points of regard to selected accessibles
    @rtype: list of L{POR}s
    @raise LookupError: When the subject accessible is dead
    '''
    try:
      sel = Interfaces.ISelection(self.accessible)
    except NotImplementedError:
      return []
    # return PORs with unique accessibles for each selected child
    return [POR(sel.getSelectedChild(i)) for i in range(sel.nSelectedChildren)]
  
  @pyLinAcc.errorToLookupError
  def getAccChildCount(self):
    '''
    Gets the number of child accessibles for the object implementing this 
    interface.
    
    @return: Number of accessible children
    @rtype: integer
    @raise LookupError: When the accessible object is dead
    '''
    return self.accessible.childCount
  
  @pyLinAcc.errorToLookupError
  def getAccAppName(self):
    '''
    Gets the name of the application application to which the subject accessible 
    belongs.
    
    @return: Name of the subject's application
    @rtype: string
    @raise LookupError: When the accessible or any parent accessible up to the
      application is dead
    '''
    app = self.accessible.getApplication()
    if app is None:
      raise LookupError
    return unicode(app.name, 'utf-8')

  @pyLinAcc.errorToLookupError
  def getAccAppID(self):
    '''
    Gets a unique ID identifying the application to which the subject accessible 
    belongs.
    
    Currently, the name and ID of the application are used. See 
    U{http://www.linuxbase.org/~gk4/a11y/idl/interfaceAccessibility_1_1Application.html#o2} 
    for information about the application ID.
    
    @return: Application's name and runtime ID
    @rtype: 2-tuple of string, integer
    @raise LookupError: When the accessible or any parent accessible up to the
      application is dead
    '''
    app = self.accessible.getApplication()
    try:
      return (app.name, app.id)
    except AttributeError:
      return None
  
  @pyLinAcc.errorToLookupError
  def getAccRoleName(self):
    '''
    Gets the accessible role name of the subject. Ignores the given item offset 
    because the default adapter assumes there are no items.
    
    @return: Localized role name
    @rtype: string
    @raise LookupError: When the accessible object is dead
    '''
    return unicode(self.accessible.getLocalizedRoleName(), 'utf-8')
    
  @pyLinAcc.errorToLookupError
  def getAccName(self):
    '''
    Gets the accessible name of the subject. Ignores the given item offset 
    because the default adapter assumes there are no items.
    
    @return: Accessible name of requested object
    @rtype: string
    @raise LookupError: When the subject accessible is dead
    '''
    return unicode(self.accessible.name, 'utf-8')
        
  @pyLinAcc.errorToLookupError
  def getAccDescription(self):
    ''' 
    Gets the accessible description of the subject. Ignores the given item 
    offset because the default adapter assumes there are no items.

    @return: Accessible description of requested object
    @rtype: string
    @raise LookupError: When the subject accessible is dead
    '''
    return unicode(self.accessible.description, 'utf-8')
  
  @pyLinAcc.errorToLookupError
  def getAccItemText(self):
    '''
    Gets the accessible name of this object. Ignores the given item offset 
    because the default adapter assumes there are no items.

    @return: Accessible name of requested object
    @rtype: string
    @raise LookupError: When the accessible object is dead
    '''
    return self.getAccName()
  
  @pyLinAcc.errorToLookupError
  def getAccTextAttr(self, name):
    '''
    Gets the value of a given attribute name. Not implemented in the base 
    adapter as it requires the Text interface to work.
    
    @param name: Name of the attribute (eg. fgcolor, bgcolor)
    @type name: string
    @return: Value of the accessible attribute
    @rtype: string
    @raise NotImplementedError: Always since the default adapter does not 
      require the Text interface
    '''
    raise NotImplementedError
  
  @pyLinAcc.errorToLookupError
  def getAccRelations(self, kind):
    '''
    Gets a list of accessibles related to the subject accessible in the manner
    given by kind.
    
    @param kind: Name specifying the kind of relations to retrieve
    @type kind: string
    @return: List of L{POR}s related to subject accessible in the manner 
      given by kind
    @rtype: list of L{POR}
    @raise LookupError: When the accessible object is dead
    '''
    # map the kind parameter to a pyLinAcc constant, or stay with the string
    kind = pyLinAcc.stringToConst('relation', kind)
    # get the entire relation set
    rs = self.accessible.getRelationSet()
    pors = []
    for r in rs:
      if r.getRelationType() == kind:
        # found one or more labels
        for i in range(r.getNTargets()):
          # get the target PORs, using IPORFactory to ensure PORs to items are
          # built properly
          por = IPORFactory(r.getTarget(i)).create()
          pors.append(por)
        break
    return pors
  
  @pyLinAcc.errorToLookupError
  def getAccStates(self):
    '''
    Gets a list of names of states indicating the current state of the given 
    accessible. Ignores the given item offset because the default adapter 
    assumes there are no items.

    @return: Names of all states
    @rtype: list
    @raise LookupError: When the accessible object is dead
    '''
    expandable = False
    states = []
    for val in self.accessible.getState().getStates():
      name = pyLinAcc.stateToString(val)
      states.append(name)
      if name == 'expandable':
        expandable = True
    # collapsed isn't always set properly, so add it if expandable but not
    # expanded
    if expandable:
      try:
        states.index('expanded')
      except ValueError:
        s = set(states)
        s.add('collapsed')
        states = list(s)
    # give some indicator of not being enabled
    try:
      states.index('enabled')
    except ValueError:
      states.append('disabled')
    return states
  
  @pyLinAcc.errorToLookupError
  def isAccTopLevelWindow(self):
    '''
    Gets whether or not the subject accessible is a top-level container in the
    accessible hierarchy.
    
    @return: Is the subject a top-level container or not?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    '''
    try:
      app = Interfaces.IApplication(self.accessible.parent)
      return True
    except NotImplementedError:
      return False
   
  @pyLinAcc.errorToLookupError
  def isAccTrivial(self):
    '''    
    Gets if the accessible should be considered trivial. This implementation
    considers an accessible trivial if the state is one of L{TRIVIAL_STATES}
    or the role is one of L{TRIVIAL_ROLES}.
    
    @return: Does the accessible consider itself trivial?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    '''
    # if it has a name, it's not trivial
    if self.accessible.name:
      return False
    # ensure no bad roles
    if self.accessible.getRole() in self.TRIVIAL_ROLES:
      return True
    ss = self.accessible.getState()
    # ensure no bad state
    for state in self.TRIVIAL_STATES:
      if ss.contains(state):
        return True
    return False
  
  @pyLinAcc.errorToLookupError
  def isAccVisible(self):
    '''
    Gets if an accessible is visible. This implementation considers an 
    accessible visible if it has all of the states in L{VISIBLE_STATES}.
    
    @return: Does the accessible consider itself visible?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    '''
    ss = self.accessible.getState()
    # ensure all good states
    for state in self.VISIBLE_STATES:
      if not ss.contains(state):
        return False
    return True

  @pyLinAcc.errorToLookupError
  def hasAccState(self, state):
    '''
    Gets if the subject has the given state. The given string state is mapped to
    the appropriate value in L{pyLinAcc.Constants} if possible, or is compared
    as a string if a constant representing the state is not found.
    
    @param state: Name of the state (e.g. 'focused', 'selected', 'selectable')
    @type state: string
    @return: Does the accessible have the given state?
    @rtype: boolean
    '''
    ss = self.accessible.getState()
    c = pyLinAcc.stringToConst('state', state)
    return c in ss.getStates()

  @pyLinAcc.errorToLookupError
  def hasAccOneState(self, *states):
    '''
    Gets if the subject has at least one of the given states. The state names
    are mapped to appropriate values in L{pyLinAcc.Constants} if possible, or
    are compared as strings if a constant representing a particular state is
    not found.
    
    @param states: List of states (e.g. 'focused', 'selected', 'selectable')
    @type states: list of string
    @return: Does the accessible have at least one of the given states?
    @rtype: boolean
    '''
    ss = self.accessible.getState()
    actual = c in ss.getStates()
    for desired in states:
      c = pyLinAcc.stringToConst('state', desired)
      if desired in actual:
        return True
    return False
  
  @pyLinAcc.errorToLookupError
  def hasAccAllStates(self, *states):
    '''
    Gets if the subject has all of the given states. The state names are
    mapped to appropriate values in L{pyLinAcc.Constants} if possible, or are
    compared as strings if a constant representing a particular state is not
    found.
    
    @param states: List of states (e.g. 'focused', 'selected', 'selectable')
    @type states: list of string
    @return: Does the accessible have all of the given states?
    @rtype: boolean
    '''
    ss = self.accessible.getState()
    actual = c in ss.getStates()
    for desired in states:
      c = pyLinAcc.stringToConst('state', desired)
      if desired not in actual:
        return False
    return True

  @pyLinAcc.errorToLookupError
  def hasAccRole(self, role):
    '''
    Gets if the subject has the given role. The given string role is mapped to
    the appropriate value in L{pyLinAcc.Constants} if possible, or is compared
    as a string if a constant representing the role is not found.
    
    @param role: Name of the role (e.g. 'terminal', 'glass pane')
    @type role: string
    @return: Does the accessible have the given role?
    @rtype: boolean
    '''
    return pyLinAcc.stringToConst('role', role) == self.accessible.getRole()
