'''
Defines a device that uses the IBM TTS synthesizer via the pyibmtts package.
pyibmtts can be obtained from U{http://ibmtts-sdk.sf.net}.

@author: Larry Weiss
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import AEOutput
import pyibmtts
from AEOutput.Constants import *
from i18n import _

class IBMSpeech(AEOutput.Speech):
  '''
  Defines a class to send output from LSR to IBM TTS using the pyibmtts wrapper
  provided with ibmtts-sdk U{http://ibmtts-sdk.sf.net}.

  To be compliant with LSR requirements, this implements the interface defined 
  in L{AEOutput.Speech}.
  
  @ivar cmd_map: Mapping from L{AEOutput.Constants} commands for speech to 
    string parameter names used by pyibmtts. This dictionary is built at 
    instantiation time instead of at module import time because it relies on
    the pyibmtts dependency which will be resolved by the L{UIRegistrar}.
  @type cmd_map: dictionary
  @ivar current_voice: Index of current voice, initialized to 1
  @type current_voice: integer
  '''  
  def __init__(self):
    '''
    Initializes the instance and it's variables, but not the device. Use 
    L{deviceInit} to initialize the device.
    '''
    # allow parent to initialize vars
    AEOutput.Speech.__init__(self)
    self.current_voice = 1
    # build the command map now that we have pyibmtts
    self.cmd_map = {CONTEXT_RATE : pyibmtts.Speed,
                    CONTEXT_VOLUME : pyibmtts.Volume,
                    CONTEXT_PITCH : pyibmtts.PitchBaseline,
                    CONTEXT_INTONATION : pyibmtts.PitchFluctuation,
                    CONTEXT_GENDER : pyibmtts.Gender,
                    CONTEXT_HEADSIZE : pyibmtts.HeadSize,
                    CONTEXT_ASPIRATION : pyibmtts.Breathiness,
                    CONTEXT_FRICATION : pyibmtts.Roughness}  
    
  def _localize(self, text):
    '''
    Converts the given unicode text to 

    @todo: PP: Temporary kludge to convert all text to Latin-1 as IBM TTS is
      expecting that encoding instead of UTF-8; use locale set by setLocale
      or current system locale in the future
    @param text: Text to localize
    @type text: unicode
    @return: Localized string
    @rtype: string
    '''
    try:
      # kludge!
      return text.encode('latin-1', 'replace')
    except (TypeError, UnicodeDecodeError), e:
      return text
  
  def _changeVoice(self, index):
    '''
    Switches to another pre-defined voice.

    @param index: Index of the voice to set
    @type index: integer
    @raise IndexError: When the voice index is out of range
    '''
    # get current parameters of active voice
    v = self.tts.getVoice(0)
    rate = v.speed
    vol = v.volume
    # try to get the voice to activate
    try:
      vnew = self.tts.getVoice(index)
    except pyibmtts.ECIException:
      raise IndexError(index)
    self.tts.setActiveVoice(vnew)
    # store the index of the now active voice
    self.current_voice = index
    # restore voice independent parameters
    v.speed = rate
    v.volume = vol

  def deviceInit(self):
    '''
    Initializes the ViaVoice speech driver through gnome-speech.

    Overrides L{AEOutput.Speech.deviceInit}

    @raise AEOutput.InitError: When the device can not be initialized
    '''
    try:
      self.tts = pyibmtts.Engine()
    except pyibmtts.ECIException:
      raise AEOutput.InitError
    # use real units and start at some speedy rate
    self.tts.realWorldUnits = True
    self.tts.getVoice(0).speed = 250

  def deviceClose(self):
    '''
    Closes the specific speech device.

    Overrides L{AEOutput.Speech.deviceClose}
    '''
    del self.tts
    
  def useThread(self):
    '''
    @return: False since IBM TTS output is asynchronous via pyibmtts
    @rtype: boolean
    '''
    return False

  def writeStop(self):
    '''
    Stops speech immediately.
    '''
    try:
      self.tts.stop()
    except pyibmtts.ECIException:
      pass

  def writeTalk(self):
    '''
    Begins synthesis of all buffered text.
    '''
    self.tts.synthesize()
    
  def writeString(self, string):
    '''
    Adds the given string to the text to be synthesized.

    Overrides L{AEOutput.Speech.writeString}
    
    @param string: Text string to buffer
    @type string: string
    '''
    self.tts.addText(self._localize(string))

  def writeCommand(self, cmd, value):
    ''' 
    Sends a command at the given specified value. Returns only once the new
    value has been set.

    Overrides L{AEOutput.Speech.writeCommand}

    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: object
    @param value: New value to set for the given command
    @type value: object
    @raise ValueError: When the command or value is invalid
    '''
    # send what we have so far at the old value
    self.writeTalk()
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      # call _changeVoice to handle voice commands
      self._changeVoice(value)  # change value to an index
    else:
      v = self.tts.getVoice(0)
      # figure out which parameter to change
      try:
        v.setParam(self.cmd_map[cmd], value)
      except (KeyError, pyibmtts.ECIException):
        raise ValueError(cmd, value)

  def deviceSpeaking(self):
    '''
    Indicates whether the device is active.

    Overrides L{AEOutput.Speech.deviceSpeaking}

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    return self.tts.speaking()

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    Overrides L{AEOutput.Base.AEOutput.getName}

    @return: The localized name for the device
    @rtype: string
    '''
    return 'IBM-TTS %d' % pyibmtts.getVersion()

  def getCommandChars(self):
    '''
    Returns an empty string since this device does not use markup for sending 
    commands by default.

    Overrides L{AEOutput.Base.AEOutput.getCommandChars}

    @return: An empty string since this implementation allows any character.
    @rtype: string
    '''
    return ''

  def getCommandName(self, cmd, value):
    '''
    Gets the user displayable (localized) name for the specified command at
    the specified value.

    Overrides L{AEOutput.Base.AEOutput.getCommandName}
    
    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @param value: Value of the command to describe
    @type value: object
    @return: Localized name for the command at the specified value
    @rtype: string
    @raise ValueError: When the command is invalid
    '''
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      try:
        # try to get the name of the voice
        v = self.tts.getVoice(value)
        return v.name
      except pyibmtts.ECIException:
        raise ValueError(value)
    else:
      # try to get a name for the given command at least
      try:
        return AEOutput.Constants.cmd_name_map[cmd]
      except KeyError:
        raise ValueError(cmd)

  def getCommandLevel(self, cmd):
    '''
    Gets the current value of the specified command.

    Overrides L{AEOutput.Base.AEOutput.getCommandLevel}

    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @return: Current value of the command
    @rtype: object
    @raise ValueError: When the command is invalid
    '''
    # voice is a special case
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      return self.current_voice
    else:
      v = self.tts.getVoice(0)
      # figure out which parameter to get
      try:
        return v.getParam(self.cmd_map[cmd])
      except (KeyError, pyibmtts.ECIException):
        raise ValueError(cmd, value)

  def getLevelCount(self, cmd):
    '''
    Gets the minimum and maximum settable values for the specified command.
    The values used in this function are taken from the IBM TTS developer
    documentation for version 6.7 of the runtime.
    
    Overrides L{AEOutput.Base.AEOutput.getLevelCount}

    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @return: Minimum and maximum settable value for the given command
    @rtype: 2-tuple of integer
    @raise ValueError: When the command is invalid
    '''
    # voice is a special case
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      return (1, 16)
    elif cmd == AEOutput.Constants.CONTEXT_RATE:
      return (70l, 1297)
    elif cmd ==  AEOutput.Constants.CONTEXT_GENDER:
      return (0,1)
    elif cmd == AEOutput.Constants.CONTEXT_PITCH:
      return (40, 442)
    elif cmd == AEOutput.Constants.CONTEXT_VOLUME:
      return (0, 65535)
    elif self.cmd_map.get(cmd) is not None:
      return (0,100)
    else:
      raise ValueError(cmd)
