'''
Defines a hierarchy of L{Task.Tools} exceptions that are raised in tools
methods. A hierarchy of exceptions is used rather than a single exception with
various error codes to support the catching of a particular subtree of the
hierarchy, a single kind of exception, or all exceptions.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
from i18n import _

class ToolsError(Exception):
  '''
  Base class for all L{Task.Tools} exceptions. This base class should not
  be instantiated directly as it does not define its own error string. It only
  exists as a catch-all for its subclass exceptions.
  
  @cvar description: Localized error description to be reported to the user when
    the Trap flag in L{LSRSettings} is True.
  @type description: string
  '''
  description = None
  def __init__(self):
    '''
    Stores a blank description and None error code using the parent Python
    Exception constructor. 
    '''
    Exception.__init__(self, self.description)
    
class UnknownError(ToolsError):
  '''
  Exception for a unknown reasons.
  '''
  description = _('Unknown task error')
   
class InputError(ToolsError):
  '''
  Exception subtree for input failures.
  '''
  description = _('Invalid input')
  
class InvalidDeviceError(InputError):
  '''
  Exception for an invalid input device reference.
  '''
  description = _('Invalid input device')

class UndefinedKeyError(InputError):
  '''
  Exception for an undefined key constant failure.
  '''
  description = _('Undefined key')

class CancelledKeyError(InputError):
  '''
  Exception for an incomplete key command registration failure.
  '''
  description = _('Cancelled key')
  
class TierError(ToolsError):
  '''
  Exception subtree for L{Tier} management failures.
  '''
  description = _('Tier management failed')

class AddPerkError(TierError):
  '''
  Exception for a push L{Perk} failure.
  '''
  description = _('Perk not found')

class RemovePerkError(TierError):
  '''
  Exception for a pop L{Perk} failure.
  '''
  description = _('Perk not loaded')

class AddTaskError(TierError):
  '''
  Exception for a register L{Task} failure.
  '''
  description = _('Task not found')
  
class RemoveTaskError(TierError):
  '''
  Exception for a unregister L{Task} failure.
  '''
  description = _('Task not loaded')

class ViewError(ToolsError):
  '''
  Exception subtree for view navigation, read, and write errors.
  '''
  description = _('Invalid view')
  
class ActionError(ViewError):
  '''
  Exception subtree for accessible action errors.
  '''
  description = _('Action failed')
  
class FocusError(ActionError):
  '''
  Exception for failures setting the focus.
  '''
  description = _('Set focus failed')
  
class SelectError(ActionError):
  '''
  Exception for failures setting the selection.
  '''
  description =  _('Set selection failed')
  
class PORError(ViewError):
  '''
  Exception for L{POR} and related accessible failures.
  '''
  description = _('Invalid point of regard')
  
class NavError(ViewError):
  '''
  Exception subtree for all navigation failures.
  '''
  description = _('Invalid navigation')

class PrevCharError(NavError):
  '''
  Exception for previous character navigation failure.
  '''
  description = _('First character')

class NextCharError(NavError):
  '''
  Exception for next char navigation failure.
  '''
  description = _('Last character')

class PrevWordError(NavError):
  '''
  Exception for previous word navigation failure.
  '''
  description = _('First word')

class NextWordError(NavError):
  '''
  Exception for next word navigation failure.
  '''
  description = _('Last word')
  
class PrevItemError(NavError):
  '''
  Exception for previous item navigation failure.
  '''
  description = _('First item')

class NextItemError(NavError):
  '''
  Exception for next item navigation failure.
  '''
  description = _('Last item')
  
class ParentItemError(NavError):
  '''
  Exception for parent item navigation failure.
  '''
  description = _('First item')