'''
Defines L{Tools} for navigating within an application and inspecting and 
manipulating its accessible objects.

@todo: PP: bug somewhere in getPrevItem; multiline gaim messages not always
  announced in full when near the edge of a window

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

from LSRInterfaces import *
from Walker import *
from Error import *
from POR import POR
import Base
import LSRConstants
import Word

class View(Base.TaskTools):
  '''
  Provides methods for reading information from accessible objects and for 
  navigating relative to a L{POR}.
  
  Methods prefixed with I{get} do not modify the L{task_por}, L{curr_item_text},
  or L{curr_item_por} variables. They only return the L{POR} of interest.
  Methods prefixed with I{move} do modify these variables. Methods prefixed with
  I{set} actually change properties of the accessible itself. 

  The exceptions to this rule are methods that start with I{getLSR} and 
  I{setLSR} which modify LSR settings.
  
  @note: Selected and editable states not being announced because they happen at
    unexpected times or on undesireable accessibles. Selection can probably be
    done properly with events. Editable we will just avoid for the time being.
  
  @cvar _state_descriptions: Mapping from state name/value pairs to localized 
    string
  @type _state_descriptions: dictionary of 2-tuple, string pairs
  '''
  _state_descriptions = {('checked', False) : _('not checked'),
                         ('enabled', False) : _('disabled'),
                         #('selected', False) : _('not selected'),
                         ('collapsed', False) : _('expanded'),
                         ('expanded', False) : _('collapsed'),
                         ('animated', False) : _('not animated'),
                         #('editable', False) : _('not editable'),
                         ('checked', True) : _('checked'),
                         #('selected', True) : _('selected'),
                         ('collapsed', True) : _('collapsed'),
                         ('expanded', True) : _('expanded'),
                         ('animated', True) : _('animated'),
                         ('disabled', True) : _('disabled'),
                         #('editable', True) : _('editable')
                         }
  
  def getAppName(self, por=None):
    '''
    Gets the accessible name of the application containing the L{POR}.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Name of the application
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      return IAccessibleInfo(por).getAccAppName()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
  
  def getAccName(self, por=None):
    '''
    Gets the accessible name of the component at the provided L{POR}. 
    When L{por} is None, the L{task_por} is used.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: The accessible name of the control of the at the point of regard
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      return IAccessibleInfo(por).getAccName()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
  
  def getAccDesc(self, por=None):
    '''
    Gets the accessible description of the component at the provided L{POR}. 
    When L{por} is None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: The accessible description of the control of the at the point of 
      regard
    @rtype: string
    @raise PORError: When the L{POR} is invalidd
    '''
    por = por or self.task_por
    try:
      return IAccessibleInfo(por).getAccDescription()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
      
  def getAccCount(self, por=None):
    '''
    Gets the number of child accessibles of the accessible indicated by
    the given L{POR}. When L{por} is None, the L{task_por} is used.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Number of accessible children
    @rtype: integer
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      ai = IAccessibleInfo(por)
      return ai.getAccChildCount()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return 0
  
  def getAccRoleName(self, por=None):
    '''
    Gets the localized role name for the accessible at the provided L{POR}. 
    When L{por} is None, the L{task_por} is used.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: The role name of the control of the at the point of regard
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      return IAccessibleInfo(por).getAccRoleName()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
    
  def getAccLabel(self, por=None):
    '''
    Gets the label for the accessible at the provided L{POR}. When L{por} is 
    None, the L{task_por} is used.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: The AccessibleName of the component that labels the component at 
      the given location.
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      # get the accessible info interface
      ai = IAccessibleInfo(por)
      # get all labelled by relations
      pors = ai.getAccRelations('labelled by')
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
    # get names of all label accessibles
    label = []
    for por in pors:
      try:
        label.append(self.getAccName(por))
      except PORError:
        # ignore POR errors
        pass
    # join names into a string
    return ' '.join(label)
  
  def hasAccState(self, state, por=None):
    '''    
    Gets if the accessible at the given L{POR} has the given state. When
    L{por} is None, the L{task_por} is used. The state is assumed to be a string
    that can be mapped to an appropriate state constant on the platform or 
    indicates an extension state that is represented by a string.
    
    @param state: Name of a state (e.g. 'focused', 'selected', 'selectable')
    @type state: string
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Does the L{POR} have the given state?
    @rtype: boolean
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      ai = IAccessibleInfo(por)
      return ai.hasAccState(state)
    except LookupError:
      raise PORError
    except NotImplementedError:
      return False
      
  def hasAccRole(self, role, por=None):
    '''    
    Gets if the accessible at the given L{POR} has the given role. When
    L{por} is None, the L{task_por} is used. The role is assumed to be a string
    that can be mapped to an appropriate role constant on the platform or 
    indicates an extension role that is represented by a string.
    
    @param role: Name of a role (e.g. 'terminal', 'glass pane', 'button')
    @type role: string
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Does the L{POR} have the given role?
    @rtype: boolean
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      ai = IAccessibleInfo(por)
      return ai.hasAccRole(role)
    except LookupError:
      raise PORError
    except NotImplementedError:
      return False
      
  def getStateText(self, por=None, name=None, value=None):
    '''    
    Gets text describing the states of the given L{POR} that might be of
    interest to the user. When L{por} is None, the L{task_por} is used. If
    name is specified, only gets the text describing the state with that
    non-translated name. If name is not specified, gets text describing all
    states of interest.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @param name: Name of the state to describe
    @type name: string
    @param value: Value of the state to describe
    @type value: string
    @return: String describing the state(s), None if named state not found
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    # look up just the give name and value
    if name is not None and value is not None:
      return self._state_descriptions.get((name, value))
    
    por = por or self.task_por
    try:
      # get all states
      states = IAccessibleInfo(por).getAccStates()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
      
    text = []
    for name in states:
      try:
        # localize all state strings
        desc = self._state_descriptions[(name, True)]
      except KeyError:
        continue
      text.append(desc)
    return ' '.join(text)
      
  def getItemText(self, por=None):
    '''
    Gets the text of the complete item that includes the specified L{POR}. When 
    L{por} is None, the L{task_por} is used. Even when the text of the current 
    point of regard is requested, it is better to use this method instead of 
    using L{curr_item_text}.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Item text at the given point of regard
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      text = IAccessibleInfo(por).getAccItemText()
      return text
    except LookupError:
      raise PORError
    except NotImplementedError:
      return ''
  
  def getLSRMode(self):
    '''
    @return: The current L{POR} mode, L{LSRConstants.MODE_POINTER},
      L{LSRConstants.MODE_FOCUS}, L{LSRConstants.MODE_EVENT}
    @rtype: integer
    '''
    return self.state.Mode
  
  def setLSRMode(self, new_mode):
    '''
    Sets the POR mode and changes the task's POR.
    
    @param new_mode: The new POR mode, L{LSRConstants.MODE_POINTER}, 
      L{LSRConstants.MODE_FOCUS}, L{LSRConstants.MODE_EVENT}
    @type new_mode: integer
    '''
    self.state.Mode = new_mode
    if new_mode == LSRConstants.MODE_POINTER:
      self.task_por = self.state.Pointer
    else:
      self.task_por = self.tier.getFocusPOR()

  def getAllSelected(self, por=None):
    '''
    Gets all of the selected items or accessible children in the accessible 
    indicated by the given L{POR}. When L{por} is None, the L{task_por} is used. 
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Points of regard to selected items or accessibles
    @rtype: list of L{POR}
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      return IAccessibleInfo(por).getAccSelection()
    except LookupError:
      raise PORError
    except NotImplementedError:
      return []
  
  def getFirstSelected(self, por=None):
    '''
    Gets the first selected items or accessible children in the accessible 
    indicated by the given L{POR}. When L{por} is None, the L{task_por} is used.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the first selected item or accessible
    @rtype: L{POR}
    @raise PORError: When the L{POR} is invalid
    '''
    all = self.getAllSelected(por)
    try:
      return all[0]
    except IndexError:
      return None
  
  def setSelected(self, por=None):
    '''
    Selects the item or accessible indicated by the given L{POR}. When L{por} is
    None, the L{task_por} is used.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @raise PORError: When the L{POR} is invalid
    @raise SelectError: When selecting the L{POR} is supported but rejected by
      the system for some reason
    '''
    por = por or self.task_por
    try:
      selected = IAccessibleAction(por).selectChild()
    except (IndexError, LookupError):
      raise PORError
    except NotImplementedError:
      # just return if selection isn't support
      return
    else:
      if not selected:
        # raise an error if selection is supported but was rejected
        raise SelectError

  def getFocus(self):
    '''
    Gets the L{POR} representing the focus of the foreground application.
    
    @return: Point of regard to the focused accessible
    @rtype: L{POR}
    '''
    return self.tier.getFocusPOR()
  
  def setFocus(self, por=None):
    '''
    Sets the system input focus to the provided L{POR}. When L{por} is None, 
    the L{task_por} is used. The character offset component of the POR may not 
    be respected for some controls.
    
    @todo: PP: add support for setting item and character offset also
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @raise PORError: When the L{POR} is invalid
    @raise FocusError: When focusing on the L{POR} is supported but rejected by
      the system for some reason
    '''
    por = por or self.task_por
    try:
      # try to set the focus
      ia = IAccessibleAction(por)
      focused = ia.setFocus()
    except LookupError:
      # the provided por was invalid
      raise PORError
    except NotImplementedError:
      # just return if setting focus is not supported
      return
    else:
      if not focused:
        # raise an error if selection is supported but was rejected
        raise FocusError
  
  def getNextAcc(self, por=None):
    '''
    Gets the L{POR} of the next accessible of a given L{POR}. When L{por} is 
    None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the next accessible
    @rtype: L{POR}
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    w = AccessibleWalker(por)
    try:
      return w.getNextPOR()
    except NotImplementedError:
      return None
      
  def getFirstPeerAcc(self, por=None):
    '''
    Gets the L{POR} of the first accessible peer of a given L{POR}. When L{por} 
    is None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the first peer accessible
    @rtype: L{POR}
    '''
    por = por or self.task_por
    try:
       p = IAccessibleNav(por).getParentAcc()
       return IAccessibleNav(p).getFirstAccChild()
    except NotImplementedError:
      return None
  
  def getLastPeerAcc(self, por=None):
    '''
    Gets the L{POR} of the last accessible peer of a given L{POR}. When L{por} 
    is None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the last peer accessible
    @rtype: L{POR}
    '''
    por = por or self.task_por
    try:
       p = IAccessibleNav(por).getParentAcc()
       return IAccessibleNav(p).getLastAccChild()
    except NotImplementedError:
      return None

  def getNextItem(self, por=None, wrap=False, only_visible=False):
    '''
    Gets the L{POR} of the next item in the given L{POR}. When L{por} is None, 
    the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @param wrap: Consider a next item outside this accessible if there is no
      next item in this accessible? Defaults to False.
    @type wrap: boolean
    @param only_visible: Only consider visible items? Defaults to False.
    @type only_visible: boolean
    @return: Point of regard to the previous item or None if not found
    @rtype: L{POR}
    '''
    por = por or self.task_por
    w = AccessibleItemWalker(por, only_visible=only_visible)
    try:
      next_por = w.getNextPOR()
    except NotImplementedError:
      return None
    if wrap:
      # return whatever the walker returned
      return next_por
    elif next_por is not None and not next_por.isSameAcc(por):
      # there is no next POR when wrapping is off
      return None
    return next_por
  
  def getNextWord(self, por=None):
    '''
    Get the L{POR} of the next word in the item indicated by the given L{POR}. 
    When L{por} is None, the L{task_por} is used.
    
    @note: This method does not respect L{LSRSettings}.Wrap.
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the next word or None if not found
    @rtype: L{POR}
    '''
    por = por or self.task_por
    # get the current item text
    try:
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      return None
    # parse the text for the next word
    prev, curr, next = Word.getContextFromString(text, self.state, por)
    if next is not None:
      # if we found a next word, return its POR
      return next.getPOR()
    else:
      return None
    
  def getNextChar(self, por=None):
    '''
    Get the L{POR} of the next character in the item indicated by the given 
    L{POR}. When L{por} is None, the L{task_por} is used.
    
    @note: This method does not respect L{LSRSettings}.Wrap.
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the next character or None if not found
    @rtype: L{POR}
    '''
    por = por or self.task_por
    try:
      # get the text of the current item
      # @todo: PP: we need an adapter method to get item text length
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      return None
    if por.char_offset < len(text)-1:
      return POR(por.accessible, por.item_offset, por.char_offset+1)
    else:
      return None
      
  def getPrevAcc(self, por=None):
    '''
    Gets the L{POR} of the previous accessible of a given L{POR}. When L{por} is 
    None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    '''
    por = por or self.task_por
    w = AccessibleWalker(por)
    try:
      return w.getPrevPOR()
    except NotImplementedError:
      return None
  
  def getPrevItem(self, por=None, wrap=False, only_visible=False):
    '''
    Gets the L{POR} of the previous item in the given L{POR}. When L{por} is 
    None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @param wrap: Consider a previous item outside this accessible if there is no
      previous item in this accessible? Defaults to False.
    @type wrap: boolean
    @param only_visible: Only consider visible items? Defaults to False.
    @type only_visible: boolean
    @return: Point of regard to the previous item or None if not found
    @rtype: L{POR}
    '''
    por = por or self.task_por
    w = AccessibleItemWalker(por, only_visible=only_visible)
    try:
      prev_por = w.getPrevPOR()  
    except NotImplementedError:
      return None
    if wrap:
      # return whatever the walker returned
      return prev_por
    elif not prev_por.isSameAcc(por):
      # there is no previous POR when wrapping is off
      return None
    return prev_por
      
  def getPrevWord(self, por=None):
    '''
    Get the L{POR} of the previous word in the item indicated by the given 
    L{POR}. When L{por} is None, the L{task_por} is used.
    
    @note: This method does not respect L{LSRSettings}.Wrap.
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the previous word or None if not found
    @rtype: L{POR}
    '''
    por = por or self.task_por
    # get the current item text; cannot use self.getItemText because it traps
    # errors that we need to know here in order to return on an invalid POR
    try:
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      return None
    # parse the text for the previous word
    prev, curr, next = Word.getContextFromString(text, self.state, por)
    if prev is not None:
      # if we found a prev word, return its POR
      return prev.getPOR()
    else:
      return None
  
  def getPrevChar(self, por=None):
    '''
    Get the L{POR} of the previous character in the item indicated by the given 
    L{POR}. When L{por} is None, the L{task_por} is used.
    
    @note: This method does not respect L{LSRSettings}.Wrap.
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard to the previous character or None if not found
    @rtype: L{POR}
    '''
    por = por or self.task_por
    if por.char_offset > 0:
      return POR(por.accessible, por.item_offset, por.char_offset-1)
    else:
      return None
  
  def getAncestorAccs(self, por=None):
    '''
    Gets a list of L{POR}s refering to the first character and first item of all
    of the readable, non-trivial ancestors of the accessible indicated by the 
    given L{POR}. When no POR is provided, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regards of all readable, non-trivial ancestors up to the
      root
    @rtype: list of L{POR}
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    w = AccessibleItemWalker(por)
    a = []
    try:
      curr = w.getParentPOR()
    except NotImplementedError:
      # the POR was bad, so stop right away
      return []
    while curr is not None:
      a.append(curr)
      try:
        curr = w.getParentPOR()
      except NotImplementedError:
        # the parent POR was bad, so stop here
        return a
    return a
  
  def getParentAcc(self, por=None):
    '''
    Gets the L{POR} of the first character and first item of the parent of the 
    provided L{POR}. When no POR is provided, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Point of regard for the parent
    @rtype: L{POR}
    '''
    por = por or self.task_por
    w = AccessibleWalker(por)
    try:
      return w.getParentPOR()
    except NotImplementedError:
      return None
      
  def getViewRootAcc(self):
    '''
    Gets a L{POR} indicating the root of the active view. This is more efficient
    than L{getRootAcc} when the root of the I{active} view is needed.
    
    @return: A point of regard to the root accessible
    @rtype: L{POR}
    '''
    return self.view_man.getView()
  
  def getRootAcc(self, por=None):
    '''
    Gets a L{POR} to the top level container of the given L{POR}.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: A point of regard to the root accessible
    @rtype: L{POR}
    @raise PORError: When L{LSRSettings}.Trap is False and the provided L{POR} 
      is invalid
    '''
    por = por or self.task_por
    w = AccessibleWalker(por)
    try:
      return w.getFirstPOR()
    except NotImplementedError:
      return None

  def getAccFromPath(self, por=None, *path):
    '''
    Gets a L{POR} representing the accessible found by treating the path 
    integers as descendant indices stating at the given L{POR}. When no POR is 
    provided, the L{task_por} is used as the starting point.
    
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @param path: List of integers indicate the path to the desired object
    @type path: list of integer
    @return: Point of regard for the descendant at the specified path
    @rtype: L{POR}
    '''
    por = por or self.task_por
    if len(path) == 0:
      return por
    try:
      # try to get the child indicated by the path
      child_por = IAccessibleNav(por).getChildAcc(path[0])
    except (IndexError, LookupError):
      return None
    # recurse with the remainder of the path
    por_from_path = self.getAccFromPath(child_por, *path[1:]) 
    return por_from_path
  
  def moveToPrevItem(self, por=None):
    '''
    Modifies the L{task_por} to the beginning of the item before the provided 
    L{POR}. When L{por} is None, the L{task_por} is used. When the POR is at 
    the first item of the application, a L{Task.Tools.Error} is raised.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @raise PORError: When the L{POR} is invalid
    @raise PrevItemError: When there is no previous item
    '''
    por = por or self.task_por
    w = AccessibleItemWalker(por)
    try:
      por = w.getPrevPOR()
    except NotImplementedError:
      raise PORError
    if por is None:
      raise PrevItemError
    else:
      self.task_por = por
    
  def moveToCurrItem(self, por=None):
    '''
    Modifies the L{task_por} to the beginning of the item that includes the 
    provided L{POR}. When L{por} is None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    '''
    por = por or self.task_por
    # reset to first character in this item
    self.task_por = POR(por.accessible, por.item_offset, 0)

  def moveToNextItem(self, por=None):
    '''
    Modifies the L{task_por} to the beginning of the item after the provided 
    L{POR}. When L{por} is None, the L{task_por} is used. When the POR is at 
    the first item of the application, a L{Task.Tools.Error} is raised.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @raise PORError: When the L{POR} is invalid
    @raise NextItemError: When there is no next item
    '''
    por = por or self.task_por
    w = AccessibleItemWalker(por)
    try:
      por = w.getNextPOR()
    except NotImplementedError:
      raise PORError
    if por is None:
      raise NextItemError
    else:
      self.task_por = por

  def moveToPrevWord(self, por=None):
    '''
    Modifies the L{task_por} to the beginning of the L{Word} before the provided
    L{POR}. When L{por} is None, the L{task_por} is used. When the POR is at the
    first L{Word} in the item and wrapping is on, the L{task_por} is
    updated to the beginning of the last L{Word} of the previous item. When
    there is no previous word, L{LSRSettings}.Wrap is False, and 
    L{LSRSettings}.Trap is False, an exception is raised.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Did the L{task_por} move to the previous item?
    @rtype: boolean
    @raise PORError: When the L{POR} is invalid
    @raise PrevWordError: When L{LSRSettings}.Wrap is False and there is no
      previous word in this item, or when there is no previous word because the 
      L{POR} is at the end of the view
    '''
    por = por or self.task_por
    # get the current item text; cannot use self.getItemText because it traps
    # errors that we need to know here in order to return on an invalid POR
    try:
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      raise PORError
    # parse the text for the previous word
    prev, curr, next = Word.getContextFromString(text, self.state, por)
    if prev is not None:
      # if we found a prev word, store its POR
      self.task_por = prev.getPOR()
      return False
    elif not self.state.Wrap:
      # if we're not wrapping, error
      raise PrevWordError
    else:
      # move to the previous item if possible
      try:
        self.moveToPrevItem(por)
      except PrevItemError:
        # no previous item means no previous word
        raise PrevWordError
      # now move to the last word in the item
      words = Word.buildWordsFromString(self.getItemText(), self.state, 
                                        self.task_por)
      try:
        # get the POR of the last word
        self.task_por = words[-1].getPOR()
      except IndexError:
        # no words, so just use the current POR for the start of the item
        pass
      return True
      
  def moveToCurrWord(self, por=None):
    '''
    Modifies the L{task_por} to the beginning of the L{Word} that includes the 
    provided L{POR}. When L{por} is None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    # get the current item text
    try:
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      raise PORError
    # parse the text for the current word
    prev, curr, next = Word.getContextFromString(text, self.state, por)
    if curr is not None:
      # if we found the current word, store its POR
      self.task_por = curr.getPOR()
    else:
      # use a POR pointing to the start of this item
      self.task_por = POR(por.accessible, por.item_offset, 0)

  def moveToNextWord(self, por=None):
    '''
    Modifies the L{task_por} to the beginning of the L{Word} after the 
    provided L{POR}. When L{por} is None, the L{task_por} is used. When the POR 
    is at the last Word in the item and wrapping is on, the L{task_por} 
    is updated to the beginning of the first Word of the next item.  When
    there is no next word, L{LSRSettings}.Wrap is False, and 
    L{LSRSettings}.Trap is False, an exception is raised.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Did the L{task_por} move to the next item?
    @rtype: boolean
    @raise PORError: When the L{POR} is invalid
    @raise NextWordError: When L{LSRSettings}.Wrap is False and there is no next
      word in this item, or when there is no next word because the L{POR} is at
      the end of the view
    '''
    por = por or self.task_por
    # get the current item text
    try:
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      raise PORError
    # parse the text for the next word
    prev, curr, next = Word.getContextFromString(text, self.state, por)
    if next is not None:
      # if we found a next word, store its POR
      self.task_por = next.getPOR()
      return False
    elif not self.state.Wrap:
      # if we're not wrapping, error
      raise NextWordError
    else:
      # move to the next item if possible
      try:
        self.moveToNextItem(por)
      except NextItemError:
        # no next item means no next word
        raise NextWordError
      # now move to the first word in the item, just use char_offset 0 because
      # blanks at the start of a word are considered part of the main part of 
      # that word
      self.task_por = POR(self.task_por.accessible, self.task_por.item_offset,0)
      return True
      
      # this code will actually build the words and get the offset of the first
      # one, but it *should* always be at zero so we probably don't need it
      #words = Word.buildWordsFromString(self.getItemText(), self.state, 
                                        #self.task_por)
      #try:
        ## get the POR of the first word
        #self.task_por = words[0].getPOR()
      #except IndexError:
        ## no words, so just use the current POR for the start of the item
        #pass

  def moveToPrevChar(self, por=None):
    '''
    Modifies the L{task_por} to the character before the provided L{POR}. When 
    L{por} is None, the L{task_por} is used. When the L{POR} is at the first 
    character in the item and wrapping is on, the L{task_por} is updated 
    to the last character of the previous item. When L{LSRSettings}.Wrap is 
    False and L{LSRSettings}.Trap is False, an exception is raised.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Did the L{task_por} move to the previous item?
    @rtype: boolean
    @raise PORError: When the L{POR} is invalid
    @raise PrevCharError: When L{LSRSettings}.Wrap is False and there is no
      previous character in this item, or when there is no previous character 
      because the L{POR} is at the start of the view
    '''
    por = por or self.task_por
    if por.char_offset > 0:
      self.task_por = POR(por.accessible, por.item_offset, por.char_offset-1)
      return False
    elif not self.state.Wrap:
      # if we're not wrapping, error
      raise PrevCharError
    else:
      # move to the previous item if possible
      try:
        self.moveToPrevItem(por)
      except PrevItemError:
        # no previous item means no previous word
        raise PrevCharError
      # now move to the last character in the previous item (which is now in the
      # task_por)
      text = self.getItemText()
      self.task_por.char_offset = max(0, len(text)-1)
      return True

  def moveToCurrChar(self, por=None):
    '''
    Modifies the L{task_por} to the character at the provided L{POR}. When 
    L{por} is None, the L{task_por} is used.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    '''
    self.task_por = por or self.task_por

  def moveToNextChar(self, por=None):
    '''
    Modifies the L{task_por} to the character after the provided L{POR}. When 
    L{por} is None, the L{task_por} is used. When the L{POR} is at the last 
    character in the item and L{LSRSettings}.Wrap is True, the L{task_por} 
    is updated to the first character of the next item. When 
    L{LSRSettings}.Wrap is False and L{LSRSettings}.Trap is False, an 
    exception is raised.

    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Did the L{task_por} move to the next item?
    @rtype: boolean
    @raise PORError: When the L{POR} is invalid
    @raise NextCharError: When L{LSRSettings}.Wrap is False and there is no
      next character in this item, or when there is no next character because 
      the L{POR} is at the end of the view
    '''
    por = por or self.task_por
    try:
      # get the text of the current item
      # @todo: PP: we need an adapter method to get item text length
      text = IAccessibleInfo(por).getAccItemText()
    except (LookupError, NotImplementedError):
      raise PORError
    if por.char_offset < len(text)-1:
      self.task_por = POR(por.accessible, por.item_offset, por.char_offset+1)
      return False
    elif not self.state.Wrap:
      # if we're not wrapping, error
      raise NextCharError
    else:
      # move to the next item if possible
      try:
        self.moveToNextItem(por)
      except NextItemError:
        # no next item means no next char
        raise NextCharError
      # we're already at the start of the next item, so just return
      return True

  def moveToPOR(self, por):
    '''
    Moves the L{task_por} to the given L{POR}. Just a convenience method. A 
    L{Perk} writer can set self.task_por = x if that is preferred.
    '''
    self.task_por = por

  def moveToRoot(self):
    '''
    Modifies the L{task_por} to the root of the view.

    @raise PORError: When L{LSRSettings}.Trap is False and the provided L{POR} 
    is invalid
    '''
    self.task_por = self.getRootAcc()

  def moveToEnd(self):
    '''
    Modifies the L{task_por} to the beginning of the last item in the current
    view.

    @raise PORError: When the L{POR} is invalid
    '''
    por = self.getRootAcc()
    w = AccessibleItemWalker(por)
    try:
      por = w.getLastPOR()
    except NotImplementedError:
      # the por we tried to walk was bad
      raise PORError
    if por is None:
      raise PORError
    else:
      self.task_por = por
  
  def getAccTextAttr(self, name, por=None):
    '''
    Gets the value of the given attribute of an accessible of a given 
    L{POR}.  When L{por} is None, the L{task_por} is used.
    
    Valid attribute names/value pairs include:
    (subject to change, gnome ATK specification):
    
    left-margin: The pixel width of the left margin
    right-margin: The pixel width of the right margin
    indent: The number of pixels that the text is indented
    invisible: Either "true" or "false" indicates whether text is visible or not
    editable: Either "true" or "false" indicates whether text is editable or not
    pixels-above-lines:	Pixels of blank space to leave above each 
    newline-terminated line.
    pixels-below-lines:	Pixels of blank space to leave below each 
    newline-terminated line.
    pixels-inside-wrap:	Pixels of blank space to leave between wrapped lines 
    inside the same newline-terminated line (paragraph).
    bg-full-height: "true" or "false" whether to make the background color for 
    each character the height of the highest font used on the current line, or 
    the height of the font used for the current character.
    rise: Number of pixels that the characters are risen above the baseline
    underline: "none", "single", "double" or "low"
    strikethrough: "true" or "false" whether the text is strikethrough
    size: The size of the characters.
    scale: The scale of the characters: a string representation of a double.
    weight: The weight of the characters.
    language: The language used
    family-name: The font family name
    bg-color: The background color: RGB value of the format "u,u,u"
    fg-color: The foreground color: RGB value of the format "u,u,u"
    bg-stipple: "true" if a GdkBitmap is set for stippling the background color.
    fg-stipple: "true" if a GdkBitmap is set for stippling the foreground color.
    wrap-mode: The wrap mode of the text, if any: "none", "char" or "word"
    direction: The direction of the text, if set: "none", "ltr" or "rtl"
    justification: The justification of the text, if set: "left", "right", 
    "center" or "fill"
    stretch: The stretch of the text, if set: "ultra_condensed", 
    "extra_condensed", "condensed", "semi_condensed", "normal", "semi_expanded",
    "expanded", "extra_expanded" or "ultra_expanded"
    variant: The capitalization of the text, if set: "normal" or "small_caps"
    style: The slant style of the text, if set: "normal", "oblique" or "italic"
    last-defined: not a valid text attribute, for finding end of enumeration
    
    @param name: text attribute name, eg. "fg-color", "justification"
    @type name: string
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: value of attribute
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      # get the info interface
      ai = IAccessibleInfo(por)
      value = ai.getAccTextAttr(name)
      return value
    except LookupError:
      # raise POR error if some accessible was dead, bad or missing
      raise PORError
    except NotImplementedError:
      return ''

  def getAccAllTextAttrs(self, por=None):
    '''
    Like {Task.Tools.View.getAccTextAttr}, but gets the values of all of the
    current attributes of an accessible of a given L{POR}.  When L{por} is None,
    the L{task_por} is used.
     
    @param por: A point of regard; use the current task's POR if none supplied
    @type por: L{POR}
    @return: Value of attribute
    @rtype: string
    @raise PORError: When the L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      # get the info interface
      ai = IAccessibleInfo(por)
      attrs = ai.getAccAllTextAttrs()
      return attrs
    except LookupError:
      # raise POR error if some accessible was dead, bad or missing
      raise PORError
    except NotImplementedError:
      return ''

  def getStartOfHardLine(self, por=None):
    '''
    From a given L{POR}, get the L{POR} of the start of the first line that
    follows the previous hard line break, or the start of the first line if it
    is the first line in the accessible. When L{por} is None, the L{task_por} is
    used.
    
    @note: This method does not respect L{LSRSettings}.Wrap.
    @param por: Point of regard; use the current task's L{POR} if none supplied
    @type por: L{POR}
    @return: Point of regard to the previous character or None if not found
    @rtype: L{POR}
    @raise PORError: When some L{POR} is invalid
    '''
    por = por or self.task_por
    try:
      prev = self.getPrevItem(por)
      while prev is not None and prev.item_offset is not None:
        prevText = self.getItemText(prev)
        if prevText.endswith('\n'): # stop at a line break
          break
        por = prev
        prev = self.getPrevItem(por)
      return POR(por.accessible, por.item_offset, 0)
    except (LookupError, PORError):
      # whatever we started with was not a valid POR
      raise PORError

  def getEndOfHardLine(self, por=None):
    '''
    From a given L{POR}, get the L{POR} of the end of the first following line
    that ends with a hard line break, or the end of the last line if it is the
    last line in the accessible. When L{por} is None, the L{task_por} is used.
    
    @note: This method does not respect L{LSRSettings}.Wrap.
    @param por: Point of regard; use the current task's L{POR} if none supplied
    @type por: L{POR}
    @return: Point of regard to the previous character or None if not found
    @rtype: L{POR}
    @raise PORError: When some L{POR} is invalid
    '''
    por = por or self.task_por
    try:    
      next = self.getNextItem(por)
      text = self.getItemText(por)
      while next is not None:
        if text.endswith('\n'): # stop at a line break
          break
        text = self.getItemText(next)
        por = next
        next = self.getNextItem(por)
      if next is None:
        # bottom line in chat window doesn't terminate in '\n'
        return POR(por.accessible, por.item_offset, len(text))
      else:
        return POR(por.accessible, por.item_offset, len(text)-1)
    except (LookupError, PORError):
      # whatever we started with was not a valid POR
      raise PORError

  def getAccTextBetween(self, start_por=None, end_por=None, only_visible=False):
    '''
    Gets all text between two point of regards. If L{start_por} is not 
    specified, defaults to using the current L{task_por}. If L{end_por} is not
    given, defaults to getting all text to the end of the current accessible.
    
    @todo: PP: what about PORs with None as item_offset or with None as text
      returned from getItemText?
    
    @param start_por: A point of regard indicating the start of the text range; 
      use the current task's POR if none supplied
    @type start_por: L{POR}
    @param end_por: A point of regard indicating the end of the text range; 
      use the POR to the end of the accessible if none supplied
    @type end_por: L{POR}
    @param only_visible: Only consider visible items? Defaults to False.
    @type only_visible: boolean
    @return: Text between the two L{POR}s
    @rtype: string
    @raise PORError: When L{LSRSettings}.Trap is False and the provided L{POR} 
      is invalid
    '''
    start_por = start_por or self.task_por
    w = AccessibleItemWalker(start_por, only_visible=only_visible)
    
    text = []
    curr = start_por
    prev = start_por
    # walk until we pass the end_por or hit the end of the current accessible
    while 1:
      if curr is None or (end_por is None and not curr.isSameAcc(start_por)):
        # stop if the end is not specified and we've gotten to the last item of
        # this accessible
        break
      elif end_por.isSameItem(curr):
        # slice up to the requested char offset in this item
        text.append(self.getItemText(curr)[:end_por.char_offset])
        break
      elif curr.isItemAfter(end_por):
        # slice up to the requested char offset in the previous item
        text.append(self.getItemText(prev)[:end_por.char_offset])
        break
      else:
        # otherwise, store all the item text
        text.append(self.getItemText(curr))
      prev = curr
      try:
        curr = w.getNextPOR()
      except NotImplementedError:
        # no POR to walk, so stop early
        break
    return ''.join(text)
