'''
Defines a class that provides a "raw" walk of all accessibles in the hiearchy.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Base
from LSRInterfaces import *

class AccessibleWalker(Base.Walker):
  '''
  Walks the accessible hierarchy exactly as it is presented. Uses the 
  L{Adapters} provided to get next, previous, parent, first, and last 
  accessibles without any additional processing.
  
  @ivar por: The starting L{POR} for the L{Walker}
  @type por: L{POR}
  '''
  def __init__(self, por):
    '''
    Stores the starting L{POR}.

    @param por: The current L{POR} for the L{Walker}
    @type por: L{POR}
    '''
    self.por = por
    
  def getCurrPOR(self):
    '''
    @return: Point of regard to the current accessible
    @rtype: L{POR}
    '''
    return self.por
  
  def getParentPOR(self):
    '''   
    @return: Point of regard to the parent of the current accessible or None if
      it does not exist
    @rtype: L{POR}
    '''
    try:
      self.por = IAccessibleNav(self.por).getParentAcc()
    except LookupError:
      return None
    return self.por
  
  def getNextPOR(self):
    '''
    @return: Point of regard to the next accessible peer of the current 
      accessible or None if it does not exist
    @rtype: L{POR}
    '''
    try:
      self.por = IAccessibleNav(self.por).getNextAcc()
    except (LookupError, IndexError):
      return None
    return self.por
  
  def getPrevPOR(self):
    '''
    @return: Point of regard to the previous accessible peer of the current 
      accessible or None if it does not exist
    @rtype: L{POR}
    '''
    try:
      self.por = IAccessibleNav(self.por).getPrevAcc()
    except LookupError:
      return None
    return self.por
  
  def getFirstPOR(self):
    '''
    @return: Point of regard to the first accessible
    @rtype: L{POR}
    '''
    por = self.por
    child = self.por
    while 1:
      try:
        parent = IAccessibleNav(por).getParentAcc()
      except LookupError:
        self.por = child
        return child
      else:
        child = por
        por = parent

  def getLastPOR(self):
    '''
    @return: Point of regard to the last accessible
    @rtype: L{POR}
    '''
    # get to the first POR first so we can traverse the very last branch
    self.getFirstPOR()
    while 1:
      try:
        self.por = IAccessibleNav(self.por).getLastAccChild()
      except LookupError:
        # stop when we can find any more children
        return self.por