'''
Defines a device that uses the IBM TTS synthesizer via the pyibmtts package.
pyibmtts can be obtained from U{http://ibmtts-sdk.sf.net}.

@var UTF16_LANGS: All supported languages that accept UTF-16 encoded strings 
  instead of CP1252 strings.
@type UTF16_LANGS: tuple

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2007 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
import AEOutput
import pyibmtts
from i18n import _

__uie__ = dict(kind='device')

UTF16_LANGS = (pyibmtts.MandarinChinese, pyibmtts.TaiwaneseMandarin, 
               pyibmtts.StandardJapanese, pyibmtts.StandardKorean, 
               pyibmtts.StandardCantonese)

class IBMSpeechStyle(AEOutput.AudioStyle):
  '''
  Overrides the base L{AEOutput.Style} class, filling in fields for supported 
  style properties with their appropriate values.
  '''
  def init(self, device):
    '''
    Create style settings based on known properties of the IBM text to speech
    engine.
    '''
    self.newEnum('Voice', 1, _('Voice'),
                 {_('Adult Male 1') : 1,
                  _('Adult Female 1'): 2,
                  _('Child 1') : 3,
                  _('Adult Male 2') : 4,
                  _('Adult Male 3') : 5,
                  _('Adult Female 2') : 6,
                  _('Elderly Female 1') : 7,
                  _('Elderly Male 1') : 8},
                 _('Name of the active voice'))
    self.newRelRange('Pitch', 220, _('Pitch'), 40, 442, 0,
                     _('Baseline voice pitch in Hertz'))
    self.newRelRange('Rate', 250, _('Rate'), 176, 1297, 0,
                     _('Speech rate in words per minute'))
    self.newRelPercent('Volume', 58982, _('Volume'), 1, 65535, 0,
                       _('Speech volume as a percentage'))
  
  def getGroups(self):
    '''
    Gets configurable absolute settings affecting all output from this device.
    
    @return: Group of all configurable settings
    @rtype: L{AEState.Setting.Group}
    '''
    g = self.newGroup()
    s = g.newGroup(_('Speech'))
    s.extend(['Pitch', 'Rate', 'Volume'])
    if self.isDefault():
      # generate a group for standard word parsing settings on the default
      # object only for now
      self._newWordGroup(g)
    return g
  
class IBMSpeech(AEOutput.Audio):
  '''
  IBM TTS via the pyibmtts wrapper.
 
  @ivar last_style: Last style object to be applied to output
  @type last_style: L{AEOutput.Style}
  '''    
  USE_THREAD = False
  STYLE = IBMSpeechStyle
  
  def _localize(self, text, style):
    '''
    Converts the given unicode text to CP1252 or UTF-16 depending on the 
    language flag in the style.

    @note: Larry says IBM TTS expects all Asian language strings to be encoded
      in UCS-2, predecessor of UTF-16, and all others in CP1252.
    @param text: Text to localize
    @type text: unicode
    @param style: Style object indicating the language and locale
    @type style: L{AEOutput.Style}
    @return: Localized string
    @rtype: string
    '''
    if self.tts.languageDialect in UTF16_LANGS:
      return text.encode('utf-16', 'ignore')
    else:
      return text.encode('cp1252', 'ignore')

  def init(self):
    '''
    Initializes the IBM TTSspeech driver through gnome-speech.
    
    @raise AEOutput.InitError: When the device can not be initialized
    '''
    self.last_style = None
    try:
      self.tts = pyibmtts.Engine()
    except pyibmtts.ECIException:
      raise AEOutput.InitError
    # use real units
    self.tts.realWorldUnits = True
      
  def createDistinctStyles(self, num_groups, num_layers):
    '''
    Creates distinct styles following the guidelines set forth in the base
    class. Only distinguishes groups using voice, not layers.
    
    @param num_groups: Number of sematic groups the requestor would like to
      represent using distinct styles
    @type num_groups: integer
    @param num_layers: Number of content origins (e.g. output originating from
      a background task versus the focus) the requestor would like to represent
      using distinct styles
    @type num_layers: integer   
    @return: Styles
    @rtype: list of L{AEOutput.Style}
    ''' 
    styles = []
    for i in xrange(num_groups*num_layers):
      # create a new style object
      s = IBMSpeechStyle(self.default_style)
      # initialize it
      s.init(self)
      v_num = (i % 8) + 1
      v = self.tts.getVoice(v_num)
      s.Voice = v_num
      # set relative pitch value
      s.Pitch = v.pitchBaseline
      # store the style for later
      styles.append(s)
    return styles

  def close(self):
    '''
    Closes the speech device.
    '''
    del self.tts

  def sendStop(self, style=None):
    '''
    Stops speech immediately.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    try:
      self.tts.stop()
    except pyibmtts.ECIException:
      pass

  def sendTalk(self, style=None):
    '''
    Begins synthesis of the buffered text.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    try:
      self.tts.synthesize()
    except pyibmtts.ECIException:
      pass
    
  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    if style.isDirty() or self.last_style != style:
      self.last_style = style
      self._applyStyle(style)
    try:
      self.tts.addText(self._localize(text, style))
    except pyibmtts.ECIException:
      pass

  def isActive(self):
    '''
    Indicates whether the device is active meaning it is busy doing output or
    has text waiting to be output.

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    try:
      return self.tts.speaking()
    except pyibmtts.Exception:
      return False

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    @return: Localized name for the device
    @rtype: string
    '''
    return 'IBM-TTS %s' % pyibmtts.getVersion()
  
  def sendIndex(self, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output.
    
    @todo: PP: implement when index working

    @param style: Style indicating channel in which the marker will be appended
    @type style: L{AEOutput.Style}
    '''
    raise NotImplementedError

  def _applyStyle(self, style):
    '''
    Applies a given style to this output device. All output following this 
    call will be presented in the given style until this method is called again
    with a different style.
    
    @param style: Style to apply to future output
    @type style: L{AEOutput.Style}
    '''
    try:
      v = self.tts.getVoice(style.Voice)
      v = self.tts.setActiveVoice(v)
    except pyibmtts.ECIException:
      # just quit if we can't even get the voice to which style params should 
      # be applied
      return
    # restore voice independent parameters
    try:
      v.speed = style.Rate
    except pyibmtts.ECIException:
      pass
    try:
      v.volume = style.Volume
    except pyibmtts.ECIException:
      pass
    try:
      v.pitchBaseline = style.Pitch
    except pyibmtts.ECIException:
      pass
