'''
Defines a L{Perk} for the Firefox web browser.

Keys registered

CapsLock-A: Read the current web page address.

@todo: PP: fix skipped nodes in backwards browsing
@todo: PP: fix save pointer bug when menus are first activated

@todo: PP: skip invisible text chunks
@todo: PP: fix extra document focus when in-page component receives focus
@todo: PP: announce POR for restored document position
@todo: PP: clear out doc_pors when document events indicate a document no
  longer exists
@todo: PP: move last focus along with pointer so link tabbing is in sync
@todo: PP: improve nested list handling

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2007 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
# import useful modules for Perks
import Perk, Task
from POR import POR
from i18n import _

__uie__ = dict(kind='perk', tier='Minefield', all_tiers=False)

DOCUMENT_ROLE = 'document frame'

class FirefoxPerk(Perk.Perk):
  '''
  Defines hotkeys to improve document browsing and chrome access for the 
  Firefox web browser.
  
  @ivar block_next_select: Stop the next selection event from propogating
  @type block_next_select: boolean
  '''
  def init(self):
    # set an audio device as the default output
    self.setPerkIdealOutput('audio')
    
    self.registerTask(RestoreBrowser())
    self.registerTask(ExtraMenuFocus())
    self.registerTask(ExtraCarets())
    self.registerTask(RememberDocumentPOR(all=True))
    self.registerTask(ReadDocumentAddress('read document address'))
    kbd = self.getInputDevice(None, 'keyboard')
    self.addInputModifiers(kbd, kbd.AEK_CAPS_LOCK)
    self.registerCommand(kbd, 'read document address', False, 
                         [kbd.AEK_CAPS_LOCK, kbd.AEK_A])
    self.from_doc = False
    
class RestoreBrowser(Task.ViewTask):
  def executeGained(self, por, **kwargs):
    self.from_doc = False

class RememberDocumentPOR(Task.FocusTask):
  '''
  Tracks the last position reviewed in a document so that it may become the
  user's L{POR} again after focus is restored to the document.
  '''  
  def init(self):
    self.doc_pors = {}
  
  def executeGained(self, por, **kwargs):
    '''
    Check if focus is being restored on an HTML container or document.
    '''
    if self.perk.from_doc:
      # nav within a doc doesn't restore pointer
      self.perk.from_doc = False
      return
    doc = self.findAccByRole(DOCUMENT_ROLE, ancestor=True)
    if doc is None:
      # nav outside a doc doesn't restore pointer
      return
    try:
      # try to restore document position
      self.task_por = self.doc_pors[doc]
    except KeyError, e:
      return
    # consume the next focus
    #self.blockNTasks(1, Task.FocusTask, lambda gained, **kwargs: gained)
    # consume the next selector
    self.blockNTasks(1, Task.SelectorTask)
    # do our own selector on the restored POR
    self.doTask('read item details', por=self.task_por)
    return False
  
  def executeLost(self, por, **kwargs):
    '''
    Store the POR in an HTML container or document.
    '''
    if self.hasAccRole(DOCUMENT_ROLE, por):
      self.perk.from_doc = True
      self.doc_pors[por] = self.getPointerPOR()
      print 'stored:', self.getPointerPOR()
      return
    doc = self.findAccByRole(DOCUMENT_ROLE, ancestor=True)
    if doc is not None:
      self.perk.from_doc = True
      self.doc_pors[doc] = self.getPointerPOR()
      print 'stored:', self.getPointerPOR()

class ExtraCarets(Task.CaretTask):
  '''
  Ignores extra caret events sent after selection of in-page content that
  that is not editable.
  '''
  def execute(self, por, **kwargs):
    if not self.hasAccState('editable'):
      return False

class ExtraMenuFocus(Task.FocusTask):
  '''
  Ignores extra menu focus events for top level menus.
  '''
  def executeGained(self, por, **kwargs):
    parent = self.getParentAcc(por)
    if self.hasAccRole('menu bar', parent):
      self.blockNTasks(1, Task.SelectorTask)
      return False

class ReadDocumentAddress(Task.InputTask):
  '''
  Reads the address of the current document.
  '''
  def execute(self, **kwargs):
    '''
    Does a search of the accessible hierarchy for the location bar, then finds
    the entry field in the bar to announce its text.
    '''
    self.stopNow()
    root = self.getViewRootAcc()
    toolbar = self.getChildAcc(1, root)
    location = self.findAccByRole('autocomplete', toolbar, depth_first=False)
    text = self.findAccByRole('entry', location, depth_first=False)
    address = self.getNextItem(text)
    self.sayItem(address)
