/*
 *  Medusa
 *
 *  Copyright (C) 1999, 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 */

#include <config.h>

#include <fcntl.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <unistd.h>

#include "medusa-conf.h"
#include "medusa-master-db.h"
#include "medusa-lock-file.h"
#include "medusa-lock-file-paths.h"
#include "medusa-index-service-private.h"

/* Creates a totally new file index, and writes it over the old one */
static void             do_full_indexing                  (char *root_directory);
/* Looks for new and deleted files on the hard drive, and updates the current index */
static void             do_fast_reindexing                (void);
static void             do_index_accounting               (void);
/* Sits on the indexing socket and waits for requests to do either of the
   above indexings */
void             wait_for_indexing_signal          (void); 
/* Parses index receipt transmissions */
void             indexing_signal_received_callback (GIOChannel *indexing_messages,
						    GIOCondition condition,
						    gpointer data); 
/* Sets up index request socket */
int              initialize_socket                 (struct sockaddr_un *daemon_address); 
/* Copies the index files over to a different location 
   for quick reindexing so that the 
   current database is usable until the updated index is ready */
void             backup_index_files                (void);
/* Moves the ready updated/new index files
   to the place where are actually read */
void             copy_new_index_files              (void);


static gboolean holding_normal_lock = FALSE;
static gboolean holding_backup_lock = FALSE;

static void
release_locks (void)
{
  if (holding_normal_lock) {
        holding_normal_lock = FALSE;
	medusa_lock_file_release (MEDUSA_FILE_INDEX_LOCK_FILE);
  }

  if (holding_backup_lock) {
        holding_backup_lock = FALSE;
	medusa_lock_file_release (MEDUSA_FILE_INDEX_BACKUP_LOCK_FILE);
  }

}

/* Main index loop.  
   Creates three process to wait for signals, 
   do fast reindexing, and do slow reindexing */
int
main (int argc, char *argv[])
{

  atexit (release_locks);
  gnome_vfs_init ();

  if (argc > 1) {
    printf ("argument is %s\n", argv[1]);
    if (strcmp (argv[1], "account") == 0) {
      do_index_accounting ();
    }
    else {
      do_full_indexing (argv[1]);
    }
  }
  else {
    do_full_indexing (ROOT_DIRECTORY);
  }
  
  /* pid_t p_number; */



  /* p_number = fork ();

  if (p_number == 0) {
     Child process should listen for reindex requests 
    wait_for_indexing_signal ();
  }
  else {
    p_number = fork ();

    if (p_number == 0) {
      Next child process reindexes file names every n minutes 
      do_fast_reindexing (); 
    }
    else {
      Parent process indexes every time interval 

      do_full_indexing ();
    }
  }
*/
  exit (1);
  return 1;
}



void
wait_for_indexing_signal (void)
{
  struct sockaddr_un *daemon_address;
  int index_request_socket;
  GIOChannel *indexing_messages;
  
  /* FIXME: I haven't worried about security concerns for now.  The only
     thing that this can do is reindex the machine.  However,
     this may be a problem in multi-user configurations.  So there
     should be some access control model here. */
  daemon_address = g_new0 (struct sockaddr_un, 1);
  index_request_socket = initialize_socket (daemon_address);
  indexing_messages = g_io_channel_unix_new (index_request_socket);

  g_io_add_watch (indexing_messages,
		  G_IO_IN,
		  (GIOFunc) indexing_signal_received_callback,
		  NULL);

}

void
do_index_accounting ()
{
  MedusaMasterDB *master_db;
  
  unlink (URI_ACCOUNT_INDEX);
  unlink (DIRECTORY_NAME_ACCOUNT_INDEX);
  unlink (FILE_NAME_ACCOUNT_INDEX);
  unlink (MIME_TYPE_ACCOUNT_INDEX);
  unlink (FILE_SYSTEM_DB_ACCOUNT_INDEX);
  unlink (TEXT_INDEX_START_ACCOUNT_INDEX);
  unlink (TEXT_INDEX_LOCATION_ACCOUNT_INDEX);
  unlink (TEXT_INDEX_WORD_ACCOUNT_INDEX);
  unlink ( TEXT_INDEX_TEMP_ACCOUNT_INDEX);
  printf ("Removed old account files\n");
  
  master_db = medusa_master_db_new ("file://" ROOT_DIRECTORY,
				    MEDUSA_DB_LOG_TEXT_INDEX_DATA,
				      URI_ACCOUNT_INDEX,
				    FILE_SYSTEM_DB_ACCOUNT_INDEX,
				    FILE_NAME_ACCOUNT_INDEX,
				    DIRECTORY_NAME_ACCOUNT_INDEX,
				    MIME_TYPE_ACCOUNT_INDEX,
				    TEXT_INDEX_START_ACCOUNT_INDEX,
				    TEXT_INDEX_LOCATION_ACCOUNT_INDEX,
				    TEXT_INDEX_WORD_ACCOUNT_INDEX,
				    TEXT_INDEX_TEMP_ACCOUNT_INDEX);
  printf ("Initialized database\n");
#ifdef NOT_DEFINED
  medusa_master_db_set_accounting_on (master_db, INDEX_ACCOUNT_FILE);
#endif
  medusa_master_db_index (master_db);
  printf ("Finished indexing \n");
  
  medusa_master_db_unref (master_db);
  unlink (URI_ACCOUNT_INDEX);
  unlink (DIRECTORY_NAME_ACCOUNT_INDEX);
  unlink (FILE_NAME_ACCOUNT_INDEX);
  unlink (MIME_TYPE_ACCOUNT_INDEX);
  unlink (FILE_SYSTEM_DB_ACCOUNT_INDEX);
  unlink (TEXT_INDEX_START_ACCOUNT_INDEX);
  unlink (TEXT_INDEX_LOCATION_ACCOUNT_INDEX);
  unlink (TEXT_INDEX_WORD_ACCOUNT_INDEX);
  unlink ( TEXT_INDEX_TEMP_ACCOUNT_INDEX);
  printf ("Removed account files\n");
}  

		     


void
do_full_indexing (char *root_directory)
{
  MedusaMasterDB *master_db;
  char *root_uri;
  int i;
  char *temp_index_name;

  for (; ;) {
    /* Grab DB lock */
    
    if (! medusa_lock_file_acquire (MEDUSA_FILE_INDEX_BACKUP_LOCK_FILE)) {
      /* FIXME: give more precise error. */
      puts ("ERROR: failed to acquire lock file.");
	  exit (-1);
    }
    holding_backup_lock = TRUE;
    
    unlink ( FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX);
    unlink ( DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX);
    unlink ( FILE_NAME_HASH_NAME BACKUP_SUFFIX);
    unlink ( MIME_TYPE_HASH_NAME BACKUP_SUFFIX);
    unlink ( URI_LIST_NAME BACKUP_SUFFIX);
    unlink ( TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX);
    unlink ( TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX);
    unlink ( TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX);
    for (i = 0; i < NUMBER_OF_TEMP_INDEXES; i++) {
      temp_index_name = g_strdup_printf ("%s%s.%d", TEXT_INDEX_TEMP_FILE_NAME, BACKUP_SUFFIX, i); 
      unlink (temp_index_name);
      g_free (temp_index_name);
    }
    printf ("Removed old backup files\n");
    
    /* Create db lock */
    root_uri = g_strdup_printf ("%s%s", "file://", root_directory);
    if (getenv ("MEDUSA_INDEX_DEBUG") != NULL) {
      master_db = medusa_master_db_new (root_uri,
					MEDUSA_DB_LOG_EVERYTHING,
					URI_LIST_NAME BACKUP_SUFFIX,
					FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX,
					FILE_NAME_HASH_NAME BACKUP_SUFFIX,
					DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX,
					MIME_TYPE_HASH_NAME BACKUP_SUFFIX,
					TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX,
					TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX,
					TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX,
					TEXT_INDEX_TEMP_FILE_NAME BACKUP_SUFFIX);
					}
      else if (getenv ("MEDUSA_INDEX_LOG_ABBREVIATED") != NULL) {
      master_db = medusa_master_db_new (root_uri,
					MEDUSA_DB_LOG_ABBREVIATED,
					URI_LIST_NAME BACKUP_SUFFIX,
					FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX,
					FILE_NAME_HASH_NAME BACKUP_SUFFIX,
					DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX,
					MIME_TYPE_HASH_NAME BACKUP_SUFFIX,
					TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX,
					TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX,
					TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX,
					TEXT_INDEX_TEMP_FILE_NAME BACKUP_SUFFIX);
      }
      else if (getenv ("MEDUSA_INDEX_LOG_ERRORS") != NULL) {
	master_db = medusa_master_db_new (root_uri,
					  MEDUSA_DB_LOG_ABBREVIATED,
					  URI_LIST_NAME BACKUP_SUFFIX,
					  FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX,
					  FILE_NAME_HASH_NAME BACKUP_SUFFIX,
					  DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX,
					  MIME_TYPE_HASH_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_TEMP_FILE_NAME BACKUP_SUFFIX);
					  }
	else {
	master_db = medusa_master_db_new (root_uri,
					  MEDUSA_DB_LOG_NOTHING,
					  URI_LIST_NAME BACKUP_SUFFIX,
					  FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX,
					  FILE_NAME_HASH_NAME BACKUP_SUFFIX,
					  DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX,
					  MIME_TYPE_HASH_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX,
					  TEXT_INDEX_TEMP_FILE_NAME BACKUP_SUFFIX);
	}

    g_free (root_uri);
    printf ("Initialized database\n");
  /* It is faster to erase the old file and reindex, I think
       at this point */
    printf ("Indexing...\n");
    medusa_master_db_index (master_db);
    printf ("Finished indexing \n");
    
    medusa_master_db_unref (master_db);
    printf ("Wrote databases to disk\n");
    
    if (! medusa_lock_file_acquire (MEDUSA_FILE_INDEX_LOCK_FILE)) {
	  /* FIXME: give more precise error. */
	  puts ("ERROR: failed to acquire lock file.");
	  exit (-1);
	}

        holding_normal_lock = TRUE;
	copy_new_index_files ();
	printf ("Done moving files to correct locations\n");
	/* Remove db locks */
	medusa_lock_file_release (MEDUSA_FILE_INDEX_LOCK_FILE);
        holding_normal_lock = FALSE;
	medusa_lock_file_release (MEDUSA_FILE_INDEX_BACKUP_LOCK_FILE);
        holding_backup_lock = FALSE;
	printf ("Done releasing locks; going to sleep\n");
	sleep (MEDUSA_FILESYSTEM_INDEXING_INTERVAL);
	
  }
}

void
do_fast_reindexing (void)
{
  MedusaMasterDB *master_db;
  for (; ;) {
    sleep (MEDUSA_FILENAME_INDEXING_INTERVAL);
#ifdef INDEX_DEBUG
    g_message ("Looking for db lock to reindex file names\n");
#endif

    /* Wait for db lock */
       if (! medusa_lock_file_acquire (MEDUSA_FILE_INDEX_BACKUP_LOCK_FILE)) { 
	 /* FIXME: give more precise error. */
	 puts ("ERROR: failed to acquire backup lock file.");
	 exit (-1);
       }
    holding_backup_lock = TRUE;
    

    backup_index_files();

    
#ifdef INDEX_DEBUG
       g_message ("Got db lock to reindex file names\n");
#endif
       
       master_db = medusa_master_db_new ("file://" ROOT_DIRECTORY,
#ifdef INDEX_DEBUG
					 MEDUSA_DB_LOG_EVERYTHING
#elif defined(INDEX_LOG_SMALL)
					 MEDUSA_DB_LOG_ABBREVIATED,
#elif defined(INDEX_LOG_ERRORS)
					 MEDUSA_DB_LOG_ERRORS,
#else
					 MEDUSA_DB_LOG_NOTHING,
#endif
					 URI_LIST_NAME BACKUP_SUFFIX,
					 FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX,
					 FILE_NAME_HASH_NAME BACKUP_SUFFIX,
					 DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX,
					 MIME_TYPE_HASH_NAME BACKUP_SUFFIX,
					 TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX,
					 TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX,
					 TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX,
					 TEXT_INDEX_TEMP_FILE_NAME BACKUP_SUFFIX);
	medusa_master_db_update (master_db);
	medusa_master_db_unref (master_db);
	
	if (! medusa_lock_file_acquire (MEDUSA_FILE_INDEX_LOCK_FILE)) { 
	  holding_normal_lock = TRUE;
	  /* FIXME: give more precise error. */
	  puts ("ERROR: failed to acquire lock file.");
	  exit (-1);
	}
	
	copy_new_index_files (); 
	/* FIXME: should check for errors here */
	
	
	/* Remove db locks */
	holding_normal_lock = FALSE;
	medusa_lock_file_release (MEDUSA_FILE_INDEX_LOCK_FILE);
	holding_backup_lock = FALSE;
	medusa_lock_file_release (MEDUSA_FILE_INDEX_BACKUP_LOCK_FILE);
  }
  }

void
indexing_signal_received_callback (GIOChannel *indexing_messages,
				   GIOCondition condition,
				   gpointer data)
{
  char command_buffer[MAX_LINE];
  char *parse_location;
  GIOError error_code;
  int length_read;

  g_assert (indexing_messages != NULL); 
  /* We never call a callback unless there is something to read
     here. */
  g_assert (condition == G_IO_IN);
  
  error_code = g_io_channel_read (indexing_messages,
				  command_buffer,
				  MAX_LINE,
				  &length_read);
  g_return_if_fail (error_code == G_IO_ERROR_NONE);
  
  parse_location = command_buffer;
  while (parse_location != NULL) {
    if (strlen (parse_location) >= strlen (MEDUSA_FILE_REINDEX_REQUEST) &&
	strncmp (parse_location, MEDUSA_FILE_REINDEX_REQUEST, strlen (MEDUSA_FILE_REINDEX_REQUEST))) {
      do_fast_reindexing ();
    }
    if (strlen (parse_location) >= strlen (MEDUSA_TOTAL_REINDEX_REQUEST) &&
	strncmp (parse_location, MEDUSA_TOTAL_REINDEX_REQUEST, strlen (MEDUSA_TOTAL_REINDEX_REQUEST))) {
      do_full_indexing (ROOT_DIRECTORY);
    }
    parse_location = strchr (parse_location, '\n');
    parse_location++;
  }
}

int
initialize_socket (struct sockaddr_un *daemon_address)
{
  int index_listening_fd; 

  index_listening_fd = socket (AF_LOCAL, SOCK_STREAM, 0);
  g_return_val_if_fail (index_listening_fd != -1, -1);
  
  daemon_address->sun_family = AF_LOCAL; 
  /* FIXME:  This number (108) sucks, but it has no #define in the header.
     What to do? (POSIX requires 100 bytes at least, here)  */
  snprintf (daemon_address->sun_path, 100, "%s", INDEX_SOCKET_PATH);
  /* If socket currently exists, delete it 
     unlink (daemon_address->sun_path); */
#ifdef INDEX_DEBUG
  g_message  ("Creating socket at %s\n", daemon_address->sun_path);
#endif
  g_return_val_if_fail (bind (index_listening_fd, (struct sockaddr *) daemon_address, 
			      SUN_LEN (daemon_address)) != -1, 4);
  
  g_return_val_if_fail (listen (index_listening_fd, 5) != -1, 5);

  return index_listening_fd;
} 


 void
backup_index_files (void)
{
  /* FIXME: should not use cp!!!! */
  system ("/bin/cp -f "  URI_LIST_NAME " " 
	   URI_LIST_NAME BACKUP_SUFFIX);
  system ("/bin/cp -f " DIRECTORY_NAME_HASH_NAME " " 
	  DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX);
  system ("/bin/cp -f "  FILE_NAME_HASH_NAME " " 
	  FILE_NAME_HASH_NAME BACKUP_SUFFIX);

  system ("/bin/cp -f " FILE_SYSTEM_DB_FILE_NAME " " 
	  FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX);
  system ("/bin/cp -f "  MIME_TYPE_HASH_NAME " " 
	   MIME_TYPE_HASH_NAME BACKUP_SUFFIX);

  system ("/bin/cp -f "  TEXT_INDEX_START_FILE_NAME " " 
	   TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX);
  system ("/bin/cp -f "  TEXT_INDEX_LOCATION_FILE_NAME " " 
	   TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX);
  system ("/bin/cp -f "  TEXT_INDEX_WORD_FILE_NAME " " 
	   TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX);
  system ("/bin/cp -f "  TEXT_INDEX_TEMP_FILE_NAME " " 
	   TEXT_INDEX_TEMP_FILE_NAME BACKUP_SUFFIX);
  
}

void
copy_new_index_files (void)
{
  char *temp_index_name;
  int i;

  unlink (URI_LIST_NAME);
  rename (URI_LIST_NAME BACKUP_SUFFIX, 
	  URI_LIST_NAME);
  
  unlink (DIRECTORY_NAME_HASH_NAME);
  rename (DIRECTORY_NAME_HASH_NAME BACKUP_SUFFIX, 
	  DIRECTORY_NAME_HASH_NAME);
  
  unlink (FILE_NAME_HASH_NAME);
  rename (FILE_NAME_HASH_NAME BACKUP_SUFFIX, 
	  FILE_NAME_HASH_NAME);
  
  unlink (FILE_SYSTEM_DB_FILE_NAME);
  rename (FILE_SYSTEM_DB_FILE_NAME BACKUP_SUFFIX, 
	  FILE_SYSTEM_DB_FILE_NAME);

  unlink (MIME_TYPE_HASH_NAME);
  rename (MIME_TYPE_HASH_NAME BACKUP_SUFFIX, 
	  MIME_TYPE_HASH_NAME);
  
  unlink (TEXT_INDEX_START_FILE_NAME);
  rename (TEXT_INDEX_START_FILE_NAME BACKUP_SUFFIX,
	  TEXT_INDEX_START_FILE_NAME);

  unlink (TEXT_INDEX_LOCATION_FILE_NAME);
  rename (TEXT_INDEX_LOCATION_FILE_NAME BACKUP_SUFFIX,
	  TEXT_INDEX_LOCATION_FILE_NAME);

  unlink (TEXT_INDEX_WORD_FILE_NAME);
  rename (TEXT_INDEX_WORD_FILE_NAME BACKUP_SUFFIX,
	  TEXT_INDEX_WORD_FILE_NAME);
  
  /* We don't need to keep the temp files around */
  for (i = 0; i < NUMBER_OF_TEMP_INDEXES; i++) {
    temp_index_name = g_strdup_printf ("%s%s.%d", TEXT_INDEX_TEMP_FILE_NAME, BACKUP_SUFFIX, i); 
    unlink (temp_index_name);
    g_free (temp_index_name);
  }

}



