/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 1999, 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 */

#include <stdlib.h>
#include <stdio.h>

#include "medusa-conf.h"
#include "medusa-file-index.h"
#include "medusa-file-index-encoders.h"
#include "medusa-file-index-queries.h"
#include "medusa-file-information.h"
#include "medusa-search-uri.h"

#define MEDUSA_FILE_INDEX_VERSION    "0.1"
#define MEDUSA_FILE_URI_PREFIX "file://"

#define FILE_INDEX_METAINFO_LENGTH 100
/* Check that the field information in the header (just the titles) matches what we
   expected. */
static MedusaRDBFieldInfo *  file_index_field_info          (void);
static MedusaQueryClauses *  file_index_clauses             (void);
static void                  read_index_info_from_database  (MedusaFileSystemDB *file_system_db);
static void                  write_index_info_to_database   (MedusaFileSystemDB *file_system_db,
                                                             int last_indexing_timex);



MedusaFileSystemDB *
medusa_file_system_db_new (char *root_directory,
                           char *db_file_name,
                           MedusaHash *file_names,
                           MedusaHash *directory_names,
                           char *mime_type_hash)
{
        MedusaFileSystemDB *file_system_db;
        MedusaRDBFile *db_file;

        file_system_db = g_new0 (MedusaFileSystemDB, 1);

        db_file = medusa_rdb_file_new (db_file_name, FILE_INDEX_METAINFO_LENGTH);

        /* Create fields and reserved first record if the file is new */
        if (medusa_rdb_file_is_empty (db_file)) {
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_URI_NUMBER_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_URI_NUMBER_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_uri_number_encode, (MedusaRDBDecodeFunc) medusa_file_database_uri_number_decode);
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_MTIME_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_MTIME_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_mtime_encode,
                                           (MedusaRDBDecodeFunc) medusa_file_database_mtime_decode);
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_OWNER_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_owner_encode,
                                           (MedusaRDBDecodeFunc) medusa_file_database_owner_decode);
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_GROUP_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_GROUP_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_group_encode,
                                           (MedusaRDBDecodeFunc) medusa_file_database_group_decode);
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_PERMISSIONS_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_PERMISSIONS_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_permissions_encode,
                                           (MedusaRDBDecodeFunc) medusa_file_database_permissions_decode);
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_SIZE_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_SIZE_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_size_encode,
                                           (MedusaRDBDecodeFunc) medusa_file_database_size_decode);
                medusa_rdb_file_add_field (db_file,
                                           MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE,
                                           MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_SIZE,
                                           (MedusaRDBEncodeFunc) medusa_file_database_mime_type_encode,
                                           (MedusaRDBDecodeFunc) medusa_file_database_mime_type_decode);
                file_system_db->file_database = medusa_rdb_table_all_rows (db_file);

                write_index_info_to_database (file_system_db, 0);
                file_system_db->version = g_strdup (MEDUSA_FILE_INDEX_VERSION);
                
        }
        else {
                /* Load the correct field info from the hard coded procedure */
                /* FIXME:  Does this introduce a leak? */
                db_file->field_info = file_index_field_info ();
                file_system_db->file_database = medusa_rdb_table_all_rows (db_file);
                read_index_info_from_database (file_system_db);
                g_return_val_if_fail (file_system_db->version != NULL, NULL);
                
        }

        file_system_db->indexing_start_time = time (NULL);
        file_system_db->root_directory = g_strdup (root_directory);
        file_system_db->directory_names = directory_names;
        file_system_db->file_names = file_names;
        file_system_db->mime_types = medusa_hash_new (mime_type_hash, 
                                                      MIME_HASH_BITS);
 
        file_system_db->clauses = file_index_clauses (); 
        return file_system_db;
}

int                  
medusa_file_system_db_get_number_of_records (MedusaFileSystemDB *db)
{
        return medusa_rdb_table_get_number_of_records (db->file_database);
}

MedusaRDBFieldInfo *
medusa_file_system_db_get_field_info (MedusaFileSystemDB *db)
{
        return db->file_database->file->field_info;
}

void 
medusa_file_system_db_free (MedusaFileSystemDB *file_system_db)
{
        /* Write last indexing time information into the file */
        write_index_info_to_database (file_system_db,
                                      file_system_db->indexing_start_time);
        medusa_rdb_table_free (file_system_db->file_database);
        g_free (file_system_db->root_directory);
        g_free (file_system_db->version);
        medusa_hash_unref (file_system_db->directory_names);
        medusa_hash_unref (file_system_db->file_names);
        medusa_hash_unref (file_system_db->mime_types);
        g_free (file_system_db);
}






void
medusa_file_system_db_index_file (MedusaFileSystemDB *file_system_db, 
                                  int uri_number,
                                  GnomeVFSFileInfo *file_info)
{
        MedusaFileAttributes *file_attributes;
        file_attributes = medusa_file_attributes_new (uri_number, file_info);
        medusa_index_file_attributes (file_attributes, file_system_db);
        medusa_file_attributes_free (file_attributes);
}

static void
write_index_info_to_database (MedusaFileSystemDB *file_system_db, int last_index_time)
{
        char *metainfo; 

        metainfo = g_strdup_printf ("%s0%d0",MEDUSA_FILE_INDEX_VERSION,
                                    last_index_time);
        medusa_rdb_file_set_metainfo (file_system_db->file_database->file,
                                      metainfo);
        g_free (metainfo);

}

static void
read_index_info_from_database (MedusaFileSystemDB *file_system_db)
{
        char *metainfo, *version;
        
        metainfo = medusa_rdb_file_get_metainfo (file_system_db->file_database->file);
        version = g_new0 (char, file_system_db->file_database->file->metainfo_length);
        sscanf (metainfo, "%s0%d0", version, (int *) &file_system_db->last_index_time);
        file_system_db->version = g_strdup (version);
        g_free (metainfo);
        g_free (version);
}
                              


static MedusaRDBFieldInfo *
file_index_field_info (void)
{
        MedusaRDBFieldInfo *db_field_info;
        db_field_info = medusa_rdb_field_info_new ();
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_URI_NUMBER_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_URI_NUMBER_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_uri_number_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_uri_number_decode);
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_MTIME_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_MTIME_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_mtime_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_mtime_decode);
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_OWNER_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_owner_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_owner_decode);
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_GROUP_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_GROUP_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_group_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_group_decode);
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_PERMISSIONS_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_PERMISSIONS_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_permissions_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_permissions_decode);
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_SIZE_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_SIZE_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_size_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_size_decode);
        medusa_rdb_field_add (db_field_info,
                              MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE,
                              MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_SIZE,
                              (MedusaRDBEncodeFunc) medusa_file_database_mime_type_encode,
                              (MedusaRDBDecodeFunc) medusa_file_database_mime_type_decode);

                                              
        return db_field_info;
}

static MedusaQueryClauses *
file_index_clauses (void)
{
        MedusaQueryClauses *clauses;

        clauses = medusa_query_clauses_new ();
        medusa_query_clauses_add_clause (clauses,
                                         "file_type",
                                         "is",
                                         (MedusaQueryFunc) medusa_file_index_is_of_type,
                                         MEDUSA_ARGUMENT_TYPE_STRING);
        medusa_query_clauses_add_clause (clauses,
                                         "mod_time",
                                         "updated",
                                         (MedusaQueryFunc) medusa_file_index_is_modified_after,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        
        medusa_query_clauses_add_clause (clauses,
                                         "mod_time",
                                         "not_updated",
                                         (MedusaQueryFunc) medusa_file_index_is_modified_before,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "owner",
                                         "is",
                                         (MedusaQueryFunc) medusa_file_index_is_owned_by,
                                         MEDUSA_ARGUMENT_TYPE_STRING);
        medusa_query_clauses_add_clause (clauses,
                                         "owner",
                                         "has_uid",
                                         (MedusaQueryFunc) medusa_file_index_has_uid,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "owner",
                                         "is_not",
                                         (MedusaQueryFunc) medusa_file_index_is_not_owned_by,
                                         MEDUSA_ARGUMENT_TYPE_STRING);
        medusa_query_clauses_add_clause (clauses,
                                         "owner",
                                         "does_not_have_uid",
                                         (MedusaQueryFunc) medusa_file_index_does_not_have_uid,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "group",
                                         "is",
                                         (MedusaQueryFunc) medusa_file_index_is_in_group,
                                         MEDUSA_ARGUMENT_TYPE_STRING);
        medusa_query_clauses_add_clause (clauses,
                                         "group",
                                         "has_gid",
                                         (MedusaQueryFunc) medusa_file_index_has_gid,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "group",
                                         "is_not",
                                         (MedusaQueryFunc) medusa_file_index_is_not_in_group,
                                         MEDUSA_ARGUMENT_TYPE_STRING);
        medusa_query_clauses_add_clause (clauses,
                                         "group",
                                         "does_not_have_uid",
                                         (MedusaQueryFunc) medusa_file_index_does_not_have_gid,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "size",
                                         "larger_than",
                                         (MedusaQueryFunc) medusa_file_index_is_larger_than,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "size",
                                         "smaller_than",
                                         (MedusaQueryFunc) medusa_file_index_is_smaller_than,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "size",
                                         "is",
                                         (MedusaQueryFunc) medusa_file_index_is_of_size,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        medusa_query_clauses_add_clause (clauses,
                                         "permissions_to_read",
                                         "include_uid",
                                         (MedusaQueryFunc) medusa_file_index_uid_can_read_file,
                                         MEDUSA_ARGUMENT_TYPE_INTEGER);
        return clauses;
        
                                                   
        
}


MedusaRDBRecordNumbers *
medusa_file_system_db_get_first_real_record (MedusaFileSystemDB *db)
{
        return db->file_database->first_record;
}

MedusaQueryClauses *
medusa_file_system_db_get_query_clauses (MedusaFileSystemDB *db)
{
  return db->clauses;
}

