/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Maciej Stachowiak <mjs@eazel.com>
 *
 *  medusa-lock-file.c -   Utility functions for managing lock files
 */

#include <config.h>

#include <medusa-lock-file.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <glib.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

/* Function for creating lock files that works properly, even on
   strange file systems like NFS. */

gboolean
medusa_lock_file_acquire (const char *file_name) 
{
        char *unique_file_name;
        char hostname [1025];
        struct stat statbuf;
        
        if (gethostname (hostname, 1024) != 0) {
                return FALSE;
        }
        
        /* Create unique file (make sure we don't try to acquire this lock
           more than once from the same process! */
        
        unique_file_name = g_strdup_printf ("%s-%s-%d", file_name, hostname, getpid ());
        
        unlink (unique_file_name);
        
        if (creat (unique_file_name, S_IREAD | S_IWRITE) != 0) {
                if (stat (unique_file_name, &statbuf) != 0) {
                        g_free (unique_file_name);
                        return FALSE;
                }
        }
        
        /* Link it to the real lock file name. */
        
        while (1) {
                if (link (unique_file_name, file_name) == 0) {
                        break;
                }
                
                if (stat (unique_file_name, &statbuf) == 0 && statbuf.st_nlink == 2) {
                        break;
                }
                
                /* FIXME: is this a reasonable interval? */
                sleep (1);
        }
        
        unlink (unique_file_name);
        
        g_free (unique_file_name);
        
        return TRUE;
}


void
medusa_lock_file_release (const char *file_name) 
{
        unlink (file_name);
}
