/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 */

#define SEARCH_URI_DEBUG

#include <glib.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "medusa-master-db.h"
#include "medusa-master-db-private.h"
#include "medusa-uri-list.h"
#include "medusa-search-uri.h"
#include "medusa-utils.h"




static char *                get_next_root_uri           (char **string);
static char **               get_search_criteria         (char *string);
static MedusaClauseClosure * clause_to_clause_closure    (char *clause,                       
                                                          MedusaMasterDB *master_db);


GList *
medusa_search_uri_to_clause_closures (char *search_uri,
                                      MedusaMasterDB *master_db)
{
        char *location;
        char *root_uri;
        char **clauses;
        GList *clause_info;
        int i;

        clause_info = NULL;
        g_return_val_if_fail (medusa_uri_is_search_uri (search_uri), NULL);
  
        location = strchr (search_uri, ':');
        location++;
        
        root_uri = get_next_root_uri (&location);
        /* We shouldn't be getting any other kinds of uri's except
           for a single root with root = "file:///" */
        g_return_val_if_fail (root_uri != NULL, NULL);
        g_return_val_if_fail (strcmp (root_uri, "file:///") == 0, NULL);
        g_return_val_if_fail (get_next_root_uri (&location) == NULL, NULL);

        location = strchr (search_uri, ']');
        location++;

        /* For now we don't except or queries here */
        g_return_val_if_fail (strchr (location, '|') == NULL, NULL);

        clauses = get_search_criteria (location);
        g_return_val_if_fail (clauses != NULL, NULL);
        
        for (i=0; clauses[i] != NULL; i++) {
                clause_info = g_list_prepend (clause_info,
                                              clause_to_clause_closure (clauses[i],
                                                                        master_db));
        }
        return clause_info;
}


static MedusaClauseClosure *
clause_to_clause_closure (char *clause,
                          MedusaMasterDB *master_db)
{
        MedusaQueryFunc evaluate;
        MedusaClauseClosure *closure;
        MedusaArgumentType type;
        char *last_word;


        closure = g_new0 (MedusaClauseClosure, 1);

        last_word = &clause[strlen(clause) - 1];
        while (*last_word != ' ') {
                last_word--;
        }
        last_word++;
        evaluate = medusa_query_clauses_get_function (medusa_uri_list_get_query_clauses (master_db->uri_list),
                                                      clause,
                                                      &type);
        if (evaluate != NULL) {
                closure->query_func = evaluate;
                closure->uri_list = master_db->uri_list;
                closure->file_system_db = NULL;
                switch (type) {
                case MEDUSA_ARGUMENT_TYPE_STRING:
                        closure->argument = g_strdup (last_word);
                        break;
                case MEDUSA_ARGUMENT_TYPE_INTEGER:
                        closure->argument = GINT_TO_POINTER(strtol (last_word, NULL, 10));
                        break;
                default:
                        g_assert_not_reached ();
                        return NULL;
                }
                closure->is_content_request = FALSE;
                return closure;
        }
        
        evaluate = medusa_query_clauses_get_function (medusa_file_system_db_get_query_clauses (master_db->file_system_db),
                                                      clause,
                                                      &type);
        if (evaluate != NULL) {
                closure->query_func = evaluate;
                closure->file_system_db = master_db->file_system_db;
                closure->uri_list = NULL;
                switch (type) {
                case MEDUSA_ARGUMENT_TYPE_STRING:
                        closure->argument = g_strdup (last_word);
                        break;
                case MEDUSA_ARGUMENT_TYPE_INTEGER:
                        closure->argument = GINT_TO_POINTER(strtol (last_word, NULL, 10));
                        break;
                }
                closure->is_content_request = FALSE;
                return closure;
        }
        if (medusa_string_has_prefix (clause, "content includes")) {
                        closure->is_content_request = TRUE;
                        closure->content_request = g_strdup (last_word);
                        closure->inclusion_query = TRUE;
                        return closure;
        }

        if (medusa_string_has_prefix (clause, "content does_not_include")) {
                        closure->is_content_request = TRUE;
                        closure->content_request = g_strdup (last_word);
                        closure->inclusion_query = FALSE;
                        return closure;
        }
        return NULL;
}

gboolean              
medusa_search_uri_contains_content_requests (char *search_uri)
{
        char *content_include, *content_does_not_include, *file_type_is;
        content_include = strstr (search_uri,"content includes ");
        content_does_not_include = strstr (search_uri,"content does_not_include ");
        file_type_is = strstr (search_uri, "file_type is ");
        return ((content_include != NULL) || (content_does_not_include != NULL)
                || (file_type_is != NULL));
}

char *                
medusa_search_uri_add_user_can_read (char *search_uri, int uid)
{
        char *new_criterion;
        char *extended_search_uri;

        new_criterion = g_strdup_printf ("permissions_to_read include_uid %d", uid);
        if (strstr (search_uri, new_criterion)) {
                g_free (new_criterion);
                return g_strdup (search_uri);
        }
        else {
                extended_search_uri = g_strdup_printf ("%s & %s", search_uri, new_criterion);
                g_free (new_criterion);
                return extended_search_uri;
        }
}

void                  
medusa_clause_closure_free (MedusaClauseClosure *closure)
{
        if (closure->is_content_request == TRUE) {
                g_free (closure->content_request);
        }
        
        g_free (closure);
}

gboolean              
medusa_clause_closure_is_content_search (gpointer data,
                                         gpointer user_data)
{
        MedusaClauseClosure *clause_closure;

        clause_closure = (MedusaClauseClosure *) data;
        g_return_val_if_fail (clause_closure != NULL, FALSE);

        g_assert (clause_closure->is_content_request == TRUE ||
                  clause_closure->is_content_request == FALSE);

        return clause_closure->is_content_request;
}

gboolean
medusa_uri_is_search_uri (char *uri)
{
        char *trimmed_uri;
        gboolean result;

        /* Remove leading spaces */
        trimmed_uri = g_strchug (uri);
        result = FALSE;
        /* FIXME:  Perhaps these prefixes should 
           be stored somewhere? */
        if (strncmp (trimmed_uri, "gnome-search:", strlen ("gnome-search:")) ==  0) {
                result = TRUE;
        }
        if (strncmp (trimmed_uri, "search:", strlen ("search")) == 0) {
                result = TRUE;
        }
        if (strncmp (trimmed_uri, "medusa:", strlen ("medusa:")) == 0) {
                result = TRUE;
        }
        return result;
}

/* Gets the next bracket enclosed segment.
   does no checking for uri validity */
static char *
get_next_root_uri (char **string)
{
        char *close_bracket;
        char *uri;
        printf ("Trying to parse string %s\n", *string);
        g_return_val_if_fail (*string != NULL, NULL);
        /* Return if there are no more uri's */
        if (*string[0] != '[') {
                return NULL;
        }
        close_bracket = strchr (*string, ']');
        g_return_val_if_fail (close_bracket, NULL);
        
        uri = g_new0 (char, (int) (close_bracket - *string));
        strncpy (uri, *string + 1, (int) (close_bracket - *string - 1));
#ifdef SEARCH_URI_DEBUG
        printf ("Next root uri is %s\n", uri);
#endif
        *string = close_bracket + 1;
        return uri;
}

static char **
get_search_criteria (char *string)
{
        g_return_val_if_fail (string != NULL, NULL);
        return g_strsplit (string, " & ", strlen (string));
}

