/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 * 
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors: Rebecca Schulman <rebecka@eazel.com>
 *  
 */


/* medusa-index-service.c -- API for externals users of the medusa
   indexing service. */

#include <config.h>
#include "medusa-index-service.h"
#include "medusa-index-service-private.h"
#include "medusa-log.h"
#include "medusa-service-private.h"


#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>



static time_t        get_last_index_update_time_by_hack                 (void);

static char *
open_index_socket_and_send_message (const char *message)
{
        int index_socket;
        int read_length;
        char *request;        
        char response[512];

        index_socket = medusa_initialize_socket_for_requests (INDEX_SOCKET_PATH);
        if (index_socket == -1) {
                medusa_log_error ("Couldn't open reindexing service socket");
                return NULL;
        }

        request = g_strdup (message);
        write (index_socket, request, strlen (request));
        g_free (request);
        
        read_length = 1;
        memset (response, 0, 512);
        read_length = read (index_socket, response, 512 * sizeof (char));
        response[511] = 0;

        close (index_socket);

        if (read_length <= 0) {
                return NULL;
        }
        return g_strdup (response);
}


MedusaIndexingRequestResult 
medusa_index_service_request_reindex (void)
{

        char *response;
        MedusaIndexingRequestResult reindex_result;

        response = open_index_socket_and_send_message (MEDUSA_REINDEX_REQUEST);

        if (response == NULL) {
                reindex_result = MEDUSA_INDEXER_ERROR_NO_RESPONSE;
        }
        else if (strstr (response, MEDUSA_REINDEX_REQUEST_ACCEPTED)) {
                reindex_result = MEDUSA_INDEXER_REQUEST_OK;
        }
        else if (strstr (response, MEDUSA_REINDEX_REQUEST_DENIED_BUSY)) {
                reindex_result = MEDUSA_INDEXER_ERROR_BUSY;
        }
        else {
                reindex_result = MEDUSA_INDEXER_ERROR_INVALID_RESPONSE;
        }

        g_free (response);
        return reindex_result;
}

MedusaIndexingStatus
medusa_index_service_is_available (void)
{
        char *response;
        MedusaIndexingRequestResult service_availability;
        
        response = open_index_socket_and_send_message (MEDUSA_INDEX_SERVICE_AVAILABILITY_REQUEST);
        
        if (response == NULL) {
                service_availability = MEDUSA_INDEXER_NOT_AVAILABLE;
        }
        else if (strstr (response, MEDUSA_INDEX_SERVICE_AVAILABILITY_UP)) {
                service_availability = MEDUSA_INDEXER_READY_TO_INDEX;
        }
        else if (strstr (response, MEDUSA_INDEX_SERVICE_AVAILABILITY_BUSY)) {
                service_availability = MEDUSA_INDEXER_ALREADY_INDEXING;
        }
        else {
                service_availability = MEDUSA_INDEXER_NOT_AVAILABLE;
        }

        g_free (response);
        return service_availability;
}

time_t                      
medusa_index_service_get_last_index_update_time  (void)
{
        return get_last_index_update_time_by_hack ();
}


/* FIXME bugzilla.eazel.com 4883: this is a bad hack */
static time_t
get_last_index_update_time_by_hack (void)
{
        struct stat index_file_info;
        int stat_return_code;
        char *location_file_name; 

        location_file_name = g_strdup_printf ("%s/%s", MEDUSA_LOCALSTATEDIR,
				       "text-index-location-file");
        stat_return_code = stat (location_file_name, &index_file_info);
        if (stat_return_code != 0) {
                return 0;
        }
        g_free (location_file_name);

        return index_file_info.st_mtime;
}
