/* packed-treeview.c
 *
 * Copyright (C) 2001 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "packed-treeview.h"
#include "marshal.h"

static void packed_treeview_class_init (PackedTreeViewClass * class);
static void packed_treeview_init (PackedTreeView * ptv);
static void packed_treeview_finalize (GObject   * object);

enum
{
	SWAP_REQUIRED,
	LAST_SIGNAL
};

struct _PackedTreeViewPrivate
{
	GtkWidget  *treeview;
	GtkWidget  *arrows_box; 
	GtkWidget  *arrow_up;
	GtkWidget  *arrow_down;	
	gboolean    right_buttons;
};

static gint packed_treeview_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
packed_treeview_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (PackedTreeViewClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) packed_treeview_class_init,
			NULL,
			NULL,
			sizeof (PackedTreeView),
			0,
			(GInstanceInitFunc) packed_treeview_init
		};		

		type = g_type_register_static (GTK_TYPE_HBOX, "PackedTreeView", &info, 0);
	}
	return type;
}

static void
packed_treeview_class_init (PackedTreeViewClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	packed_treeview_signals[SWAP_REQUIRED] =
		g_signal_new ("swap_required",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (PackedTreeViewClass, swap_required),
			      NULL, NULL,
			      marshal_VOID__POINTER_POINTER, G_TYPE_NONE, 2,
			      G_TYPE_POINTER, G_TYPE_POINTER);

	class->swap_required = NULL;
	object_class->finalize = packed_treeview_finalize;
}

static void
packed_treeview_init (PackedTreeView * ptv)
{
	ptv->priv = g_new0 (PackedTreeViewPrivate, 1);
	ptv->priv->treeview = NULL;
	ptv->priv->arrows_box = NULL;
	ptv->priv->arrow_up = NULL;
	ptv->priv->arrow_down = NULL;
	ptv->priv->right_buttons = FALSE;
}

/* filalize the widget setup */
static void packed_treeview_post_init (PackedTreeView * ptv);

GtkWidget *
packed_treeview_new (GtkWidget *treeview, gboolean buttons_go_right)
{
	GObject   *obj;
	PackedTreeView *ptv;

	g_return_val_if_fail (treeview && GTK_IS_TREE_VIEW (treeview), NULL);

	obj = g_object_new (PACKED_TREEVIEW_TYPE, NULL);
	ptv = PACKED_TREEVIEW (obj);

	ptv->priv->treeview = treeview;
	ptv->priv->right_buttons = buttons_go_right;

	/* building the actual interface */
	packed_treeview_post_init (ptv);

	return GTK_WIDGET (obj);
}

static void
packed_treeview_finalize (GObject   * object)
{
	PackedTreeView *ptv;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_PACKED_TREEVIEW (object));

	ptv = PACKED_TREEVIEW (object);
	if (ptv->priv) {
		g_free (ptv->priv);
		ptv->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}


static void up_button_click_cb (GtkWidget *button, PackedTreeView * ptv);
static void dn_button_click_cb (GtkWidget *button, PackedTreeView * ptv);
static void select_changed_cb (GtkTreeSelection *select, PackedTreeView * ptv);
static void 
packed_treeview_post_init (PackedTreeView * ptv)
{
	GtkTreeSelection *select;
	GtkWidget *bb, *button, *arrow, *sw;
	
	/* arrows button box */
	bb = gtk_vbutton_box_new ();
	gtk_button_box_set_child_size (GTK_BUTTON_BOX (bb), 15, 15);
	gtk_container_set_border_width (GTK_CONTAINER (bb), GNOME_PAD);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	if (ptv->priv->right_buttons)
		gtk_box_pack_end (GTK_BOX (ptv), bb, FALSE, TRUE, 0);
	else
		gtk_box_pack_start (GTK_BOX (ptv), bb, FALSE, TRUE, 0);
	ptv->priv->arrows_box = bb;

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (up_button_click_cb), ptv);
	arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);
	ptv->priv->arrow_up = button;

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (dn_button_click_cb), ptv);
	arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);
	ptv->priv->arrow_down = button;

	gtk_widget_show_all (bb);

	/* treeview in ist scrolled window */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (sw), ptv->priv->treeview);

	gtk_box_pack_start (GTK_BOX (ptv), sw, TRUE, TRUE, 0);
	gtk_widget_show_all (sw);

	/* selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptv->priv->treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
			  G_CALLBACK (select_changed_cb), ptv);

	/* initial sensitiveness of the arrow buttons */
	gtk_widget_set_sensitive (ptv->priv->arrow_up, FALSE);
	gtk_widget_set_sensitive (ptv->priv->arrow_down, FALSE);
}

static void require_move(PackedTreeView * ptv, gint offset);
static void 
up_button_click_cb (GtkWidget *button, PackedTreeView * ptv)
{
	require_move (ptv, -1);
}

static void 
dn_button_click_cb (GtkWidget *button, PackedTreeView * ptv)
{
	require_move (ptv, 1);
}

static void 
packed_treeview_refresh_arrows (PackedTreeView *ptv);
static void
require_move(PackedTreeView * ptv, gint offset)
{
	GtkTreeIter iter1, iter2;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	g_return_if_fail (offset);

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptv->priv->treeview));
	if (gtk_tree_selection_get_selected (select, &model, &iter1)) {
		GtkTreePath *path;
		gint i;
		gboolean error = FALSE;

		path = gtk_tree_model_get_path (model, &iter1);
		i = 0;
		while ((i != offset) && !error) {
			if (offset > 0) {
				gtk_tree_path_next (path);
				i++;
			}
			else {
				error = !gtk_tree_path_prev (path);
				i--;
			}
		}
		
		if (!error)
			error = !gtk_tree_model_get_iter (model, &iter2, path);
		gtk_tree_path_free (path);

		if (error) {
			/* FIXME: emit a warning */
		}
		else {
#ifdef debug_signal
			g_print (">> 'SWAP_REQUIRED' from require_move\n");
#endif
			g_signal_emit (G_OBJECT (ptv), packed_treeview_signals[SWAP_REQUIRED], 0,
				       &iter1, &iter2);
#ifdef debug_signal
			g_print ("<< 'SWAP_REQUIRED' from require_move\n");
#endif
		}
		packed_treeview_refresh_arrows (ptv);
	}
	else {
		return;
	}
}

static void 
select_changed_cb (GtkTreeSelection *select, PackedTreeView *ptv)
{
	packed_treeview_refresh_arrows (ptv);
}

static void 
packed_treeview_refresh_arrows (PackedTreeView *ptv)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptv->priv->treeview));
	if (gtk_tree_selection_get_selected (select, &model, &iter)) {
		GtkTreePath *path;
		gboolean okup=FALSE, okdn = FALSE;

		/* up button */
		path = gtk_tree_model_get_path (model, &iter);
		if (gtk_tree_path_prev (path))
			okup = TRUE;
		gtk_tree_path_free (path);

		/* down button */
		if (gtk_tree_model_iter_next (model, &iter))
			okdn = TRUE;
	
		gtk_widget_set_sensitive (ptv->priv->arrow_up, okup);
 		gtk_widget_set_sensitive (ptv->priv->arrow_down, okdn);
	}
	else {
		gtk_widget_set_sensitive (ptv->priv->arrow_up, FALSE);
 		gtk_widget_set_sensitive (ptv->priv->arrow_down, FALSE);
	}
}

void
packed_treeview_set_show_arrows (PackedTreeView *ptv, gboolean show)
{
	g_return_if_fail (ptv && IS_PACKED_TREEVIEW (ptv));

	if (show)
		gtk_widget_show (ptv->priv->arrows_box);
	else
		gtk_widget_hide (ptv->priv->arrows_box);
}
