/* mg-data-cell-renderer-combo.c
 * Copyright (C) 2000  Red Hat, Inc.,  Jonathan Blandford <jrb@redhat.com>
 * Copyright (C) 2003  Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <libgda/libgda.h>
#include "mg-data-cell-renderer-combo.h"
#include <libmergeant/marshal.h>
#include <libmergeant/mg-conf.h>
#include <libmergeant/mg-server.h>
#include <libmergeant/mg-data-entry.h>
#include <libmergeant/mg-data-handler.h>
#include <libmergeant/mg-server-data-type.h>
#include <libmergeant/utility.h>
#include <libmergeant/mg-query.h>
#include <libmergeant/mg-parameter.h>
#include <libmergeant/mg-entity.h>
#include <libmergeant/mg-field.h>
#include <libmergeant/mg-result-set.h>
#include <gdk/gdkkeysyms.h>

static void mg_data_cell_renderer_combo_init       (MgDataCellRendererCombo      *celltext);
static void mg_data_cell_renderer_combo_class_init (MgDataCellRendererComboClass *class);
static void mg_data_cell_renderer_combo_dispose    (GObject                  *object);
static void mg_data_cell_renderer_combo_finalize   (GObject                  *object);

static void mg_data_cell_renderer_combo_get_property  (GObject                  *object,
				       guint                     param_id,
				       GValue                   *value,
				       GParamSpec               *pspec);
static void mg_data_cell_renderer_combo_set_property  (GObject                  *object,
				       guint                     param_id,
				       const GValue             *value,
				       GParamSpec               *pspec);
static void mg_data_cell_renderer_combo_get_size   (GtkCellRenderer          *cell,
						      GtkWidget                *widget,
						      GdkRectangle             *cell_area,
						      gint                     *x_offset,
						      gint                     *y_offset,
						      gint                     *width,
						      gint                     *height);
static void mg_data_cell_renderer_combo_render     (GtkCellRenderer          *cell,
						      GdkWindow                *window,
						      GtkWidget                *widget,
						      GdkRectangle             *background_area,
						      GdkRectangle             *cell_area,
						      GdkRectangle             *expose_area,
						      GtkCellRendererState      flags);

static gboolean mg_data_cell_renderer_combo_activate  (GtkCellRenderer            *cell,
						       GdkEvent                   *event,
						       GtkWidget                  *widget,
						       const gchar                *path,
						       GdkRectangle               *background_area,
						       GdkRectangle               *cell_area,
						       GtkCellRendererState        flags);

static void dependency_param_changed_cb (MgParameter *param, MgDataCellRendererCombo *datacell);
static void clean_values_pk (MgDataCellRendererCombo *datacell);
static GdaValue *prepare_signal_changed_value (MgDataCellRendererCombo *datacell, gint row);

enum {
	CHANGED,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_VALUES,
	PROP_VALUES_COMPLETE,
	PROP_VALUE_ATTRIBUTES,
	PROP_TO_BE_DELETED,
	PROP_SET_DEFAULT_IF_INVALID
};

enum {
	ROW_NULL_VALUE = -1, /* don't change the ids... */
	ROW_NA         = -2,
	ROW_ERROR      = -3
};

#define SYMBOL_WIDTH 12


/* Note:
 * the current_row is a curson indicating the currently selected row in the ccore->resultset.
 * The negative values have the following meaning:
 * -1: a NULL value: ROW_NULL_VALUE
 * -2: no link with the recordset has been made, the information is not available: ROW_NA
 * -3: an error: ROW_ERROR
 */
struct _MgDataCellRendererComboPrivate
{
	ComboCore *ccore;
	GList     *strings; /* list of strings for the possible values to be chosen among */
	guint      attributes;
	gboolean   to_be_deleted;
	gboolean   set_default_if_invalid;
	gboolean   emit_pk_values_only;

	/* keeping track of the current displayed values */
	gint       current_row;
	GList     *values_pk; /* list of PK fields GdaValue */

	/* window which pops up when the cell renderer is activated */
	GtkWidget *popup_window;
	GtkWidget *treeview;
	gchar     *current_path;
};

typedef struct 
{
	/* text renderer */
	gulong focus_out_id;
} MgDataCellRendererComboInfo;
#define MG_DATA_CELL_RENDERER_COMBO_INFO_KEY "mg_data_cell_renderer_combo_info_key"



static GObjectClass *parent_class = NULL;
static guint text_cell_renderer_combo_signals [LAST_SIGNAL];

#define MG_DATA_CELL_RENDERER_COMBO_PATH "mg_data_cell_renderer_combo_path"

GType
mg_data_cell_renderer_combo_get_type (void)
{
	static GType cell_text_type = 0;

	if (!cell_text_type) {
		static const GTypeInfo cell_text_info =	{
			sizeof (MgDataCellRendererComboClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) mg_data_cell_renderer_combo_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (MgDataCellRendererCombo),
			0,              /* n_preallocs */
			(GInstanceInitFunc) mg_data_cell_renderer_combo_init,
		};
		
		cell_text_type =
			g_type_register_static (GTK_TYPE_CELL_RENDERER_TEXT, "MgDataCellRendererCombo",
						&cell_text_info, 0);
	}

	return cell_text_type;
}


/* this function insures that the mode of the cell renderer is GTK_CELL_RENDERER_MODE_ACTIVATABLE
 * because this mode is changed by the GtkCellRendererText class which we inherits, but the mode we are
 * using the GTK_CELL_RENDERER_MODE_ACTIVATABLE.
 */
static void repair_cell_mode (GtkCellRenderer *renderer, GParamSpec *pspec, gpointer data)
{
	renderer->mode = GTK_CELL_RENDERER_MODE_ACTIVATABLE;
}

static void
mg_data_cell_renderer_combo_init (MgDataCellRendererCombo *datacell)
{
	GTK_CELL_RENDERER (datacell)->mode = GTK_CELL_RENDERER_MODE_ACTIVATABLE;
	datacell->priv = g_new0 (MgDataCellRendererComboPrivate, 1);
	datacell->priv->ccore = NULL;
	datacell->priv->attributes = 0;
	datacell->priv->current_row = ROW_NA;
	datacell->priv->values_pk = NULL;
	datacell->priv->strings = NULL;
	datacell->priv->popup_window = NULL;
	datacell->priv->treeview = NULL;
	datacell->priv->current_path = NULL;
	datacell->priv->set_default_if_invalid = FALSE;
	datacell->priv->emit_pk_values_only = FALSE;
	g_signal_connect (G_OBJECT (datacell), "notify",
			  G_CALLBACK (repair_cell_mode), NULL);
}

static void
mg_data_cell_renderer_combo_class_init (MgDataCellRendererComboClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

    	object_class->dispose = mg_data_cell_renderer_combo_dispose;
	object_class->finalize = mg_data_cell_renderer_combo_finalize;

	object_class->get_property = mg_data_cell_renderer_combo_get_property;
	object_class->set_property = mg_data_cell_renderer_combo_set_property;

	cell_class->get_size = mg_data_cell_renderer_combo_get_size;
	cell_class->render = mg_data_cell_renderer_combo_render;
	cell_class->activate = mg_data_cell_renderer_combo_activate;
  
	g_object_class_install_property (object_class,
					 PROP_VALUES,
					 g_param_spec_pointer ("values",
							       _("Values limited to PK fields"),
							       _("GList of GdaValue to render, limited to PK fields"),
							       G_PARAM_WRITABLE));

	g_object_class_install_property (object_class,
					 PROP_VALUES_COMPLETE,
					 g_param_spec_pointer ("values_complete",
							       _("Values"),
							       _("GList of GdaValue to render, not limited to PK fields "),
							       G_PARAM_WRITABLE));
  
	g_object_class_install_property (object_class,
					 PROP_VALUE_ATTRIBUTES,
					 g_param_spec_uint ("value_attributes", NULL, NULL,
                                                            0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_TO_BE_DELETED,
					 g_param_spec_boolean ("to_be_deleted", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));

        g_object_class_install_property (object_class, PROP_SET_DEFAULT_IF_INVALID,
					 g_param_spec_boolean ("set_default_if_invalid", NULL, NULL, FALSE,
                                                               (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	  
	text_cell_renderer_combo_signals [CHANGED] =
		g_signal_new ("changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (MgDataCellRendererComboClass, changed),
			      NULL, NULL,
			      marshal_VOID__STRING_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_POINTER);

}

static void
mg_data_cell_renderer_combo_dispose (GObject *object)
{
	MgDataCellRendererCombo *datacell = MG_DATA_CELL_RENDERER_COMBO (object);

	if (datacell->priv->popup_window) {
		gtk_widget_destroy (datacell->priv->popup_window);
		datacell->priv->popup_window = NULL;
	}

	if (datacell->priv->ccore) {
		utility_combo_free_core (datacell->priv->ccore);
		datacell->priv->ccore = NULL;
	}

	clean_values_pk (datacell);

	/* parent class */
	parent_class->dispose (object);
}

static void
mg_data_cell_renderer_combo_finalize (GObject *object)
{
	MgDataCellRendererCombo *datacell = MG_DATA_CELL_RENDERER_COMBO (object);

	if (datacell->priv) {
		if (datacell->priv->strings) {
			GList *list = datacell->priv->strings;
			while (list) {
				g_free (list->data);
				list = g_list_next (list);
			}
			g_list_free (datacell->priv->strings);
		}

		if (datacell->priv->current_path)
			g_free (datacell->priv->current_path);

		g_free (datacell->priv);
		datacell->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void
mg_data_cell_renderer_combo_get_property (GObject        *object,
					    guint           param_id,
					    GValue         *value,
					    GParamSpec     *pspec)
{
	MgDataCellRendererCombo *datacell = MG_DATA_CELL_RENDERER_COMBO (object);

	switch (param_id) {
	case PROP_VALUE_ATTRIBUTES:
		g_value_set_uint (value, datacell->priv->attributes);
		break;
	case PROP_SET_DEFAULT_IF_INVALID:
		g_value_set_boolean (value, datacell->priv->set_default_if_invalid);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}


static gint         compute_row_from_values (MgDataCellRendererCombo *datacell, GList *values);
static const gchar *compute_string_from_row (MgDataCellRendererCombo *datacell, gint row);
static void
mg_data_cell_renderer_combo_set_property (GObject      *object,
					  guint         param_id,
					  const GValue *value,
					  GParamSpec   *pspec)
{
	MgDataCellRendererCombo *datacell = MG_DATA_CELL_RENDERER_COMBO (object);

	switch (param_id) {
	case PROP_VALUES:
		if (value) {	
			GList *gvalues = g_value_get_pointer (value);
			if (gvalues) {
				gint row;
				if (! datacell->priv->ccore->data_model)
					utility_combo_compute_model (datacell->priv->ccore);

				row = compute_row_from_values (datacell, gvalues);
				datacell->priv->current_row = row;
				g_object_set (G_OBJECT (object), "text", 
					      compute_string_from_row (datacell, row), NULL);
			}
			else
				g_object_set (G_OBJECT (object), "text", "", NULL);
		}
		else
			g_object_set (G_OBJECT (object), "text", "", NULL);
			      
		g_object_notify (object, "values");
		break;
	case PROP_VALUES_COMPLETE:
		if (value) {	
			GList *gvalues = g_value_get_pointer (value);
			if (gvalues) {
				GSList *list;
				gchar *str;
				GdaValue *val;

				datacell->priv->current_row = ROW_NA;
				str = utility_combo_compute_display_string (datacell->priv->ccore, gvalues);
				g_object_set (G_OBJECT (object), "text", str, NULL);
				g_free (str);

				/* extracting the PK values */
				clean_values_pk (datacell);

				list = datacell->priv->ccore->nodes;
				while (list) {
					val = g_list_nth_data (gvalues, COMBO_NODE (list->data)->position);
					datacell->priv->values_pk = g_list_append (datacell->priv->values_pk, 
										   gda_value_copy (val));
					list = g_slist_next (list);
				}
			}
			else
				g_object_set (G_OBJECT (object), "text", "", NULL);
		}
		else
			g_object_set (G_OBJECT (object), "text", "", NULL);
			      
		g_object_notify (object, "values_complete");
		break;
	case PROP_VALUE_ATTRIBUTES:
		datacell->priv->attributes = g_value_get_uint (value);
		break;
	case PROP_TO_BE_DELETED:
		datacell->priv->to_be_deleted = g_value_get_boolean (value);
		break;
	case PROP_SET_DEFAULT_IF_INVALID:
		datacell->priv->set_default_if_invalid = g_value_get_boolean (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

/**
 * mg_data_cell_renderer_combo_get_first_value_available
 * @datacell: a #MgDataCellRendererCombo renderer
 *
 * Get the first value available from the possible values listed in the combo
 * box. %NULL is returned if there is not any value available.
 *
 * Returns: a new #GdaValue, or %NULL
 */
GdaValue *
mg_data_cell_renderer_combo_get_first_value_available (MgDataCellRendererCombo *datacell)
{
	GdaValue *value = NULL;
	g_return_val_if_fail (datacell && IS_MG_DATA_CELL_RENDERER_COMBO (datacell), NULL);

	if (! datacell->priv->ccore->data_model)
		utility_combo_compute_model (datacell->priv->ccore);
	
	if (datacell->priv->ccore->data_model_valid) 
		value = prepare_signal_changed_value (datacell, 0);
	return value;
}

static void
clean_values_pk (MgDataCellRendererCombo *datacell)
{
	GList *list = datacell->priv->values_pk;

	if (!datacell->priv->values_pk)
		return;

	while (list) {
		gda_value_free (list->data);
		list = g_list_next (list);
	}
	g_list_free (datacell->priv->values_pk);
	datacell->priv->values_pk = NULL;
}

static const gchar *
compute_string_from_row (MgDataCellRendererCombo *datacell, gint row)
{
	gchar *retval = NULL;

	g_return_val_if_fail (row != ROW_NA, "");

	switch (row) {
	case ROW_ERROR:
		retval = _("Error");
		break;
	case ROW_NULL_VALUE:
		retval = "";
		break;
	default:
		if (! datacell->priv->strings)
			datacell->priv->strings = utility_combo_compute_choice_strings (datacell->priv->ccore);

		retval = g_list_nth_data (datacell->priv->strings, row);
		break;
	}


	return retval;
}



/**
 * compute_row_from_values
 *
 * Find the selected row in ccore->resultset from the provided values
 * Returns: ROW_NULL_VALUE, ROW_ERROR, or the row number if no error
 */
static gint
compute_row_from_values (MgDataCellRendererCombo *datacell, GList *values)
{
	gint retval = ROW_NULL_VALUE; /* corresponds to NULL */
	gboolean allnull = TRUE;
	GList *list;

	/* try to determine if all the values are NULL or of type GDA_VALUE_TYPE_NULL */
	list = values;
	while (list && allnull) {
		if (list->data && (gda_value_get_type ((GdaValue *)list->data) != GDA_VALUE_TYPE_NULL))
			allnull = FALSE;
		list = g_list_next (list);
	}
	
	/* actual values settings */
	if (!allnull) {
		g_return_val_if_fail (g_list_length (values) == g_slist_length (datacell->priv->ccore->nodes), retval);
		
		if (datacell->priv->ccore->data_model_valid) {
			gint row = 0, nbrows;
			
			nbrows = gda_data_model_get_n_rows (datacell->priv->ccore->data_model);
			while ((row < nbrows) && (retval == ROW_NULL_VALUE)) {
				GSList *nodes = datacell->priv->ccore->nodes;
				GList *argptr = values;
				gboolean equal = TRUE;
				ComboNode *node;
				const GdaValue *model_value, *arg_value;
				
				while (argptr && equal) {
					GdaValueType type1=GDA_VALUE_TYPE_NULL, type2=GDA_VALUE_TYPE_NULL;
					
					node = COMBO_NODE (nodes->data);
					model_value = gda_data_model_get_value_at (datacell->priv->ccore->data_model, 
										   node->position, row);
					arg_value = (GdaValue*) (argptr->data);
					
					if (arg_value)
						type1 = gda_value_get_type (arg_value);
					if (model_value)
						type2 = gda_value_get_type (model_value);
					if (type1 != type2)
						equal = FALSE;
					else
						if (type1 != GDA_VALUE_TYPE_NULL)
							equal = !gda_value_compare (model_value, arg_value);
					
					nodes = g_slist_next (nodes);
					argptr = g_list_next (argptr);
				}
				
				if (equal) 
					retval = row;
				row++;
			}

			if (retval == ROW_NULL_VALUE)
				retval = ROW_ERROR;
		}
	}

	return retval;
}


/**
 * mg_data_cell_renderer_combo_new
 * @conf: a #MgConf object
 * @context:
 * @node:
 * 
 * Creates a new #MgDataCellRendererCombo.
 * 
 * Return value: the new cell renderer
 **/
GtkCellRenderer *
mg_data_cell_renderer_combo_new (MgConf *conf, MgContext *context, MgContextNode *node)
{
	GObject *obj;
	MgDataCellRendererCombo *datacell;
	MgQueryType qtype;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (context && IS_MG_CONTEXT (context), NULL);
	g_return_val_if_fail (node, NULL);
	g_return_val_if_fail (g_slist_find (context->nodes, node), NULL);
	g_return_val_if_fail (node->query, NULL);
	qtype = mg_query_get_query_type (node->query);
	g_return_val_if_fail ((qtype == MG_QUERY_TYPE_SELECT) ||
			      (qtype == MG_QUERY_TYPE_UNION) ||
			      (qtype == MG_QUERY_TYPE_INTERSECT), NULL);
	

	obj = g_object_new (MG_DATA_CELL_RENDERER_COMBO_TYPE, NULL);
	datacell = MG_DATA_CELL_RENDERER_COMBO (obj);

	/* create the ComboCore structure */
	datacell->priv->ccore = utility_combo_initialize_core (conf, context, node, 
							      G_CALLBACK (dependency_param_changed_cb),
							      datacell);
	
	/* create the data model */
	utility_combo_compute_model (datacell->priv->ccore);

	/* find an intelligent original value: the ones provided by the parameters themselves */
	/* null_possible = TRUE; */
/* 	values = NULL; */
/* 	list = datacell->priv->ccore->nodes; */
/* 	while (list) { */
/* 		const GdaValue *init = mg_parameter_get_value (COMBO_NODE (list->data)->param); */
/* 		values = g_list_append (values, init); */

/* 		if (mg_parameter_get_not_null (COMBO_NODE (list->data)->param)) */
/* 			null_possible = FALSE; */
		
/* 		list = g_slist_next (list); */
/* 	} */
/* 	datacell->priv->null_possible = null_possible; */
/* 	display_datacell_model (datacell); */
/* 	mg_entry_datacell_set_values (datacell, values); */
/* 	g_list_free (values); */

	return GTK_CELL_RENDERER (obj);
}

static void
dependency_param_changed_cb (MgParameter *param, MgDataCellRendererCombo *datacell)
{
	utility_combo_destroy_model (datacell->priv->ccore);
	datacell->priv->current_row = ROW_NA;
	if (datacell->priv->popup_window) {
		gtk_widget_destroy (datacell->priv->popup_window);
		datacell->priv->popup_window = NULL;
	}
}


static void
mg_data_cell_renderer_combo_get_size (GtkCellRenderer *cell,
				      GtkWidget       *widget,
				      GdkRectangle    *cell_area,
				      gint            *x_offset,
				      gint            *y_offset,
				      gint            *width,
				      gint            *height)
{
	gint calc_width;
	gint calc_height;
	gint expander_size;

	/* get the size as calculated by the GtkCellRendererText */
	GtkCellRendererClass *text_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TEXT);
	(text_class->get_size) (cell, widget, cell_area, x_offset, y_offset, width, height);
	
	/* Add more space for the popdonw menu symbol */
	gtk_widget_style_get (widget, "expander_size", &expander_size, NULL);
	calc_width = (gint) cell->xpad * 2 + expander_size;
	calc_height = (gint) cell->ypad * 2 + expander_size;
	
	if (width)
		*width += calc_width;
	
	if (height && (*height < calc_height))
		*height = calc_height;
}

static void
mg_data_cell_renderer_combo_render (GtkCellRenderer      *cell,
				    GdkWindow            *window,
				    GtkWidget            *widget,
				    GdkRectangle         *background_area,
				    GdkRectangle         *cell_area,
				    GdkRectangle         *expose_area,
				    GtkCellRendererState  flags)
	
{
	GtkStateType state = 0;	
	gint expander_size;

	/* render the text as for the GtkCellRendererText */
	GtkCellRendererClass *text_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TEXT);
	(text_class->render) (cell, window, widget, background_area, cell_area, expose_area, flags);

	/* render the popdown menu symbol */
	if ((flags & GTK_CELL_RENDERER_SELECTED) == GTK_CELL_RENDERER_SELECTED)	{
		if (GTK_WIDGET_HAS_FOCUS (widget))
			state = GTK_STATE_SELECTED;
		else
			state = GTK_STATE_ACTIVE;
	}
	else {
		if (GTK_CELL_RENDERER_TEXT (cell)->editable)
			state = GTK_STATE_NORMAL;
		else
			state = GTK_STATE_INSENSITIVE;
	}
	gtk_widget_style_get (widget, "expander_size", &expander_size, NULL);

	gtk_paint_expander (widget->style,
			    window, state,
			    cell_area, 
			    widget,
			    "expander",
			    cell_area->x + cell_area->width - cell->xpad - expander_size/2.,
			    cell_area->y + cell_area->height - cell->ypad - expander_size/2. ,
			    GTK_EXPANDER_EXPANDED);

	if (MG_DATA_CELL_RENDERER_COMBO (cell)->priv->to_be_deleted)
		gtk_paint_hline (widget->style,
				 window, GTK_STATE_SELECTED,
				 cell_area, 
				 widget,
				 "hline",
				 cell_area->x + cell->xpad, cell_area->x + cell_area->width - cell->xpad,
				 cell_area->y + cell_area->height / 2.);
}

static gboolean popup_grab_on_window (GdkWindow *window, guint32 activate_time);
static gint popup_window_delete_popup (GtkWidget *widget, MgDataCellRendererCombo *datacell);
static gint popup_window_key_press_popup (GtkWidget *widget, GdkEventKey *event, MgDataCellRendererCombo *datacell);
static gint popup_window_button_press_popup (GtkWidget *widget, GdkEventButton *event, MgDataCellRendererCombo *datacell);
static GtkWidget *popup_create_treeview (MgDataCellRendererCombo *datacell, gint *nbrows, gint *nbcols);
static void popup_treeview_set_size_pos (GtkWidget *treeview, GtkWidget *sw,
					 gint total_nb_cols, gint total_nb_rows, 
					 GdkRectangle *cell_area,
					 MgDataCellRendererCombo *datacell);
static gboolean 
mg_data_cell_renderer_combo_activate  (GtkCellRenderer            *cell,
				       GdkEvent                   *event,
				       GtkWidget                  *widget,
				       const gchar                *path,
				       GdkRectangle               *background_area,
				       GdkRectangle               *cell_area,
				       GtkCellRendererState        flags)
{
	MgDataCellRendererCombo *datacell;
	gint x, y;

	datacell = MG_DATA_CELL_RENDERER_COMBO (cell);

	/* popup window */
	if (!datacell->priv->popup_window) {
		gint total_nb_rows, total_nb_cols;
		GtkWidget *window, *sw, *treeview;

		window = gtk_window_new (GTK_WINDOW_POPUP);
		gtk_widget_set_events (window, gtk_widget_get_events (window) | GDK_KEY_PRESS_MASK);
		gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
		g_signal_connect (G_OBJECT (window), "delete_event",
				  G_CALLBACK (popup_window_delete_popup), datacell);
		g_signal_connect (G_OBJECT (window), "key_press_event",
				  G_CALLBACK (popup_window_key_press_popup), datacell);
		g_signal_connect (G_OBJECT (window), "button_press_event",
				  G_CALLBACK (popup_window_button_press_popup), datacell);

		/* treeview inside the popup */
		sw = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_OUT);
		gtk_container_add (GTK_CONTAINER (window), sw);
		
		treeview = popup_create_treeview (datacell, &total_nb_rows, &total_nb_cols);
		gtk_container_add (GTK_CONTAINER (sw), treeview);
		g_object_set_data (G_OBJECT (treeview), "window", window);
		popup_treeview_set_size_pos (treeview, sw, total_nb_cols, total_nb_rows, cell_area, datacell);
				
		datacell->priv->popup_window = window;
		datacell->priv->treeview = treeview;
	}

	/* try to compute the right value */
	if (datacell->priv->current_row == ROW_NA) {
		gint row;
		if (! datacell->priv->ccore->data_model)
			utility_combo_compute_model (datacell->priv->ccore);
		
		row = compute_row_from_values (datacell, datacell->priv->values_pk);
		datacell->priv->current_row = row;
	}

	/* setting the right value */
	if (datacell->priv->current_row >= -1) {
		GtkTreePath *path;
		if (datacell->priv->current_row >= 0) {
			if (datacell->priv->attributes & MG_DATA_ENTRY_CAN_BE_NULL)
				path = gtk_tree_path_new_from_indices (datacell->priv->current_row + 1, -1);
			else
				path = gtk_tree_path_new_from_indices (datacell->priv->current_row, -1);
		}
		else
			path = gtk_tree_path_new_from_indices (0, -1);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (datacell->priv->treeview), path, NULL, FALSE);
		gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (datacell->priv->treeview), path, NULL, TRUE, 0.0, 0.0);
		gtk_tree_path_free (path);
	}

	/* keeping track of the path passed to this function */
	if (datacell->priv->current_path)
		g_free (datacell->priv->current_path);
	datacell->priv->current_path = g_strdup (path);

	/* position the popup window and show it */
	gdk_window_get_origin (gtk_tree_view_get_bin_window (GTK_TREE_VIEW (widget)), &x, &y);
	x += background_area->x; 
	y += background_area->y + background_area->height;
	if (x < 0) x = 0;
	if (y < 0) y = 0;
	gtk_window_move (GTK_WINDOW (datacell->priv->popup_window), x, y);
	gtk_widget_show_all (datacell->priv->popup_window);

	/* Temporarily grab pointer and keyboard, copied from GnomeDateEdit */
        if (!popup_grab_on_window (widget->window, gtk_get_current_event_time ()))
                return FALSE;
	
        gtk_grab_add (datacell->priv->popup_window);
        gtk_widget_show (datacell->priv->popup_window);
        gtk_widget_grab_focus (datacell->priv->treeview);
        popup_grab_on_window (datacell->priv->popup_window->window, gtk_get_current_event_time ());

	return TRUE;
}


/* we want to calculate the appropriate size for the tree view only once it has been realized */
static void
popup_treeview_set_size_pos (GtkWidget *treeview, GtkWidget *sw, 
			     gint total_nb_cols, gint total_nb_rows, 
			     GdkRectangle *cell_area,
			     MgDataCellRendererCombo *datacell)
{
	gint i;
	gint width = 0, height = 0;
	GtkTreeViewColumn *column = NULL;
	
	/* REM: as long as the widget is not realized, the following width and height calculations are useless... */
	/* for (i = 0; i < total_nb_cols; i++) { */
/* 		GtkTreePath *path = gtk_tree_path_new_from_indices (0, -1); */
/* 		GdkRectangle rect; */

/* 		column = gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), i); */
/* 		gtk_tree_view_get_background_area (GTK_TREE_VIEW (treeview), path, column, &rect); */
/* 		width = +rect.width; */
/* 		gtk_tree_path_free (path); */
/* 	} */

/* 	for (i = 0; (i < total_nb_rows) && (i < 15); i++) { */
/* 		GtkTreePath *path = gtk_tree_path_new_from_indices (i, -1); */
/* 		GdkRectangle rect; */

/* 		gtk_tree_view_get_background_area (GTK_TREE_VIEW (treeview), path, column, &rect); */
/* 		height += rect.height; */
/* 		gtk_tree_path_free (path); */
/* 	} */

	if (width < cell_area->width)
		width = cell_area->width;
	if (height < cell_area->height * 15)
		height = cell_area->height * 15;

	gtk_widget_set_size_request (sw, width, height);	
}

static void popup_cell_render_set_attributes (GtkTreeViewColumn *tree_column,
					      GtkCellRenderer *cell,
					      GtkTreeModel *tree_model,
					      GtkTreeIter *iter, MgDataCellRendererCombo *datacell);
static void popup_tree_selection_changed_cb (GtkTreeSelection *selection, MgDataCellRendererCombo *datacell);
static gboolean popup_tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, MgDataCellRendererCombo *datacell);
static GtkWidget *
popup_create_treeview (MgDataCellRendererCombo *datacell, gint *nbrows, gint *nbcols)
{
	GtkWidget *treeview;
	GtkTreeStore *tree_model;
	GtkTreeSelection *select;

	/* if no data model exists, then calculate a new one */
	if (! datacell->priv->ccore->data_model) 
		utility_combo_compute_model (datacell->priv->ccore);

	/* data model : one column in which we store the data model's row number */
	if (datacell->priv->ccore->data_model_valid) {
		gint row_count = 0;
		GtkTreeIter iter;
		gint i;

		tree_model = gtk_tree_store_new (1, G_TYPE_INT);

		/* storing each row number */
		row_count = mg_resultset_get_nbtuples (datacell->priv->ccore->resultset);
		for (i = 0; i < row_count; i++) {
			gtk_tree_store_append (tree_model, &iter, NULL);
			gtk_tree_store_set (tree_model, &iter, 0, i, -1);
		}

		/* if NULL is possible, we add an empty line */
		if (datacell->priv->attributes & MG_DATA_ENTRY_CAN_BE_NULL) {
			gtk_tree_store_prepend (tree_model, &iter, NULL);
			gtk_tree_store_set (tree_model, &iter, 0, -1, -1);
			row_count++;
		}

		if (nbrows)
			*nbrows = row_count;
	}
	else {
		GtkTreeIter iter;
		gchar *str;
		const GdaValue *value;

		tree_model = gtk_tree_store_new (1, G_TYPE_STRING);
		gtk_tree_store_append (tree_model, &iter, NULL);
		value = gda_data_model_get_value_at (datacell->priv->ccore->data_model, 0, 0);
		str = gda_value_stringify (value);
		gtk_tree_store_set (tree_model, &iter, 0, str, -1);
		g_free (str);
		if (nbrows)
			*nbrows = 1;
	}

	/* view itself */
	treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (tree_model));
	g_object_unref (G_OBJECT (tree_model));

	/* add columns */
	if (datacell->priv->ccore->data_model_valid) {
		gint *mask, masksize;

		mask = datacell->priv->ccore->mask;
		masksize = datacell->priv->ccore->masksize;
		if (mask) {
			GtkCellRenderer *renderer;
			gint j;
			MgField *field;
			MgDataHandler *dh;

			for (j=0; j<masksize; j++) {
				field = mg_entity_get_field_by_index (MG_ENTITY (datacell->priv->ccore->query),
								      mask [j]);
				g_assert (field);
				dh = mg_server_data_type_get_handler (mg_field_get_data_type (field));
				
				renderer = gtk_cell_renderer_text_new ();
				g_object_set_data (G_OBJECT (renderer), "data_handler", dh);
				g_object_set_data (G_OBJECT (renderer), "colnum", GINT_TO_POINTER (mask [j]));
				g_object_set (G_OBJECT (renderer), "size-points", 8., NULL);
				gtk_tree_view_insert_column_with_data_func (GTK_TREE_VIEW (treeview), -1, 
									    NULL, renderer,
									    (GtkTreeCellDataFunc) 
									    popup_cell_render_set_attributes, 
									    datacell, NULL);
			}
		}
		if (nbcols)
			*nbcols = masksize;
	}
	else {
		GtkCellRenderer *renderer;

		renderer = gtk_cell_renderer_text_new ();
		g_object_set (G_OBJECT (renderer), "size-points", 8., "editable", FALSE, NULL);
		gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview), 0, NULL,
							     renderer, "text", 0, NULL);
		if (nbcols)
			*nbcols = 1;
	}
	
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), FALSE);

	/* selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
			  G_CALLBACK (popup_tree_selection_changed_cb), datacell);
	g_signal_connect_after (G_OBJECT (treeview), "event",
				G_CALLBACK (popup_tree_view_event_cb), datacell);

	return treeview;
}

static void
popup_cell_render_set_attributes (GtkTreeViewColumn *tree_column,
				  GtkCellRenderer *cell,
				  GtkTreeModel *tree_model,
				  GtkTreeIter *iter, MgDataCellRendererCombo *datacell)
{
	MgDataHandler *dh;
	gint colnum, rownum;
	const GdaValue *value;
	gchar *str;

	dh = g_object_get_data (G_OBJECT (cell), "data_handler");
	colnum = GPOINTER_TO_INT (g_object_get_data  (G_OBJECT (cell), "colnum"));
	gtk_tree_model_get (tree_model, iter, 0, &rownum, -1);
	if (rownum >= 0) {
		value = gda_data_model_get_value_at (datacell->priv->ccore->data_model, colnum, rownum);
		str = mg_data_handler_get_str_from_value (dh, value);
		g_object_set (G_OBJECT (cell), "text", str, NULL);
		g_free (str);
	}
	else
		g_object_set (G_OBJECT (cell), "text", "", NULL);
}

static GdaValue *
prepare_signal_changed_value (MgDataCellRendererCombo *datacell, gint row)
{
	GdaValue *value;
	
	if (row >= 0) {
		/* making list of values for the selected row */
		GList *values = NULL;
		if (datacell->priv->emit_pk_values_only) {
			GSList *list;
			list = datacell->priv->ccore->nodes;
			while (list) {
				ComboNode *node = COMBO_NODE (list->data);
				values = g_list_append (values,
							gda_data_model_get_value_at (datacell->priv->ccore->data_model, 
										     node->position, row));
				list = g_slist_next (list);
			}
		}
		else {
			gint i, nbcols;
			nbcols = gda_data_model_get_n_columns (datacell->priv->ccore->data_model);
			for (i=0; i<nbcols; i++) {
				values = g_list_append (values, 
							gda_data_model_get_value_at (datacell->priv->ccore->data_model,
										     i, row));
			}
		}
		value = gda_value_new_list ((GdaValueList *) values);
		if (values) g_list_free (values);
	}
	else
		value = gda_value_new_null ();	
	
	return value;
}

static void
popup_tree_selection_changed_cb (GtkTreeSelection *selection,
				 MgDataCellRendererCombo *datacell)
{
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (datacell), "to_be_closed"))) {
		GtkTreeIter iter;
		GtkTreeModel *model;

		if (datacell->priv->ccore->data_model_valid &&
		    gtk_tree_selection_get_selected (selection, &model, &iter))	{
			GdaValue *value;
			gint row;
			
			/* retreive selected row */
			gtk_tree_model_get (model, &iter, 0, &row, -1);

			/* making list of values for the selected row */
			value = prepare_signal_changed_value (datacell, row);
			g_signal_emit (G_OBJECT (datacell), text_cell_renderer_combo_signals[CHANGED], 0, 
				       datacell->priv->current_path, value);
			gda_value_free (value);
		}
		g_object_set_data (G_OBJECT (datacell), "to_be_closed", NULL);
		gtk_grab_remove (datacell->priv->popup_window);
		gtk_widget_hide (datacell->priv->popup_window);
	}
}

static gboolean
popup_tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, MgDataCellRendererCombo *datacell)
{
	if ((event->type == GDK_BUTTON_PRESS) && (((GdkEventButton*) event)->button == 1)) {
		/* during the next call to popup_tree_selection_changed_cb(), use the selection
		 * which will be available at that time */
		g_object_set_data (G_OBJECT (datacell), "to_be_closed", GINT_TO_POINTER (TRUE));
		return FALSE;
	}

	if ((event->type == GDK_KEY_PRESS) && (((GdkEventKey*) event)->keyval == GDK_Return)){
		/* Use the selection as it is to fetch the right selected row => call ourselves
		* the popup_tree_selection_changed_cb() function */
		GtkTreeSelection *select;
		select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
		g_object_set_data (G_OBJECT (datacell), "to_be_closed", GINT_TO_POINTER (TRUE));

		popup_tree_selection_changed_cb (select, datacell);
		return FALSE;
	}

	return FALSE;
}

static gboolean
popup_grab_on_window (GdkWindow *window, guint32 activate_time)
{
        if ((gdk_pointer_grab (window, TRUE,
                               GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
                               GDK_POINTER_MOTION_MASK,
                               NULL, NULL, activate_time) == 0)) {
                if (gdk_keyboard_grab (window, TRUE,
                                       activate_time) == 0)
                        return TRUE;
                else {
                        gdk_pointer_ungrab (activate_time);
                        return FALSE;
                }
        }
        return FALSE;
}

static gint
popup_window_delete_popup (GtkWidget *widget, MgDataCellRendererCombo *datacell)
{
	gtk_grab_remove (widget);
	gtk_widget_hide (widget);

	return TRUE;
}

static gint
popup_window_key_press_popup (GtkWidget *widget, GdkEventKey *event, MgDataCellRendererCombo *datacell)
{
	if (event->keyval != GDK_Escape)
                return FALSE;

        g_signal_stop_emission_by_name (widget, "key_press_event");
	gtk_grab_remove (widget);
	gtk_widget_hide (widget);

        return TRUE;
}

static gint
popup_window_button_press_popup (GtkWidget *widget, GdkEventButton *event, MgDataCellRendererCombo *datacell)
{
	GtkWidget *child;
	
        child = gtk_get_event_widget ((GdkEvent *) event);
        /* We don't ask for button press events on the grab widget, so
         *  if an event is reported directly to the grab widget, it must
         *  be on a window outside the application (and thus we remove
         *  the popup window). Otherwise, we check if the widget is a child
         *  of the grab widget, and only remove the popup window if it
         *  is not.
         */
        if (child != widget) {
                while (child) {
                        if (child == widget)
                                return FALSE;
                        child = child->parent;
                }
        }
	
	gtk_grab_remove (widget);
	gtk_widget_hide (widget);
	
        return TRUE;
}
