/* data-handler.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "data-handler.h"
#include "data-handler-private.h"

static void data_handler_class_init (DataHandlerClass * class);
static void data_handler_init (DataHandler * dh);
static void data_handler_dispose (GObject   * object);


typedef struct _MissingDataNode MissingDataNode;
typedef struct _ExecUi          ExecUi;



/*
 *
 * Main functions for the object
 *
 */


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass   *parent_class = NULL;

guint
data_handler_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (DataHandlerClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) data_handler_class_init,
			NULL,
			NULL,
			sizeof (DataHandler),
			0,
			(GInstanceInitFunc) data_handler_init
		};		
		
		type = g_type_register_static (G_TYPE_OBJECT, "DataHandler", &info, 0);
	}

	return type;
}

static void
data_handler_class_init (DataHandlerClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = data_handler_dispose;
}

static void
data_handler_init (DataHandler * dh)
{
	dh->priv = g_new0 (DataHandlerPriv, 1);
	dh->priv->descr = NULL;
	dh->priv->detailled_descr = NULL;
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 0;
	dh->priv->valid_gda_types = NULL;
	dh->priv->expand_widget = FALSE;
	dh->priv->widget_from_value = NULL;
	dh->priv->value_from_widget = NULL;
	dh->priv->widget_update = NULL;
	dh->priv->sql_from_value = NULL;
	dh->priv->str_from_value = NULL;
	dh->priv->disposed = FALSE;
}

GObject *
data_handler_new ()
{
	GObject   *obj;

	obj = g_object_new (DATA_HANDLER_TYPE, NULL);

	return obj;
}

static void data_handler_plugin_release (DataHandler *dh);
static void
data_handler_dispose (GObject   * object)
{
	DataHandler *dh;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_DATA_HANDLER (object));

	dh = DATA_HANDLER (object);

#ifdef debug
	g_print ("Disposing of DataHandler %p: %s\n", dh, dh->priv->descr);
#endif

	if (dh->priv) {
		dh->priv->disposed = TRUE;

		/* if plugin, free the plugin */
		if (dh->priv->lib_handle)
			data_handler_plugin_release (dh);

		if (dh->priv->valid_gda_types)
			g_free (dh->priv->valid_gda_types);
		
		g_free (dh->priv);
		dh->priv = NULL;
	}
	

	/* for the parent class */
        parent_class->dispose (object);
}

static void 
data_handler_plugin_release (DataHandler *dh)
{
	void (*release_struct) (DataHandler * dh);

	g_module_symbol (dh->priv->lib_handle, "release_struct", (gpointer *) &release_struct);
	if (release_struct)
		(release_struct) (dh);
	g_module_close (dh->priv->lib_handle);
	dh->priv->lib_handle = NULL;

	if (dh->priv->plugin_file)
		g_free (dh->priv->plugin_file);
}

void 
data_handler_init_plugin (DataHandler *dh, GModule *lib, const gchar * plugin_file)
{
	g_return_if_fail (dh && IS_DATA_HANDLER (dh));
	g_return_if_fail (lib);
	g_return_if_fail (plugin_file);
	
	dh->priv->lib_handle = lib;
	if (dh->priv->plugin_file)
		g_free (dh->priv->plugin_file);
	dh->priv->plugin_file = g_strdup (plugin_file);

#ifdef debug
	g_print ("\tNew plugin (%p/%p): %s(V%s)\n", dh, 
		 dh->priv->widget_from_value, dh->priv->plugin_name, dh->priv->version);
#endif
}

/*
 * Handler's main functionnalities
 */
GtkWidget *
data_handler_get_widget_from_value  (DataHandler * dh, const GdaValue * value)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return (dh->priv->widget_from_value)(dh, value);
}


GdaValue *
data_handler_get_value_from_widget  (DataHandler * dh, GtkWidget *widget)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (widget && GTK_IS_WIDGET (widget), NULL);

	return (dh->priv->value_from_widget)(dh, widget);
}

void
data_handler_widget_update (DataHandler * dh, GtkWidget * wid, 
			    const GdaValue * value, gboolean value_as_init)
{
	g_return_if_fail (dh && IS_DATA_HANDLER (dh));
	g_return_if_fail (wid && GTK_IS_WIDGET (wid));

	(dh->priv->widget_update)(dh, wid, value, value_as_init);
	
}

gchar *
data_handler_get_sql_from_value (DataHandler * dh, const GdaValue * value)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return (dh->priv->sql_from_value)(dh, value);
}

gchar *
data_handler_get_str_from_value     (DataHandler * dh, const GdaValue * value)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return (dh->priv->str_from_value)(dh, value);
}



/*
 * Getting information 
 */
guint 
data_handler_get_nb_gda_type (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), 0);

	return dh->priv->nb_gda_type;
}

GdaValueType 
data_handler_get_gda_type (DataHandler * dh, guint index)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), GDA_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail (index < dh->priv->nb_gda_type, GDA_VALUE_TYPE_UNKNOWN);

	return dh->priv->valid_gda_types [index];
}

const gchar *
data_handler_get_descr         (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return dh->priv->descr;
}

const gchar *
data_handler_get_descr_detail  (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return dh->priv->detailled_descr;
}

const gchar *
data_handler_get_version       (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return dh->priv->version;
}

gboolean 
data_handler_is_plugin (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), FALSE);
	
	return dh->priv->lib_handle ? TRUE : FALSE;
}

const gchar *
data_handler_get_plugin_name   (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return dh->priv->plugin_name;
}

const gchar *
data_handler_get_plugin_file   (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return dh->priv->plugin_file;
}

gchar *
data_handler_get_key   (DataHandler * dh)
{
	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	return dh->priv->get_unique_key ();
}
