/*
 *
 * bg-ext-handler.c :
 *
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <hestgray@ionet.net>
 *
 */

#include <libgnome/libgnome.h>
#include "bg-ext-handler.h"
#include "editor.h"

static MtmExtHandlerClass *bg_ext_handler_parent_class;

struct _BgExtHandlerPrivate
{
	GList *configs;
	MtmExt *ext;
	GHashTable *descs;
	GList *strs;
};

enum
{
	ARG_0,
	ARG_COLOR1,
	ARG_COLOR2,
	ARG_COLOR_GRADIENT,
	ARG_GRADIENT_TYPE,
	ARG_WALLPAPER_LAYOUT,
	ARG_EXT_CONTEXT
};

static GtkEnumValue _bg_wp_type_values[] = {
	{ WALLPAPER_TILED, "WALLPAPER_TILED", "Tiled" },
	{ WALLPAPER_CENTERED, "WALLPAPER_CENTERED", "Centered" },
	{ WALLPAPER_SCALED, "WALLPAPER_SCALED", "Scaled" },
	{ WALLPAPER_SCALED_KEEP, "WALLPAPER_SCALED_KEEP", "Scaled - keep aspect"} ,
	{ 0, NULL, NULL }
};

static GtkEnumValue _bg_grad_type_values[] = {
	{ VERTICAL, "VERTICAL", "Vertical" },
	{ HORIZONTAL, "HORIZONTAL", "Horizontal" },
	{ 0, NULL, NULL }
};

GtkType bg_wp_get_type (void);
GtkType bg_grad_get_type (void);

static gchar* bg_ext_handler_describe_arg (MtmExtHandler *handler, gchar *arg);

GtkType
bg_wp_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		type = gtk_type_register_enum ("BgWallpaperType",
			_bg_wp_type_values);
	}

	return type;
}

GtkType
bg_grad_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		type = gtk_type_register_enum ("BgGradientType",
			_bg_grad_type_values);
	}

	return type;
}

static void bg_ext_handler_set_arg (GtkObject *object, GtkArg *arg,
	guint arg_id);
static void bg_ext_handler_get_arg (GtkObject *object, GtkArg *arg,
	guint arg_id);

static void
bg_ext_handler_destroy (GtkObject *object)
{
	BgExtHandler *handler = BG_EXT_HANDLER (object);
	GList *list;

	if (handler->priv->configs)
	{
		list = handler->priv->configs;
		while (list)
		{
			bgconfig_destroy (list->data);
			list = list->next;
		}
	
		g_list_free (handler->priv->configs);
	}
	
	g_hash_table_destroy (handler->priv->descs);
	g_list_foreach (handler->priv->strs, (GFunc) g_free, NULL);
	g_list_free (handler->priv->strs);

	g_free (handler->priv);

	if (GTK_OBJECT_CLASS (bg_ext_handler_parent_class)->destroy)
		(*GTK_OBJECT_CLASS (bg_ext_handler_parent_class)->destroy)(object);
}

static void
bg_ext_handler_class_init (GtkObjectClass *object_class)
{
	bg_ext_handler_parent_class = gtk_type_class (mtm_ext_handler_get_type ());

	object_class->set_arg = bg_ext_handler_set_arg;
	object_class->get_arg = bg_ext_handler_get_arg;

	gtk_object_add_arg_type ("BgExtHandler::Color1", GTK_TYPE_GDK_COLOR,
		GTK_ARG_READWRITE, ARG_COLOR1);
	gtk_object_add_arg_type ("BgExtHandler::Color2", GTK_TYPE_GDK_COLOR,
		GTK_ARG_READWRITE, ARG_COLOR2);
	gtk_object_add_arg_type ("BgExtHandler::ColorGradient", GTK_TYPE_BOOL,
		GTK_ARG_READWRITE, ARG_COLOR_GRADIENT);
	gtk_object_add_arg_type ("BgExtHandler::GradientType",
		bg_grad_get_type (), GTK_ARG_READWRITE, ARG_GRADIENT_TYPE);
	gtk_object_add_arg_type ("BgExtHandler::WallpaperLayout",
		bg_wp_get_type (), GTK_ARG_READWRITE, ARG_WALLPAPER_LAYOUT);
	gtk_object_add_arg_type ("BgExtHandler::ext_context",
		MTM_EXT_TYPE, GTK_ARG_READWRITE, ARG_EXT_CONTEXT);

	object_class->destroy = bg_ext_handler_destroy;
}

static void
bg_ext_handler_init (GtkObject *object)
{
	BgExtHandler *handler = BG_EXT_HANDLER (object);
	int i = 0;
	gchar *key, *val;
	static gchar* args[] = {"Color1", _("Primary Color"), "Color2", _("Secondary Color"), "ColorGradient", _("Use Gradient"), "GradientType", _("Gradient Type"), "WallpaperLayout", _("Wallpaper Layout Type"), NULL};

	MTM_EXT_HANDLER (handler)->describe_arg = bg_ext_handler_describe_arg;

	handler->priv = g_new0 (BgExtHandlerPrivate, 1);
	
	handler->priv->descs = g_hash_table_new (g_str_hash, g_str_equal);

	while (args[i] != NULL)
	{
		key = g_strdup (args[i]);
		handler->priv->strs = g_list_prepend (handler->priv->strs, key);

		val = g_strdup (args[i + 1]);
		handler->priv->strs = g_list_prepend (handler->priv->strs, val);

		g_hash_table_insert (handler->priv->descs, key, val);
		i += 2;
	}
}

GtkType
bg_ext_handler_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		GtkTypeInfo info =
		{
			"BgExtHandler",
			sizeof (BgExtHandler),
			sizeof (BgExtHandlerClass),
			(GtkClassInitFunc) bg_ext_handler_class_init,
			(GtkObjectInitFunc) bg_ext_handler_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (mtm_ext_handler_get_type (), &info);
	};

	return type;
}

BgExtHandler*
bg_ext_handler_new (MtmEnv *env)
{
	BgExtHandler *handler;

	g_return_val_if_fail (env != NULL, NULL);

	handler = gtk_type_new (bg_ext_handler_get_type ());
	MTM_STATEFUL (handler)->env = env;
	return handler;
}

static void bg_ext_handler_set_arg (GtkObject *object, GtkArg *arg,
	guint arg_id)
{
	BgExtHandler *handler = BG_EXT_HANDLER (object);
	BgConfig *config;
	GdkColor *color;

	switch (arg_id)
	{
		case ARG_COLOR1:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			color = GTK_VALUE_BOXED (*arg);
			config->color1 = *color;
			break;
		case ARG_COLOR2:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			color = GTK_VALUE_BOXED (*arg);
			config->color2 = *color;
			break;
		case ARG_COLOR_GRADIENT:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			config->gradient = GTK_VALUE_BOOL (*arg);
			break;
		case ARG_GRADIENT_TYPE:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			config->gradtype = GTK_VALUE_ENUM (*arg);
			break;
		case ARG_WALLPAPER_LAYOUT:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			config->align = GTK_VALUE_ENUM (*arg);
			break;
		case ARG_EXT_CONTEXT:
			handler->priv->ext = MTM_EXT (GTK_VALUE_OBJECT (*arg));
			config = gtk_object_get_data (
				GTK_OBJECT (handler->priv->ext), "bgconfig");
			if (!config)
			{
				gchar *file;
				file = g_strconcat (g_get_home_dir (),
					"/.gnome/Background", NULL);
				config = bgconfig_load (file);
				g_free (file);
				handler->priv->configs = g_list_append (
					handler->priv->configs, config);
				gtk_object_set_data (
					GTK_OBJECT (handler->priv->ext),
					"bgconfig", config);
			}
			break;
	}
}

static void bg_ext_handler_get_arg (GtkObject *object, GtkArg *arg,
	guint arg_id)
{
	BgExtHandler *handler = BG_EXT_HANDLER (object);
	BgConfig *config;

	switch (arg_id)
	{
		case ARG_COLOR1:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			GTK_VALUE_BOXED (*arg) = &(config->color1);
			break;
		case ARG_COLOR2:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			GTK_VALUE_BOXED (*arg) = &(config->color2);
			break;
		case ARG_COLOR_GRADIENT:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			GTK_VALUE_BOOL (*arg) = config->gradient;
			break;
		case ARG_GRADIENT_TYPE:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			GTK_VALUE_ENUM (*arg) = config->gradtype;
			break;
		case ARG_WALLPAPER_LAYOUT:
			config = bg_ext_handler_get_config (handler);
			g_return_if_fail (config != NULL);
			GTK_VALUE_ENUM (*arg) = config->align;
			break;
		case ARG_EXT_CONTEXT:
			GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (
				handler->priv->ext);
			break;
	}
}

BgConfig*
bg_ext_handler_get_config (BgExtHandler *handler)
{
	g_return_val_if_fail (handler != NULL, NULL);
	g_return_val_if_fail (handler->priv->ext != NULL, NULL);
	return gtk_object_get_data (
		GTK_OBJECT (handler->priv->ext), "bgconfig");	
}

static gchar*
bg_ext_handler_describe_arg (MtmExtHandler *handler, gchar *arg)
{
	g_return_val_if_fail (arg != NULL, NULL);

	return g_strdup (g_hash_table_lookup (BG_EXT_HANDLER (handler)->priv->descs, arg));
}
