/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <pan/base/pan-glib-extensions.h>


/***
****
****  STRINGS
****
***/

/* null-safe strstr */
gchar*
pan_strstr (const gchar * s1,
            const gchar * s2)
{
	g_return_val_if_fail (s1!=NULL, NULL);
	g_return_val_if_fail (s2!=NULL, NULL);

	return strstr (s1, s2);
}

/* null-safe strcmp.  NULLs < strings. */
gint 
pan_strcmp (const gchar * a,
            const gchar * b)
{
	gint retval;

	if (a == b)
		retval = 0;
	else if (!a)
		retval = -1;
	else if (!b)
		retval = 1;
	else
		retval = strcmp (a, b);

	return retval;
}

gchar*
pan_stristr (const char * string, const char * pattern)
{
	size_t slen;
	size_t plen;
	char * start = (gchar*) string;

	g_return_val_if_fail (string!=NULL, NULL);
	g_return_val_if_fail (pattern!=NULL, NULL);

 	slen = strlen (string);
	plen = strlen (pattern);

	for (; slen>=plen; ++start, --slen)
	{
		gchar * sptr;
		gchar * pptr;

		/* find start of pattern in string */
		while (toupper(*start) != toupper(*pattern))
		{
			++start;
			--slen;

			/* if pattern longer than string */
			if (slen < plen)
				return NULL;
		}

		sptr = start;
		pptr = (char *) pattern;

		while (toupper(*sptr) == toupper(*pptr))
		{
			++sptr;
			++pptr;

			if (!*pptr)
				return start;
		}
	}

	return NULL;
}

gboolean
string_ends_with (const gchar * str,
                  const gchar * end)
{
	size_t str_len;
	size_t end_len;
	gboolean retval = FALSE;

	g_return_val_if_fail (end!=NULL, FALSE);
	g_return_val_if_fail (str!=NULL, FALSE);

	str_len = strlen (str);
	end_len = strlen (end);
	if (end_len <= str_len)
		retval = !memcmp(end, str+str_len-end_len, end_len);
 
	return retval;
}


void
replace_gstr (char   ** target_gfree_old,
              char    * assign_from_me)
{
	char * gfree_old;

 	g_return_if_fail (target_gfree_old != NULL);

	gfree_old = *target_gfree_old;
	*target_gfree_old = assign_from_me;
	g_free (gfree_old);
}

/**
 * Replaces the search string inside original with the replace string.
 * This should be safe with overlapping elements of text, though I haven't
 * not tested it with elaborately cruel cases.
 */
char*
pan_substitute (const char * original,
                const char * search,
                const char * replace)
{
	size_t slen;		/* length of search */
	size_t rlen;		/* length of replace */
	size_t tlen;		/* length of target (predicted) */
	gint i;
	const gchar * o;
	const gchar * pchar;
	gchar * t;
	gchar * retval = NULL;

	g_return_val_if_fail (original!=NULL, NULL);
	g_return_val_if_fail (*original!='\0', NULL);
	g_return_val_if_fail (search!=NULL, NULL);
	g_return_val_if_fail (*search!='\0', NULL);
	g_return_val_if_fail (replace!=NULL, NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	/* calculate the length */

	i = 0;
	tlen = 0;
	pchar = original;
	while ((pchar = pan_strstr (pchar, search))) {
		i++;
		pchar += slen;
	}
	tlen = strlen(original) + i*(rlen - slen);

	/**
	***  Make the substitution.
	**/

	o = original;
	t = retval = g_malloc(tlen + 1);
	while ((pchar = pan_strstr (o, search))) {
		(void) memcpy (t, o, (size_t)(pchar-o));
		t += pchar-o;
		(void) memcpy (t, replace, (size_t)rlen);
		t += rlen;
		o = pchar + slen;
	}
	(void) strcpy ( t, o );

	pan_warn_if_fail (strlen(retval)==tlen);

	return retval;
}

/***
****
****  GLIB AUGMENTATION
****
***/

void
pan_g_ptr_array_append (GPtrArray   * target,
                        gpointer    * source_ptr,
                        guint         source_qty)
{
	guint old_len;

	g_return_if_fail (target!=NULL);
	g_return_if_fail (source_ptr!=NULL);
	g_return_if_fail (source_qty>0);
       
	old_len = target->len;
	g_ptr_array_set_size (target, old_len+source_qty);
	memcpy (target->pdata+old_len, source_ptr, sizeof(gpointer)*source_qty);
}

void
pan_g_ptr_array_assign  (GPtrArray  * target,
                         gpointer   * source_ptr,
                         guint        source_qty)
{
	g_return_if_fail (target!=NULL);
	g_return_if_fail (source_qty==0 || source_ptr!=NULL);

	g_ptr_array_set_size (target, source_qty);
	memcpy (target->pdata, source_ptr, sizeof(gpointer)*source_qty);
}

GPtrArray*
pan_g_ptr_array_dup     (const GPtrArray  * source)
{
	GPtrArray * retval;

	g_return_val_if_fail (source!=NULL, NULL);

	retval = g_ptr_array_new ();
	pan_g_ptr_array_assign (retval, source->pdata, source->len);
	return retval;
}


static void
pan_hash_to_ptr_array_ghfunc (gpointer key, gpointer val, gpointer data)
{
	g_ptr_array_add ((GPtrArray*)data, val);
}
void
pan_hash_to_ptr_array  (GHashTable   * hash,
                        GPtrArray    * fillme)
{
	g_return_if_fail (fillme!=NULL);
	g_ptr_array_set_size (fillme, 0);

	g_return_if_fail (hash!=NULL);
	pan_g_ptr_array_reserve (fillme, g_hash_table_size(hash));
	g_hash_table_foreach (hash, pan_hash_to_ptr_array_ghfunc, fillme);
}


void
pan_g_ptr_array_reserve (GPtrArray  * a,
                         int          n)
{
	int len;

	g_return_if_fail (a!=NULL);

	len = a->len;
	g_ptr_array_set_size (a, MAX(len, n));
	a->len = len;
}

void
pan_g_ptr_array_insert (GPtrArray   * a,
                        gpointer      ptr,
                        int           index)
{
        g_return_if_fail (a!=NULL);

        if (index<0 || index>=a->len)
	{
                g_ptr_array_add (a, ptr);
	}
        else
        {
		pan_g_ptr_array_reserve (a, a->len+1);
                g_memmove (&a->pdata[index+1],
                           &a->pdata[index],
                           sizeof(gpointer)*(a->len-index));
                a->pdata[index] = ptr;
		++a->len;
        }
}

void
pan_g_ptr_array_foreach (GPtrArray   * a,
                         GFunc         func,
                         gpointer      user_data)
{
	guint i;

	g_return_if_fail (a!=NULL);
	g_return_if_fail (func!=NULL);

	for (i=0; i!=a->len; ++i)
	{
		gpointer call_data = g_ptr_array_index (a, i);
		(*func)(call_data, user_data);
	}
}

void
pan_g_string_replace (GString       * string,
                      const char    * search,
		      const char    * replace)
{
	const gchar * pch;
	size_t slen;
	size_t rlen;
	size_t offset;

	g_return_if_fail (string != NULL);
	g_return_if_fail (is_nonempty_string(search));
	g_return_if_fail (replace != NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	offset = 0;
	while ((pch = pan_strstr (string->str + offset, search)) && *pch) {
		const gint pos = pch - string->str;
		g_string_erase (string, pos, slen);
		g_string_insert (string, pos, replace);
		offset = pos + rlen;
	}
}

void
pan_g_string_strstrip (GString * string)
{
	g_return_if_fail (string != NULL);

	if (string->len != 0)
	{
		g_strstrip (string->str);
		string->len = strlen (string->str);
	}
}

void
pan_g_string_append_len  (GString       * str,
                          const gchar   * appendme,
                          guint           len)
{
	gchar * pch;

	g_return_if_fail (str!=NULL);
	g_return_if_fail (appendme!=NULL);

	pch = g_strndup (appendme, len);
	g_string_append (str, pch);
	g_free (pch);
}

/***
****
****  TOKENS
****
***/

 
void
skip_next_token (const char   * pch,
                 const char     delimiter,
                 const char  ** setme_next_token)
{
	if (is_nonempty_string(pch))
	{
		while (*pch && *pch!=delimiter)
			++pch;
		if (*pch==delimiter)
			++pch; /* skip past delimiter */
		if (setme_next_token)
			*setme_next_token = pch;
	}
}
 
const gchar*
get_next_token_range (const gchar   * pch,
                      const gchar     delimiter,
                      const gchar  ** setme_next_token,
                      const gchar  ** setme_start,
                      gint          * setme_len)
{
	const gchar * start = pch;

	if (pch!=NULL)
		while (*pch && *pch!=delimiter)
			++pch;

	*setme_start = start;
	*setme_len = pch==NULL ? 0 : pch-start;
	*setme_next_token = *pch==delimiter ? pch+1 : pch;
	return is_nonempty_string(start) ? start : NULL;
}


glong
get_next_token_long (const char   * pch,
                     const char     delimiter,
                     const char  ** setme_next_token)
{
	glong retval = 0;

	if (is_nonempty_string(pch))
	{
		const gchar * m;
		const gchar * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, atoi may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((int)*m)) {
				retval = atol (m);
				break;
			}
		}
	}

	return retval;
}

gulong
get_next_token_ulong (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token)
{
	gulong retval = 0;

	if (is_nonempty_string(pch))
	{
		const gchar * m;
		const gchar * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, strtoul may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((int)*m)) {
				retval = strtoul (m, NULL, 10);
				break;
			}
		}
	}

	return retval;
}

int
get_next_token_int (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	int retval = 0;

	if (is_nonempty_string(pch))
	{
		const gchar * m;
		const gchar * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, atoi may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((int)*m)) {
				retval = atoi(m);
				break;
			}
		}
	}

	return retval;
}

char*
get_next_token_str (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	char* retval = NULL;
	register const char* end = NULL;

	if (is_nonempty_string(pch)) {
		end = pch;
		while (is_nonempty_string(end) && *end!=delimiter)
			++end;
	}

	if (end != NULL)
		retval = g_strndup (pch, end-pch);

	if (setme_next_token)
		*setme_next_token = is_nonempty_string(end) ? end+1 : end;

	return retval;
}

gboolean
get_next_token_g_str (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token,
                      GString      * setme)
{
	register const char* end = pch;
	gboolean retval;

	while (is_nonempty_string(end) && *end!=delimiter)
		++end;

	if (end != pch)
	{
		gchar * tmp = g_strndup (pch, end-pch);
		g_string_assign (setme, tmp);
		g_free (tmp);
		retval = TRUE;
	}
	else
	{
		g_string_truncate (setme, 0);
		retval = end==NULL ? FALSE : *end==delimiter;
	}

	if (setme_next_token)
		*setme_next_token = is_nonempty_string(end) ? end+1 : end;

	return retval;
}


/***
****
****  UNSORTED
****
***/


int
lower_bound (const void         * key,
             const void         * base,
             size_t               n,
             size_t               size,
             int                  (*compare)(const void *, const void *),
             gboolean           * exact_match )
{
	register int low = 0;
	register int high = n - 1;

	while (low<=high)
	{
		const int mid = (unsigned)(low+high)/2u;
		const void * checkme = (void*)(((char*)(base))+(mid*size));
		const int comp = (*compare)(key,checkme);

		if (comp>0) low = mid+1;
		else if (comp<0 ) high = mid-1;
		else {
			if ( exact_match!=NULL )
				*exact_match = TRUE;
			return mid;
		}
	}

	if (exact_match!=NULL)
		*exact_match = FALSE;

	return low;
}


void
commatize_ulong (gulong    num,
		 char    * setme)
{
	char buf[32], *src=buf;
	int len;
	int i;

	sprintf (buf, "%ld", num);
	len = strlen (buf);
	i = len % 3;
	for (;;)
	{
		while (*src && i--)
			*setme++ = *src++;
		if (!*src) {
			*setme = '\0';
			return;
		}
		else if (src!=buf)
		   *setme++ = ',';
		i = 3;
	}

	pan_warn_if_reached ();
}


/***
****
****  XML
****
***/

gchar*
pan_str_escape (const gchar * str)
{
	gint size_needed;
	gchar * retval;
	gchar * out;
	const gchar * in;

	/* sanity clause */
	if (!is_nonempty_string (str))
		return g_strdup ("");

	/* how big must the buffer be? */
	size_needed = 1;
	for (in=str; *in; ++in) {
		gint inc;
		switch (*in) {
			case '&': inc = 5; break;
			case '>': case '<': inc = 4; break;
			case '"': inc = 6; break;
			default: inc = 1; break;
		}
		size_needed += inc;
	}

	/* build the output string */
	retval = out = g_malloc (size_needed);
	for (in=str; *in; ++in) {
		switch (*in) {
			case '&': memcpy (out, "&amp;", 5); out += 5; break;
			case '>': memcpy (out, "&gt;", 4); out += 4; break;
			case '<': memcpy (out, "&lt;", 4); out += 4; break;
			case '"': memcpy (out, "&quot;", 6); out += 6; break;
			default: *out++ = *in; break;
		}
	}
	*out = '\0';

	return retval;
}

gchar*
pan_str_unescape (const gchar * escaped)
{
	const gchar * src = escaped;
	gchar * buf = g_new0 (gchar, strlen(escaped)+1);
	gchar * tgt = buf;

	while (*src)
	{
		if (*src!='&') { *tgt++ = *src++; }
		else if (!strncmp(src,"&lt;",4)) { *tgt++ = '<';  src+=4; }
		else if (!strncmp(src,"&gt;",4))   { *tgt++ = '>';  src+=4; }
		else if (!strncmp(src,"&amp;",5))  { *tgt++ = '&';  src+=5; }
		else if (!strncmp(src,"&apos;",6)) { *tgt++ = '\''; src+=6; }
		else if (!strncmp(src,"&quot;",6)) { *tgt++ = '"';  src+=6; }
		else *tgt++ = *src++;
	}

	*tgt = '\0';

	return buf;
}
