/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <gnome.h>

#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/queue.h>
#include <pan/task-bodies.h>
#include <pan/task-decode.h>
#include <pan/save.h>
#include <pan/text.h>
#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static void save_file_cb (GtkWidget* entry, gpointer user_data);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static gchar * last_save_as_path = NULL;

static gboolean append_groupname = FALSE;

static gboolean back_of_queue = TRUE;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

static gint
save_article_cb (gpointer obj, gpointer arg, gpointer data)
{
	gint status;
	debug_enter ("save_article_cb");

	status = GPOINTER_TO_INT(arg);
	if (status == TASK_SUCCESS)
	{
		guint i;
		const TaskBodies * task = TASK_BODIES(obj);
		const GPtrArray * articles = task->articles;
		for (i=0; i<articles->len; ++i)
		{
			const Article * article = ARTICLE(g_ptr_array_index(task->articles,i));
			gchar * msg = article_get_message (article);
			GtkWidget * gfs;

			pan_lock ();
			gfs = gtk_file_selection_new (_("Save As..."));
			gtk_object_set_data_full (GTK_OBJECT(gfs), "msg", msg, g_free);
			gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(gfs)->ok_button),
					    "clicked", save_file_cb, (gpointer)gfs );
			gtk_signal_connect_object (GTK_OBJECT(GTK_FILE_SELECTION(gfs)->cancel_button),
						   "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						   (gpointer)gfs);
			gtk_widget_show(gfs);
			pan_unlock ();
		}
	}

	debug_exit ("save_article_cb");
	return 0;
}

void
save_article (Article * article)
{
	Task * task;
	GPtrArray * array;

	/* sanity clause */
	g_return_if_fail (article!=NULL);
	g_return_if_fail (article->group!=NULL);

	/* ensure we have the article body */
       	array = g_ptr_array_new ();
	g_ptr_array_add (array, article);
	task = TASK(task_bodies_new (article->group, array));
	g_ptr_array_free (array, TRUE);
	if (task != NULL) {
		pan_callback_add (task->task_ran_callback, save_article_cb, NULL);
		queue_add (task);
	}
}

void
save_current_article (void)
{
	Article * a = get_current_article ();
	if (a != NULL)
		save_article (a);
}

/************
*************  PRIVATE ROUTINES
************/

static void
real_save_file (GtkFileSelection *gfs)
{
	FILE *fp;
	const gchar * msg = gtk_object_get_data (GTK_OBJECT(gfs), "msg");
	const gchar * fname = gtk_object_get_data (GTK_OBJECT(gfs), "filename");
	gchar * tmp;
      
	/* open the file... */	
	fp = fopen (fname, "w+");
	if (!fp) {
		pan_error_dialog (_("Unable to create file \"%s\""), fname);
		return;
	}

	/* write the file... */
	tmp = pan_substitute (msg, "\r", "");
	fputs (tmp, fp);
	g_free (tmp);

	/* cleanup */
	log_add_va (LOG_INFO, "Saved file \"%s\"", fname);
	fclose(fp);
	gtk_widget_destroy (GTK_WIDGET(gfs));
}

static void
yes_im_sure_cb(gint reply, gpointer data)
{
	if (reply==0) /* yes */
		real_save_file (GTK_FILE_SELECTION(data));
}

/*****
******
*****/

static void
save_file_cb(GtkWidget* entry, gpointer user_data)
{
	GtkFileSelection *gfs = GTK_FILE_SELECTION(user_data);
	gchar *filename = gtk_file_selection_get_filename(gfs);
	if (filename && *filename)
	{
		gtk_object_set_data(GTK_OBJECT(gfs),"filename",filename);
		if (file_exists(filename))
		{
			GtkWidget *w = NULL;
			w = gnome_question_dialog_parented (
				_("File Exists.  Overwrite this file?"),
				yes_im_sure_cb, gfs,
				GTK_WINDOW(Pan.window) );
		}
		else
		{
			real_save_file (gfs);
		}
	}
}

/*****
******
*****/

typedef struct
{
	GtkWidget * default_dir_rb;
	GtkWidget * group_default_dir_rb;
	GtkWidget * user_specified_rb;
	GtkWidget * user_specified_entry;
	GtkWidget * encoded_filename_rb;
	GtkWidget * subject_filename_rb;
	GtkWidget * user_defined_filename_rb;
	GtkWidget * append_groupname_tb;
	GtkWidget * front_of_queue_rb;
	GtkWidget * back_of_queue_rb;
	GtkWidget * custom_filename_rb;
	GtkWidget * custom_filename_entry;

	Group  *group;

	GSList *gslists_of_unowned_articles;
}
SaveAsDialogStruct;

static void
save_attachment_as_dialog_clicked_cb (GnomeDialog      * dialog,
                                      int                button,
                                      gpointer           user_data)
{
	SaveAsDialogStruct* save = (SaveAsDialogStruct*)user_data;
	GSList * l;
	GSList * filenames = NULL;
	GSList * filename = NULL;
	gchar *path;

	/* cancel button */
	if (button==1)
	{
		pan_lock();
		gtk_widget_destroy (GTK_WIDGET(dialog));
		pan_unlock();
		return;
	}

	/* get path */
	path = NULL;
	if (gtk_toggle_button_get_active( /* user-specified directory */
			GTK_TOGGLE_BUTTON(save->user_specified_rb)))
		path = gtk_editable_get_chars (
			GTK_EDITABLE(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY (save->user_specified_entry))), 0, -1);
	else if ((save->group_default_dir_rb!=NULL) /* group preferences */
		 && gtk_toggle_button_get_active(
			 GTK_TOGGLE_BUTTON(save->group_default_dir_rb)))
		path = g_strdup(save->group->download_dir);
	else path = g_strdup (download_dir); /* global preferences */

	/* append the groupname, if desired */
	if (gtk_toggle_button_get_active(
			GTK_TOGGLE_BUTTON(save->append_groupname_tb)))
	{
		gchar * pch = g_strdup_printf ("%s/%s",
					       path,
					       save->group->name);
		g_free (path);
		path = pch;
	}

	/* get filenames */
	if (gtk_toggle_button_get_active( /* use encoded filename */
			GTK_TOGGLE_BUTTON(save->encoded_filename_rb)))
	{
		for (l=save->gslists_of_unowned_articles; l!=NULL; l=l->next)
			filenames = g_slist_append(filenames, NULL);
	}
	else if (gtk_toggle_button_get_active( /* use subject as filename */
			GTK_TOGGLE_BUTTON(save->subject_filename_rb)))
	{
		for (l=save->gslists_of_unowned_articles; l!=NULL; l=l->next)
		{
			/* use the subject of the first article in the list */
			GSList * article_parts = (GSList*)l->data;
			const Article* article = ARTICLE(article_parts->data);
			char* name = g_strdup (article_get_subject(article));
			filenames = g_slist_append (filenames, name);
		}
	}
	else /* user-specified filename */
	{
		filenames = g_slist_append (filenames,
			gtk_editable_get_chars (GTK_EDITABLE(save->custom_filename_entry), 0, -1));
	}

	/* queue 'em */
	filename = filenames;
	for (l=save->gslists_of_unowned_articles; l!=NULL; l=l->next)
	{
		int queue_index;
		GSList * l2 = (GSList*) l->data;
		PanObject * task;

		/* add to queue */
		queue_index = back_of_queue ? -1 : 0;
		task = task_decode_new(
			save->group,
			article_get_subject(ARTICLE(l2->data)),
			path,
			(const char*)(filename->data),
			l2);
		task_decode_validate_and_queue_or_self_destruct (
			TASK_DECODE(task),
			queue_index);

		if (filename->next != NULL)
			filename = filename->next;
	}

	/* cleanup */
	g_free (path);
	g_slist_foreach (filenames, (GFunc)g_free, NULL);
	g_slist_free (filenames);
	pan_lock();
	gtk_widget_destroy (GTK_WIDGET(dialog));
	pan_unlock();
}

static void
priority_toggled_cb (GtkToggleButton    * togglebutton,
                     gpointer              user_data)
{
	SaveAsDialogStruct *save = (SaveAsDialogStruct*) user_data;
	gboolean b = gtk_toggle_button_get_active (togglebutton);
	back_of_queue = togglebutton==GTK_TOGGLE_BUTTON(save->back_of_queue_rb) ? b : !b;
}

static void
append_groupname_toggled_cb (GtkToggleButton   *  togglebutton,
                             gpointer             user_data)
{
	append_groupname = gtk_toggle_button_get_active(togglebutton);
}


static void
save_attachment_as_dialog_destroy_cb (GtkObject    * o,
                                      gpointer       user_data)
{
	GSList * l;
	SaveAsDialogStruct *save = (SaveAsDialogStruct*) user_data;

	/* free the save-as-dialog utility structure */
	for (l=save->gslists_of_unowned_articles; l!=NULL; l=l->next)
	{
		GSList* l2 = (GSList*) l->data;
		/*g_slist_foreach (l2, (GFunc)article_free, NULL);*/
		g_slist_free (l2);
	}

	/* cleanup */
	g_slist_free (save->gslists_of_unowned_articles);
	g_free (save);
}

static void
user_specified_path_changed_cb (GtkEditable   * editable,
                                gpointer        user_data)
{
	replace_gstr (&last_save_as_path,
	              gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1));
}

static gint
entry_tickled_cb (GtkWidget * w, gpointer * event, gpointer user_data)
{
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(user_data), TRUE);
	return FALSE;
}

static void
create_save_attachment_as_dialog (Group * group,
                                  GSList * gslists_of_unowned_articles)
{
	gchar * pch;
	GtkWidget * w;
	GtkWidget * hbox;
	GtkWidget * vbox;
	GtkWidget * frame;
	GtkWidget * entry;
	GtkWidget * dialog;
	SaveAsDialogStruct * data;

	data = g_new0 (SaveAsDialogStruct, 1);
	data->group = group;
	data->gslists_of_unowned_articles = gslists_of_unowned_articles;

	pan_lock ();

	/* create & parent the dialog */
	dialog = gnome_dialog_new (_("Pan: Save Attachments As"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gnome_dialog_set_default (GNOME_DIALOG(dialog), 1); /* cancel */

	/* "Location" pane */
	frame = gtk_frame_new (_("Location"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (FALSE, 0);

	/* global default directory */
	w = gtk_radio_button_new_with_label (NULL, download_dir);
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
	data->default_dir_rb = w;

	/* group default directory */
	if (is_nonempty_string (group->download_dir)) {
		w = gtk_radio_button_new_with_label_from_widget (
			GTK_RADIO_BUTTON(w), group->download_dir);
		gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
		data->group_default_dir_rb = w;
	}

	/* user-specified directory */
	hbox = gtk_hbox_new (FALSE, 0);
	w = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON(w));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	data->user_specified_rb = w;
	entry = gnome_file_entry_new ("Save As", _("Download Directory"));
	gtk_signal_connect (
		GTK_OBJECT(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry))),
		"focus_in_event",
		GTK_SIGNAL_FUNC(entry_tickled_cb),
		data->user_specified_rb); 
	gtk_signal_connect (
		GTK_OBJECT(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry))),
		"button_press_event",
		GTK_SIGNAL_FUNC(entry_tickled_cb),
		data->user_specified_rb); 
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (entry), TRUE);
	gtk_widget_set_usize (GTK_WIDGET(entry), 400, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w),
				      last_save_as_path!=NULL);
	if (!is_nonempty_string(last_save_as_path))
		replace_gstr (&last_save_as_path, g_strdup(download_dir));
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (entry))), last_save_as_path);
	gtk_box_pack_start (GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	data->user_specified_entry = entry;
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);


	/* append group name to the path? */
	pch = g_strdup_printf (_("Add ``/%s'' to path"), group->name);
	w = gtk_check_button_new_with_label (pch);
	g_free (pch);
	data->append_groupname_tb = w;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), append_groupname);
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, GNOME_PAD_SMALL);

	/* add "Location" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);




	/* "Filename" pane */
	frame = gtk_frame_new (_("Filename"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (TRUE, 0);

	/* use encoded filename */
	w = gtk_radio_button_new_with_label (NULL, _("Use Encoded Filename"));
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->encoded_filename_rb = w;

	/* use subject */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Article Subject"));
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->subject_filename_rb = w;

	/* user-specified filenames */
	hbox = gtk_hbox_new (FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Custom Filename"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	data->custom_filename_rb = w;
	w = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	data->custom_filename_entry = w;
	gtk_signal_connect (GTK_OBJECT(w), "focus_in_event",
	                    GTK_SIGNAL_FUNC(entry_tickled_cb),
	                    data->custom_filename_rb); 
	gtk_signal_connect (GTK_OBJECT(w), "button_press_event",
	                    GTK_SIGNAL_FUNC(entry_tickled_cb),
	                    data->custom_filename_rb); 
	gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	/* add "Filename" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);



	/* "Priority" pane */
	frame = gtk_frame_new (_("Priority"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (TRUE, 0);

	/* front of queue */
	w = gtk_radio_button_new_with_label (NULL, _("Add Task to the Front of the Queue"));
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), !back_of_queue);
	data->front_of_queue_rb = w;

	/* back of queue (default) */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Add Task to the Back of the Queue"));
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), back_of_queue);
	data->back_of_queue_rb = w;

	/* add "Priority" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);




	/* prep the dialog */
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
		GTK_SIGNAL_FUNC(save_attachment_as_dialog_clicked_cb),
		data);
	gtk_signal_connect (GTK_OBJECT(dialog), "destroy",
		GTK_SIGNAL_FUNC(save_attachment_as_dialog_destroy_cb),
		data); 
	gtk_signal_connect (GTK_OBJECT(data->append_groupname_tb), "toggled",
		append_groupname_toggled_cb, data);
	gtk_signal_connect (GTK_OBJECT(data->front_of_queue_rb), "toggled",
		priority_toggled_cb, data);
	gtk_signal_connect (GTK_OBJECT(data->back_of_queue_rb), "toggled",
		priority_toggled_cb, data);
	gtk_signal_connect (
		GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (entry))), "changed",
		user_specified_path_changed_cb, NULL);

	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock ();
}



void
save_attachment_as (Group* group,
		    GSList* gslists_of_unowned_articles)
{
	create_save_attachment_as_dialog (group,
					  gslists_of_unowned_articles);
}

