/*
 * Pan - A Newsreader for X
 * Copyright (C) 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>

#include <pan/globals.h>
#include <pan/group.h>
#include <pan/queue.h>
#include <pan/task-headers.h>
#include <pan/util.h>

#include <pan/dialogs/dialogs.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	Group * group;

	GtkWidget * dialog;
	GtkWidget * all_headers_rb;
	GtkWidget * all_new_headers_rb;
	GtkWidget * latest_n_headers_rb;
	GtkWidget * n_headers_spinbutton;
	GtkWidget * download_bodies_cb;
}
DownloadHeadersDialogStruct;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
download_headers_clicked_cb (GnomeDialog * dialog, int num, gpointer data)
{
	DownloadHeadersDialogStruct * ui = (DownloadHeadersDialogStruct*)data;

	if (num == 0) /* okay */
	{
		PanObject * task = NULL;

		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->all_headers_rb)))
		{
			task = task_headers_new (ui->group, HEADERS_ALL);
		}
		else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->all_new_headers_rb)))
		{
			task = task_headers_new (ui->group, HEADERS_NEW);
		}
		else /* N headers */
		{
			int sample_size = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(ui->n_headers_spinbutton));
			task = task_headers_new_sample (ui->group, sample_size, FALSE);
		}

		if (task != NULL)
		{
			const gboolean download_bodies = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->download_bodies_cb));
			TASK_HEADERS(task)->download_bodies = download_bodies;
			queue_add (TASK(task));
		}
	}

	/* cleanup */
	gtk_widget_destroy (GTK_WIDGET(dialog));
	g_free (ui);
}

static int
spin_tickled_cb (GtkWidget * w, gpointer * event, gpointer user_data)
{
	/* if a user clicked in the spinbutton window,
	   select the spinbutton radiobutton for them. */
	DownloadHeadersDialogStruct * data = (DownloadHeadersDialogStruct*) user_data;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(data->latest_n_headers_rb), TRUE);
	return FALSE;
}

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

void
dialog_download_headers (Group * group)
{
	DownloadHeadersDialogStruct * dialog;
	GtkAdjustment * adj;
	GtkWidget * vbox;
	GtkWidget * hbox;
	GtkWidget * w;
	gchar * s;

	/* sanity checks */
	g_return_if_fail (group!=NULL);
	g_return_if_fail (!group_is_folder (group));
	g_return_if_fail (is_nonempty_string (group->name));

	/* create dialog */
	dialog = g_new0 (DownloadHeadersDialogStruct, 1);
	dialog->group = group;
	pan_lock ();
	s = g_strdup_printf (
		_("Download Headers for group `%s'"),
		group_get_readable_name(group));
	dialog->dialog = gnome_dialog_new (s,
		GNOME_STOCK_BUTTON_OK,
		GNOME_STOCK_BUTTON_CANCEL,
		NULL);
	g_free (s);

	/**
	*** headers
	**/

	/* all new headers */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_radio_button_new_with_label (NULL, _("Download New Headers"));
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	dialog->all_new_headers_rb = w;

	/* all headers, if group->articles is NULL, this is the default*/
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Download All Headers"));
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	dialog->all_headers_rb = w;
	if (!group->article_qty) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
	}
		
	/* some new headers */
	hbox = gtk_hbox_new (FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Download"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	dialog->latest_n_headers_rb = w;
	adj = GTK_ADJUSTMENT(gtk_adjustment_new (
		150, 0, INT_MAX, 50, 50, 1));
	w = gtk_spin_button_new (adj, 1, 0);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	dialog->n_headers_spinbutton = w;
	gtk_box_pack_start (GTK_BOX(hbox), gtk_label_new (_("Most Recent Headers")), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog->dialog)->vbox), vbox);
	gtk_signal_connect (GTK_OBJECT(w), "focus_in_event", GTK_SIGNAL_FUNC(spin_tickled_cb), dialog);
	gtk_signal_connect (GTK_OBJECT(w), "button_press_event", GTK_SIGNAL_FUNC(spin_tickled_cb), dialog);

	/* 'download bodies too?' togglebutton */
	w = gtk_check_button_new_with_label (_("Also Download Article Bodies"));
	dialog->download_bodies_cb = w;
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, GNOME_PAD_BIG);

	gtk_signal_connect (GTK_OBJECT(dialog->dialog), "clicked", GTK_SIGNAL_FUNC (download_headers_clicked_cb), dialog);
	gnome_dialog_set_parent (GNOME_DIALOG(dialog->dialog),
	                         GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog->dialog);
	pan_unlock ();
}
