/*
 * Pan - A Newsreader for X
 * Copyright (C) 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>  

#include "util.h"
#include "util-wrap.h"

/****
*****  LINE WRAPPING
****/

static void
split_line_to_leader_and_content (const gchar * line,
                                  GString     * setme_leader,
                                  GString     * setme_content)
{
	g_string_truncate (setme_leader, 0);
	g_string_truncate (setme_content, 0);

	for (; is_nonempty_string(line) && (*line=='>' || isspace((int)*line)); ++line)
		g_string_append_c (setme_leader, *line);
	g_string_assign (setme_content, line);
}

typedef struct
{
	gchar * leader;
	gchar * content;
}
Paragraph;

static Paragraph*
paragraph_new (const gchar * leader, const gchar * content)
{
	Paragraph * p = g_new0 (Paragraph, 1);
	p->leader = g_strdup (leader);
	p->content = g_strstrip (g_strdup (content));
	return p;
}

static void
paragraph_free (Paragraph * p)
{
	g_free (p->leader);
	g_free (p->content);
	g_free (p);
}

/**
 * FIXME: doesn't handle paragraphs whose first lines are indented
 */
static GPtrArray*
get_paragraphs (const gchar * body, gint wrap_column)
{
	GPtrArray * lines = g_ptr_array_new ();
	GPtrArray * paragraphs = g_ptr_array_new ();

	/* build an array of lines that are split between leader & content */
	if (1)
	{
		GString * line = g_string_new (NULL);
		GString * leader = g_string_new (NULL);
		GString * content = g_string_new (NULL);

		while (get_next_token_g_str (body, '\n', &body, line))
		{
			split_line_to_leader_and_content (line->str, leader, content);
			g_ptr_array_add (lines, paragraph_new (leader->str, content->str));
		}

		/* add an empty line to make the paragraph-making loop go smoothly */
		g_ptr_array_add (lines, paragraph_new ("", ""));

		/* cleanup step */
		g_string_free (line, TRUE);
		g_string_free (leader, TRUE);
		g_string_free (content, TRUE);
	}

	/* merge the lines into paragraphs */
	if (1)
	{
		gint i;
		gint prev_content_len = 0;
		GString * cur_leader = g_string_new (NULL);
		GString * cur_content = g_string_new (NULL);

		for (i=0; i<lines->len; ++i)
		{
			const Paragraph * p = (const Paragraph*) g_ptr_array_index (lines, i);
			gboolean paragraph_end = TRUE;
			gboolean hard_break = FALSE;

			if (!cur_content->len || !pan_strcmp (p->leader, cur_leader->str))
				paragraph_end = FALSE;

			if (!is_nonempty_string(p->content)) {
				hard_break = prev_content_len!=0;
				paragraph_end = TRUE;
			}

			/* we usually don't want to wrap really short lines */
			if (prev_content_len!=0 && prev_content_len<(wrap_column/2))
				paragraph_end = TRUE;

			if (paragraph_end) /* the new line is a new paragraph, so save the old line */
			{
				g_ptr_array_add (paragraphs, paragraph_new (cur_leader->str, cur_content->str));
				g_string_assign (cur_leader, p->leader);
				g_string_assign (cur_content, p->content);

				if (hard_break)
					g_ptr_array_add (paragraphs, paragraph_new (cur_leader->str, ""));
			}
			else /* append to the content */
			{
				if (cur_content->len)
					g_string_append_c (cur_content, ' ');
				g_string_assign (cur_leader, p->leader);
				g_string_append (cur_content, p->content);
			}

			prev_content_len = is_nonempty_string(p->content) ? strlen (p->content) : 0;
		}

		/* cleanup step */
		g_string_free (cur_leader, TRUE);
		g_string_free (cur_content, TRUE);
	}

	/* remember that empty line we added back up at the top?  We remove it now */
	if (paragraphs->len>0)
		paragraph_free ((Paragraph*) g_ptr_array_remove_index (paragraphs, paragraphs->len-1));

	/* cleanup */
	pan_g_ptr_array_foreach (lines, (GFunc)paragraph_free, NULL);
	g_ptr_array_free (lines, TRUE);

	return paragraphs;
}

static gchar*
wrap_long_lines (gchar   * string,
                 gint      maxlen)
{
	gchar * linefeed_here;
	gchar * pch;
	gchar * line_start;


	/* walk through the entire string */
	linefeed_here = NULL;
	for (line_start=pch=string; *pch; )
	{
		/* a linefeed could go here; remember this space */
		if (isspace((int)*pch) || *pch=='\n')
			linefeed_here = pch;

		/* line's too long; add a linefeed if we can */
		if (pch-line_start>=maxlen && linefeed_here!=NULL)
		{
			*linefeed_here = '\n';
			pch = line_start = linefeed_here + 1;
			linefeed_here = NULL;
		}
		else ++pch;
	}

	return string;
}

static void
fill_paragraph (const Paragraph   * p,
                GPtrArray         * setme,
                gint                wrap_column)
{
	GString * str = g_string_new (NULL);
	gchar * pch = g_strconcat (p->content, "\n", NULL);
	const gchar * march = pch;
	const gint wrap_len = wrap_column - strlen(p->leader);

	wrap_long_lines (pch, wrap_len);
	while ((get_next_token_g_str (march, '\n', &march, str)))
		g_ptr_array_add (setme, g_strconcat(p->leader, str->str, NULL));

	g_free (pch);
	g_string_free (str, TRUE);
}

gchar*
fill_body (const gchar * body, gint wrap_column)
{
	GPtrArray* array = g_ptr_array_new ();
	GPtrArray* paragraphs;
	gchar * new_body;
	gchar * tmp_body;
	gchar * sig;
	gint i;

	/* sanity checks */
	g_return_val_if_fail (is_nonempty_string(body), NULL);

	/* get a temp copy of the body -- we don't wrap the signature. */
	tmp_body = pan_substitute (body, "\r", "");
	sig = strstr (tmp_body, "\n-- \n");
	if (sig != NULL)
		*sig = '\0';

	/* fill the paragraphs */
       	paragraphs = get_paragraphs (tmp_body, wrap_column);
	for (i=0; i<paragraphs->len; ++i)
		fill_paragraph ((Paragraph*)g_ptr_array_index(paragraphs,i), array, wrap_column);

	/* make a single string of all filled lines */
	g_ptr_array_add (array, NULL);
	if (1) {
		gint i;
		GString * s = g_string_new (NULL);
		for (i=0; g_ptr_array_index(array,i)!=NULL; ++i) {
			g_string_append (s, (const char*)g_ptr_array_index(array,i));
			g_string_append_c (s, '\n');
		}
		g_string_truncate (s, s->len-1);
		new_body = s->str;
		g_string_free (s, FALSE);
	}

	/* if we had a sig, put it back in */
	if (sig != NULL) {
		*sig = '\n';
		replace_gstr (&new_body, g_strjoin("\n",new_body, sig, NULL));
	}

	/* cleanup */
	pan_g_ptr_array_foreach (paragraphs, (GFunc)paragraph_free, NULL);
	g_ptr_array_free (paragraphs, TRUE);
	pan_g_ptr_array_foreach (array, (GFunc)g_free, NULL);
	g_ptr_array_free (array, TRUE);
	g_free (tmp_body);

	return new_body;
}


/***
****
***/


gchar*
mute_quoted_text_in_body (const gchar * text)
{
	gchar * retval;
	const gchar * march;
	gboolean last_line_was_quote;
	GString * buf;
	GString * line;

	g_return_val_if_fail (text!=NULL, NULL);

	march = text;
	buf = g_string_new (NULL);
	line = g_string_new (NULL);
	last_line_was_quote = FALSE;
	while (get_next_token_g_str (march, '\n', &march, line))
	{
		const gboolean is_quote = line->len!=0 && line->str[0]=='>';

		/* if we're starting a quote muting, say so */
		if (!is_quote)
		{
			g_string_append (buf, line->str);
			g_string_append_c (buf, '\n');
		}
		else if (!last_line_was_quote)
		{
			g_string_append (buf, _("> [quoted text muted]"));
			g_string_append_c (buf, '\n');
		}

		last_line_was_quote = is_quote;
	}

	retval = buf->str;
	g_string_free (buf, FALSE);
	g_string_free (line, TRUE);
	return retval;
}


gchar*
rot13 (const gchar * text)
{
	gchar * retval = NULL;
	const char * a = "abcdefghijklmnopqrstuvwxyz";

	if (text != NULL)
	{
		gchar * pch;

		retval = g_strdup (text);

		for (pch=retval; *pch!='\0'; ++pch)
		{
			if (isalpha((int)*pch))
			{
				const gboolean uc = isupper((int)*pch);
				*pch = tolower(*pch);
				*pch = a[((*pch-'a')+13)%26];
				if (uc)
					*pch = toupper(*pch);
			}
		}
	}

	return retval;
}

/***
****
****
***/

void
wrap_regression_tests (void)
{
	const gchar * test_type;

	if (1)
	{
		gint wrap_column;
		const gchar * in;
		gchar * out;
		const gchar * expected_out;
		gint test = 1;

		test_type = "fill test";

		wrap_column = 50;
		in = "> a\n\n> b";
		out = fill_body (in, wrap_column);
		expected_out = "> a\n\n> b";
		regression_test_result_va (!strcmp(out,expected_out),
		                           test_type, test++,
		                           "wrap `%s' expected `%s', got `%s'",
					   pan_substitute (in, "\n", "\\n"),
					   pan_substitute (expected_out, "\n", "\\n"),
					   pan_substitute (out, "\n", "\\n"));

		wrap_column = 50;
		in = "> a\n>\n> b";
		out = fill_body (in, wrap_column);
		expected_out = "> a\n>\n> b";
		regression_test_result_va (!strcmp(out,expected_out),
		                           test_type, test++,
		                           "wrap `%s' expected `%s', got `%s'",
					   pan_substitute (in, "\n", "\\n"),
					   pan_substitute (expected_out, "\n", "\\n"),
					   pan_substitute (out, "\n", "\\n"));

		wrap_column = 50;
		in = "> a\n> b\n> c";
		out = fill_body (in, wrap_column);
		expected_out = "> a\n> b\n> c";
		regression_test_result_va (!strcmp(out,expected_out),
		                           test_type, test++,
		                           "wrap `%s' expected `%s', got `%s'",
					   pan_substitute (in, "\n", "\\n"),
					   pan_substitute (expected_out, "\n", "\\n"),
					   pan_substitute (out, "\n", "\\n"));
	}
}
