%{

/*
 This file is part of pybliographer
 
 Copyright (C) 1998 Frederic GOBRY
 Email : gobry@idiap.ch
 	   
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2 
 of the License, or (at your option) any later version.
   
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 
 $Id: bibparse.y,v 1.8 1998/10/28 13:29:11 gobry Exp $
*/

#include "bibtex.h"

extern void bibtex_parser_initialize (bibtex_file *);
extern void bibtex_parser_continue (bibtex_file *);
extern void bibtex_parser_finish (bibtex_file *);

extern int yydebug;

static bibtex_entry * entry = NULL;
static int start_line;
static bibtex_file * bfile;

static void nop (void) { return ;}

void bibtex_next_line (void) { entry->length ++; }

void bibtex_analyzer_initialize (bibtex_file * file)  {
  bibtex_parser_initialize (file);
}

void bibtex_analyzer_finish (bibtex_file * file)  {
  bibtex_parser_finish (file);
}

bibtex_entry * 
parse_bibtex (bibtex_file * file) {
  int ret;

  bfile = file;

  yydebug = file->debug;
  start_line = file->line;

  entry = g_new (bibtex_entry, 1);

  entry->length = 0;
  entry->name = entry->type = entry->body = NULL;
  entry->table = g_hash_table_new (g_str_hash, g_str_equal);

  bibtex_parser_continue (file);
  ret = yyparse ();

  tmp_string_free ();

  if (ret != 0) {
    bibtex_free_entry (entry);
    entry = NULL;
  }

  return entry;
}

 void yyerror (char * s) {
   fprintf (stderr,"error: %d: %s\n", start_line + entry->length, s);
 }
 void yywarning (char * s) {
   fprintf (stderr,"warning: %d: %s\n", start_line + entry->length, s);
 }

%}	

%union{
  char * text;
  bibtex_entry * entry;
}

%token end_of_file
%token <text> L_NAME
%token <text> L_DIGIT
%token <text> L_QUOTED

%type <entry> entry
%type <entry> values
%type <entry> value

%type <text> content
%type <text> inner
%type <text> inner2
%type <text> text_part
%type <text> ext_text_part

%left L_QUOTED
%left <text> '('
%left <text> ')'
%left <text> ','
%left <text> '%'
%left <text> '='
%left <text> '#'
%left <text> '@'
%left <text> '"'
%left <text> '{'
%left <text> '}'
%left <text> '\n'
%left COMPOSE

%% 

/* Les deux types d'entrees */
entry:	  '@' L_NAME '{' values '}' { entry->type = g_strdup ($2); g_strdown (entry->type); YYACCEPT; }
        | '@' L_NAME '(' values ')' { entry->type = g_strdup ($2); g_strdown (entry->type); YYACCEPT; }
	| end_of_file		    { bfile->eof = TRUE; YYABORT; }
	;

/* La liste des valeurs */
values:				    { nop (); }
	| value ',' values
	| value				
	;

/* Une valeur */
value:	  L_NAME '=' content { 

  char * name;
  char * value;
  int len;

  len = strlen ($1);
  while ($1[len - 1] == ' ') {
    $1[len - 1] = '\0';
    len --;
  }

  name = g_hash_table_lookup (entry->table, $1);
  if (name) {
    yywarning ("field redefinition");
    g_free (name);

    name = $1;
  }
  else {
    name = g_strdup ($1);
  }
  value = g_strdup ($3);
  len = strlen (value);

  if (value [len - 1] == '\n') value [len - 1] = '\0';
  g_strdown (name);

  g_hash_table_insert (entry->table, name, value); }

	| L_NAME			{ 

  if (entry->name) {
    yyerror ("name redefinition");
    YYABORT; 
  }
  entry->name = g_strdup ($1); }
	;

content:    ext_text_part '#' content	{ $$ = merge_strings ($1, $3); tmp_string ($$); } 
	  | ext_text_part		
	  ;

ext_text_part:   L_DIGIT { $$ = $1; }
	       | '{' inner '}' { $$ = g_strconcat ($1, $2, $3, NULL); tmp_string ($$); }
	       | L_NAME {

	    gchar * val;
	    gchar * key;
	    
	    if ($1 [0] == '{' || $1 [0] == '"') {
	      $$ = $1;
	    }
	    else {
	      key = g_strdup ($1);
	      tmp_string (key);
	      g_strdown (key);
	      
	      val = g_hash_table_lookup (bfile->table, key);
	      
	      if (val) {
		$$ = g_strdup (val);
		tmp_string ($$);
	      }
	      else {
		yywarning ("string undefined");
		$$ = "\"\"";
	      }
	    }
}
	       | '"' inner2 '"'		{ $$ = g_strconcat ($1, $2, $3, NULL); tmp_string ($$); }
	       ;

text_part:   L_DIGIT			
	   | L_NAME	     		
	   | '{' inner '}'		{ $$ = g_strconcat ($1, $2, $3, NULL); tmp_string ($$); }
	   ;

inner:				{ $$ = ""; }
       | L_QUOTED inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '('  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | ')'  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | ','  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '"'  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '%'  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '='  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '#'  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '@'  inner		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | ' '  inner		{ $$ = g_strconcat (" ", $2, NULL); tmp_string ($$); }
       | '\n' inner		{ $$ = g_strconcat (" ", $2, NULL); bibtex_next_line (); tmp_string ($$);  }
       | text_part inner 	{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
;

inner2:				{ $$ = ""; }
       | L_QUOTED inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '('  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | ')'  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | ','  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '%'  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '='  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '#'  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | '@'  inner2		{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
       | ' '  inner2		{ $$ = g_strconcat (" ", $2, NULL); tmp_string ($$); }
       | '\n' inner2		{ $$ = g_strconcat (" ", $2, NULL); bibtex_next_line (); tmp_string ($$);  }
       | text_part inner2 	{ $$ = g_strconcat ($1, $2, NULL); tmp_string ($$); }
;

%%     
	
