/* valasemanticanalyzer.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include "valasemanticanalyzer.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valascope.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaattribute.h>
#include <vala/valastatement.h>
#include <vala/valaarray.h>
#include <vala/valalockable.h>
#include <vala/valatypeparameter.h>
#include <vala/valaliteral.h>
#include <vala/valaenumvalue.h>
#include <vala/valanamespacereference.h>
#include <vala/valanamespace.h>
#include <vala/valapointer.h>
#include <vala/valaenum.h>
#include <vala/valainvokable.h>

struct _ValaSemanticAnalyzerPrivate {
	gboolean _memory_management;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaSourceFile* current_source_file;
	ValaTypeReference* current_return_type;
	ValaClass* current_class;
	ValaStruct* current_struct;
	GeeCollection* current_using_directives;
	ValaTypeReference* bool_type;
	ValaTypeReference* string_type;
	ValaTypeReference* int_type;
	ValaTypeReference* uint_type;
	ValaTypeReference* ulong_type;
	ValaTypeReference* unichar_type;
	ValaTypeReference* type_type;
	ValaDataType* pointer_type;
	ValaDataType* object_type;
	ValaDataType* initially_unowned_type;
	ValaDataType* glist_type;
	ValaDataType* gslist_type;
	ValaDataType* gerror_type;
	ValaDataType* iterable_type;
	ValaDataType* iterator_type;
	ValaDataType* list_type;
	ValaDataType* map_type;
	gint next_lambda_id;
};
#define VALA_SEMANTIC_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerPrivate))
enum  {
	VALA_SEMANTIC_ANALYZER_DUMMY_PROPERTY,
	VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT
};
static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface, GeeCollection* list);
static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t);
static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr);
static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr);
static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr);
static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr);
static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr);
static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr);
static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_symbol (ValaSemanticAnalyzer* self, ValaSymbol* sym);
static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type);
static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GeeCollection* params, GeeCollection* args);
static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr);
static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr);
static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type);
static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GeeCollection* types);
static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self);
static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_semantic_analyzer_parent_class = NULL;
static void vala_semantic_analyzer_dispose (GObject * obj);


ValaSemanticAnalyzer* vala_semantic_analyzer_new (gboolean manage_memory)
{
	ValaSemanticAnalyzer * self;
	self = g_object_newv (VALA_TYPE_SEMANTIC_ANALYZER, 0, NULL);
	vala_semantic_analyzer_set_memory_management (self, manage_memory);
	return self;
}


/**
 * Analyze and check code in the specified context.
 *
 * @param context a code context
 */
void vala_semantic_analyzer_analyze (ValaSemanticAnalyzer* self, ValaCodeContext* context)
{
	ValaSymbol* __temp0;
	ValaTypeReference* __temp1;
	ValaDataType* __temp2;
	ValaTypeReference* __temp3;
	ValaDataType* __temp4;
	ValaDataType* __temp5;
	ValaTypeReference* __temp6;
	ValaDataType* __temp7;
	ValaTypeReference* __temp8;
	ValaDataType* __temp9;
	ValaTypeReference* __temp10;
	ValaDataType* __temp11;
	ValaTypeReference* __temp12;
	ValaDataType* __temp13;
	ValaSymbol* glib_ns;
	ValaSymbol* gee_ns;
	ValaSymbol* __temp26;
	ValaSymbol* __temp25;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	__temp0 = NULL;
	self->priv->root_symbol = (__temp0 = VALA_SYMBOL (g_object_ref (vala_code_context_get_root (context))), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), __temp0);
	__temp1 = NULL;
	self->priv->bool_type = (__temp1 = vala_type_reference_new (), (self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL))), __temp1);
	__temp2 = NULL;
	vala_type_reference_set_data_type (self->priv->bool_type, (__temp2 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "bool"))));
	(__temp2 == NULL ? NULL : (__temp2 = (g_object_unref (__temp2), NULL)));
	__temp3 = NULL;
	self->priv->string_type = (__temp3 = vala_type_reference_new (), (self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL))), __temp3);
	__temp4 = NULL;
	vala_type_reference_set_data_type (self->priv->string_type, (__temp4 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "string"))));
	(__temp4 == NULL ? NULL : (__temp4 = (g_object_unref (__temp4), NULL)));
	__temp5 = NULL;
	self->priv->pointer_type = (__temp5 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "pointer")), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), __temp5);
	__temp6 = NULL;
	self->priv->int_type = (__temp6 = vala_type_reference_new (), (self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL))), __temp6);
	__temp7 = NULL;
	vala_type_reference_set_data_type (self->priv->int_type, (__temp7 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "int"))));
	(__temp7 == NULL ? NULL : (__temp7 = (g_object_unref (__temp7), NULL)));
	__temp8 = NULL;
	self->priv->uint_type = (__temp8 = vala_type_reference_new (), (self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL))), __temp8);
	__temp9 = NULL;
	vala_type_reference_set_data_type (self->priv->uint_type, (__temp9 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "uint"))));
	(__temp9 == NULL ? NULL : (__temp9 = (g_object_unref (__temp9), NULL)));
	__temp10 = NULL;
	self->priv->ulong_type = (__temp10 = vala_type_reference_new (), (self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL))), __temp10);
	__temp11 = NULL;
	vala_type_reference_set_data_type (self->priv->ulong_type, (__temp11 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "ulong"))));
	(__temp11 == NULL ? NULL : (__temp11 = (g_object_unref (__temp11), NULL)));
	__temp12 = NULL;
	self->priv->unichar_type = (__temp12 = vala_type_reference_new (), (self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL))), __temp12);
	__temp13 = NULL;
	vala_type_reference_set_data_type (self->priv->unichar_type, (__temp13 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "unichar"))));
	(__temp13 == NULL ? NULL : (__temp13 = (g_object_unref (__temp13), NULL)));
	/* TODO: don't require GLib namespace in semantic analyzer*/
	glib_ns = vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "GLib");
	if (glib_ns != NULL) {
		ValaDataType* __temp14;
		ValaDataType* __temp15;
		ValaTypeReference* __temp16;
		ValaDataType* __temp17;
		ValaDataType* __temp18;
		ValaDataType* __temp19;
		ValaDataType* __temp20;
		__temp14 = NULL;
		self->priv->object_type = (__temp14 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->priv->object_type == NULL ? NULL : (self->priv->object_type = (g_object_unref (self->priv->object_type), NULL))), __temp14);
		__temp15 = NULL;
		self->priv->initially_unowned_type = (__temp15 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "InitiallyUnowned")), (self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL))), __temp15);
		__temp16 = NULL;
		self->priv->type_type = (__temp16 = vala_type_reference_new (), (self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL))), __temp16);
		__temp17 = NULL;
		vala_type_reference_set_data_type (self->priv->type_type, (__temp17 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Type"))));
		(__temp17 == NULL ? NULL : (__temp17 = (g_object_unref (__temp17), NULL)));
		__temp18 = NULL;
		self->priv->glist_type = (__temp18 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "List")), (self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL))), __temp18);
		__temp19 = NULL;
		self->priv->gslist_type = (__temp19 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "SList")), (self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL))), __temp19);
		__temp20 = NULL;
		self->priv->gerror_type = (__temp20 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Error")), (self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL))), __temp20);
	}
	gee_ns = vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "Gee");
	if (gee_ns != NULL) {
		ValaDataType* __temp21;
		ValaDataType* __temp22;
		ValaDataType* __temp23;
		ValaDataType* __temp24;
		__temp21 = NULL;
		self->priv->iterable_type = (__temp21 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Iterable")), (self->priv->iterable_type == NULL ? NULL : (self->priv->iterable_type = (g_object_unref (self->priv->iterable_type), NULL))), __temp21);
		__temp22 = NULL;
		self->priv->iterator_type = (__temp22 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Iterator")), (self->priv->iterator_type == NULL ? NULL : (self->priv->iterator_type = (g_object_unref (self->priv->iterator_type), NULL))), __temp22);
		__temp23 = NULL;
		self->priv->list_type = (__temp23 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "List")), (self->priv->list_type == NULL ? NULL : (self->priv->list_type = (g_object_unref (self->priv->list_type), NULL))), __temp23);
		__temp24 = NULL;
		self->priv->map_type = (__temp24 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Map")), (self->priv->map_type == NULL ? NULL : (self->priv->map_type = (g_object_unref (self->priv->map_type), NULL))), __temp24);
	}
	__temp26 = NULL;
	__temp25 = NULL;
	self->priv->current_symbol = (__temp26 = (__temp25 = self->priv->root_symbol, (__temp25 == NULL ? NULL : g_object_ref (__temp25))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp26);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
	(gee_ns == NULL ? NULL : (gee_ns = (g_object_unref (gee_ns), NULL)));
}


static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file)
{
	ValaSemanticAnalyzer * self;
	ValaSourceFile* __temp27;
	GeeCollection* __temp28;
	GeeCollection* __temp29;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	__temp27 = NULL;
	self->priv->current_source_file = (__temp27 = g_object_ref (file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), __temp27);
	__temp28 = NULL;
	self->priv->current_using_directives = (__temp28 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), __temp28);
	self->priv->next_lambda_id = 0;
	vala_source_file_accept_children (file, VALA_CODE_VISITOR (self));
	__temp29 = NULL;
	self->priv->current_using_directives = (__temp29 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), __temp29);
}


static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp30;
	ValaClass* __temp31;
	GeeList* prerequisites;
	GeeList* missing_prereqs;
	ValaSymbol* __temp49;
	ValaSymbol* __temp48;
	ValaClass* __temp50;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	__temp30 = NULL;
	self->priv->current_symbol = (__temp30 = VALA_SYMBOL (g_object_ref (cl)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp30);
	__temp31 = NULL;
	self->priv->current_class = (__temp31 = g_object_ref (cl), (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp31);
	if (vala_class_get_base_class (cl) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_class_get_base_class (cl)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
	}
	{
		GeeCollection* base_type_reference_collection;
		GeeIterator* base_type_reference_it;
		base_type_reference_collection = vala_class_get_base_types (cl);
		base_type_reference_it = gee_iterable_iterator (GEE_ITERABLE (base_type_reference_collection));
		while (gee_iterator_next (base_type_reference_it)) {
			ValaTypeReference* base_type_reference;
			base_type_reference = gee_iterator_get (base_type_reference_it);
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (base_type_reference)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
				(base_type_reference == NULL ? NULL : (base_type_reference = (g_object_unref (base_type_reference), NULL)));
			}
		}
		(base_type_reference_collection == NULL ? NULL : (base_type_reference_collection = (g_object_unref (base_type_reference_collection), NULL)));
		(base_type_reference_it == NULL ? NULL : (base_type_reference_it = (g_object_unref (base_type_reference_it), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	/* gather all prerequisites */
	prerequisites = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), GEE_COLLECTION (prerequisites));
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	/* check whether all prerequisites are met */
	missing_prereqs = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal));
	{
		GeeList* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = prerequisites;
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaDataType* prereq;
			prereq = gee_iterator_get (prereq_it);
			{
				if (!vala_semantic_analyzer_class_is_a (self, cl, prereq)) {
					char* __temp32;
					__temp32 = NULL;
					gee_list_insert (missing_prereqs, 0, (__temp32 = vala_symbol_get_full_name (VALA_SYMBOL (prereq))));
					(__temp32 = (g_free (__temp32), NULL));
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
			}
		}
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
	/* report any missing prerequisites */
	if (gee_collection_get_size (((GeeCollection*) missing_prereqs)) > 0) {
		char* __temp33;
		char* __temp34;
		char* error_string;
		gboolean first;
		char* __temp37;
		vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
		__temp33 = NULL;
		__temp34 = NULL;
		error_string = (__temp34 = g_strdup_printf ("%s: some prerequisites (", (__temp33 = vala_symbol_get_full_name (VALA_SYMBOL (cl)))), (__temp33 = (g_free (__temp33), NULL)), __temp34);
		first = TRUE;
		{
			GeeList* s_collection;
			GeeIterator* s_it;
			s_collection = missing_prereqs;
			s_it = gee_iterable_iterator (GEE_ITERABLE (s_collection));
			while (gee_iterator_next (s_it)) {
				char* s;
				s = gee_iterator_get (s_it);
				{
					if (first) {
						char* __temp35;
						__temp35 = NULL;
						error_string = (__temp35 = g_strdup_printf ("%s`%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp35);
						first = FALSE;
					} else {
						char* __temp36;
						__temp36 = NULL;
						error_string = (__temp36 = g_strdup_printf ("%s, `%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp36);
					}
					(s = (g_free (s), NULL));
				}
			}
			(s_it == NULL ? NULL : (s_it = (g_object_unref (s_it), NULL)));
		}
		__temp37 = NULL;
		error_string = (__temp37 = g_strconcat (error_string, (") are not met"), NULL), (error_string = (g_free (error_string), NULL)), __temp37);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) cl)), error_string);
		(error_string = (g_free (error_string), NULL));
	}
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					ValaInterface* __temp38;
					ValaInterface* iface;
					__temp38 = NULL;
					iface = (__temp38 = VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), (__temp38 == NULL ? NULL : g_object_ref (__temp38)));
					{
						GeeCollection* m_collection;
						GeeIterator* m_it;
						m_collection = vala_interface_get_methods (iface);
						m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
						while (gee_iterator_next (m_it)) {
							ValaMethod* m;
							m = gee_iterator_get (m_it);
							{
								/* We do not need to do expensive equality checking here since this is done
								 * already. We only need to guarantee the symbols are present.
								 
								 check methods */
								if (vala_method_get_is_abstract (m)) {
									ValaSymbol* sym;
									sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) cl)), vala_symbol_get_name (((ValaSymbol*) m)));
									if (sym == NULL || !(VALA_IS_METHOD (sym)) || vala_method_get_base_interface_method ((VALA_METHOD (sym))) != m) {
										char* __temp41;
										char* __temp40;
										char* __temp39;
										vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
										__temp41 = NULL;
										__temp40 = NULL;
										__temp39 = NULL;
										vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) cl)), (__temp41 = g_strdup_printf ("`%s' does not implement interface method `%s'", (__temp39 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (__temp40 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
										(__temp41 = (g_free (__temp41), NULL));
										(__temp40 = (g_free (__temp40), NULL));
										(__temp39 = (g_free (__temp39), NULL));
									}
									(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								}
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							}
						}
						(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
						(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
					}
					(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	/* all abstract symbols defined in base classes have to be implemented in non-abstract classes
	 * VAPI classes don't have to specify overridden methods
	 */
	if (!vala_class_get_is_abstract (cl) && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) cl))))) {
		ValaClass* __temp42;
		ValaClass* base_class;
		__temp42 = NULL;
		base_class = (__temp42 = vala_class_get_base_class (cl), (__temp42 == NULL ? NULL : g_object_ref (__temp42)));
		while (base_class != NULL && vala_class_get_is_abstract (base_class)) {
			ValaClass* __temp47;
			ValaClass* __temp46;
			{
				GeeCollection* m_collection;
				GeeIterator* m_it;
				m_collection = vala_class_get_methods (base_class);
				m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
				while (gee_iterator_next (m_it)) {
					ValaMethod* m;
					m = gee_iterator_get (m_it);
					{
						if (vala_method_get_is_abstract (m)) {
							ValaSymbol* sym;
							sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) cl)), vala_symbol_get_name (((ValaSymbol*) m)));
							if (sym == NULL || !(VALA_IS_METHOD (sym)) || vala_method_get_base_method ((VALA_METHOD (sym))) != m) {
								char* __temp45;
								char* __temp44;
								char* __temp43;
								vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
								__temp45 = NULL;
								__temp44 = NULL;
								__temp43 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) cl)), (__temp45 = g_strdup_printf ("`%s' does not implement abstract method `%s'", (__temp43 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (__temp44 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
								(__temp45 = (g_free (__temp45), NULL));
								(__temp44 = (g_free (__temp44), NULL));
								(__temp43 = (g_free (__temp43), NULL));
							}
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					}
				}
				(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
				(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
			}
			__temp47 = NULL;
			__temp46 = NULL;
			base_class = (__temp47 = (__temp46 = vala_class_get_base_class (base_class), (__temp46 == NULL ? NULL : g_object_ref (__temp46))), (base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL))), __temp47);
		}
		(base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL)));
	}
	__temp49 = NULL;
	__temp48 = NULL;
	self->priv->current_symbol = (__temp49 = (__temp48 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp48 == NULL ? NULL : g_object_ref (__temp48))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp49);
	__temp50 = NULL;
	self->priv->current_class = (__temp50 = NULL, (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp50);
	(prerequisites == NULL ? NULL : (prerequisites = (g_object_unref (prerequisites), NULL)));
	(missing_prereqs == NULL ? NULL : (missing_prereqs = (g_object_unref (missing_prereqs), NULL)));
}


static void vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface, GeeCollection* list)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	g_return_if_fail (list == NULL || GEE_IS_COLLECTION (list));
	{
		GeeCollection* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = vala_interface_get_prerequisites (iface);
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaTypeReference* prereq;
			prereq = gee_iterator_get (prereq_it);
			{
				ValaDataType* __temp51;
				ValaDataType* type;
				__temp51 = NULL;
				type = (__temp51 = vala_type_reference_get_data_type (prereq), (__temp51 == NULL ? NULL : g_object_ref (__temp51)));
				/* skip on previous errors */
				if (type == NULL) {
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					continue;
				}
				gee_collection_add (list, type);
				if (VALA_IS_INTERFACE (type)) {
					vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (type), list);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
}


static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CLASS (cl), FALSE);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	if (VALA_DATA_TYPE (cl) == t) {
		return TRUE;
	}
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				if (VALA_IS_CLASS (vala_type_reference_get_data_type (base_type))) {
					if (vala_semantic_analyzer_class_is_a (self, VALA_CLASS (vala_type_reference_get_data_type (base_type)), t)) {
						gboolean __temp53;
						return (__temp53 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), __temp53);
					}
				} else {
					if (vala_type_reference_get_data_type (base_type) == t) {
						gboolean __temp54;
						return (__temp54 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), __temp54);
					}
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	return FALSE;
}


static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp56;
	ValaStruct* __temp57;
	ValaSymbol* __temp59;
	ValaSymbol* __temp58;
	ValaStruct* __temp60;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	__temp56 = NULL;
	self->priv->current_symbol = (__temp56 = VALA_SYMBOL (g_object_ref (st)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp56);
	__temp57 = NULL;
	self->priv->current_struct = (__temp57 = g_object_ref (st), (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp57);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	__temp59 = NULL;
	__temp58 = NULL;
	self->priv->current_symbol = (__temp59 = (__temp58 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp58 == NULL ? NULL : g_object_ref (__temp58))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp59);
	__temp60 = NULL;
	self->priv->current_struct = (__temp60 = NULL, (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp60);
}


static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp61;
	ValaClass* prereq_class;
	ValaSymbol* __temp70;
	ValaSymbol* __temp69;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	__temp61 = NULL;
	self->priv->current_symbol = (__temp61 = VALA_SYMBOL (g_object_ref (iface)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp61);
	{
		GeeCollection* prerequisite_reference_collection;
		GeeIterator* prerequisite_reference_it;
		prerequisite_reference_collection = vala_interface_get_prerequisites (iface);
		prerequisite_reference_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_reference_collection));
		while (gee_iterator_next (prerequisite_reference_it)) {
			ValaTypeReference* prerequisite_reference;
			prerequisite_reference = gee_iterator_get (prerequisite_reference_it);
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (prerequisite_reference)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
				(prerequisite_reference == NULL ? NULL : (prerequisite_reference = (g_object_unref (prerequisite_reference), NULL)));
			}
		}
		(prerequisite_reference_collection == NULL ? NULL : (prerequisite_reference_collection = (g_object_unref (prerequisite_reference_collection), NULL)));
		(prerequisite_reference_it == NULL ? NULL : (prerequisite_reference_it = (g_object_unref (prerequisite_reference_it), NULL)));
	}
	/* check prerequisites */
	prereq_class = NULL;
	{
		GeeCollection* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = vala_interface_get_prerequisites (iface);
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaTypeReference* prereq;
			prereq = gee_iterator_get (prereq_it);
			{
				ValaDataType* __temp62;
				ValaDataType* class_or_interface;
				__temp62 = NULL;
				class_or_interface = (__temp62 = vala_type_reference_get_data_type (prereq), (__temp62 == NULL ? NULL : g_object_ref (__temp62)));
				/* skip on previous errors */
				if (class_or_interface == NULL) {
					vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
					(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
					continue;
				}
				/* interfaces are not allowed to have multiple instantiable prerequisites */
				if (VALA_IS_CLASS (class_or_interface)) {
					ValaClass* __temp68;
					ValaClass* __temp67;
					if (prereq_class != NULL) {
						char* __temp66;
						char* __temp65;
						char* __temp64;
						char* __temp63;
						vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
						__temp66 = NULL;
						__temp65 = NULL;
						__temp64 = NULL;
						__temp63 = NULL;
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) iface)), (__temp66 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", (__temp63 = vala_symbol_get_full_name (VALA_SYMBOL (iface))), (__temp64 = vala_symbol_get_full_name (VALA_SYMBOL (class_or_interface))), (__temp65 = vala_symbol_get_full_name (VALA_SYMBOL (prereq_class))))));
						(__temp66 = (g_free (__temp66), NULL));
						(__temp65 = (g_free (__temp65), NULL));
						(__temp64 = (g_free (__temp64), NULL));
						(__temp63 = (g_free (__temp63), NULL));
						(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
						(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
						(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
						(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
						(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
						return;
					}
					__temp68 = NULL;
					__temp67 = NULL;
					prereq_class = (__temp68 = (__temp67 = VALA_CLASS (class_or_interface), (__temp67 == NULL ? NULL : g_object_ref (__temp67))), (prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL))), __temp68);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
			}
		}
		(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
	if (prereq_class == NULL) {
		ValaTypeReference* obj_type;
		/* default to GObject */
		obj_type = vala_type_reference_new ();
		vala_type_reference_set_data_type (obj_type, self->priv->object_type);
		vala_interface_prepend_prerequisite (iface, obj_type);
		(obj_type == NULL ? NULL : (obj_type = (g_object_unref (obj_type), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	__temp70 = NULL;
	__temp69 = NULL;
	self->priv->current_symbol = (__temp70 = (__temp69 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp69 == NULL ? NULL : g_object_ref (__temp69))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp70);
	(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
}


static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	vala_code_node_accept_children (VALA_CODE_NODE (cb), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	if (!vala_source_file_get_pkg (self->priv->current_source_file)) {
		if (vala_constant_get_initializer (c) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (c), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) c)), "A const field requires a initializer to be provided");
		}
	}
}


static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_FIELD (f));
	vala_code_node_accept_children (VALA_CODE_NODE (f), VALA_CODE_VISITOR (self));
	if (f->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	} else {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
}


static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp71;
	ValaTypeReference* __temp73;
	ValaTypeReference* __temp72;
	ValaAttribute* init_attr;
	ValaSymbol* __temp75;
	ValaSymbol* __temp74;
	ValaTypeReference* __temp76;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_METHOD (m));
	__temp71 = NULL;
	self->priv->current_symbol = (__temp71 = VALA_SYMBOL (g_object_ref (m)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp71);
	__temp73 = NULL;
	__temp72 = NULL;
	self->priv->current_return_type = (__temp73 = (__temp72 = vala_method_get_return_type (m), (__temp72 == NULL ? NULL : g_object_ref (__temp72))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp73);
	init_attr = vala_code_node_get_attribute (VALA_CODE_NODE (m), "ModuleInit");
	if (init_attr != NULL) {
		vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) m)))), m);
	}
	if (vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL) {
		/* is null if it is void or a reference to a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_method_get_return_type (m))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	}
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	__temp75 = NULL;
	__temp74 = NULL;
	self->priv->current_symbol = (__temp75 = (__temp74 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp74 == NULL ? NULL : g_object_ref (__temp74))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp75);
	__temp76 = NULL;
	self->priv->current_return_type = (__temp76 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp76);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol))) {
		ValaMethod* __temp77;
		ValaMethod* up_method;
		ValaTypeReference* __temp79;
		ValaTypeReference* __temp78;
		/* lambda expressions produce nested methods */
		__temp77 = NULL;
		up_method = (__temp77 = VALA_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol)), (__temp77 == NULL ? NULL : g_object_ref (__temp77)));
		__temp79 = NULL;
		__temp78 = NULL;
		self->priv->current_return_type = (__temp79 = (__temp78 = vala_method_get_return_type (up_method), (__temp78 == NULL ? NULL : g_object_ref (__temp78))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp79);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (self->priv->current_symbol)) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, m, VALA_CLASS (self->priv->current_symbol));
			if (vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				vala_semantic_analyzer_find_base_class_method (self, m, VALA_CLASS (self->priv->current_symbol));
				if (vala_method_get_base_method (m) == NULL) {
					char* __temp81;
					char* __temp80;
					__temp81 = NULL;
					__temp80 = NULL;
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), (__temp81 = g_strdup_printf ("%s: no suitable method found to override", (__temp80 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
					(__temp81 = (g_free (__temp81), NULL));
					(__temp80 = (g_free (__temp80), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (self->priv->current_symbol)) {
			if (vala_method_get_is_abstract (m) || vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				char* __temp83;
				char* __temp82;
				__temp83 = NULL;
				__temp82 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), (__temp83 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (__temp82 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
				(__temp83 = (g_free (__temp83), NULL));
				(__temp82 = (g_free (__temp82), NULL));
				(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
				return;
			}
		}
	}
	(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
}


static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) cl)), vala_symbol_get_name (((ValaSymbol*) m)));
	if (VALA_IS_METHOD (sym)) {
		ValaMethod* __temp84;
		ValaMethod* base_method;
		__temp84 = NULL;
		base_method = (__temp84 = VALA_METHOD (sym), (__temp84 == NULL ? NULL : g_object_ref (__temp84)));
		if (vala_method_get_is_abstract (base_method) || vala_method_get_is_virtual (base_method)) {
			if (!vala_method_equals (m, base_method)) {
				char* __temp87;
				char* __temp86;
				char* __temp85;
				vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
				__temp87 = NULL;
				__temp86 = NULL;
				__temp85 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), (__temp87 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp85 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (__temp86 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
				(__temp87 = (g_free (__temp87), NULL));
				(__temp86 = (g_free (__temp86), NULL));
				(__temp85 = (g_free (__temp85), NULL));
				(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			vala_method_set_base_method (m, base_method);
			(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_method (self, m, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = vala_class_get_base_types (cl);
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				/* FIXME report error if multiple possible base methods are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) vala_type_reference_get_data_type (type))), vala_symbol_get_name (((ValaSymbol*) m)));
					if (VALA_IS_METHOD (sym)) {
						ValaMethod* __temp88;
						ValaMethod* base_method;
						__temp88 = NULL;
						base_method = (__temp88 = VALA_METHOD (sym), (__temp88 == NULL ? NULL : g_object_ref (__temp88)));
						if (vala_method_get_is_abstract (base_method)) {
							if (!vala_method_equals (m, base_method)) {
								char* __temp91;
								char* __temp90;
								char* __temp89;
								vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
								__temp91 = NULL;
								__temp90 = NULL;
								__temp89 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), (__temp91 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp89 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (__temp90 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
								(__temp91 = (g_free (__temp91), NULL));
								(__temp90 = (g_free (__temp90), NULL));
								(__temp89 = (g_free (__temp89), NULL));
								(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
								(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
								return;
							}
							vala_method_set_base_interface_method (m, base_method);
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
							(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp92;
	ValaSymbol* __temp96;
	ValaTypeReference* __temp98;
	ValaTypeReference* __temp97;
	ValaSymbol* __temp100;
	ValaSymbol* __temp99;
	ValaTypeReference* __temp101;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	__temp92 = NULL;
	vala_method_set_return_type (VALA_METHOD (m), (__temp92 = vala_type_reference_new ()));
	(__temp92 == NULL ? NULL : (__temp92 = (g_object_unref (__temp92), NULL)));
	vala_type_reference_set_data_type (vala_method_get_return_type (((ValaMethod*) m)), VALA_DATA_TYPE (vala_symbol_get_parent_symbol (((ValaSymbol*) m))));
	vala_type_reference_set_transfers_ownership (vala_method_get_return_type (((ValaMethod*) m)), TRUE);
	if (VALA_IS_CLASS (self->priv->current_symbol)) {
		ValaClass* __temp93;
		ValaClass* cl;
		/* check for floating reference*/
		__temp93 = NULL;
		cl = (__temp93 = VALA_CLASS (self->priv->current_symbol), (__temp93 == NULL ? NULL : g_object_ref (__temp93)));
		while (cl != NULL) {
			ValaClass* __temp95;
			ValaClass* __temp94;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				vala_type_reference_set_floating_reference (vala_method_get_return_type (((ValaMethod*) m)), TRUE);
				break;
			}
			__temp95 = NULL;
			__temp94 = NULL;
			cl = (__temp95 = (__temp94 = vala_class_get_base_class (cl), (__temp94 == NULL ? NULL : g_object_ref (__temp94))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp95);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
	__temp96 = NULL;
	self->priv->current_symbol = (__temp96 = VALA_SYMBOL (g_object_ref (m)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp96);
	__temp98 = NULL;
	__temp97 = NULL;
	self->priv->current_return_type = (__temp98 = (__temp97 = vala_method_get_return_type (((ValaMethod*) m)), (__temp97 == NULL ? NULL : g_object_ref (__temp97))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp98);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	__temp100 = NULL;
	__temp99 = NULL;
	self->priv->current_symbol = (__temp100 = (__temp99 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp99 == NULL ? NULL : g_object_ref (__temp99))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp100);
	__temp101 = NULL;
	self->priv->current_return_type = (__temp101 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp101);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol))) {
		ValaMethod* __temp102;
		ValaMethod* up_method;
		ValaTypeReference* __temp104;
		ValaTypeReference* __temp103;
		/* lambda expressions produce nested methods */
		__temp102 = NULL;
		up_method = (__temp102 = VALA_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol)), (__temp102 == NULL ? NULL : g_object_ref (__temp102)));
		__temp104 = NULL;
		__temp103 = NULL;
		self->priv->current_return_type = (__temp104 = (__temp103 = vala_method_get_return_type (up_method), (__temp103 == NULL ? NULL : g_object_ref (__temp103))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp104);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (vala_method_get_is_abstract (((ValaMethod*) m)) || vala_method_get_is_virtual (((ValaMethod*) m)) || vala_method_get_overrides (((ValaMethod*) m))) {
		char* __temp106;
		char* __temp105;
		__temp106 = NULL;
		__temp105 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), (__temp106 = g_strdup_printf ("The creation method `%s' cannot be marked as override, virtual, or abstract", (__temp105 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
		(__temp106 = (g_free (__temp106), NULL));
		(__temp105 = (g_free (__temp105), NULL));
		return;
	}
	if (vala_method_get_body (((ValaMethod*) m)) != NULL && self->priv->current_class != NULL) {
		gint n_params;
		n_params = 0;
		{
			GeeCollection* stmt_collection;
			GeeIterator* stmt_it;
			stmt_collection = vala_block_get_statements (vala_method_get_body (((ValaMethod*) m)));
			stmt_it = gee_iterable_iterator (GEE_ITERABLE (stmt_collection));
			while (gee_iterator_next (stmt_it)) {
				ValaStatement* stmt;
				stmt = gee_iterator_get (stmt_it);
				{
					ValaProperty* __temp107;
					gboolean __temp108;
					ValaProperty* __temp109;
					gboolean __temp110;
					__temp107 = NULL;
					if ((__temp108 = !(VALA_IS_EXPRESSION_STATEMENT (stmt)) || (__temp107 = vala_expression_statement_assigned_property ((VALA_EXPRESSION_STATEMENT (stmt)))) == NULL, (__temp107 == NULL ? NULL : (__temp107 = (g_object_unref (__temp107), NULL))), __temp108)) {
						vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "class creation methods only allow property assignment statements");
						(stmt == NULL ? NULL : (stmt = (g_object_unref (stmt), NULL)));
						(stmt_collection == NULL ? NULL : (stmt_collection = (g_object_unref (stmt_collection), NULL)));
						(stmt_it == NULL ? NULL : (stmt_it = (g_object_unref (stmt_it), NULL)));
						return;
					}
					__temp109 = NULL;
					if ((__temp110 = vala_property_accessor_get_construction (vala_property_get_set_accessor ((__temp109 = vala_expression_statement_assigned_property ((VALA_EXPRESSION_STATEMENT (stmt)))))), (__temp109 == NULL ? NULL : (__temp109 = (g_object_unref (__temp109), NULL))), __temp110)) {
						n_params++;
					}
					(stmt == NULL ? NULL : (stmt = (g_object_unref (stmt), NULL)));
				}
			}
			(stmt_collection == NULL ? NULL : (stmt_collection = (g_object_unref (stmt_collection), NULL)));
			(stmt_it == NULL ? NULL : (stmt_it = (g_object_unref (stmt_it), NULL)));
		}
		vala_creation_method_set_n_construction_params (m, n_params);
	}
}


static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
	vala_code_node_accept_children (VALA_CODE_NODE (p), VALA_CODE_VISITOR (self));
	if (!vala_formal_parameter_get_ellipsis (p)) {
		if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (vala_formal_parameter_get_construct_parameter (p)) {
		ValaBlock* __temp111;
		ValaBlock* method_body;
		ValaMemberAccess* left;
		ValaMemberAccess* right;
		ValaSymbol* prop_sym;
		ValaExpressionStatement* __temp115;
		ValaAssignment* __temp114;
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_parent_symbol (((ValaSymbol*) p))))) {
			vala_code_node_set_error (VALA_CODE_NODE (p), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) p)), "construct parameters are only allowed in type creation methods");
			return;
		}
		__temp111 = NULL;
		method_body = (__temp111 = vala_method_get_body (((ValaMethod*) (VALA_CREATION_METHOD (vala_symbol_get_parent_symbol (((ValaSymbol*) p)))))), (__temp111 == NULL ? NULL : g_object_ref (__temp111)));
		left = vala_member_access_new_simple (vala_symbol_get_name (((ValaSymbol*) p)), NULL);
		right = vala_member_access_new_simple (vala_symbol_get_name (((ValaSymbol*) p)), NULL);
		/* try to lookup the requested property */
		prop_sym = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (self->priv->current_class), vala_symbol_get_name (((ValaSymbol*) p)));
		if (!(VALA_IS_PROPERTY (prop_sym))) {
			char* __temp113;
			char* __temp112;
			vala_code_node_set_error (VALA_CODE_NODE (p), TRUE);
			__temp113 = NULL;
			__temp112 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) p)), (__temp113 = g_strdup_printf ("class `%s' does not contain a property named `%s'", (__temp112 = vala_symbol_get_full_name (VALA_SYMBOL (self->priv->current_class))), vala_symbol_get_name (((ValaSymbol*) p)))));
			(__temp113 = (g_free (__temp113), NULL));
			(__temp112 = (g_free (__temp112), NULL));
			(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
			(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
			(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
			(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
			return;
		}
		vala_expression_set_symbol_reference (VALA_EXPRESSION (left), prop_sym);
		vala_expression_set_symbol_reference (VALA_EXPRESSION (right), VALA_SYMBOL (p));
		__temp115 = NULL;
		__temp114 = NULL;
		vala_block_add_statement (method_body, VALA_STATEMENT ((__temp115 = vala_expression_statement_new (VALA_EXPRESSION ((__temp114 = vala_assignment_new (VALA_EXPRESSION (left), VALA_EXPRESSION (right), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL))), NULL))));
		(__temp115 == NULL ? NULL : (__temp115 = (g_object_unref (__temp115), NULL)));
		(__temp114 == NULL ? NULL : (__temp114 = (g_object_unref (__temp114), NULL)));
		(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
		(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
		(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
		(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
	}
}


static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) cl)), vala_symbol_get_name (((ValaSymbol*) prop)));
	if (VALA_IS_PROPERTY (sym)) {
		ValaProperty* __temp116;
		ValaProperty* base_property;
		__temp116 = NULL;
		base_property = (__temp116 = VALA_PROPERTY (sym), (__temp116 == NULL ? NULL : g_object_ref (__temp116)));
		if (vala_property_get_is_abstract (base_property) || vala_property_get_is_virtual (base_property)) {
			if (!vala_property_equals (prop, base_property)) {
				char* __temp119;
				char* __temp118;
				char* __temp117;
				vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
				__temp119 = NULL;
				__temp118 = NULL;
				__temp117 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) prop)), (__temp119 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp117 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (__temp118 = vala_symbol_get_full_name (VALA_SYMBOL (base_property))))));
				(__temp119 = (g_free (__temp119), NULL));
				(__temp118 = (g_free (__temp118), NULL));
				(__temp117 = (g_free (__temp117), NULL));
				(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			vala_property_set_base_property (prop, base_property);
			(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_property (self, prop, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = vala_class_get_base_types (cl);
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				/* FIXME report error if multiple possible base properties are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) vala_type_reference_get_data_type (type))), vala_symbol_get_name (((ValaSymbol*) prop)));
					if (VALA_IS_PROPERTY (sym)) {
						ValaProperty* __temp120;
						ValaProperty* base_property;
						__temp120 = NULL;
						base_property = (__temp120 = VALA_PROPERTY (sym), (__temp120 == NULL ? NULL : g_object_ref (__temp120)));
						if (vala_property_get_is_abstract (base_property)) {
							if (!vala_property_equals (prop, base_property)) {
								char* __temp123;
								char* __temp122;
								char* __temp121;
								vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
								__temp123 = NULL;
								__temp122 = NULL;
								__temp121 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) prop)), (__temp123 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp121 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (__temp122 = vala_symbol_get_full_name (VALA_SYMBOL (base_property))))));
								(__temp123 = (g_free (__temp123), NULL));
								(__temp122 = (g_free (__temp122), NULL));
								(__temp121 = (g_free (__temp121), NULL));
								(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
								(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
								return;
							}
							vala_property_set_base_interface_property (prop, base_property);
							(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
							(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp124;
	ValaSymbol* __temp126;
	ValaSymbol* __temp125;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	__temp124 = NULL;
	self->priv->current_symbol = (__temp124 = VALA_SYMBOL (g_object_ref (prop)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp124);
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
	__temp126 = NULL;
	__temp125 = NULL;
	self->priv->current_symbol = (__temp126 = (__temp125 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp125 == NULL ? NULL : g_object_ref (__temp125))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp126);
	if (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)) != NULL) {
		/* is null if it references a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (((ValaSymbol*) prop)))) {
		ValaClass* __temp127;
		ValaClass* cl;
		__temp127 = NULL;
		cl = (__temp127 = VALA_CLASS (vala_symbol_get_parent_symbol (((ValaSymbol*) prop))), (__temp127 == NULL ? NULL : g_object_ref (__temp127)));
		vala_semantic_analyzer_find_base_interface_property (self, prop, cl);
		if (vala_property_get_is_virtual (prop) || vala_property_get_overrides (prop)) {
			vala_semantic_analyzer_find_base_class_property (self, prop, cl);
			if (vala_property_get_base_property (prop) == NULL) {
				char* __temp129;
				char* __temp128;
				vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
				__temp129 = NULL;
				__temp128 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) prop)), (__temp129 = g_strdup_printf ("%s: no suitable property found to override", (__temp128 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
				(__temp129 = (g_free (__temp129), NULL));
				(__temp128 = (g_free (__temp128), NULL));
			}
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp142;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	vala_property_accessor_set_prop (acc, VALA_PROPERTY (self->priv->current_symbol));
	if (vala_property_accessor_get_readable (acc)) {
		ValaTypeReference* __temp130;
		__temp130 = NULL;
		self->priv->current_return_type = (__temp130 = g_object_ref (vala_property_get_type_reference (vala_property_accessor_get_prop (acc))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp130);
	} else {
		ValaTypeReference* __temp131;
		/* void*/
		__temp131 = NULL;
		self->priv->current_return_type = (__temp131 = vala_type_reference_new (), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp131);
	}
	if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) acc))))) {
		if (vala_property_accessor_get_body (acc) == NULL && !vala_property_get_interface_only (vala_property_accessor_get_prop (acc)) && !vala_property_get_is_abstract (vala_property_accessor_get_prop (acc))) {
			ValaBlock* __temp132;
			/* no accessor body specified, insert default body */
			__temp132 = NULL;
			vala_property_accessor_set_body (acc, (__temp132 = vala_block_new (NULL)));
			(__temp132 == NULL ? NULL : (__temp132 = (g_object_unref (__temp132), NULL)));
			if (vala_property_accessor_get_readable (acc)) {
				ValaReturnStatement* __temp135;
				ValaMemberAccess* __temp134;
				char* __temp133;
				__temp135 = NULL;
				__temp134 = NULL;
				__temp133 = NULL;
				vala_block_add_statement (vala_property_accessor_get_body (acc), VALA_STATEMENT ((__temp135 = vala_return_statement_new (VALA_EXPRESSION ((__temp134 = vala_member_access_new_simple ((__temp133 = g_strdup_printf ("_%s", vala_symbol_get_name (((ValaSymbol*) vala_property_accessor_get_prop (acc))))), NULL))), NULL))));
				(__temp135 == NULL ? NULL : (__temp135 = (g_object_unref (__temp135), NULL)));
				(__temp134 == NULL ? NULL : (__temp134 = (g_object_unref (__temp134), NULL)));
				(__temp133 = (g_free (__temp133), NULL));
			} else {
				ValaExpressionStatement* __temp140;
				ValaAssignment* __temp139;
				ValaMemberAccess* __temp138;
				ValaMemberAccess* __temp137;
				char* __temp136;
				__temp140 = NULL;
				__temp139 = NULL;
				__temp138 = NULL;
				__temp137 = NULL;
				__temp136 = NULL;
				vala_block_add_statement (vala_property_accessor_get_body (acc), VALA_STATEMENT ((__temp140 = vala_expression_statement_new (VALA_EXPRESSION ((__temp139 = vala_assignment_new (VALA_EXPRESSION ((__temp137 = vala_member_access_new_simple ((__temp136 = g_strdup_printf ("_%s", vala_symbol_get_name (((ValaSymbol*) vala_property_accessor_get_prop (acc))))), NULL))), VALA_EXPRESSION ((__temp138 = vala_member_access_new_simple ("value", NULL))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL))), NULL))));
				(__temp140 == NULL ? NULL : (__temp140 = (g_object_unref (__temp140), NULL)));
				(__temp139 == NULL ? NULL : (__temp139 = (g_object_unref (__temp139), NULL)));
				(__temp138 == NULL ? NULL : (__temp138 = (g_object_unref (__temp138), NULL)));
				(__temp137 == NULL ? NULL : (__temp137 = (g_object_unref (__temp137), NULL)));
				(__temp136 = (g_free (__temp136), NULL));
			}
		}
		if (vala_property_accessor_get_writable (acc) || vala_property_accessor_get_construction (acc)) {
			ValaFormalParameter* __temp141;
			__temp141 = NULL;
			vala_property_accessor_set_value_parameter (acc, (__temp141 = vala_formal_parameter_new ("value", vala_property_get_type_reference (vala_property_accessor_get_prop (acc)), NULL)));
			(__temp141 == NULL ? NULL : (__temp141 = (g_object_unref (__temp141), NULL)));
			vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) vala_property_accessor_get_body (acc))), vala_symbol_get_name (((ValaSymbol*) vala_property_accessor_get_value_parameter (acc))), VALA_SYMBOL (vala_property_accessor_get_value_parameter (acc)));
		}
	}
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
	__temp142 = NULL;
	self->priv->current_return_type = (__temp142 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp142);
}


static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	vala_code_node_accept_children (VALA_CODE_NODE (sig), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c)
{
	ValaSemanticAnalyzer * self;
	ValaFormalParameter* __temp144;
	ValaTypeReference* __temp143;
	ValaSymbol* __temp145;
	ValaSymbol* __temp147;
	ValaSymbol* __temp146;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	__temp144 = NULL;
	__temp143 = NULL;
	vala_constructor_set_this_parameter (c, (__temp144 = vala_formal_parameter_new ("this", (__temp143 = vala_type_reference_new ()), NULL)));
	(__temp144 == NULL ? NULL : (__temp144 = (g_object_unref (__temp144), NULL)));
	(__temp143 == NULL ? NULL : (__temp143 = (g_object_unref (__temp143), NULL)));
	vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_constructor_get_this_parameter (c)), VALA_DATA_TYPE (self->priv->current_symbol));
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) c)), vala_symbol_get_name (((ValaSymbol*) vala_constructor_get_this_parameter (c))), VALA_SYMBOL (vala_constructor_get_this_parameter (c)));
	vala_symbol_set_owner (VALA_SYMBOL (c), vala_symbol_get_scope (self->priv->current_symbol));
	__temp145 = NULL;
	self->priv->current_symbol = (__temp145 = VALA_SYMBOL (g_object_ref (c)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp145);
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	__temp147 = NULL;
	__temp146 = NULL;
	self->priv->current_symbol = (__temp147 = (__temp146 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp146 == NULL ? NULL : g_object_ref (__temp146))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp147);
}


static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp148;
	ValaSymbol* __temp150;
	ValaSymbol* __temp149;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	vala_symbol_set_owner (VALA_SYMBOL (d), vala_symbol_get_scope (self->priv->current_symbol));
	__temp148 = NULL;
	self->priv->current_symbol = (__temp148 = VALA_SYMBOL (g_object_ref (d)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp148);
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
	__temp150 = NULL;
	__temp149 = NULL;
	self->priv->current_symbol = (__temp150 = (__temp149 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp149 == NULL ? NULL : g_object_ref (__temp149))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp150);
}


static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
}


static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp151;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	vala_symbol_set_owner (VALA_SYMBOL (b), vala_symbol_get_scope (self->priv->current_symbol));
	__temp151 = NULL;
	self->priv->current_symbol = (__temp151 = VALA_SYMBOL (g_object_ref (b)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp151);
}


static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp153;
	ValaSymbol* __temp152;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (b);
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = gee_iterator_get (decl_it);
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	__temp153 = NULL;
	__temp152 = NULL;
	self->priv->current_symbol = (__temp153 = (__temp152 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp152 == NULL ? NULL : g_object_ref (__temp152))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp153);
}


static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl)
{
	ValaSemanticAnalyzer * self;
	ValaBlock* __temp160;
	ValaBlock* block;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_type_reference (decl) == NULL) {
		ValaTypeReference* __temp154;
		/* var type */
		if (vala_variable_declarator_get_initializer (decl) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) decl)), "var declaration not allowed without initializer");
			return;
		}
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) decl)), "var declaration not allowed with non-typed initializer");
			return;
		}
		__temp154 = NULL;
		vala_variable_declarator_set_type_reference (decl, (__temp154 = vala_type_reference_copy (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))));
		(__temp154 = (g_object_unref (__temp154), NULL));
		vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (decl), (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) == NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))));
		vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (decl), FALSE);
	}
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_variable_declarator_get_initializer (decl)))) {
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) decl)), "expression type not allowed as initializer");
				return;
			}
			if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
				ValaMethod* __temp155;
				ValaMethod* m;
				ValaCallback* __temp156;
				ValaCallback* cb;
				__temp155 = NULL;
				m = (__temp155 = VALA_METHOD (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl))), (__temp155 == NULL ? NULL : g_object_ref (__temp155)));
				__temp156 = NULL;
				cb = (__temp156 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp156 == NULL ? NULL : g_object_ref (__temp156)));
				/* check whether method matches callback type */
				if (!vala_callback_matches_method (cb, m)) {
					char* __temp159;
					char* __temp158;
					char* __temp157;
					vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
					__temp159 = NULL;
					__temp158 = NULL;
					__temp157 = NULL;
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) decl)), (__temp159 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp157 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (__temp158 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
					(__temp159 = (g_free (__temp159), NULL));
					(__temp158 = (g_free (__temp158), NULL));
					(__temp157 = (g_free (__temp157), NULL));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					return;
				}
				vala_expression_set_static_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) decl)), "expression type not allowed as initializer");
				return;
			}
		}
		if (vala_semantic_analyzer_get_memory_management (self)) {
			if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))) {
				/* rhs transfers ownership of the expression */
				if (!vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
					/* lhs doesn't own the value */
					vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) decl)), "Invalid assignment from owned expression to unowned variable");
					return;
				}
			}
		}
	}
	if (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	vala_scope_add (vala_symbol_get_scope (self->priv->current_symbol), vala_symbol_get_name (((ValaSymbol*) decl)), VALA_SYMBOL (decl));
	__temp160 = NULL;
	block = (__temp160 = VALA_BLOCK (self->priv->current_symbol), (__temp160 == NULL ? NULL : g_object_ref (__temp160)));
	vala_block_add_local_variable (block, decl);
	vala_symbol_set_active (VALA_SYMBOL (decl), TRUE);
	(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (((ValaExpression*) list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) list))))) {
		ValaArray* __temp161;
		ValaArray* edt;
		GeeCollection* inits;
		gint rank;
		ValaTypeReference* child_type;
		/* initializer is used as array initializer */
		__temp161 = NULL;
		edt = (__temp161 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) list)))), (__temp161 == NULL ? NULL : g_object_ref (__temp161)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank ((VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) list))))));
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (((ValaExpression*) list)));
		if (rank > 1) {
			ValaArray* __temp162;
			__temp162 = NULL;
			vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp162 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1))));
			(__temp162 = (g_object_unref (__temp162), NULL));
		} else {
			vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt));
		}
		{
			GeeCollection* e_collection;
			GeeIterator* e_it;
			e_collection = inits;
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = gee_iterator_get (e_it);
				{
					ValaTypeReference* __temp163;
					__temp163 = NULL;
					vala_expression_set_expected_type (e, (__temp163 = vala_type_reference_copy (child_type)));
					(__temp163 = (g_object_unref (__temp163), NULL));
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_object_unref (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (((ValaExpression*) list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) list))))) {
		ValaArray* __temp164;
		ValaArray* edt;
		GeeCollection* inits;
		gint rank;
		ValaTypeReference* child_type;
		gboolean error;
		__temp164 = NULL;
		edt = (__temp164 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) list)))), (__temp164 == NULL ? NULL : g_object_ref (__temp164)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank (edt);
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (((ValaExpression*) list)));
		error = FALSE;
		if (rank > 1) {
			ValaArray* __temp165;
			__temp165 = NULL;
			vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp165 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1))));
			(__temp165 = (g_object_unref (__temp165), NULL));
			{
				GeeCollection* e_collection;
				GeeIterator* e_it;
				e_collection = inits;
				e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
				while (gee_iterator_next (e_it)) {
					ValaExpression* e;
					e = gee_iterator_get (e_it);
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
							continue;
						}
						if (!(VALA_IS_INITIALIZER_LIST (e))) {
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) e)), "Initializer list expected");
							(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
							continue;
						}
						if (!vala_type_reference_equals (vala_expression_get_static_type (e), child_type)) {
							char* __temp166;
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							__temp166 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) e)), (__temp166 = g_strdup_printf ("Expected initializer list of type `%s' but got `%s'", vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_data_type (child_type))), vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))));
							(__temp166 = (g_free (__temp166), NULL));
						}
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
					}
				}
				(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
			}
		} else {
			vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt));
			{
				GeeCollection* e_collection;
				GeeIterator* e_it;
				e_collection = inits;
				e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
				while (gee_iterator_next (e_it)) {
					ValaExpression* e;
					e = gee_iterator_get (e_it);
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
							continue;
						}
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (e), child_type)) {
							char* __temp167;
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							__temp167 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) e)), (__temp167 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_data_type (child_type))), vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))));
							(__temp167 = (g_free (__temp167), NULL));
						}
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
					}
				}
				(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
			}
		}
		if (!error) {
			/* everything seems to be correct */
			vala_expression_set_static_type (VALA_EXPRESSION (list), vala_expression_get_expected_type (((ValaExpression*) list)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_object_unref (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	if (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)))) {
		vala_report_warning (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Short-living reference");
	}
	vala_code_node_set_tree_can_fail (VALA_CODE_NODE (stmt), vala_code_node_get_tree_can_fail (((ValaCodeNode*) vala_expression_statement_get_expression (stmt))));
}


static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_if_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_if_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_if_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_while_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_while_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_for_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_for_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* __temp168;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp168 = NULL;
	vala_foreach_statement_set_variable_declarator (stmt, (__temp168 = vala_variable_declarator_new (vala_foreach_statement_get_variable_name (stmt), NULL, NULL)));
	(__temp168 == NULL ? NULL : (__temp168 = (g_object_unref (__temp168), NULL)));
	vala_variable_declarator_set_type_reference (vala_foreach_statement_get_variable_declarator (stmt), vala_foreach_statement_get_type_reference (stmt));
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) vala_foreach_statement_get_body (stmt))), vala_foreach_statement_get_variable_name (stmt), VALA_SYMBOL (vala_foreach_statement_get_variable_declarator (stmt)));
	vala_block_add_local_variable (vala_foreach_statement_get_body (stmt), vala_foreach_statement_get_variable_declarator (stmt));
	vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_variable_declarator (stmt)), TRUE);
}


static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* __temp170;
	char* __temp169;
	ValaTypeReference* __temp171;
	ValaDataType* __temp172;
	ValaDataType* collection_type;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	__temp170 = NULL;
	__temp169 = NULL;
	vala_foreach_statement_set_collection_variable_declarator (stmt, (__temp170 = vala_variable_declarator_new ((__temp169 = g_strdup_printf ("%s_collection", vala_foreach_statement_get_variable_name (stmt))), NULL, NULL)));
	(__temp170 == NULL ? NULL : (__temp170 = (g_object_unref (__temp170), NULL)));
	(__temp169 = (g_free (__temp169), NULL));
	__temp171 = NULL;
	vala_variable_declarator_set_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt), (__temp171 = vala_type_reference_copy (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt)))));
	(__temp171 = (g_object_unref (__temp171), NULL));
	vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt)), FALSE);
	vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt)), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))));
	vala_block_add_local_variable (VALA_BLOCK (stmt), vala_foreach_statement_get_collection_variable_declarator (stmt));
	vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_collection_variable_declarator (stmt)), TRUE);
	__temp172 = NULL;
	collection_type = (__temp172 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))), (__temp172 == NULL ? NULL : g_object_ref (__temp172)));
	if (self->priv->iterable_type != NULL && (collection_type == self->priv->iterable_type || vala_data_type_is_subtype_of (collection_type, self->priv->iterable_type))) {
		ValaVariableDeclarator* __temp174;
		char* __temp173;
		ValaTypeReference* __temp175;
		__temp174 = NULL;
		__temp173 = NULL;
		vala_foreach_statement_set_iterator_variable_declarator (stmt, (__temp174 = vala_variable_declarator_new ((__temp173 = g_strdup_printf ("%s_it", vala_foreach_statement_get_variable_name (stmt))), NULL, NULL)));
		(__temp174 == NULL ? NULL : (__temp174 = (g_object_unref (__temp174), NULL)));
		(__temp173 = (g_free (__temp173), NULL));
		__temp175 = NULL;
		vala_variable_declarator_set_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt), (__temp175 = vala_type_reference_new ()));
		(__temp175 == NULL ? NULL : (__temp175 = (g_object_unref (__temp175), NULL)));
		vala_type_reference_set_data_type (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), self->priv->iterator_type);
		vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), TRUE);
		vala_type_reference_add_type_argument (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), vala_foreach_statement_get_type_reference (stmt));
		vala_block_add_local_variable (VALA_BLOCK (stmt), vala_foreach_statement_get_iterator_variable_declarator (stmt));
		vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_iterator_variable_declarator (stmt)), TRUE);
	} else {
		if (!(VALA_IS_ARRAY (collection_type) || collection_type == self->priv->glist_type || collection_type == self->priv->gslist_type)) {
			vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Collection not iterable");
			(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
			return;
		}
	}
	(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	if (self->priv->current_return_type == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Return not allowed in this context");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) == NULL && vala_type_reference_get_data_type (self->priv->current_return_type) != NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Return without value in non-void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_data_type (self->priv->current_return_type) == NULL && vala_type_reference_get_type_parameter (self->priv->current_return_type) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Return with value in void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type)) {
		char* __temp178;
		char* __temp177;
		char* __temp176;
		__temp178 = NULL;
		__temp177 = NULL;
		__temp176 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), (__temp178 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", (__temp176 = vala_type_reference_to_string (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)))), (__temp177 = vala_type_reference_to_string (self->priv->current_return_type)))));
		(__temp178 = (g_free (__temp178), NULL));
		(__temp177 = (g_free (__temp177), NULL));
		(__temp176 = (g_free (__temp176), NULL));
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt))) && vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_report_warning (vala_code_node_get_source_reference (((ValaCodeNode*) stmt)), "Local variable with strong reference used as return value and method return type hasn't been declared to transfer ownership");
	}
}


static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause)
{
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* __temp179;
	ValaTypeReference* __temp180;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
	if (vala_type_reference_get_data_type (vala_catch_clause_get_type_reference (clause)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_catch_clause_get_type_reference (clause))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp179 = NULL;
	vala_catch_clause_set_variable_declarator (clause, (__temp179 = vala_variable_declarator_new (vala_catch_clause_get_variable_name (clause), NULL, NULL)));
	(__temp179 == NULL ? NULL : (__temp179 = (g_object_unref (__temp179), NULL)));
	__temp180 = NULL;
	vala_variable_declarator_set_type_reference (vala_catch_clause_get_variable_declarator (clause), (__temp180 = vala_type_reference_new ()));
	(__temp180 == NULL ? NULL : (__temp180 = (g_object_unref (__temp180), NULL)));
	vala_type_reference_set_data_type (vala_variable_declarator_get_type_reference (vala_catch_clause_get_variable_declarator (clause)), self->priv->gerror_type);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) vala_catch_clause_get_body (clause))), vala_catch_clause_get_variable_name (clause), VALA_SYMBOL (vala_catch_clause_get_variable_declarator (clause)));
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


/**
 * Visit operation called for lock statements.
 *
 * @param stmt a lock statement
 */
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_LOCK_STATEMENT (stmt));
	/* resource must be a member access and denote a Lockable */
	if (!(VALA_IS_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt)) && VALA_IS_LOCKABLE (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_lock_statement_get_resource (stmt))), "Expression is either not a member access or does not denote a lockable member");
		return;
	}
	/* parent symbol must be the current class */
	if (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))) != VALA_SYMBOL (self->priv->current_class)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_lock_statement_get_resource (stmt))), "Only members of the current class are lockable");
	}
	vala_lockable_set_lock_used ((VALA_LOCKABLE (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))), TRUE);
}


static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL) {
		ValaTypeReference* __temp181;
		ValaArray* __temp182;
		__temp181 = NULL;
		vala_expression_set_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)), (__temp181 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr))));
		(__temp181 = (g_object_unref (__temp181), NULL));
		__temp182 = NULL;
		vala_type_reference_set_data_type (vala_expression_get_expected_type (((ValaExpression*) vala_array_creation_expression_get_initializer_list (expr))), VALA_DATA_TYPE ((__temp182 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) vala_array_creation_expression_get_initializer_list (expr)))), vala_array_creation_expression_get_rank (expr)))));
		(__temp182 = (g_object_unref (__temp182), NULL));
	}
}


/* FIXME: add element type to type_argument
*
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	GeeCollection* size;
	ValaTypeReference* __temp183;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	size = GEE_COLLECTION (vala_array_creation_expression_get_sizes (expr));
	/* check for errors in the size list */
	if (size != NULL) {
		{
			GeeCollection* e_collection;
			GeeIterator* e_it;
			e_collection = size;
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = gee_iterator_get (e_it);
				{
					if (vala_expression_get_static_type (e) == NULL) {
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
						(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
						return;
					} else {
						/* return on previous error */
						if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) e)), "Expression of integer type expected");
						}
					}
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
		if (vala_code_node_get_error (((ValaCodeNode*) expr))) {
			(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
			return;
		}
	}
	/* check for wrong elements inside the initializer */
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL && vala_expression_get_static_type (((ValaExpression*) vala_array_creation_expression_get_initializer_list (expr))) == NULL) {
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		return;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (expr) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Cannot determine the element type of the created array");
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		return;
	}
	__temp183 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp183 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr))));
	(__temp183 = (g_object_unref (__temp183), NULL));
	if (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)) != NULL) {
		ValaArray* __temp184;
		__temp184 = NULL;
		vala_type_reference_set_data_type (vala_expression_get_static_type (((ValaExpression*) expr)), VALA_DATA_TYPE ((__temp184 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr)))));
		(__temp184 = (g_object_unref (__temp184), NULL));
	} else {
		ValaArray* __temp185;
		__temp185 = NULL;
		vala_type_reference_set_data_type (vala_expression_get_static_type (((ValaExpression*) expr)), VALA_DATA_TYPE ((__temp185 = vala_type_parameter_get_array (vala_type_reference_get_type_parameter (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr)))));
		(__temp185 = (g_object_unref (__temp185), NULL));
	}
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), TRUE);
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), TRUE);
	vala_type_reference_add_type_argument (vala_expression_get_static_type (((ValaExpression*) expr)), vala_array_creation_expression_get_element_type (expr));
	(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
}


static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_BOOLEAN_LITERAL (expr));
	vala_literal_set_static_type (VALA_LITERAL (expr), self->priv->bool_type);
}


static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp186;
	ValaDataType* __temp187;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (expr));
	__temp186 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (__temp186 = vala_type_reference_new ()));
	(__temp186 == NULL ? NULL : (__temp186 = (g_object_unref (__temp186), NULL)));
	__temp187 = NULL;
	vala_type_reference_set_data_type (vala_literal_get_static_type (((ValaLiteral*) expr)), (__temp187 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "char"))));
	(__temp187 == NULL ? NULL : (__temp187 = (g_object_unref (__temp187), NULL)));
}


static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp188;
	ValaDataType* __temp190;
	char* __temp189;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (expr));
	__temp188 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (__temp188 = vala_type_reference_new ()));
	(__temp188 == NULL ? NULL : (__temp188 = (g_object_unref (__temp188), NULL)));
	__temp190 = NULL;
	__temp189 = NULL;
	vala_type_reference_set_data_type (vala_literal_get_static_type (((ValaLiteral*) expr)), (__temp190 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), (__temp189 = vala_integer_literal_get_type_name (expr))))));
	(__temp190 == NULL ? NULL : (__temp190 = (g_object_unref (__temp190), NULL)));
	(__temp189 = (g_free (__temp189), NULL));
}


static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp191;
	ValaDataType* __temp193;
	char* __temp192;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_REAL_LITERAL (expr));
	__temp191 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (__temp191 = vala_type_reference_new ()));
	(__temp191 == NULL ? NULL : (__temp191 = (g_object_unref (__temp191), NULL)));
	__temp193 = NULL;
	__temp192 = NULL;
	vala_type_reference_set_data_type (vala_literal_get_static_type (((ValaLiteral*) expr)), (__temp193 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), (__temp192 = vala_real_literal_get_type_name (expr))))));
	(__temp193 == NULL ? NULL : (__temp193 = (g_object_unref (__temp193), NULL)));
	(__temp192 = (g_free (__temp192), NULL));
}


static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp194;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_STRING_LITERAL (expr));
	__temp194 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (__temp194 = vala_type_reference_copy (self->priv->string_type)));
	(__temp194 = (g_object_unref (__temp194), NULL));
	vala_type_reference_set_non_null (vala_literal_get_static_type (((ValaLiteral*) expr)), TRUE);
}


static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp195;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_NULL_LITERAL (expr));
	/* empty TypeReference represents null */
	__temp195 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (__temp195 = vala_type_reference_new ()));
	(__temp195 == NULL ? NULL : (__temp195 = (g_object_unref (__temp195), NULL)));
}


static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_LITERAL_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_literal_get_static_type (vala_literal_expression_get_literal (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_symbol (ValaSemanticAnalyzer* self, ValaSymbol* sym)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	if (VALA_IS_FIELD (sym)) {
		ValaField* __temp196;
		ValaField* f;
		ValaTypeReference* __temp197;
		__temp196 = NULL;
		f = (__temp196 = VALA_FIELD (sym), (__temp196 == NULL ? NULL : g_object_ref (__temp196)));
		__temp197 = NULL;
		return (__temp197 = g_object_ref (vala_field_get_type_reference (f)), (f == NULL ? NULL : (f = (g_object_unref (f), NULL))), __temp197);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_CONSTANT (sym)) {
			ValaConstant* __temp198;
			ValaConstant* c;
			ValaTypeReference* __temp199;
			__temp198 = NULL;
			c = (__temp198 = VALA_CONSTANT (sym), (__temp198 == NULL ? NULL : g_object_ref (__temp198)));
			__temp199 = NULL;
			return (__temp199 = g_object_ref (vala_constant_get_type_reference (c)), (c == NULL ? NULL : (c = (g_object_unref (c), NULL))), __temp199);
			(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
		} else {
			if (VALA_IS_PROPERTY (sym)) {
				ValaProperty* __temp200;
				ValaProperty* prop;
				ValaTypeReference* type;
				ValaTypeReference* __temp201;
				__temp200 = NULL;
				prop = (__temp200 = VALA_PROPERTY (sym), (__temp200 == NULL ? NULL : g_object_ref (__temp200)));
				type = vala_type_reference_copy (vala_property_get_type_reference (prop));
				vala_type_reference_set_takes_ownership (type, FALSE);
				__temp201 = NULL;
				return (__temp201 = type, (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), __temp201);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(type = (g_object_unref (type), NULL));
			} else {
				if (VALA_IS_FORMAL_PARAMETER (sym)) {
					ValaFormalParameter* __temp202;
					ValaFormalParameter* p;
					ValaTypeReference* __temp203;
					ValaTypeReference* __temp204;
					__temp202 = NULL;
					p = (__temp202 = VALA_FORMAL_PARAMETER (sym), (__temp202 == NULL ? NULL : g_object_ref (__temp202)));
					__temp203 = NULL;
					__temp204 = NULL;
					return (__temp204 = (__temp203 = vala_formal_parameter_get_type_reference (p), (__temp203 == NULL ? NULL : g_object_ref (__temp203))), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), __temp204);
					(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				} else {
					if (VALA_IS_TYPE_REFERENCE (sym)) {
						ValaTypeReference* __temp205;
						__temp205 = NULL;
						return (__temp205 = VALA_TYPE_REFERENCE (sym), (__temp205 == NULL ? NULL : g_object_ref (__temp205)));
					} else {
						if (VALA_IS_VARIABLE_DECLARATOR (sym)) {
							ValaVariableDeclarator* __temp207;
							ValaVariableDeclarator* decl;
							ValaTypeReference* __temp208;
							ValaTypeReference* __temp209;
							__temp207 = NULL;
							decl = (__temp207 = VALA_VARIABLE_DECLARATOR (sym), (__temp207 == NULL ? NULL : g_object_ref (__temp207)));
							__temp208 = NULL;
							__temp209 = NULL;
							return (__temp209 = (__temp208 = vala_variable_declarator_get_type_reference (decl), (__temp208 == NULL ? NULL : g_object_ref (__temp208))), (decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL))), __temp209);
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						} else {
							if (VALA_IS_ENUM_VALUE (sym)) {
								ValaTypeReference* type;
								type = vala_type_reference_new ();
								vala_type_reference_set_data_type (type, VALA_DATA_TYPE (vala_symbol_get_parent_symbol (sym)));
								return type;
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							}
						}
					}
				}
			}
		}
	}
	return VALA_TYPE_REFERENCE (NULL);
}


ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name)
{
	ValaSymbol* result;
	gpointer __temp222;
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	result = vala_scope_lookup (vala_symbol_get_scope (sym), name);
	if (result != NULL) {
		return result;
	}
	if (VALA_IS_CLASS (sym)) {
		ValaClass* __temp213;
		ValaClass* cl;
		__temp213 = NULL;
		cl = (__temp213 = VALA_CLASS (sym), (__temp213 == NULL ? NULL : g_object_ref (__temp213)));
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaTypeReference* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					ValaSymbol* __temp214;
					__temp214 = NULL;
					result = (__temp214 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (base_type)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp214);
					if (result != NULL) {
						ValaSymbol* __temp215;
						__temp215 = NULL;
						return (__temp215 = result, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp215);
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			ValaStruct* __temp216;
			ValaStruct* st;
			__temp216 = NULL;
			st = (__temp216 = VALA_STRUCT (sym), (__temp216 == NULL ? NULL : g_object_ref (__temp216)));
			{
				GeeCollection* base_type_collection;
				GeeIterator* base_type_it;
				base_type_collection = vala_struct_get_base_types (st);
				base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
				while (gee_iterator_next (base_type_it)) {
					ValaTypeReference* base_type;
					base_type = gee_iterator_get (base_type_it);
					{
						ValaSymbol* __temp217;
						__temp217 = NULL;
						result = (__temp217 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (base_type)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp217);
						if (result != NULL) {
							ValaSymbol* __temp218;
							__temp218 = NULL;
							return (__temp218 = result, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp218);
						}
						(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
					}
				}
				(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
				(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				ValaInterface* __temp219;
				ValaInterface* iface;
				__temp219 = NULL;
				iface = (__temp219 = VALA_INTERFACE (sym), (__temp219 == NULL ? NULL : g_object_ref (__temp219)));
				{
					GeeCollection* prerequisite_collection;
					GeeIterator* prerequisite_it;
					prerequisite_collection = vala_interface_get_prerequisites (iface);
					prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
					while (gee_iterator_next (prerequisite_it)) {
						ValaTypeReference* prerequisite;
						prerequisite = gee_iterator_get (prerequisite_it);
						{
							ValaSymbol* __temp220;
							__temp220 = NULL;
							result = (__temp220 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (prerequisite)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp220);
							if (result != NULL) {
								ValaSymbol* __temp221;
								__temp221 = NULL;
								return (__temp221 = result, (prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL))), (prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL))), (prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL))), (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), __temp221);
							}
							(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
						}
					}
					(prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL)));
					(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
				}
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	return VALA_SYMBOL ((__temp222 = NULL, (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp222));
	(result == NULL ? NULL : (result = (g_object_unref (result), NULL)));
}


static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp223;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	__temp223 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp223 = vala_type_reference_copy (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr)))));
	(__temp223 = (g_object_unref (__temp223), NULL));
	/* don't call g_object_ref_sink on inner and outer expression*/
	vala_type_reference_set_floating_reference (vala_expression_get_static_type (((ValaExpression*) expr)), FALSE);
}


static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* base_symbol;
	ValaSymbol* __temp241;
	ValaSymbol* member;
	ValaMemberAccessibility access;
	ValaTypeReference* __temp249;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	base_symbol = NULL;
	if (vala_member_access_get_inner (expr) == NULL) {
		ValaSymbol* __temp225;
		ValaSymbol* __temp224;
		ValaSymbol* __temp226;
		ValaSymbol* sym;
		__temp225 = NULL;
		__temp224 = NULL;
		base_symbol = (__temp225 = (__temp224 = self->priv->current_symbol, (__temp224 == NULL ? NULL : g_object_ref (__temp224))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp225);
		__temp226 = NULL;
		sym = (__temp226 = self->priv->current_symbol, (__temp226 == NULL ? NULL : g_object_ref (__temp226)));
		while (sym != NULL && vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL) {
			ValaSymbol* __temp227;
			ValaSymbol* __temp229;
			ValaSymbol* __temp228;
			__temp227 = NULL;
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp227 = vala_semantic_analyzer_symbol_lookup_inherited (sym, vala_member_access_get_member_name (expr))));
			(__temp227 == NULL ? NULL : (__temp227 = (g_object_unref (__temp227), NULL)));
			__temp229 = NULL;
			__temp228 = NULL;
			sym = (__temp229 = (__temp228 = vala_symbol_get_parent_symbol (sym), (__temp228 == NULL ? NULL : g_object_ref (__temp228))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp229);
		}
		if (vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL) {
			{
				GeeCollection* ns_collection;
				GeeIterator* ns_it;
				ns_collection = self->priv->current_using_directives;
				ns_it = gee_iterable_iterator (GEE_ITERABLE (ns_collection));
				while (gee_iterator_next (ns_it)) {
					ValaNamespaceReference* ns;
					ns = gee_iterator_get (ns_it);
					{
						ValaSymbol* local_sym;
						local_sym = vala_scope_lookup (vala_symbol_get_scope (vala_namespace_reference_get_namespace_symbol (ns)), vala_member_access_get_member_name (expr));
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference (((ValaExpression*) expr)) != NULL) {
								char* __temp232;
								char* __temp231;
								char* __temp230;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								__temp232 = NULL;
								__temp231 = NULL;
								__temp230 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp232 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_member_access_get_member_name (expr), (__temp230 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (((ValaExpression*) expr)))), (__temp231 = vala_symbol_get_full_name (local_sym)))));
								(__temp232 = (g_free (__temp232), NULL));
								(__temp231 = (g_free (__temp231), NULL));
								(__temp230 = (g_free (__temp230), NULL));
								(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
								(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
								(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
								return;
							}
							vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), local_sym);
						}
						(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
						(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
					}
				}
				(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
			}
		}
		(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	} else {
		if (vala_code_node_get_error (((ValaCodeNode*) vala_member_access_get_inner (expr)))) {
			/* if there was an error in the inner expression, skip this check */
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr)) || VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
			ValaSymbol* __temp234;
			ValaSymbol* __temp233;
			__temp234 = NULL;
			__temp233 = NULL;
			base_symbol = (__temp234 = (__temp233 = vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)), (__temp233 == NULL ? NULL : g_object_ref (__temp233))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp234);
			if (VALA_IS_NAMESPACE (base_symbol) || VALA_IS_DATA_TYPE (base_symbol)) {
				ValaSymbol* __temp235;
				__temp235 = NULL;
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp235 = vala_scope_lookup (vala_symbol_get_scope (base_symbol), vala_member_access_get_member_name (expr))));
				(__temp235 == NULL ? NULL : (__temp235 = (g_object_unref (__temp235), NULL)));
			}
		}
		if (vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
			ValaSymbol* __temp237;
			ValaDataType* __temp236;
			ValaSymbol* __temp238;
			__temp237 = NULL;
			__temp236 = NULL;
			base_symbol = (__temp237 = VALA_SYMBOL ((__temp236 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))), (__temp236 == NULL ? NULL : g_object_ref (__temp236)))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp237);
			__temp238 = NULL;
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp238 = vala_semantic_analyzer_symbol_lookup_inherited (base_symbol, vala_member_access_get_member_name (expr))));
			(__temp238 == NULL ? NULL : (__temp238 = (g_object_unref (__temp238), NULL)));
		}
	}
	if (vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL) {
		char* __temp240;
		char* __temp239;
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		__temp240 = NULL;
		__temp239 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp240 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", vala_member_access_get_member_name (expr), (__temp239 = vala_symbol_get_full_name (base_symbol)))));
		(__temp240 = (g_free (__temp240), NULL));
		(__temp239 = (g_free (__temp239), NULL));
		(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
		return;
	}
	__temp241 = NULL;
	member = (__temp241 = vala_expression_get_symbol_reference (((ValaExpression*) expr)), (__temp241 == NULL ? NULL : g_object_ref (__temp241)));
	access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	if (VALA_IS_FIELD (member)) {
		access = (VALA_FIELD (member))->access;
	} else {
		if (VALA_IS_METHOD (member)) {
			access = (VALA_METHOD (member))->access;
		}
	}
	if (access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		ValaDataType* __temp242;
		ValaDataType* target_type;
		gboolean in_target_type;
		__temp242 = NULL;
		target_type = (__temp242 = VALA_DATA_TYPE (vala_symbol_get_parent_symbol (member)), (__temp242 == NULL ? NULL : g_object_ref (__temp242)));
		in_target_type = FALSE;
		{
			ValaSymbol* this_symbol;
			ValaSymbol* __temp246;
			ValaSymbol* __temp245;
			ValaSymbol* __temp244;
			ValaSymbol* __temp243;
			this_symbol = NULL;
			__temp246 = NULL;
			__temp245 = NULL;
			__temp244 = NULL;
			__temp243 = NULL;
			for (this_symbol = (__temp244 = (__temp243 = self->priv->current_symbol, (__temp243 == NULL ? NULL : g_object_ref (__temp243))), (this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL))), __temp244); this_symbol != NULL; this_symbol = (__temp246 = (__temp245 = vala_symbol_get_parent_symbol (this_symbol), (__temp245 == NULL ? NULL : g_object_ref (__temp245))), (this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL))), __temp246)) {
				if (VALA_SYMBOL (target_type) == this_symbol) {
					in_target_type = TRUE;
					break;
				}
			}
			(this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL)));
		}
		if (!in_target_type) {
			char* __temp248;
			char* __temp247;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			__temp248 = NULL;
			__temp247 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp248 = g_strdup_printf ("Access to private member `%s' denied", (__temp247 = vala_symbol_get_full_name (member)))));
			(__temp248 = (g_free (__temp248), NULL));
			(__temp247 = (g_free (__temp247), NULL));
			(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
			return;
		}
		(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_expression_get_symbol_reference (((ValaExpression*) expr)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	__temp249 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp249 = vala_semantic_analyzer_get_static_type_for_symbol (self, vala_expression_get_symbol_reference (((ValaExpression*) expr)))));
	(__temp249 == NULL ? NULL : (__temp249 = (g_object_unref (__temp249), NULL)));
	(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
	(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
}


static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expression_type), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expected_type), FALSE);
	/* only null is compatible to null */
	if (vala_type_reference_get_data_type (expected_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL) {
		return (vala_type_reference_get_data_type (expression_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL);
	}
	if (vala_type_reference_get_data_type (expression_type) == NULL) {
		/* null can be cast to any reference or array type or pointer type */
		if (vala_type_reference_get_type_parameter (expected_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_is_out (expected_type) || VALA_IS_POINTER (vala_type_reference_get_data_type (expected_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* null is not compatible with any other type (i.e. value types) */
	if (vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
		/* any reference or array type or pointer type can be cast to a generic pointer */
		if (vala_type_reference_get_type_parameter (expression_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expression_type)) || VALA_IS_POINTER (vala_type_reference_get_data_type (expression_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expression_type)) || vala_type_reference_get_data_type (expression_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* temporarily ignore type parameters */
	if (vala_type_reference_get_type_parameter (expected_type) != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) != VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type))) {
		return FALSE;
	}
	if (VALA_IS_ENUM (vala_type_reference_get_data_type (expression_type)) && vala_type_reference_get_data_type (expected_type) == vala_type_reference_get_data_type (self->priv->int_type)) {
		return TRUE;
	}
	if (vala_type_reference_get_data_type (expression_type) == vala_type_reference_get_data_type (expected_type)) {
		return TRUE;
	}
	if (VALA_IS_STRUCT (vala_type_reference_get_data_type (expression_type)) && VALA_IS_STRUCT (vala_type_reference_get_data_type (expected_type))) {
		ValaStruct* __temp259;
		ValaStruct* expr_struct;
		ValaStruct* __temp260;
		ValaStruct* expect_struct;
		__temp259 = NULL;
		expr_struct = (__temp259 = VALA_STRUCT (vala_type_reference_get_data_type (expression_type)), (__temp259 == NULL ? NULL : g_object_ref (__temp259)));
		__temp260 = NULL;
		expect_struct = (__temp260 = VALA_STRUCT (vala_type_reference_get_data_type (expected_type)), (__temp260 == NULL ? NULL : g_object_ref (__temp260)));
		/* integer types may be implicitly cast to floating point types */
		if (vala_struct_is_integer_type (expr_struct) && vala_struct_is_floating_type (expect_struct)) {
			gboolean __temp261;
			return (__temp261 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp261);
		}
		if ((vala_struct_is_integer_type (expr_struct) && vala_struct_is_integer_type (expect_struct)) || (vala_struct_is_floating_type (expr_struct) && vala_struct_is_floating_type (expect_struct))) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean __temp262;
				return (__temp262 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp262);
			}
		}
		(expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL)));
		(expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL)));
	}
	return vala_data_type_is_subtype_of (vala_type_reference_get_data_type (expression_type), vala_type_reference_get_data_type (expected_type));
}


static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp264;
	ValaSymbol* msym;
	GeeCollection* params;
	GeeCollection* args;
	GeeIterator* arg_it;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_invocation_expression_get_call (expr)))) {
		/* if method resolving didn't succeed, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	__temp264 = NULL;
	msym = (__temp264 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp264 == NULL ? NULL : g_object_ref (__temp264)));
	if (msym == NULL) {
		/* if no symbol found, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		return;
	}
	params = NULL;
	if (VALA_IS_INVOKABLE (msym)) {
		ValaInvokable* __temp265;
		ValaInvokable* m;
		__temp265 = NULL;
		m = (__temp265 = VALA_INVOKABLE (msym), (__temp265 == NULL ? NULL : g_object_ref (__temp265)));
		if (vala_invokable_is_invokable (m)) {
			GeeCollection* __temp266;
			__temp266 = NULL;
			params = (__temp266 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_object_unref (params), NULL))), __temp266);
		} else {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "invocation not supported in this context");
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
			(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
			return;
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "invocation not supported in this context");
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
		return;
	}
	args = vala_invocation_expression_get_argument_list (expr);
	arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				if (gee_iterator_next (arg_it)) {
					ValaExpression* arg;
					arg = gee_iterator_get (arg_it);
					/* store expected type for callback parameters */
					vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param));
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
	(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
	(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
}


static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GeeCollection* params, GeeCollection* args)
{
	ValaExpression* prev_arg;
	GeeIterator* arg_it;
	ValaAttribute* __temp267;
	gboolean __temp268;
	gboolean diag;
	gboolean ellipsis;
	gint i;
	gboolean __temp286;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), FALSE);
	g_return_val_if_fail (VALA_IS_SYMBOL (msym), FALSE);
	g_return_val_if_fail (params == NULL || GEE_IS_COLLECTION (params), FALSE);
	g_return_val_if_fail (args == NULL || GEE_IS_COLLECTION (args), FALSE);
	prev_arg = NULL;
	arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
	__temp267 = NULL;
	diag = (__temp268 = ((__temp267 = vala_code_node_get_attribute (VALA_CODE_NODE (msym), "Diagnostics")) != NULL), (__temp267 == NULL ? NULL : (__temp267 = (g_object_unref (__temp267), NULL))), __temp268);
	ellipsis = FALSE;
	i = 0;
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				/* header file necessary if we need to cast argument */
				if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL) {
					vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
				}
				if (!gee_iterator_next (arg_it)) {
					if (vala_formal_parameter_get_default_expression (param) == NULL) {
						char* __temp270;
						char* __temp269;
						gboolean __temp271;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						__temp270 = NULL;
						__temp269 = NULL;
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp270 = g_strdup_printf ("Too few arguments, method `%s' does not take %d arguments", (__temp269 = vala_symbol_get_full_name (msym)), gee_collection_get_size (args))));
						(__temp270 = (g_free (__temp270), NULL));
						(__temp269 = (g_free (__temp269), NULL));
						return (__temp271 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), __temp271);
					}
				} else {
					ValaExpression* arg;
					ValaExpression* __temp279;
					ValaExpression* __temp278;
					arg = gee_iterator_get (arg_it);
					if (vala_expression_get_static_type (arg) == NULL) {
						/* disallow untyped arguments except for type inference of callbacks*/
						if (!(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))))) {
							char* __temp272;
							gboolean __temp273;
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							__temp272 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp272 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
							(__temp272 = (g_free (__temp272), NULL));
							return (__temp273 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), __temp273);
						}
					} else {
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param))) {
							char* __temp276;
							char* __temp275;
							char* __temp274;
							gboolean __temp277;
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							__temp276 = NULL;
							__temp275 = NULL;
							__temp274 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp276 = g_strdup_printf ("Argument %d: Cannot convert from `%s' to `%s'", i + 1, (__temp274 = vala_type_reference_to_string (vala_expression_get_static_type (arg))), (__temp275 = vala_type_reference_to_string (vala_formal_parameter_get_type_reference (param))))));
							(__temp276 = (g_free (__temp276), NULL));
							(__temp275 = (g_free (__temp275), NULL));
							(__temp274 = (g_free (__temp274), NULL));
							return (__temp277 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), __temp277);
						}
					}
					__temp279 = NULL;
					__temp278 = NULL;
					prev_arg = (__temp279 = (__temp278 = arg, (__temp278 == NULL ? NULL : g_object_ref (__temp278))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), __temp279);
					i++;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	if (!ellipsis && gee_iterator_next (arg_it)) {
		char* __temp281;
		char* __temp280;
		gboolean __temp282;
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		__temp281 = NULL;
		__temp280 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp281 = g_strdup_printf ("Too many arguments, method `%s' does not take %d arguments", (__temp280 = vala_symbol_get_full_name (msym)), gee_collection_get_size (args))));
		(__temp281 = (g_free (__temp281), NULL));
		(__temp280 = (g_free (__temp280), NULL));
		return (__temp282 = FALSE, (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), __temp282);
	}
	if (diag && prev_arg != NULL) {
		ValaExpression* __temp283;
		ValaExpression* format_arg;
		__temp283 = NULL;
		format_arg = (__temp283 = prev_arg, (__temp283 == NULL ? NULL : g_object_ref (__temp283)));
		if (VALA_IS_LITERAL_EXPRESSION (format_arg)) {
			ValaStringLiteral* __temp284;
			ValaStringLiteral* format_lit;
			char* __temp285;
			__temp284 = NULL;
			format_lit = (__temp284 = VALA_STRING_LITERAL (vala_literal_expression_get_literal ((VALA_LITERAL_EXPRESSION (format_arg)))), (__temp284 == NULL ? NULL : g_object_ref (__temp284)));
			__temp285 = NULL;
			vala_string_literal_set_value (format_lit, (__temp285 = g_strdup_printf ("\"%s:%d: %s", vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) expr)))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (((ValaCodeNode*) expr))), g_utf8_offset_to_pointer (vala_string_literal_get_value (format_lit), 1))));
			(__temp285 = (g_free (__temp285), NULL));
			(format_lit == NULL ? NULL : (format_lit = (g_object_unref (format_lit), NULL)));
		}
		(format_arg == NULL ? NULL : (format_arg = (g_object_unref (format_arg), NULL)));
	}
	return (__temp286 = TRUE, (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), __temp286);
	(prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL)));
	(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
}


static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp287;
	ValaSymbol* msym;
	ValaTypeReference* ret_type;
	GeeCollection* params;
	GeeCollection* __temp295;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) expr))) {
		return;
	}
	__temp287 = NULL;
	msym = (__temp287 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp287 == NULL ? NULL : g_object_ref (__temp287)));
	ret_type = NULL;
	params = NULL;
	if (VALA_IS_INVOKABLE (msym)) {
		ValaInvokable* __temp288;
		ValaInvokable* m;
		ValaTypeReference* __temp289;
		GeeCollection* __temp290;
		__temp288 = NULL;
		m = (__temp288 = VALA_INVOKABLE (msym), (__temp288 == NULL ? NULL : g_object_ref (__temp288)));
		__temp289 = NULL;
		ret_type = (__temp289 = vala_invokable_get_return_type (m), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp289);
		__temp290 = NULL;
		params = (__temp290 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_object_unref (params), NULL))), __temp290);
		if (vala_type_reference_get_data_type (ret_type) == NULL && vala_type_reference_get_type_parameter (ret_type) == NULL) {
			/* void return type*/
			if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (((ValaCodeNode*) expr))))) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "invocation of void method not allowed as expression");
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				return;
			}
		}
		/* resolve generic return values*/
		if (vala_type_reference_get_type_parameter (ret_type) != NULL) {
			ValaMemberAccess* __temp291;
			ValaMemberAccess* ma;
			if (!(VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)))) {
				vala_report_error (vala_code_node_get_source_reference ((VALA_CODE_NODE (m))), "internal error: unsupported generic return value");
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				return;
			}
			__temp291 = NULL;
			ma = (__temp291 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (__temp291 == NULL ? NULL : g_object_ref (__temp291)));
			if (vala_member_access_get_inner (ma) == NULL) {
				/* TODO resolve generic return values within the type hierarchy if possible*/
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "internal error: resolving generic return values within type hierarchy not supported yet");
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				return;
			} else {
				ValaTypeReference* __temp292;
				__temp292 = NULL;
				ret_type = (__temp292 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_member_access_get_inner (ma)), msym, ret_type, VALA_CODE_NODE (expr)), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp292);
				if (ret_type == NULL) {
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
					return;
				}
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	if (VALA_IS_METHOD (msym)) {
		ValaMethod* __temp293;
		ValaMethod* m;
		GeeCollection* __temp294;
		__temp293 = NULL;
		m = (__temp293 = VALA_METHOD (msym), (__temp293 == NULL ? NULL : g_object_ref (__temp293)));
		__temp294 = NULL;
		vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (gee_collection_get_size ((__temp294 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (((ValaExpression*) expr))));
		(__temp294 == NULL ? NULL : (__temp294 = (g_object_unref (__temp294), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	vala_expression_set_static_type (VALA_EXPRESSION (expr), ret_type);
	__temp295 = NULL;
	vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), msym, params, (__temp295 = vala_invocation_expression_get_argument_list (expr)));
	(__temp295 == NULL ? NULL : (__temp295 = (g_object_unref (__temp295), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
	(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
}


ValaTypeReference* vala_semantic_analyzer_get_actual_type (ValaTypeReference* derived_instance_type, ValaSymbol* generic_member, ValaTypeReference* generic_type, ValaCodeNode* node_reference)
{
	ValaTypeReference* __temp296;
	ValaTypeReference* instance_type;
	gint param_index;
	GeeList* __temp313;
	ValaTypeReference* __temp314;
	ValaTypeReference* actual_type;
	ValaTypeReference* __temp317;
	ValaTypeReference* __temp318;
	g_return_val_if_fail (derived_instance_type == NULL || VALA_IS_TYPE_REFERENCE (derived_instance_type), NULL);
	g_return_val_if_fail (generic_member == NULL || VALA_IS_SYMBOL (generic_member), NULL);
	g_return_val_if_fail (generic_type == NULL || VALA_IS_TYPE_REFERENCE (generic_type), NULL);
	g_return_val_if_fail (node_reference == NULL || VALA_IS_CODE_NODE (node_reference), NULL);
	__temp296 = NULL;
	instance_type = (__temp296 = derived_instance_type, (__temp296 == NULL ? NULL : g_object_ref (__temp296)));
	while (VALA_SYMBOL (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_parent_symbol (generic_member)) {
		GeeCollection* base_types;
		/* trace type arguments back to the datatype where the method has been declared*/
		base_types = NULL;
		if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
			ValaClass* __temp297;
			ValaClass* cl;
			GeeCollection* __temp298;
			__temp297 = NULL;
			cl = (__temp297 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (__temp297 == NULL ? NULL : g_object_ref (__temp297)));
			__temp298 = NULL;
			base_types = (__temp298 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), __temp298);
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		} else {
			if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
				ValaInterface* __temp299;
				ValaInterface* iface;
				GeeCollection* __temp300;
				__temp299 = NULL;
				iface = (__temp299 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (__temp299 == NULL ? NULL : g_object_ref (__temp299)));
				__temp300 = NULL;
				base_types = (__temp300 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), __temp300);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			} else {
				gpointer __temp301;
				vala_report_error (vala_code_node_get_source_reference (node_reference), "internal error: unsupported generic type");
				vala_code_node_set_error (node_reference, TRUE);
				return VALA_TYPE_REFERENCE ((__temp301 = NULL, (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp301));
			}
		}
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = base_types;
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaTypeReference* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					ValaSymbol* __temp302;
					gboolean __temp303;
					__temp302 = NULL;
					if ((__temp303 = (__temp302 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (base_type)), vala_symbol_get_name (generic_member))) != NULL, (__temp302 == NULL ? NULL : (__temp302 = (g_object_unref (__temp302), NULL))), __temp303)) {
						ValaTypeReference* instance_base_type;
						ValaTypeReference* __temp309;
						ValaTypeReference* __temp308;
						/* construct a new type reference for the base type with correctly linked type arguments*/
						instance_base_type = vala_type_reference_new ();
						vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type));
						{
							GeeList* type_arg_collection;
							GeeIterator* type_arg_it;
							type_arg_collection = vala_type_reference_get_type_arguments (base_type);
							type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
							while (gee_iterator_next (type_arg_it)) {
								ValaTypeReference* type_arg;
								type_arg = gee_iterator_get (type_arg_it);
								{
									if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
										gint param_index;
										ValaTypeReference* __temp307;
										GeeList* __temp306;
										/* link to type argument of derived type*/
										param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (type_arg))));
										if (param_index == -1) {
											char* __temp304;
											gpointer __temp305;
											__temp304 = NULL;
											vala_report_error (vala_code_node_get_source_reference (node_reference), (__temp304 = g_strdup_printf ("internal error: unknown type parameter %s", vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (type_arg))))));
											(__temp304 = (g_free (__temp304), NULL));
											vala_code_node_set_error (node_reference, TRUE);
											return VALA_TYPE_REFERENCE ((__temp305 = NULL, (type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL))), (type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL))), (type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL))), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp305));
										}
										__temp307 = NULL;
										__temp306 = NULL;
										type_arg = (__temp307 = gee_list_get ((__temp306 = vala_type_reference_get_type_arguments (instance_type)), param_index), (type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL))), __temp307);
										(__temp306 == NULL ? NULL : (__temp306 = (g_object_unref (__temp306), NULL)));
									}
									vala_type_reference_add_type_argument (instance_base_type, type_arg);
									(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
								}
							}
							(type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL)));
							(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
						}
						__temp309 = NULL;
						__temp308 = NULL;
						instance_type = (__temp309 = (__temp308 = instance_base_type, (__temp308 == NULL ? NULL : g_object_ref (__temp308))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp309);
						(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		(base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL)));
	}
	if (VALA_SYMBOL (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_parent_symbol (generic_member)) {
		gpointer __temp310;
		vala_report_error (vala_code_node_get_source_reference (node_reference), "internal error: generic type parameter tracing not supported yet");
		vala_code_node_set_error (node_reference, TRUE);
		return VALA_TYPE_REFERENCE ((__temp310 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp310));
	}
	param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (generic_type))));
	if (param_index == -1) {
		char* __temp311;
		gpointer __temp312;
		__temp311 = NULL;
		vala_report_error (vala_code_node_get_source_reference (node_reference), (__temp311 = g_strdup_printf ("internal error: unknown type parameter %s", vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (generic_type))))));
		(__temp311 = (g_free (__temp311), NULL));
		vala_code_node_set_error (node_reference, TRUE);
		return VALA_TYPE_REFERENCE ((__temp312 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp312));
	}
	__temp313 = NULL;
	__temp314 = NULL;
	actual_type = (__temp314 = VALA_TYPE_REFERENCE (gee_list_get ((__temp313 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (__temp313 == NULL ? NULL : (__temp313 = (g_object_unref (__temp313), NULL))), __temp314);
	if (actual_type == NULL) {
		char* __temp315;
		gpointer __temp316;
		__temp315 = NULL;
		vala_report_error (vala_code_node_get_source_reference (node_reference), (__temp315 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (generic_type))))));
		(__temp315 = (g_free (__temp315), NULL));
		vala_code_node_set_error (node_reference, TRUE);
		return VALA_TYPE_REFERENCE ((__temp316 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), __temp316));
	}
	__temp317 = NULL;
	actual_type = (__temp317 = vala_type_reference_copy (actual_type), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), __temp317);
	vala_type_reference_set_transfers_ownership (actual_type, vala_type_reference_get_takes_ownership (actual_type) && vala_type_reference_get_transfers_ownership (generic_type));
	vala_type_reference_set_takes_ownership (actual_type, vala_type_reference_get_takes_ownership (actual_type) && vala_type_reference_get_takes_ownership (generic_type));
	__temp318 = NULL;
	return (__temp318 = actual_type, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp318);
	(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
	(actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr)
{
	ValaSemanticAnalyzer * self;
	ValaDataType* __temp319;
	ValaDataType* container_type;
	gboolean index_int_type_check;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_ELEMENT_ACCESS (expr));
	if (vala_expression_get_static_type (vala_element_access_get_container (expr)) == NULL) {
		/* don't proceed if a child expression failed */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	__temp319 = NULL;
	container_type = (__temp319 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))), (__temp319 == NULL ? NULL : g_object_ref (__temp319)));
	index_int_type_check = TRUE;
	/* assign a static_type when possible */
	if (VALA_IS_ARRAY (container_type)) {
		GeeList* args;
		ValaTypeReference* __temp321;
		args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (expr)));
		if (gee_collection_get_size (((GeeCollection*) args)) != 1) {
			char* __temp320;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			__temp320 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp320 = g_strdup_printf ("internal error: array reference with %d type arguments, expected 1", gee_collection_get_size (((GeeCollection*) args)))));
			(__temp320 = (g_free (__temp320), NULL));
			(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
			(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
			return;
		}
		__temp321 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp321 = gee_list_get (args, 0)));
		(__temp321 == NULL ? NULL : (__temp321 = (g_object_unref (__temp321), NULL)));
		(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
	} else {
		if (container_type == vala_type_reference_get_data_type (self->priv->string_type)) {
			GeeList* __temp322;
			gboolean __temp323;
			__temp322 = NULL;
			if ((__temp323 = gee_collection_get_size (((GeeCollection*) (__temp322 = vala_element_access_get_indices (expr)))) != 1, (__temp322 == NULL ? NULL : (__temp322 = (g_object_unref (__temp322), NULL))), __temp323)) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Element access with more than one dimension is not supported for strings");
				(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
				return;
			}
			vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->unichar_type);
		} else {
			if (container_type != NULL && self->priv->list_type != NULL && self->priv->map_type != NULL && (container_type == self->priv->list_type || vala_data_type_is_subtype_of (container_type, self->priv->list_type) || container_type == self->priv->map_type || vala_data_type_is_subtype_of (container_type, self->priv->map_type))) {
				GeeCollection* indices;
				GeeIterator* indices_it;
				ValaExpression* index;
				ValaSymbol* get_sym;
				ValaMethod* __temp324;
				ValaMethod* get_method;
				GeeCollection* get_params;
				GeeIterator* get_params_it;
				ValaFormalParameter* get_param;
				ValaTypeReference* __temp325;
				ValaTypeReference* index_type;
				ValaTypeReference* __temp331;
				ValaTypeReference* __temp330;
				indices = GEE_COLLECTION (vala_element_access_get_indices (expr));
				if (gee_collection_get_size (indices) != 1) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Element access with more than one dimension is not supported for the specified type");
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					return;
				}
				indices_it = gee_iterable_iterator (GEE_ITERABLE (indices));
				gee_iterator_next (indices_it);
				index = gee_iterator_get (indices_it);
				index_int_type_check = FALSE;
				get_sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) container_type)), "get");
				if (!(VALA_IS_METHOD (get_sym))) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "invalid get method in specified collection type");
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
					(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
					(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					return;
				}
				__temp324 = NULL;
				get_method = (__temp324 = VALA_METHOD (get_sym), (__temp324 == NULL ? NULL : g_object_ref (__temp324)));
				get_params = vala_invokable_get_parameters (VALA_INVOKABLE (get_method));
				get_params_it = gee_iterable_iterator (GEE_ITERABLE (get_params));
				gee_iterator_next (get_params_it);
				get_param = gee_iterator_get (get_params_it);
				__temp325 = NULL;
				index_type = (__temp325 = vala_formal_parameter_get_type_reference (get_param), (__temp325 == NULL ? NULL : g_object_ref (__temp325)));
				if (vala_type_reference_get_type_parameter (index_type) != NULL) {
					ValaTypeReference* __temp326;
					__temp326 = NULL;
					index_type = (__temp326 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_element_access_get_container (expr)), VALA_SYMBOL (get_method), vala_formal_parameter_get_type_reference (get_param), VALA_CODE_NODE (expr)), (index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL))), __temp326);
				}
				if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (index), index_type)) {
					char* __temp329;
					char* __temp328;
					char* __temp327;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					__temp329 = NULL;
					__temp328 = NULL;
					__temp327 = NULL;
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp329 = g_strdup_printf ("index expression: Cannot convert from `%s' to `%s'", (__temp327 = vala_type_reference_to_string (vala_expression_get_static_type (index))), (__temp328 = vala_type_reference_to_string (index_type)))));
					(__temp329 = (g_free (__temp329), NULL));
					(__temp328 = (g_free (__temp328), NULL));
					(__temp327 = (g_free (__temp327), NULL));
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
					(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
					(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
					(get_method == NULL ? NULL : (get_method = (g_object_unref (get_method), NULL)));
					(get_params == NULL ? NULL : (get_params = (g_object_unref (get_params), NULL)));
					(get_params_it == NULL ? NULL : (get_params_it = (g_object_unref (get_params_it), NULL)));
					(get_param == NULL ? NULL : (get_param = (g_object_unref (get_param), NULL)));
					(index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL)));
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					return;
				}
				__temp331 = NULL;
				__temp330 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp331 = vala_type_reference_copy ((__temp330 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_element_access_get_container (expr)), VALA_SYMBOL (get_method), vala_method_get_return_type (get_method), VALA_CODE_NODE (expr))))));
				(__temp331 = (g_object_unref (__temp331), NULL));
				(__temp330 == NULL ? NULL : (__temp330 = (g_object_unref (__temp330), NULL)));
				vala_type_reference_set_takes_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), FALSE);
				(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
				(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
				(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
				(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
				(get_method == NULL ? NULL : (get_method = (g_object_unref (get_method), NULL)));
				(get_params == NULL ? NULL : (get_params = (g_object_unref (get_params), NULL)));
				(get_params_it == NULL ? NULL : (get_params_it = (g_object_unref (get_params_it), NULL)));
				(get_param == NULL ? NULL : (get_param = (g_object_unref (get_param), NULL)));
				(index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL)));
			} else {
				char* __temp333;
				char* __temp332;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				__temp333 = NULL;
				__temp332 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp333 = g_strdup_printf ("The expression `%s' does not denote an Array", (__temp332 = vala_type_reference_to_string (vala_expression_get_static_type (vala_element_access_get_container (expr)))))));
				(__temp333 = (g_free (__temp333), NULL));
				(__temp332 = (g_free (__temp332), NULL));
			}
		}
	}
	if (index_int_type_check) {
		{
			GeeList* e_collection;
			GeeIterator* e_it;
			e_collection = vala_element_access_get_indices (expr);
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = gee_iterator_get (e_it);
				{
					/* check if the index is of type integer 
					 don't proceed if a child expression failed */
					if (vala_expression_get_static_type (e) == NULL) {
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
						(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
						(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
						return;
					}
					/* check if the index is of type integer */
					if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) e)), "Expression of integer type expected");
					}
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
	}
	(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_BASE_ACCESS (expr));
	if (self->priv->current_class == NULL) {
		GeeCollection* __temp338;
		GeeIterator* __temp339;
		GeeIterator* base_type_it;
		ValaTypeReference* __temp340;
		if (self->priv->current_struct == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Base access invalid outside of class and struct");
			return;
		} else {
			GeeCollection* __temp334;
			gboolean __temp335;
			__temp334 = NULL;
			if ((__temp335 = gee_collection_get_size ((__temp334 = vala_struct_get_base_types (self->priv->current_struct))) != 1, (__temp334 == NULL ? NULL : (__temp334 = (g_object_unref (__temp334), NULL))), __temp335)) {
				char* __temp337;
				GeeCollection* __temp336;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				__temp337 = NULL;
				__temp336 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp337 = g_strdup_printf ("Base access invalid without base type %d", gee_collection_get_size ((__temp336 = vala_struct_get_base_types (self->priv->current_struct))))));
				(__temp337 = (g_free (__temp337), NULL));
				(__temp336 == NULL ? NULL : (__temp336 = (g_object_unref (__temp336), NULL)));
				return;
			}
		}
		__temp338 = NULL;
		__temp339 = NULL;
		base_type_it = (__temp339 = gee_iterable_iterator (GEE_ITERABLE ((__temp338 = vala_struct_get_base_types (self->priv->current_struct)))), (__temp338 == NULL ? NULL : (__temp338 = (g_object_unref (__temp338), NULL))), __temp339);
		gee_iterator_next (base_type_it);
		__temp340 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp340 = gee_iterator_get (base_type_it)));
		(__temp340 == NULL ? NULL : (__temp340 = (g_object_unref (__temp340), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	} else {
		ValaTypeReference* __temp341;
		__temp341 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp341 = vala_type_reference_new ()));
		(__temp341 == NULL ? NULL : (__temp341 = (g_object_unref (__temp341), NULL)));
		vala_type_reference_set_data_type (vala_expression_get_static_type (((ValaExpression*) expr)), VALA_DATA_TYPE (vala_class_get_base_class (self->priv->current_class)));
	}
	vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_type_reference_get_data_type (vala_expression_get_static_type (((ValaExpression*) expr)))));
}


static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_POSTFIX_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_postfix_expression_get_inner (expr)));
}


static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaDataType* type;
	ValaTypeReference* __temp363;
	GeeCollection* __temp370;
	gboolean __temp371;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	type = NULL;
	if (vala_object_creation_expression_get_type_reference (expr) == NULL) {
		ValaSymbol* __temp342;
		ValaSymbol* constructor_sym;
		ValaSymbol* __temp343;
		ValaSymbol* type_sym;
		GeeCollection* type_args;
		ValaTypeReference* __temp358;
		if (vala_object_creation_expression_get_member_name (expr) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Incomplete object creation expression");
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (((ValaExpression*) vala_object_creation_expression_get_member_name (expr))) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		__temp342 = NULL;
		constructor_sym = (__temp342 = vala_expression_get_symbol_reference (((ValaExpression*) vala_object_creation_expression_get_member_name (expr))), (__temp342 == NULL ? NULL : g_object_ref (__temp342)));
		__temp343 = NULL;
		type_sym = (__temp343 = vala_expression_get_symbol_reference (((ValaExpression*) vala_object_creation_expression_get_member_name (expr))), (__temp343 == NULL ? NULL : g_object_ref (__temp343)));
		type_args = vala_member_access_get_type_arguments (vala_object_creation_expression_get_member_name (expr));
		if (VALA_IS_METHOD (constructor_sym)) {
			ValaSymbol* __temp345;
			ValaSymbol* __temp344;
			ValaMethod* __temp346;
			ValaMethod* constructor;
			GeeCollection* __temp349;
			__temp345 = NULL;
			__temp344 = NULL;
			type_sym = (__temp345 = (__temp344 = vala_symbol_get_parent_symbol (constructor_sym), (__temp344 == NULL ? NULL : g_object_ref (__temp344))), (type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL))), __temp345);
			__temp346 = NULL;
			constructor = (__temp346 = VALA_METHOD (constructor_sym), (__temp346 == NULL ? NULL : g_object_ref (__temp346)));
			if (!(VALA_IS_CREATION_METHOD (constructor_sym))) {
				char* __temp348;
				char* __temp347;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				__temp348 = NULL;
				__temp347 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp348 = g_strdup_printf ("`%s' is not a creation method", (__temp347 = vala_symbol_get_full_name (VALA_SYMBOL (constructor))))));
				(__temp348 = (g_free (__temp348), NULL));
				(__temp347 = (g_free (__temp347), NULL));
				(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
				(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
				(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (constructor));
			__temp349 = NULL;
			type_args = (__temp349 = vala_member_access_get_type_arguments ((VALA_MEMBER_ACCESS (vala_member_access_get_inner (vala_object_creation_expression_get_member_name (expr))))), (type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL))), __temp349);
			(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
		} else {
			if (VALA_IS_ENUM_VALUE (constructor_sym)) {
				ValaSymbol* __temp351;
				ValaSymbol* __temp350;
				__temp351 = NULL;
				__temp350 = NULL;
				type_sym = (__temp351 = (__temp350 = vala_symbol_get_parent_symbol (constructor_sym), (__temp350 == NULL ? NULL : g_object_ref (__temp350))), (type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL))), __temp351);
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), constructor_sym);
			}
		}
		if (VALA_IS_CLASS (type_sym) || VALA_IS_STRUCT (type_sym)) {
			ValaDataType* __temp353;
			ValaDataType* __temp352;
			__temp353 = NULL;
			__temp352 = NULL;
			type = (__temp353 = (__temp352 = VALA_DATA_TYPE (type_sym), (__temp352 == NULL ? NULL : g_object_ref (__temp352))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp353);
		} else {
			if (VALA_IS_ENUM (type_sym) && vala_enum_get_error_domain ((VALA_ENUM (type_sym)))) {
				ValaDataType* __temp355;
				ValaDataType* __temp354;
				__temp355 = NULL;
				__temp354 = NULL;
				type = (__temp355 = (__temp354 = VALA_DATA_TYPE (type_sym), (__temp354 == NULL ? NULL : g_object_ref (__temp354))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp355);
			} else {
				char* __temp357;
				char* __temp356;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				__temp357 = NULL;
				__temp356 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp357 = g_strdup_printf ("`%s' is not a class, struct, or error domain", (__temp356 = vala_symbol_get_full_name (type_sym)))));
				(__temp357 = (g_free (__temp357), NULL));
				(__temp356 = (g_free (__temp356), NULL));
				(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
				(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
		}
		__temp358 = NULL;
		vala_object_creation_expression_set_type_reference (expr, (__temp358 = vala_type_reference_new ()));
		(__temp358 == NULL ? NULL : (__temp358 = (g_object_unref (__temp358), NULL)));
		vala_type_reference_set_data_type (vala_object_creation_expression_get_type_reference (expr), type);
		{
			GeeCollection* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = type_args;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaTypeReference* type_arg;
				type_arg = gee_iterator_get (type_arg_it);
				{
					vala_type_reference_add_type_argument (vala_object_creation_expression_get_type_reference (expr), type_arg);
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
		(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
		(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
		(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
	} else {
		ValaDataType* __temp360;
		ValaDataType* __temp359;
		__temp360 = NULL;
		__temp359 = NULL;
		type = (__temp360 = (__temp359 = vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)), (__temp359 == NULL ? NULL : g_object_ref (__temp359))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp360);
	}
	if (!vala_data_type_is_reference_type (type) && !(VALA_IS_ENUM (type))) {
		char* __temp362;
		char* __temp361;
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		__temp362 = NULL;
		__temp361 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp362 = g_strdup_printf ("Can't create instance of value type `%s'", (__temp361 = vala_type_reference_to_string (vala_object_creation_expression_get_type_reference (expr))))));
		(__temp362 = (g_free (__temp362), NULL));
		(__temp361 = (g_free (__temp361), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (type), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	__temp363 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp363 = vala_type_reference_copy (vala_object_creation_expression_get_type_reference (expr))));
	(__temp363 = (g_object_unref (__temp363), NULL));
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), TRUE);
	if (VALA_IS_CLASS (type)) {
		ValaClass* __temp364;
		ValaClass* cl;
		__temp364 = NULL;
		cl = (__temp364 = VALA_CLASS (type), (__temp364 == NULL ? NULL : g_object_ref (__temp364)));
		if (vala_class_get_is_abstract (cl)) {
			char* __temp366;
			char* __temp365;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL);
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			__temp366 = NULL;
			__temp365 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp366 = g_strdup_printf ("Can't create instance of abstract class `%s'", (__temp365 = vala_symbol_get_full_name (VALA_SYMBOL (cl))))));
			(__temp366 = (g_free (__temp366), NULL));
			(__temp365 = (g_free (__temp365), NULL));
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL) {
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_class_get_default_construction_method (cl)));
		}
		while (cl != NULL) {
			ValaClass* __temp368;
			ValaClass* __temp367;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				vala_type_reference_set_floating_reference (vala_expression_get_static_type (((ValaExpression*) expr)), TRUE);
				break;
			}
			__temp368 = NULL;
			__temp367 = NULL;
			cl = (__temp368 = (__temp367 = vala_class_get_base_class (cl), (__temp367 == NULL ? NULL : g_object_ref (__temp367))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp368);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (type)) {
			ValaStruct* __temp369;
			ValaStruct* st;
			__temp369 = NULL;
			st = (__temp369 = VALA_STRUCT (type), (__temp369 == NULL ? NULL : g_object_ref (__temp369)));
			if (vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL) {
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_struct_get_default_construction_method (st)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		}
	}
	__temp370 = NULL;
	if ((__temp371 = vala_expression_get_symbol_reference (((ValaExpression*) expr)) == NULL && gee_collection_get_size ((__temp370 = vala_object_creation_expression_get_argument_list (expr))) != 0, (__temp370 == NULL ? NULL : (__temp370 = (g_object_unref (__temp370), NULL))), __temp371)) {
		char* __temp373;
		char* __temp372;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL);
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		__temp373 = NULL;
		__temp372 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp373 = g_strdup_printf ("No arguments allowed when constructing type `%s'", (__temp372 = vala_symbol_get_full_name (VALA_SYMBOL (type))))));
		(__temp373 = (g_free (__temp373), NULL));
		(__temp372 = (g_free (__temp372), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	if (VALA_IS_METHOD (vala_expression_get_symbol_reference (((ValaExpression*) expr)))) {
		ValaMethod* __temp374;
		ValaMethod* m;
		GeeCollection* __temp376;
		GeeCollection* __temp375;
		GeeCollection* __temp377;
		__temp374 = NULL;
		m = (__temp374 = VALA_METHOD (vala_expression_get_symbol_reference (((ValaExpression*) expr))), (__temp374 == NULL ? NULL : g_object_ref (__temp374)));
		__temp376 = NULL;
		__temp375 = NULL;
		vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), VALA_SYMBOL (m), (__temp375 = vala_invokable_get_parameters (VALA_INVOKABLE (m))), (__temp376 = vala_object_creation_expression_get_argument_list (expr)));
		(__temp376 == NULL ? NULL : (__temp376 = (g_object_unref (__temp376), NULL)));
		(__temp375 == NULL ? NULL : (__temp375 = (g_object_unref (__temp375), NULL)));
		__temp377 = NULL;
		vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (gee_collection_get_size ((__temp377 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (((ValaExpression*) expr))));
		(__temp377 == NULL ? NULL : (__temp377 = (g_object_unref (__temp377), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		if (VALA_IS_ENUM (type)) {
			GeeCollection* __temp378;
			gboolean __temp379;
			__temp378 = NULL;
			if ((__temp379 = gee_collection_get_size ((__temp378 = vala_object_creation_expression_get_argument_list (expr))) == 0, (__temp378 == NULL ? NULL : (__temp378 = (g_object_unref (__temp378), NULL))), __temp379)) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Too few arguments, errors need at least 1 argument");
			} else {
				GeeCollection* __temp380;
				GeeIterator* __temp381;
				GeeIterator* arg_it;
				ValaExpression* ex;
				__temp380 = NULL;
				__temp381 = NULL;
				arg_it = (__temp381 = gee_iterable_iterator (GEE_ITERABLE ((__temp380 = vala_object_creation_expression_get_argument_list (expr)))), (__temp380 == NULL ? NULL : (__temp380 = (g_object_unref (__temp380), NULL))), __temp381);
				gee_iterator_next (arg_it);
				ex = gee_iterator_get (arg_it);
				if (vala_expression_get_static_type (ex) == NULL || !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (ex), self->priv->string_type)) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Invalid type for argument 1");
				}
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ex == NULL ? NULL : (ex = (g_object_unref (ex), NULL)));
			}
		}
	}
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_SIZEOF_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->ulong_type);
}


static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_TYPEOF_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->type_type);
}


static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	ValaStruct* __temp383;
	ValaStruct* st;
	gboolean __temp384;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	__temp383 = NULL;
	st = (__temp383 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp383 == NULL ? NULL : g_object_ref (__temp383)));
	return (__temp384 = vala_struct_is_integer_type (st) || vala_struct_is_floating_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp384);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	ValaStruct* __temp386;
	ValaStruct* st;
	gboolean __temp387;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	__temp386 = NULL;
	st = (__temp386 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp386 == NULL ? NULL : g_object_ref (__temp386)));
	return (__temp387 = vala_struct_is_integer_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp387);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_unary_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
		/* integer or floating point type*/
		if (!vala_semantic_analyzer_is_numeric_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
			char* __temp389;
			char* __temp388;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			__temp389 = NULL;
			__temp388 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp389 = g_strdup_printf ("Operator not supported for `%s'", (__temp388 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
			(__temp389 = (g_free (__temp389), NULL));
			(__temp388 = (g_free (__temp388), NULL));
			return;
		}
		vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
				char* __temp391;
				char* __temp390;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				__temp391 = NULL;
				__temp390 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp391 = g_strdup_printf ("Operator not supported for `%s'", (__temp390 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
				(__temp391 = (g_free (__temp391), NULL));
				(__temp390 = (g_free (__temp390), NULL));
				return;
			}
			vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
					char* __temp393;
					char* __temp392;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					__temp393 = NULL;
					__temp392 = NULL;
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp393 = g_strdup_printf ("Operator not supported for `%s'", (__temp392 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
					(__temp393 = (g_free (__temp393), NULL));
					(__temp392 = (g_free (__temp392), NULL));
					return;
				}
				vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
					ValaMemberAccess* ma;
					ValaMemberAccess* old_value;
					ValaLiteralExpression* __temp397;
					ValaIntegerLiteral* __temp396;
					ValaBinaryExpression* __temp398;
					ValaBinaryExpression* bin;
					ValaAssignment* assignment;
					/* integer type*/
					if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
						char* __temp395;
						char* __temp394;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						__temp395 = NULL;
						__temp394 = NULL;
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp395 = g_strdup_printf ("Operator not supported for `%s'", (__temp394 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
						(__temp395 = (g_free (__temp395), NULL));
						(__temp394 = (g_free (__temp394), NULL));
						return;
					}
					ma = vala_semantic_analyzer_find_member_access (self, vala_unary_expression_get_inner (expr));
					if (ma == NULL) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Prefix operators not supported for this expression");
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
					__temp397 = NULL;
					__temp396 = NULL;
					__temp398 = NULL;
					bin = (__temp398 = vala_binary_expression_new ((vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT ? VALA_BINARY_OPERATOR_PLUS : VALA_BINARY_OPERATOR_MINUS), VALA_EXPRESSION (old_value), VALA_EXPRESSION ((__temp397 = vala_literal_expression_new (VALA_LITERAL ((__temp396 = vala_integer_literal_new ("1", NULL))), NULL))), NULL), (__temp397 == NULL ? NULL : (__temp397 = (g_object_unref (__temp397), NULL))), (__temp396 == NULL ? NULL : (__temp396 = (g_object_unref (__temp396), NULL))), __temp398);
					assignment = vala_assignment_new (VALA_EXPRESSION (ma), VALA_EXPRESSION (bin), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL);
					vala_code_node_replace (vala_code_node_get_parent_node (((ValaCodeNode*) expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (assignment));
					vala_code_node_accept (VALA_CODE_NODE (assignment), VALA_CODE_VISITOR (self));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
					return;
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF) {
						/* value type*/
						vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
					} else {
						if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
							/* reference type*/
							vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
						} else {
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "internal error: unsupported unary operator");
							return;
						}
					}
				}
			}
		}
	}
}


static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* __temp399;
		ValaParenthesizedExpression* pe;
		ValaMemberAccess* __temp400;
		__temp399 = NULL;
		pe = (__temp399 = VALA_PARENTHESIZED_EXPRESSION (expr), (__temp399 == NULL ? NULL : g_object_ref (__temp399)));
		__temp400 = NULL;
		return (__temp400 = vala_semantic_analyzer_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL))), __temp400);
		(pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* __temp401;
		__temp401 = NULL;
		return (__temp401 = VALA_MEMBER_ACCESS (expr), (__temp401 == NULL ? NULL : g_object_ref (__temp401)));
	}
	return VALA_MEMBER_ACCESS (NULL);
}


static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CAST_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) == NULL && vala_type_reference_get_type_parameter (vala_cast_expression_get_type_reference (expr)) == NULL) {
		return;
	}
	/* if type resolving didn't succeed, skip this check 
	 FIXME: check whether cast is allowed*/
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_cast_expression_get_type_reference (expr));
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_cast_expression_get_inner (expr))));
}


static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr)
{
	ValaSemanticAnalyzer * self;
	ValaPointer* __temp404;
	ValaPointer* pointer;
	ValaTypeReference* __temp405;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_POINTER_INDIRECTION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_pointer_indirection_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (!(VALA_IS_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Pointer indirection not supported for this expression");
		return;
	}
	__temp404 = NULL;
	pointer = (__temp404 = VALA_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), (__temp404 == NULL ? NULL : g_object_ref (__temp404)));
	__temp405 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp405 = vala_type_reference_new ()));
	(__temp405 == NULL ? NULL : (__temp405 = (g_object_unref (__temp405), NULL)));
	vala_type_reference_set_data_type (vala_expression_get_static_type (((ValaExpression*) expr)), vala_pointer_get_referent_type (pointer));
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr))));
	(pointer == NULL ? NULL : (pointer = (g_object_unref (pointer), NULL)));
}


static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp406;
	ValaPointer* __temp407;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_ADDRESSOF_EXPRESSION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_addressof_expression_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Address-of operator not supported for this expression");
		return;
	}
	__temp406 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp406 = vala_type_reference_new ()));
	(__temp406 == NULL ? NULL : (__temp406 = (g_object_unref (__temp406), NULL)));
	__temp407 = NULL;
	vala_type_reference_set_data_type (vala_expression_get_static_type (((ValaExpression*) expr)), VALA_DATA_TYPE ((__temp407 = vala_data_type_get_pointer (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))))));
	(__temp407 = (g_object_unref (__temp407), NULL));
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))));
}


static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp408;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_reference_transfer_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (!(VALA_IS_MEMBER_ACCESS (vala_reference_transfer_expression_get_inner (expr)) || VALA_IS_ELEMENT_ACCESS (vala_reference_transfer_expression_get_inner (expr)))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Reference transfer not supported for this expression");
		return;
	}
	if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "No reference to be transferred");
		return;
	}
	__temp408 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp408 = vala_type_reference_copy (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))));
	(__temp408 = (g_object_unref (__temp408), NULL));
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), TRUE);
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (((ValaExpression*) expr)), FALSE);
}


static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type)
{
	ValaStruct* __temp410;
	ValaStruct* left;
	ValaStruct* __temp411;
	ValaStruct* right;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (left_type), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (right_type), NULL);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (left_type))) || !(VALA_IS_STRUCT (vala_type_reference_get_data_type (right_type)))) {
		return VALA_TYPE_REFERENCE (NULL);
	}
	/* at least one operand not struct*/
	__temp410 = NULL;
	left = (__temp410 = VALA_STRUCT (vala_type_reference_get_data_type (left_type)), (__temp410 == NULL ? NULL : g_object_ref (__temp410)));
	__temp411 = NULL;
	right = (__temp411 = VALA_STRUCT (vala_type_reference_get_data_type (right_type)), (__temp411 == NULL ? NULL : g_object_ref (__temp411)));
	if ((!vala_struct_is_floating_type (left) && !vala_struct_is_integer_type (left)) || (!vala_struct_is_floating_type (right) && !vala_struct_is_integer_type (right))) {
		gpointer __temp412;
		return VALA_TYPE_REFERENCE ((__temp412 = NULL, (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp412));
	}
	/* at least one operand not numeric*/
	if (vala_struct_is_floating_type (left) == vala_struct_is_floating_type (right)) {
		/* both operands integer or floating type*/
		if (vala_struct_get_rank (left) >= vala_struct_get_rank (right)) {
			ValaTypeReference* __temp413;
			__temp413 = NULL;
			return (__temp413 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp413);
		} else {
			ValaTypeReference* __temp414;
			__temp414 = NULL;
			return (__temp414 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp414);
		}
	} else {
		/* one integer and one floating type operand*/
		if (vala_struct_is_floating_type (left)) {
			ValaTypeReference* __temp415;
			__temp415 = NULL;
			return (__temp415 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp415);
		} else {
			ValaTypeReference* __temp416;
			__temp416 = NULL;
			return (__temp416 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp416);
		}
	}
	(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
	(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
}


static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	if (vala_code_node_get_error (((ValaCodeNode*) vala_binary_expression_get_left (expr))) || vala_code_node_get_error (((ValaCodeNode*) vala_binary_expression_get_right (expr)))) {
		/* if there were any errors in inner expressions, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		ValaMemberAccess* __temp417;
		ValaInvocationExpression* __temp418;
		ValaInvocationExpression* concat_call;
		if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->string_type)) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Operands must be strings");
			return;
		}
		/* string concatenation: convert to a.concat (b) */
		__temp417 = NULL;
		__temp418 = NULL;
		concat_call = (__temp418 = vala_invocation_expression_new (VALA_EXPRESSION ((__temp417 = vala_member_access_new (vala_binary_expression_get_left (expr), "concat", NULL))), NULL), (__temp417 == NULL ? NULL : (__temp417 = (g_object_unref (__temp417), NULL))), __temp418);
		vala_invocation_expression_add_argument (concat_call, vala_binary_expression_get_right (expr));
		vala_code_node_replace (vala_code_node_get_parent_node (((ValaCodeNode*) expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (concat_call));
		vala_code_node_accept (VALA_CODE_NODE (concat_call), VALA_CODE_VISITOR (self));
		(concat_call == NULL ? NULL : (concat_call = (g_object_unref (concat_call), NULL)));
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
			ValaTypeReference* __temp419;
			__temp419 = NULL;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp419 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))));
			(__temp419 == NULL ? NULL : (__temp419 = (g_object_unref (__temp419), NULL)));
			if (vala_expression_get_static_type (((ValaExpression*) expr)) == NULL) {
				char* __temp422;
				char* __temp421;
				char* __temp420;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				__temp422 = NULL;
				__temp421 = NULL;
				__temp420 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp422 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp420 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp421 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
				(__temp422 = (g_free (__temp422), NULL));
				(__temp421 = (g_free (__temp421), NULL));
				(__temp420 = (g_free (__temp420), NULL));
				return;
			}
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
				ValaTypeReference* __temp423;
				__temp423 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp423 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))));
				(__temp423 == NULL ? NULL : (__temp423 = (g_object_unref (__temp423), NULL)));
				if (vala_expression_get_static_type (((ValaExpression*) expr)) == NULL) {
					char* __temp426;
					char* __temp425;
					char* __temp424;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					__temp426 = NULL;
					__temp425 = NULL;
					__temp424 = NULL;
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp426 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp424 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp425 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
					(__temp426 = (g_free (__temp426), NULL));
					(__temp425 = (g_free (__temp425), NULL));
					(__temp424 = (g_free (__temp424), NULL));
					return;
				}
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
					if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
						ValaMemberAccess* __temp427;
						ValaInvocationExpression* __temp428;
						ValaInvocationExpression* cmp_call;
						ValaLiteralExpression* __temp430;
						ValaIntegerLiteral* __temp429;
						/* string comparison: convert to a.collate (b) OP 0 */
						__temp427 = NULL;
						__temp428 = NULL;
						cmp_call = (__temp428 = vala_invocation_expression_new (VALA_EXPRESSION ((__temp427 = vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL))), NULL), (__temp427 == NULL ? NULL : (__temp427 = (g_object_unref (__temp427), NULL))), __temp428);
						vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
						vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call));
						__temp430 = NULL;
						__temp429 = NULL;
						vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp430 = vala_literal_expression_new (VALA_LITERAL ((__temp429 = vala_integer_literal_new ("0", NULL))), NULL))));
						(__temp430 == NULL ? NULL : (__temp430 = (g_object_unref (__temp430), NULL)));
						(__temp429 == NULL ? NULL : (__temp429 = (g_object_unref (__temp429), NULL)));
						vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
						(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
					} else {
						ValaTypeReference* resulting_type;
						resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)));
						if (resulting_type == NULL) {
							char* __temp433;
							char* __temp432;
							char* __temp431;
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							__temp433 = NULL;
							__temp432 = NULL;
							__temp431 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp433 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", (__temp431 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp432 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
							(__temp433 = (g_free (__temp433), NULL));
							(__temp432 = (g_free (__temp432), NULL));
							(__temp431 = (g_free (__temp431), NULL));
							(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
							return;
						}
						(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
					}
					vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
						/* relational operation */
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr))) && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
							char* __temp436;
							char* __temp435;
							char* __temp434;
							__temp436 = NULL;
							__temp435 = NULL;
							__temp434 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), (__temp436 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible, comparison would always evaluate to false", (__temp434 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))), (__temp435 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))))));
							(__temp436 = (g_free (__temp436), NULL));
							(__temp435 = (g_free (__temp435), NULL));
							(__temp434 = (g_free (__temp434), NULL));
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							return;
						}
						if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
							ValaMemberAccess* __temp437;
							ValaInvocationExpression* __temp438;
							ValaInvocationExpression* cmp_call;
							ValaLiteralExpression* __temp440;
							ValaIntegerLiteral* __temp439;
							/* string comparison: convert to a.collate (b) OP 0 */
							__temp437 = NULL;
							__temp438 = NULL;
							cmp_call = (__temp438 = vala_invocation_expression_new (VALA_EXPRESSION ((__temp437 = vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL))), NULL), (__temp437 == NULL ? NULL : (__temp437 = (g_object_unref (__temp437), NULL))), __temp438);
							vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
							vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call));
							__temp440 = NULL;
							__temp439 = NULL;
							vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp440 = vala_literal_expression_new (VALA_LITERAL ((__temp439 = vala_integer_literal_new ("0", NULL))), NULL))));
							(__temp440 == NULL ? NULL : (__temp440 = (g_object_unref (__temp440), NULL)));
							(__temp439 == NULL ? NULL : (__temp439 = (g_object_unref (__temp439), NULL)));
							vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
							(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
						}
						vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
							/* integer type or flags type*/
							vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_binary_expression_get_left (expr)));
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
								if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) != vala_type_reference_get_data_type (self->priv->bool_type) || vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
									vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
									vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) expr)), "Operands must be boolean");
								}
								vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
							} else {
								g_assert_not_reached ();
							}
						}
					}
				}
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_TYPE_CHECK (expr));
	if (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)) == NULL) {
		/* if type resolving didn't succeed, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
}


static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GeeCollection* types)
{
	gboolean null_found;
	gboolean class_or_iface_found;
	gboolean type_param_found;
	gboolean ref_struct_found;
	gboolean val_struct_found;
	gboolean enum_found;
	gboolean callback_found;
	ValaTypeReference* base_type;
	ValaTypeReference* last_type;
	ValaTypeReference* __temp454;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (types == NULL || GEE_IS_COLLECTION (types), NULL);
	null_found = FALSE;
	class_or_iface_found = FALSE;
	type_param_found = FALSE;
	ref_struct_found = FALSE;
	val_struct_found = FALSE;
	enum_found = FALSE;
	callback_found = FALSE;
	base_type = NULL;
	last_type = NULL;
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = types;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				ValaTypeReference* __temp442;
				ValaTypeReference* __temp441;
				__temp442 = NULL;
				__temp441 = NULL;
				last_type = (__temp442 = (__temp441 = type, (__temp441 == NULL ? NULL : g_object_ref (__temp441))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp442);
				if (vala_code_node_get_error (((ValaCodeNode*) type))) {
					ValaTypeReference* __temp443;
					ValaTypeReference* __temp444;
					__temp443 = NULL;
					base_type = (__temp443 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp443);
					vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
					__temp444 = NULL;
					return (__temp444 = base_type, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp444);
				}
				if (vala_type_reference_get_data_type (type) == NULL && vala_type_reference_get_type_parameter (type) == NULL) {
					if (!null_found) {
						null_found = TRUE;
						if (val_struct_found || enum_found) {
							vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							break;
						}
					}
				} else {
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (type)) || VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
						if (!class_or_iface_found) {
							class_or_iface_found = TRUE;
							if (type_param_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
								vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								break;
							}
						}
					} else {
						if (vala_type_reference_get_type_parameter (type) != NULL) {
							if (!type_param_found) {
								type_param_found = TRUE;
								if (class_or_iface_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
									vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
									(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
									break;
								}
							}
						} else {
							if (VALA_IS_STRUCT (vala_type_reference_get_data_type (type))) {
								ValaStruct* __temp445;
								ValaStruct* st;
								__temp445 = NULL;
								st = (__temp445 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp445 == NULL ? NULL : g_object_ref (__temp445)));
								if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
									if (!ref_struct_found) {
										ref_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || val_struct_found || enum_found || callback_found) {
											vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
											(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
											(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
											break;
										}
									}
								} else {
									if (!val_struct_found) {
										val_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || enum_found || callback_found) {
											vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
											(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
											(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
											break;
										}
									}
								}
								(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
							} else {
								if (VALA_IS_ENUM (vala_type_reference_get_data_type (type))) {
									if (!enum_found) {
										enum_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found) {
											vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
											(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
											break;
										}
									}
								} else {
									if (VALA_IS_CALLBACK (vala_type_reference_get_data_type (type))) {
										if (!callback_found) {
											callback_found = TRUE;
											if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found || enum_found) {
												vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
												(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
												break;
											}
										}
									} else {
										ValaTypeReference* __temp446;
										char* __temp448;
										char* __temp447;
										ValaTypeReference* __temp449;
										__temp446 = NULL;
										base_type = (__temp446 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp446);
										vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
										__temp448 = NULL;
										__temp447 = NULL;
										vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) type)), (__temp448 = g_strdup_printf ("internal error: unsupported type `%s'", (__temp447 = vala_type_reference_to_string (type)))));
										(__temp448 = (g_free (__temp448), NULL));
										(__temp447 = (g_free (__temp447), NULL));
										__temp449 = NULL;
										return (__temp449 = base_type, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp449);
									}
								}
							}
						}
					}
				}
				if (base_type == NULL) {
					ValaTypeReference* __temp450;
					__temp450 = NULL;
					base_type = (__temp450 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp450);
					vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type));
					vala_type_reference_set_type_parameter (base_type, vala_type_reference_get_type_parameter (type));
					vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (type));
					vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (type));
					vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (type));
				} else {
					if (vala_type_reference_get_data_type (base_type) != vala_type_reference_get_data_type (type)) {
						if (vala_semantic_analyzer_is_type_compatible (self, type, base_type)) {
						} else {
							if (vala_semantic_analyzer_is_type_compatible (self, base_type, type)) {
								vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type));
							} else {
								vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								break;
							}
						}
					}
					vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (base_type) && vala_type_reference_get_non_null (type));
					vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (base_type) && vala_type_reference_get_is_null (type));
					/* if one subexpression transfers ownership, all subexpressions must transfer ownership
					 FIXME add ref calls to subexpressions that don't transfer ownership*/
					vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (base_type) || vala_type_reference_get_transfers_ownership (type));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	if (base_type != NULL && vala_code_node_get_error (((ValaCodeNode*) base_type))) {
		char* __temp453;
		char* __temp452;
		char* __temp451;
		__temp453 = NULL;
		__temp452 = NULL;
		__temp451 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) last_type)), (__temp453 = g_strdup_printf ("`%s' is incompatible with `%s'", (__temp451 = vala_type_reference_to_string (last_type)), (__temp452 = vala_type_reference_to_string (base_type)))));
		(__temp453 = (g_free (__temp453), NULL));
		(__temp452 = (g_free (__temp452), NULL));
		(__temp451 = (g_free (__temp451), NULL));
	}
	__temp454 = NULL;
	return (__temp454 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp454);
	(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
	(last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr)
{
	ValaSemanticAnalyzer * self;
	GeeList* types;
	ValaTypeReference* __temp455;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_CONDITIONAL_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_conditional_expression_get_condition (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_conditional_expression_get_condition (expr))), "Condition must be boolean");
		return;
	}
	/* FIXME: support memory management */
	types = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	gee_collection_add (GEE_COLLECTION (types), vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)));
	gee_collection_add (GEE_COLLECTION (types), vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)));
	__temp455 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp455 = vala_semantic_analyzer_compute_common_base_type (self, GEE_COLLECTION (types))));
	(__temp455 == NULL ? NULL : (__temp455 = (g_object_unref (__temp455), NULL)));
	(types == NULL ? NULL : (types = (g_object_unref (types), NULL)));
}


static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self)
{
	char* result;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	result = g_strdup_printf ("__lambda%d", self->priv->next_lambda_id);
	self->priv->next_lambda_id++;
	return result;
	(result = (g_free (result), NULL));
}


static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self)
{
	ValaSymbol* __temp457;
	ValaSymbol* sym;
	gpointer __temp462;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	__temp457 = NULL;
	sym = (__temp457 = self->priv->current_symbol, (__temp457 == NULL ? NULL : g_object_ref (__temp457)));
	while (sym != NULL) {
		ValaSymbol* __temp461;
		ValaSymbol* __temp460;
		if (VALA_IS_METHOD (sym)) {
			ValaMethod* __temp458;
			ValaMethod* __temp459;
			__temp458 = NULL;
			__temp459 = NULL;
			return (__temp459 = (__temp458 = VALA_METHOD (sym), (__temp458 == NULL ? NULL : g_object_ref (__temp458))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp459);
		}
		__temp461 = NULL;
		__temp460 = NULL;
		sym = (__temp461 = (__temp460 = vala_symbol_get_parent_symbol (sym), (__temp460 == NULL ? NULL : g_object_ref (__temp460))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp461);
	}
	return VALA_METHOD ((__temp462 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp462));
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self)
{
	ValaSymbol* __temp463;
	ValaSymbol* sym;
	gboolean __temp467;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	__temp463 = NULL;
	sym = (__temp463 = self->priv->current_symbol, (__temp463 == NULL ? NULL : g_object_ref (__temp463)));
	while (sym != NULL) {
		ValaSymbol* __temp466;
		ValaSymbol* __temp465;
		if (VALA_IS_CONSTRUCTOR (sym)) {
			gboolean __temp464;
			return (__temp464 = TRUE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp464);
		}
		__temp466 = NULL;
		__temp465 = NULL;
		sym = (__temp466 = (__temp465 = vala_symbol_get_parent_symbol (sym), (__temp465 == NULL ? NULL : g_object_ref (__temp465))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp466);
	}
	return (__temp467 = FALSE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp467);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l)
{
	ValaSemanticAnalyzer * self;
	gboolean in_instance_method;
	ValaMethod* current_method;
	ValaCallback* __temp468;
	ValaCallback* cb;
	ValaMethod* __temp470;
	char* __temp469;
	GeeCollection* lambda_params;
	GeeIterator* lambda_param_it;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (l));
	if (vala_expression_get_expected_type (((ValaExpression*) l)) == NULL || !(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) l)))))) {
		vala_code_node_set_error (VALA_CODE_NODE (l), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) l)), "lambda expression not allowed in this context");
		return;
	}
	in_instance_method = FALSE;
	current_method = vala_semantic_analyzer_find_current_method (self);
	if (current_method != NULL) {
		in_instance_method = vala_method_get_instance (current_method);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (self);
	}
	__temp468 = NULL;
	cb = (__temp468 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (((ValaExpression*) l)))), (__temp468 == NULL ? NULL : g_object_ref (__temp468)));
	__temp470 = NULL;
	__temp469 = NULL;
	vala_lambda_expression_set_method (l, (__temp470 = vala_method_new ((__temp469 = vala_semantic_analyzer_get_lambda_name (self)), vala_callback_get_return_type (cb), NULL)));
	(__temp470 == NULL ? NULL : (__temp470 = (g_object_unref (__temp470), NULL)));
	(__temp469 = (g_free (__temp469), NULL));
	vala_method_set_instance (vala_lambda_expression_get_method (l), vala_callback_get_instance (cb) && in_instance_method);
	vala_symbol_set_owner (VALA_SYMBOL (vala_lambda_expression_get_method (l)), vala_symbol_get_scope (self->priv->current_symbol));
	lambda_params = vala_lambda_expression_get_parameters (l);
	lambda_param_it = gee_iterable_iterator (GEE_ITERABLE (lambda_params));
	{
		GeeCollection* cb_param_collection;
		GeeIterator* cb_param_it;
		cb_param_collection = vala_callback_get_parameters (cb);
		cb_param_it = gee_iterable_iterator (GEE_ITERABLE (cb_param_collection));
		while (gee_iterator_next (cb_param_it)) {
			ValaFormalParameter* cb_param;
			cb_param = gee_iterator_get (cb_param_it);
			{
				char* lambda_param;
				ValaFormalParameter* param;
				if (!gee_iterator_next (lambda_param_it)) {
					(cb_param == NULL ? NULL : (cb_param = (g_object_unref (cb_param), NULL)));
					break;
				}
				/* lambda expressions are allowed to have less parameters */
				lambda_param = gee_iterator_get (lambda_param_it);
				param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_type_reference (cb_param), NULL);
				vala_method_add_parameter (vala_lambda_expression_get_method (l), param);
				(cb_param == NULL ? NULL : (cb_param = (g_object_unref (cb_param), NULL)));
				(lambda_param = (g_free (lambda_param), NULL));
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(cb_param_collection == NULL ? NULL : (cb_param_collection = (g_object_unref (cb_param_collection), NULL)));
		(cb_param_it == NULL ? NULL : (cb_param_it = (g_object_unref (cb_param_it), NULL)));
	}
	if (gee_iterator_next (lambda_param_it)) {
		/* lambda expressions may not expect more parameters */
		vala_code_node_set_error (VALA_CODE_NODE (l), TRUE);
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) l)), "lambda expression: too many parameters");
		(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		(lambda_params == NULL ? NULL : (lambda_params = (g_object_unref (lambda_params), NULL)));
		(lambda_param_it == NULL ? NULL : (lambda_param_it = (g_object_unref (lambda_param_it), NULL)));
		return;
	}
	if (vala_lambda_expression_get_expression_body (l) != NULL) {
		ValaBlock* block;
		block = vala_block_new (NULL);
		vala_scope_set_parent_scope (vala_symbol_get_scope (((ValaSymbol*) block)), vala_symbol_get_scope (((ValaSymbol*) vala_lambda_expression_get_method (l))));
		if (vala_type_reference_get_data_type (vala_method_get_return_type (vala_lambda_expression_get_method (l))) != NULL) {
			ValaReturnStatement* __temp471;
			__temp471 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp471 = vala_return_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp471 == NULL ? NULL : (__temp471 = (g_object_unref (__temp471), NULL)));
		} else {
			ValaExpressionStatement* __temp472;
			__temp472 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp472 = vala_expression_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp472 == NULL ? NULL : (__temp472 = (g_object_unref (__temp472), NULL)));
		}
		vala_method_set_body (vala_lambda_expression_get_method (l), block);
		(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
	} else {
		vala_method_set_body (vala_lambda_expression_get_method (l), vala_lambda_expression_get_statement_body (l));
	}
	vala_symbol_set_owner (VALA_SYMBOL (vala_method_get_body (vala_lambda_expression_get_method (l))), vala_symbol_get_scope (((ValaSymbol*) vala_lambda_expression_get_method (l))));
	/* lambda expressions should be usable like MemberAccess of a method */
	vala_expression_set_symbol_reference (VALA_EXPRESSION (l), VALA_SYMBOL (vala_lambda_expression_get_method (l)));
	(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	(lambda_params == NULL ? NULL : (lambda_params = (g_object_unref (lambda_params), NULL)));
	(lambda_param_it == NULL ? NULL : (lambda_param_it = (g_object_unref (lambda_param_it), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp473;
		ValaMemberAccess* ma;
		__temp473 = NULL;
		ma = (__temp473 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp473 == NULL ? NULL : g_object_ref (__temp473)));
		if (vala_code_node_get_error (((ValaCodeNode*) ma)) || vala_expression_get_symbol_reference (((ValaExpression*) ma)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		/* if no symbol found, skip this check */
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (((ValaExpression*) ma)))) {
			ValaSignal* __temp474;
			ValaSignal* sig;
			ValaTypeReference* __temp475;
			ValaCallback* __temp476;
			__temp474 = NULL;
			sig = (__temp474 = VALA_SIGNAL (vala_expression_get_symbol_reference (((ValaExpression*) ma))), (__temp474 == NULL ? NULL : g_object_ref (__temp474)));
			__temp475 = NULL;
			vala_expression_set_expected_type (vala_assignment_get_right (a), (__temp475 = vala_type_reference_new ()));
			(__temp475 == NULL ? NULL : (__temp475 = (g_object_unref (__temp475), NULL)));
			__temp476 = NULL;
			vala_type_reference_set_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a)), VALA_DATA_TYPE ((__temp476 = vala_signal_get_callback (sig))));
			(__temp476 = (g_object_unref (__temp476), NULL));
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
		} else {
			/* do nothing*/
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a))) {
			} else {
				/* do nothing*/
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), "unsupported lvalue in assignment");
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self;
	self = ((ValaSemanticAnalyzer*) base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (vala_code_node_get_error (((ValaCodeNode*) a)) || vala_code_node_get_error (((ValaCodeNode*) vala_assignment_get_left (a))) || vala_code_node_get_error (((ValaCodeNode*) vala_assignment_get_right (a)))) {
		vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
		return;
	}
	if (vala_assignment_get_operator (a) != VALA_ASSIGNMENT_OPERATOR_SIMPLE && VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp477;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		__temp477 = NULL;
		ma = (__temp477 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp477 == NULL ? NULL : g_object_ref (__temp477)));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference (((ValaExpression*) ma))))) {
			ValaMemberAccess* old_value;
			ValaParenthesizedExpression* __temp478;
			ValaBinaryExpression* __temp479;
			ValaBinaryExpression* bin;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			__temp478 = NULL;
			__temp479 = NULL;
			bin = (__temp479 = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, VALA_EXPRESSION (old_value), VALA_EXPRESSION ((__temp478 = vala_parenthesized_expression_new (vala_assignment_get_right (a), vala_code_node_get_source_reference (((ValaCodeNode*) vala_assignment_get_right (a)))))), NULL), (__temp478 == NULL ? NULL : (__temp478 = (g_object_unref (__temp478), NULL))), __temp479);
			if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (a, VALA_EXPRESSION (bin));
			vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
			vala_assignment_set_operator (a, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
			(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp480;
		ValaMemberAccess* ma;
		__temp480 = NULL;
		ma = (__temp480 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp480 == NULL ? NULL : g_object_ref (__temp480)));
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (((ValaExpression*) ma)))) {
			ValaSignal* __temp481;
			ValaSignal* sig;
			ValaMethod* __temp482;
			ValaMethod* m;
			__temp481 = NULL;
			sig = (__temp481 = VALA_SIGNAL (vala_expression_get_symbol_reference (((ValaExpression*) ma))), (__temp481 == NULL ? NULL : g_object_ref (__temp481)));
			if (vala_expression_get_symbol_reference (vala_assignment_get_right (a)) == NULL) {
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_assignment_get_right (a))), "unsupported expression for signal handler");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			__temp482 = NULL;
			m = (__temp482 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (a))), (__temp482 == NULL ? NULL : g_object_ref (__temp482)));
			if (vala_method_get_instance (m) && m->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
				/* TODO: generate wrapper function */
				vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE);
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) vala_assignment_get_right (a))), "public instance methods not yet supported as signal handlers");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			if (vala_method_get_instance (m)) {
				/* instance signal handlers must have the self
				 * parameter at the end
				 * do not use G_CONNECT_SWAPPED as this would
				 * rearrange the parameters for instance
				 * methods and non-instance methods
				 */
				vala_method_set_instance_last (m, TRUE);
			}
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference (((ValaExpression*) ma)))) {
				ValaProperty* __temp483;
				ValaProperty* prop;
				__temp483 = NULL;
				prop = (__temp483 = VALA_PROPERTY (vala_expression_get_symbol_reference (((ValaExpression*) ma))), (__temp483 == NULL ? NULL : g_object_ref (__temp483)));
				if (vala_property_get_set_accessor (prop) == NULL) {
					char* __temp485;
					char* __temp484;
					vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE);
					__temp485 = NULL;
					__temp484 = NULL;
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) ma)), (__temp485 = g_strdup_printf ("Property `%s' is read-only", (__temp484 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
					(__temp485 = (g_free (__temp485), NULL));
					(__temp484 = (g_free (__temp484), NULL));
					(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					return;
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (((ValaExpression*) ma))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
					ValaVariableDeclarator* __temp486;
					ValaVariableDeclarator* decl;
					ValaMemberAccess* __temp487;
					ValaMemberAccess* right_ma;
					__temp486 = NULL;
					decl = (__temp486 = VALA_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (((ValaExpression*) ma))), (__temp486 == NULL ? NULL : g_object_ref (__temp486)));
					__temp487 = NULL;
					right_ma = (__temp487 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp487 == NULL ? NULL : g_object_ref (__temp487)));
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (((ValaExpression*) right_ma))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
						ValaMethod* __temp488;
						ValaMethod* m;
						ValaCallback* __temp489;
						ValaCallback* cb;
						__temp488 = NULL;
						m = (__temp488 = VALA_METHOD (vala_expression_get_symbol_reference (((ValaExpression*) right_ma))), (__temp488 == NULL ? NULL : g_object_ref (__temp488)));
						__temp489 = NULL;
						cb = (__temp489 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp489 == NULL ? NULL : g_object_ref (__temp489)));
						/* check whether method matches callback type */
						if (!vala_callback_matches_method (cb, m)) {
							char* __temp492;
							char* __temp491;
							char* __temp490;
							vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
							__temp492 = NULL;
							__temp491 = NULL;
							__temp490 = NULL;
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), (__temp492 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp490 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (__temp491 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
							(__temp492 = (g_free (__temp492), NULL));
							(__temp491 = (g_free (__temp491), NULL));
							(__temp490 = (g_free (__temp490), NULL));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						vala_expression_set_static_type (vala_assignment_get_right (a), vala_variable_declarator_get_type_reference (decl));
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					} else {
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), "Assignment: Invalid callback assignment attempt");
						(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
					(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
				} else {
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference (((ValaExpression*) ma))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
						ValaField* __temp493;
						ValaField* f;
						ValaMemberAccess* __temp494;
						ValaMemberAccess* right_ma;
						__temp493 = NULL;
						f = (__temp493 = VALA_FIELD (vala_expression_get_symbol_reference (((ValaExpression*) ma))), (__temp493 == NULL ? NULL : g_object_ref (__temp493)));
						__temp494 = NULL;
						right_ma = (__temp494 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp494 == NULL ? NULL : g_object_ref (__temp494)));
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (((ValaExpression*) right_ma))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) {
							ValaMethod* __temp495;
							ValaMethod* m;
							ValaCallback* __temp496;
							ValaCallback* cb;
							__temp495 = NULL;
							m = (__temp495 = VALA_METHOD (vala_expression_get_symbol_reference (((ValaExpression*) right_ma))), (__temp495 == NULL ? NULL : g_object_ref (__temp495)));
							__temp496 = NULL;
							cb = (__temp496 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (__temp496 == NULL ? NULL : g_object_ref (__temp496)));
							/* check whether method matches callback type */
							if (!vala_callback_matches_method (cb, m)) {
								char* __temp499;
								char* __temp498;
								char* __temp497;
								vala_code_node_set_error (VALA_CODE_NODE (f), TRUE);
								__temp499 = NULL;
								__temp498 = NULL;
								__temp497 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), (__temp499 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp497 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (__temp498 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
								(__temp499 = (g_free (__temp499), NULL));
								(__temp498 = (g_free (__temp498), NULL));
								(__temp497 = (g_free (__temp497), NULL));
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
								(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
								(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							vala_expression_set_static_type (vala_assignment_get_right (a), vala_field_get_type_reference (f));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						} else {
							vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
							vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), "Assignment: Invalid callback assignment attempt");
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
					} else {
						if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL && vala_expression_get_static_type (vala_assignment_get_right (a)) != NULL) {
							if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
								char* __temp502;
								char* __temp501;
								char* __temp500;
								/* if there was an error on either side,
								 * i.e. a.{left|right}.static_type == null, skip type check */
								vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
								__temp502 = NULL;
								__temp501 = NULL;
								__temp500 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), (__temp502 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp500 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp501 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
								(__temp502 = (g_free (__temp502), NULL));
								(__temp501 = (g_free (__temp501), NULL));
								(__temp500 = (g_free (__temp500), NULL));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							if (vala_semantic_analyzer_get_memory_management (self)) {
								if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
									/* rhs transfers ownership of the expression */
									if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
										/* lhs doesn't own the value */
										vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
										vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), "Invalid assignment from owned expression to unowned variable");
									}
								} else {
									if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
									}
								}
							}
						}
					}
				}
			}
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		/* lhs wants to own the value
		 * rhs doesn't transfer the ownership
		 * code generator needs to add reference
		 * increment calls */
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
			ValaElementAccess* __temp503;
			ValaElementAccess* ea;
			__temp503 = NULL;
			ea = (__temp503 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (a)), (__temp503 == NULL ? NULL : g_object_ref (__temp503)));
			if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				char* __temp506;
				char* __temp505;
				char* __temp504;
				/* if there was an error on either side,
				 * i.e. a.{left|right}.static_type == null, skip type check */
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				__temp506 = NULL;
				__temp505 = NULL;
				__temp504 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), (__temp506 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp504 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp505 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
				(__temp506 = (g_free (__temp506), NULL));
				(__temp505 = (g_free (__temp505), NULL));
				(__temp504 = (g_free (__temp504), NULL));
				(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
				return;
			}
			if (vala_semantic_analyzer_get_memory_management (self)) {
				if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
					GeeList* args;
					ValaTypeReference* element_type;
					/* rhs transfers ownership of the expression */
					args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (ea)));
					if (gee_collection_get_size (((GeeCollection*) args)) != 1) {
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) ea)), "internal error: array reference without type arguments");
						(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					element_type = gee_list_get (args, 0);
					if (!vala_type_reference_get_takes_ownership (element_type)) {
						/* lhs doesn't own the value */
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) a)), "Invalid assignment from owned expression to unowned variable");
						(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
						(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
					(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
				} else {
					if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
					}
				}
			}
			(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
		} else {
			return;
		}
	}
	/* lhs wants to own the value
	 * rhs doesn't transfer the ownership
	 * code generator needs to add reference
	 * increment calls */
	if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL) {
		ValaTypeReference* __temp507;
		__temp507 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (a), (__temp507 = vala_type_reference_copy (vala_expression_get_static_type (vala_assignment_get_left (a)))));
		(__temp507 = (g_object_unref (__temp507), NULL));
		if (VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (((ValaCodeNode*) a)))) {
			/* Gee.List.get () transfers ownership but void function Gee.List.set () doesn't*/
			vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (((ValaExpression*) a)), FALSE);
		}
	} else {
		vala_expression_set_static_type (VALA_EXPRESSION (a), NULL);
	}
}


gboolean vala_semantic_analyzer_get_memory_management (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	return self->priv->_memory_management;
}


void vala_semantic_analyzer_set_memory_management (ValaSemanticAnalyzer* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	self->priv->_memory_management = value;
}


static void vala_semantic_analyzer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_semantic_analyzer_get_memory_management (self));
		break;
	}
}


static void vala_semantic_analyzer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		vala_semantic_analyzer_set_memory_management (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_semantic_analyzer_class_init (ValaSemanticAnalyzerClass * klass)
{
	vala_semantic_analyzer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSemanticAnalyzerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_semantic_analyzer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_semantic_analyzer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_semantic_analyzer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_semantic_analyzer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_semantic_analyzer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_semantic_analyzer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_semantic_analyzer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_semantic_analyzer_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_semantic_analyzer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_semantic_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_semantic_analyzer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_semantic_analyzer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_semantic_analyzer_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_semantic_analyzer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_semantic_analyzer_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_semantic_analyzer_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_semantic_analyzer_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_semantic_analyzer_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_semantic_analyzer_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_block = vala_semantic_analyzer_real_visit_begin_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_block = vala_semantic_analyzer_real_visit_end_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_semantic_analyzer_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_initializer_list = vala_semantic_analyzer_real_visit_begin_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_initializer_list = vala_semantic_analyzer_real_visit_end_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_semantic_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_semantic_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_semantic_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_semantic_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_foreach_statement = vala_semantic_analyzer_real_visit_begin_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_foreach_statement = vala_semantic_analyzer_real_visit_end_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_semantic_analyzer_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_semantic_analyzer_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_semantic_analyzer_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_semantic_analyzer_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_semantic_analyzer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_array_creation_expression = vala_semantic_analyzer_real_visit_begin_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_array_creation_expression = vala_semantic_analyzer_real_visit_end_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_semantic_analyzer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_semantic_analyzer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_semantic_analyzer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_semantic_analyzer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_semantic_analyzer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_semantic_analyzer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_literal_expression = vala_semantic_analyzer_real_visit_literal_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_semantic_analyzer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_semantic_analyzer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_invocation_expression = vala_semantic_analyzer_real_visit_begin_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_semantic_analyzer_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_semantic_analyzer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_semantic_analyzer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_semantic_analyzer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_object_creation_expression = vala_semantic_analyzer_real_visit_end_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_semantic_analyzer_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_semantic_analyzer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_semantic_analyzer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_semantic_analyzer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_semantic_analyzer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_semantic_analyzer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_semantic_analyzer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_semantic_analyzer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_semantic_analyzer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_semantic_analyzer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_lambda_expression = vala_semantic_analyzer_real_visit_begin_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_assignment = vala_semantic_analyzer_real_visit_begin_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_semantic_analyzer_real_visit_end_assignment;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_semantic_analyzer_init (ValaSemanticAnalyzer * self)
{
	self->priv = VALA_SEMANTIC_ANALYZER_GET_PRIVATE (self);
	self->priv->next_lambda_id = 0;
}


static void vala_semantic_analyzer_dispose (GObject * obj)
{
	ValaSemanticAnalyzer * self;
	ValaSemanticAnalyzerClass * klass;
	GObjectClass * parent_class;
	self = VALA_SEMANTIC_ANALYZER (obj);
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL)));
	(self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL)));
	(self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL)));
	(self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL)));
	(self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL)));
	(self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL)));
	(self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL)));
	(self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL)));
	(self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL)));
	(self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->object_type == NULL ? NULL : (self->priv->object_type = (g_object_unref (self->priv->object_type), NULL)));
	(self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL)));
	(self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL)));
	(self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL)));
	(self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL)));
	(self->priv->iterable_type == NULL ? NULL : (self->priv->iterable_type = (g_object_unref (self->priv->iterable_type), NULL)));
	(self->priv->iterator_type == NULL ? NULL : (self->priv->iterator_type = (g_object_unref (self->priv->iterator_type), NULL)));
	(self->priv->list_type == NULL ? NULL : (self->priv->list_type = (g_object_unref (self->priv->list_type), NULL)));
	(self->priv->map_type == NULL ? NULL : (self->priv->map_type = (g_object_unref (self->priv->map_type), NULL)));
	klass = VALA_SEMANTIC_ANALYZER_CLASS (g_type_class_peek (VALA_TYPE_SEMANTIC_ANALYZER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_semantic_analyzer_get_type (void)
{
	static GType vala_semantic_analyzer_type_id = 0;
	if (G_UNLIKELY (vala_semantic_analyzer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSemanticAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_semantic_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSemanticAnalyzer), 0, (GInstanceInitFunc) vala_semantic_analyzer_init };
		vala_semantic_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSemanticAnalyzer", &g_define_type_info, 0);
	}
	return vala_semantic_analyzer_type_id;
}




