/* valalambdaexpression.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valalambdaexpression.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valablock.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaLambdaExpressionPrivate {
	ValaExpression* _expression_body;
	ValaBlock* _statement_body;
	ValaMethod* _method;
	GeeList* parameters;
};
#define VALA_LAMBDA_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_LAMBDA_EXPRESSION, ValaLambdaExpressionPrivate))
enum  {
	VALA_LAMBDA_EXPRESSION_DUMMY_PROPERTY,
	VALA_LAMBDA_EXPRESSION_EXPRESSION_BODY,
	VALA_LAMBDA_EXPRESSION_STATEMENT_BODY,
	VALA_LAMBDA_EXPRESSION_METHOD
};
static void vala_lambda_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_lambda_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_lambda_expression_parent_class = NULL;
static void vala_lambda_expression_dispose (GObject * obj);


/**
 * Creates a new lambda expression.
 *
 * @param expression_body  expression body
 * @param source_reference reference to source code
 * @return                 newly created lambda expression
 */
ValaLambdaExpression* vala_lambda_expression_new (ValaExpression* expression_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression * self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (expression_body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_LAMBDA_EXPRESSION, 0, NULL);
	vala_lambda_expression_set_expression_body (self, expression_body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Creates a new lambda expression with statement body.
 *
 * @param statement_body   statement body
 * @param source_reference reference to source code
 * @return                 newly created lambda expression
 */
ValaLambdaExpression* vala_lambda_expression_new_with_statement_body (ValaBlock* statement_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression * self;
	g_return_val_if_fail (VALA_IS_BLOCK (statement_body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_LAMBDA_EXPRESSION, 0, NULL);
	vala_lambda_expression_set_statement_body (self, statement_body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends implicitly typed parameter.
 *
 * @param param parameter name
 */
void vala_lambda_expression_add_parameter (ValaLambdaExpression* self, const char* param) {
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (self));
	g_return_if_fail (param != NULL);
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
}


/**
 * Returns copy of parameter list.
 *
 * @return parameter list
 */
GeeCollection* vala_lambda_expression_get_parameters (ValaLambdaExpression* self) {
	g_return_val_if_fail (VALA_IS_LAMBDA_EXPRESSION (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, GEE_COLLECTION (self->priv->parameters)));
}


static void vala_lambda_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_lambda_expression (visitor, self);
}


static void vala_lambda_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_lambda_expression_get_method (self) == NULL) {
		if (vala_lambda_expression_get_expression_body (self) != NULL) {
			vala_code_node_accept (VALA_CODE_NODE (vala_lambda_expression_get_expression_body (self)), visitor);
			vala_code_visitor_visit_end_full_expression (visitor, vala_lambda_expression_get_expression_body (self));
		} else {
			if (vala_lambda_expression_get_statement_body (self) != NULL) {
				vala_code_node_accept (VALA_CODE_NODE (vala_lambda_expression_get_statement_body (self)), visitor);
			}
		}
	} else {
		vala_code_node_accept (VALA_CODE_NODE (vala_lambda_expression_get_method (self)), visitor);
	}
}


ValaExpression* vala_lambda_expression_get_expression_body (ValaLambdaExpression* self) {
	g_return_val_if_fail (VALA_IS_LAMBDA_EXPRESSION (self), NULL);
	return self->priv->_expression_body;
}


void vala_lambda_expression_set_expression_body (ValaLambdaExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_expression_body == NULL ? NULL : (self->priv->_expression_body = (g_object_unref (self->priv->_expression_body), NULL))), _tmp2);
}


ValaBlock* vala_lambda_expression_get_statement_body (ValaLambdaExpression* self) {
	g_return_val_if_fail (VALA_IS_LAMBDA_EXPRESSION (self), NULL);
	return self->priv->_statement_body;
}


void vala_lambda_expression_set_statement_body (ValaLambdaExpression* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_statement_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_statement_body == NULL ? NULL : (self->priv->_statement_body = (g_object_unref (self->priv->_statement_body), NULL))), _tmp2);
}


ValaMethod* vala_lambda_expression_get_method (ValaLambdaExpression* self) {
	g_return_val_if_fail (VALA_IS_LAMBDA_EXPRESSION (self), NULL);
	return self->priv->_method;
}


void vala_lambda_expression_set_method (ValaLambdaExpression* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_method == NULL ? NULL : (self->priv->_method = (g_object_unref (self->priv->_method), NULL))), _tmp2);
}


static void vala_lambda_expression_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (object);
	switch (property_id) {
		case VALA_LAMBDA_EXPRESSION_EXPRESSION_BODY:
		g_value_set_object (value, vala_lambda_expression_get_expression_body (self));
		break;
		case VALA_LAMBDA_EXPRESSION_STATEMENT_BODY:
		g_value_set_object (value, vala_lambda_expression_get_statement_body (self));
		break;
		case VALA_LAMBDA_EXPRESSION_METHOD:
		g_value_set_object (value, vala_lambda_expression_get_method (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_lambda_expression_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (object);
	switch (property_id) {
		case VALA_LAMBDA_EXPRESSION_EXPRESSION_BODY:
		vala_lambda_expression_set_expression_body (self, g_value_get_object (value));
		break;
		case VALA_LAMBDA_EXPRESSION_STATEMENT_BODY:
		vala_lambda_expression_set_statement_body (self, g_value_get_object (value));
		break;
		case VALA_LAMBDA_EXPRESSION_METHOD:
		vala_lambda_expression_set_method (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_lambda_expression_class_init (ValaLambdaExpressionClass * klass) {
	vala_lambda_expression_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaLambdaExpressionPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_lambda_expression_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_lambda_expression_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_lambda_expression_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_lambda_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_lambda_expression_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_LAMBDA_EXPRESSION_EXPRESSION_BODY, g_param_spec_object ("expression-body", "expression-body", "expression-body", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_LAMBDA_EXPRESSION_STATEMENT_BODY, g_param_spec_object ("statement-body", "statement-body", "statement-body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_LAMBDA_EXPRESSION_METHOD, g_param_spec_object ("method", "method", "method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_lambda_expression_init (ValaLambdaExpression * self) {
	self->priv = VALA_LAMBDA_EXPRESSION_GET_PRIVATE (self);
	self->priv->parameters = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
}


static void vala_lambda_expression_dispose (GObject * obj) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (obj);
	(self->priv->_expression_body == NULL ? NULL : (self->priv->_expression_body = (g_object_unref (self->priv->_expression_body), NULL)));
	(self->priv->_statement_body == NULL ? NULL : (self->priv->_statement_body = (g_object_unref (self->priv->_statement_body), NULL)));
	(self->priv->_method == NULL ? NULL : (self->priv->_method = (g_object_unref (self->priv->_method), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	G_OBJECT_CLASS (vala_lambda_expression_parent_class)->dispose (obj);
}


GType vala_lambda_expression_get_type (void) {
	static GType vala_lambda_expression_type_id = 0;
	if (G_UNLIKELY (vala_lambda_expression_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaLambdaExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_lambda_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaLambdaExpression), 0, (GInstanceInitFunc) vala_lambda_expression_init };
		vala_lambda_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaLambdaExpression", &g_define_type_info, 0);
	}
	return vala_lambda_expression_type_id;
}




